/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/* --------------------------------------------------------------------
                                                                        
   peaklck - test on peak-locking effects in PIV analyses.              
                                                                        
   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  


---------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <gpiv.h>

/* #define PARFILE  "peaklck.par" */	/* Parameter file name */
#define PARFILE  "gpivrc"	/* Parameter file name */

#define HELP "Optains histograms from PIV displacements: \n\
Peaklck tests PIV data on the so-called peak-locking effect  \n\
by printing/displaying statistics of the particle displacements at \n\
sub-pixel level. Uhisto and vhisto output histograms of horizontal \n\
and vertical displacements"

#define USAGE "\
Usage: peaklck | uhisto  | vhisto \n\
               [-n N] [-f filename] [-g] [-h | --help] [-n] [-p | --print] \n\
               [-v | --version] < stdin > stdout \n\
keys: \n\
-n N:                  override number of bins \n\
-f filename:           files (without .piv extension) instaed of stdin and \n\
                       stdout \n\
-g:                    graphical visualization of with gnuplot \n\
-h | --help:           this on-line help \n\
-p | --print:          print parameters and other info to stdout \n\
-v | --version:        shows version number and exits successfully \n\
"

#define RCSID "$Id: peaklck.c,v 2.8 2006/01/31 14:18:04 gerber Exp $"


/*
 * General global parameters
 */
gboolean print_par = FALSE;
gboolean fname_logic = FALSE;

#define GNUPLOT_DISPLAY_COLOR "DarkRed"
#define GNUPLOT_DISPLAY_SIZE 250

int gnuplot = 0, gnuplot_logic = FALSE;

/*
 * Global variable declarations
 */
GpivPivData in_data;
GpivBinData klass;

/*
 * Function prototypes
 */

void command_args(int argc, char *argv[], 
                  char fname[GPIV_MAX_CHARS],
		  GpivValidPar * piv_valid_par
                  )
/*-----------------------------------------------------------------------------
 * Command line argument handling
 */
{
    char c;
    int argc_next;


    while (--argc > 0 && (*++argv)[0] == '-') {
	argc_next = 0;
	while ((argc_next == 0) && (c = *++argv[0])) {
	    switch (c) {
	    case 'n':           /* default: nbins=10 */
		klass.nbins = atoi(*++argv);
		klass.nbins_logic = TRUE;
		argc_next = 1;
		--argc;
		break;

	    case 'f':
		strcpy(fname, *++argv);
		fname_logic = TRUE;
		argc_next = 1;
		--argc;
		break;

	    case 'g':	        /* graphic output with gnuplot */
	        gnuplot = 1;
	        gnuplot_logic = TRUE;
		break;

	    case 'h':
		printf("%s\n", RCSID);
		printf("%s\n", HELP);
		printf("%s\n", USAGE);
		exit(1);

	    case 'p':
		print_par = TRUE;
		break;

	    case 'v':
		printf("%s\n", RCSID);
		exit(1);
		break;
/*
 * long option keys
 */
	    case '-':
		if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);

                } else if (strcmp("-print", *argv) == 0) {
		    print_par = TRUE;

                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);

                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}

		argc_next = 1;
		break;

	    default:
		gpiv_error("%s: unknown option: %s", RCSID, *argv);
		break;
	    }
	}
    }
    if (argc != 0) {
 	gpiv_error("%s: unknown argument: %s", RCSID, *argv);
    }

}



int main(int argc, char *argv[])
/*-----------------------------------------------------------------------------
 * main routine to tests PIV data on peak-locking effects
 */
{
    char *err_msg = NULL, *c = NULL;
    FILE *fp_par_dat;
    char fname[GPIV_MAX_CHARS], 
        fname_in[GPIV_MAX_CHARS],
	fname_parameter[GPIV_MAX_CHARS], 
        fname_out[GPIV_MAX_CHARS];
    char d_line[GPIV_MAX_LINES][GPIV_MAX_CHARS], 
        c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    char title[GPIV_MAX_CHARS];

    int nd_lines = 0, nc_lines = 0, scale = 0;
    GpivValidPar piv_valid_par, piv_valid_par_default;
    gint operator = 10;

    if ((c = strstr(argv[0], "peaklck")) != NULL) {
/*         operator = PEAKLCK; */
    } else if ((c = strstr(argv[0], "uhisto")) != NULL) {
        operator = GPIV_U;
    } else if ((c = strstr(argv[0], "vhisto")) != NULL) {
        operator = GPIV_V;
    } else {
        gpiv_error("%s: unvalid program name or symlink", argv[0]);
    }


/*
 * Setting parameters
 */
    gpiv_valid_parameters_logic(&piv_valid_par, FALSE);
    command_args(argc, argv, fname, &piv_valid_par);
    if (print_par) {
      printf("# %s\n# Command line options:\n", RCSID);
      gpiv_valid_print_parameters(piv_valid_par);
    }
    if (klass.nbins_logic == FALSE)
        klass.nbins = GPIV_NBINS_DEFAULT;


/*     gpiv_valid_default_parameters(&piv_valid_par_default, TRUE); */



    if (fname_logic == TRUE) {
/*
 * Generating filenames
 */
      gpiv_io_make_fname(fname, GPIV_EXT_PIV, fname_in);
      if (print_par) printf("# Input file: %s\n", fname_in);

      if (operator == GPIV_U || operator == GPIV_V) {
          gpiv_io_make_fname(fname, GPIV_EXT_UVHISTO, fname_out);
      } else {
          gpiv_io_make_fname(fname, GPIV_EXT_PLK, fname_out);
      }
      if (print_par) printf("# Output file: %s\n", fname_out);
/*
 * Prints command line parameters to par-file
 */
      gpiv_io_make_fname(fname, GPIV_EXT_PAR, fname_parameter);
      if (print_par)
          printf("# Data parameter file: %s\n", fname_parameter);

	if ((fp_par_dat = fopen(fname_parameter, "a")) == NULL) {
	    gpiv_error("%s: failure opening %s for input", 
		    RCSID, fname_parameter);
	}
	fprintf(fp_par_dat, "\n\n# %s\n# Command line options:\n", RCSID);
	gpiv_valid_fprint_parameters(fp_par_dat, piv_valid_par );


/*
 * Reading parametes from PARFILE (and writing to data par-file)
 */
        gpiv_scan_parameter(GPIV_VALID_PAR_KEY, PARFILE, &piv_valid_par, 
                            print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_VALID_PAR_KEY, &piv_valid_par, 
                                     print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
        gpiv_valid_fprint_parameters(fp_par_dat, piv_valid_par);
	fclose(fp_par_dat);


/*
 *  Reading PIV data
 */
	if ((scale = gpiv_fcount_pivdata(fname_in, &in_data)) == -1) {
	    fprintf(stderr,
		    "\n%s: Failure calling gpiv_count_pivdata\n",
		    RCSID);
	    return 1;
	}

    } else {
/*
 * use stdin, stdout
 */
        gpiv_scan_parameter(GPIV_VALID_PAR_KEY, PARFILE, &piv_valid_par, 
                            print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_VALID_PAR_KEY, &piv_valid_par, 
                                     print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
	if ((scale =
	     gpiv_count_pivdata(&in_data, d_line, &nd_lines)) == -1) {
	    fprintf(stderr,
		    "\n%s: Failure calling gpiv_count_pivdata\n",
		    RCSID);
	    return 1;
	}
    }

/*     if ((err_msg =  */
    gpiv_valid_check_parameters_read(&piv_valid_par, piv_valid_par_default);
/*         != NULL) gpiv_error ("%s: %s", RCSID, err_msg); */


/*
 * Now the parameters are known, data matrices can be defined by memory 
 * allocation
 */
    if (print_par)
        g_message("klass.nbins=%d", klass.nbins);
    gpiv_alloc_bindata(&klass);
    gpiv_alloc_pivdata(&in_data);


    if (fname_logic == TRUE) {
	gpiv_fread_pivdata(fname_in, &in_data, c_line, &nc_lines);
    } else {
	gpiv_read_pivdata(&in_data, d_line, nd_lines, c_line, &nc_lines);
    }

/*
 * Calling program function 
 */
    if (operator == GPIV_U || operator == GPIV_V) {
        gpiv_post_uvhisto(in_data, &klass, operator);	  
    } else {
        gpiv_valid_peaklck(in_data, &klass);
    }

    if (fname_logic == TRUE) {
	gpiv_fprint_histo(fname_out, &klass, c_line, nc_lines,
			  scale, RCSID);
    } else {
	gpiv_print_histo(&klass, c_line, nc_lines, scale, RCSID);
    }

/*
 * Graphical output with gnuplot
 */
    g_message("main:: 0");
    if (gnuplot && fname_logic) {
        if (operator == GPIV_U ) {
            snprintf(title, GPIV_MAX_CHARS, "Histogram U-velocities of %s", fname);
          } else if (operator == GPIV_V ) {
            snprintf(title, GPIV_MAX_CHARS, "Histogram V-velocities of %s", fname);
        } else {
            snprintf(title, GPIV_MAX_CHARS, "Peak locking statistics of %s", fname);
        }
        if ((err_msg = 
             gpiv_histo_gnuplot(fname_out, title, GNUPLOT_DISPLAY_COLOR, 
                                GNUPLOT_DISPLAY_SIZE))
            != NULL) gpiv_error ("%s: %s", RCSID, err_msg);   
    }


/*
 * Freeing allocated memory of matrices
 */
    gpiv_free_pivdata(&in_data);
    gpiv_free_bindata(&klass);

    exit(0);

}
