/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/* --------------------------------------------------------------------

   errvec - searches the erroneous vectors in PIV data and
   substitutes with new values, if possible

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  


---------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <gpiv.h>

/*#define  PARFILE "errvec.par" */	/* Parameter file name */
#define PARFILE "gpivrc"	/* Parameter file name */
#define USAGE "\
Usage: errvec [-a] [-b] [|--no_b] [-n N] [-c] [-f filename] [-g] [-h | --help] \n\
              [-p | --print] [-r int] [-s int] [-t float] [-u | --surround] \n\
              [-v | --version] < stdin > stdout \n\
\n\
keys: \n\
-a:                    Automatic threshold; calculates threshhold from \n\
                       residu histogram \n\
-b:                    calculates residu statistics to print histogram of \n\
                       N bins \n\
--no_b:                suppresses printing median residu histogram \n\
-n N:                  defines number of histogram of bins N for -b or -a \n\
-f filename:           files (without .piv extension) instead of stdin and \n\
                       stdout \n\
-g:                    graphical visualization of residu statistcs with \n\
                       gnuplot (needs -f)\n\
-h | --help:           this on-line help \n\
-p | --print:          print parameters to stdout  \n\
-r N:                  residu type calculated from: snr (0), median  (1) \n\
                       or normalized median (2) \n\
-s N:                  substitution of erroneous vector by: nothing (0), \n\
                       local mean from the surroundings (1), \n\
                       the median of the surroundings (2), next highest \n\
                       correlation peak (3) (needs -f) \n\
-t F:                  threshhold of residus to be accepted \n\
-u | --surround N:     odd number representing the surrounding grid \n\
                       points to be used for local mean and median values. \n\
-v | --version:        version number \n\
"


#define HELP  "\
errvec searches the erroneous vectors in a PIV data set and eventually \n\
substitutes them with new values."

#define RCSID "$Id: errvec.c,v 2.10 2006/01/31 14:18:04 gerber Exp $"

/*
 * Global variables for the parameters
 */
gboolean print_par = FALSE;
gboolean fname_logic = FALSE;

/*
 *  Why does gpiv_count_pivdata work if struct is defined here, 
 *  but not in main?
 */
GpivPivData in_data, out_data;
GpivBinData klass;
GpivLinRegData linreg;

#define GNUPLOT_DISPLAY_COLOR "DarkSlateGray"
#define GNUPLOT_DISPLAY_SIZE 250

int gnuplot = 0, gnuplot_logic = FALSE;


/*
 * Function prototypes
 */

int 
command_args(int argc, 
             char *argv[], 
             char fname[GPIV_MAX_CHARS],
             GpivValidPar * piv_valid_par
             )
/* ----------------------------------------------------------------------------
 * Command line argument handling
 */
{
    char c;
    int argc_next;


    while (--argc > 0 && (*++argv)[0] == '-') {
	argc_next = 0;
/*
 * argc_next is set to 1 if the next cmd line argument has to be searched for; 
 * in case that the command line argument concerns more than one char or cmd 
 * line argument needs a parameter
 */
	while (argc_next == 0 && (c = *++argv[0])) {
	    switch (c) {
	    case 'v':
/*
 * Use Revision Control System (RCS) for version
 */
		printf("%s\n", RCSID);
		exit(0);
		break;

	    case 'h':
		printf("%s\n", RCSID);
		printf("%s\n", HELP);
		printf("%s\n", USAGE);
#ifdef DEBUG
		printf("\n%s\n", USAGE_DEBUG);
#endif
		exit(0);
		break;
/*
 * Automatic threshold value
 */
	    case 'a':
                piv_valid_par->auto_thresh = 1;
                piv_valid_par->auto_thresh_logic = TRUE;
                piv_valid_par->residu_max_logic = TRUE;
		break;

	    case 'b':		/*  calculates median values */
		/*  to print out to stdout in histogram of N */
                piv_valid_par->res_stats = 1;
                piv_valid_par->res_stats_logic = TRUE;
                break;

	    case 'f':
		strcpy(fname, *++argv);
		fname_logic = TRUE;
		argc_next = 1;
		--argc;
		break;
/*
 * Visualize residu statistics with gnuplot
 */
	    case 'g':
	        gnuplot = 1;
		gnuplot_logic = TRUE;
/*
 * also set to calculate median values
 */
		piv_valid_par->res_stats = 1;
		piv_valid_par->res_stats_logic = TRUE;
		break;

	    case 'n':
                klass.nbins = atoi(*++argv);
                argc_next = 1;
                --argc;
		break;

	    case 'p':
	        print_par = TRUE;
		break;
/*
 *  residu type: snr or median
 */
	    case 'r':
		piv_valid_par->residu_type = atoi(*++argv);
		--argc;
		piv_valid_par->residu_type_logic = TRUE;
		argc_next = 1;
		break;
/*
 * substitution by mean, median or next highest cov. peak
 */
	    case 's':
		piv_valid_par->subst_type = atoi(*++argv);
		piv_valid_par->subst_type_logic = TRUE;
		--argc;
		argc_next = 1;
		break;
/*
 * maximum residu value allowed for 
 */
	    case 't':
                piv_valid_par->residu_max = atof(*++argv);
                piv_valid_par->residu_max_logic = TRUE;
                --argc;
                argc_next = 1;
		break;


	    case 'u':
                piv_valid_par->neighbors = atoi(*++argv);
                piv_valid_par->neighbors_logic = TRUE;
                argc_next = 1;
                --argc;
		break;

/*
 * long option keys
 */
	    case '-':
		if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);
                } else if (strcmp("-print", *argv) == 0) {
		    print_par = TRUE;
                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);
                } else if (strcmp("-no_b", *argv) == 0) {
		    piv_valid_par->res_stats = 0;
		    piv_valid_par->res_stats_logic = TRUE;
                } else if (strcmp("-surround", *argv) == 0) {
                    piv_valid_par->neighbors = atoi(*++argv);
                    piv_valid_par->neighbors_logic = TRUE;
                argc_next = 1;
                --argc;
                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
		break;

	    default:
#ifdef DEBUG
		printf("\n%s\n", USAGE_DEBUG);
#endif
		gpiv_error(USAGE);
		break;
	    }
	}
    }

    if (argc != 0) {
#ifdef DEBUG
	printf("\n%s", USAGE_DEBUG);
#endif
	gpiv_error("%s: %s\n", RCSID, USAGE);
    }
    return 0;
}



int 
make_fname(char *fname, 
           char *fname_in_piv, 
           char *fname_parameter,
           char *fname_out, 
           GpivValidPar piv_valid_par
           )
/* ----------------------------------------------------------------------------
 * generates filenames
 */
{
    if (fname_logic == FALSE) {
	gpiv_error("%s: Filename has to be set", RCSID);
    }

    gpiv_io_make_fname(fname, GPIV_EXT_PAR, fname_parameter);
    if (print_par)
        printf("# Data parameter file: %s\n", fname_parameter);

    gpiv_io_make_fname(fname, GPIV_EXT_PIV, fname_in_piv);
    if (print_par)
	printf("# Input data file: %s\n", fname_in_piv);

    if (piv_valid_par.res_stats == 1) {
        gpiv_io_make_fname(fname, GPIV_EXT_ERR_STAT, fname_out);
    } else {
        gpiv_io_make_fname(fname, GPIV_EXT_ERR_PIV, fname_out);
    }
    if (print_par)
	printf("# Output file: %s\n", fname_out);

    return 0;
}



int 
main(int argc, 
     char *argv[]
     )
/* ----------------------------------------------------------------------------
 * main routine to calculate erroneous vectors in a PIV data set
 */
{
    char *err_msg = NULL;
    FILE *fp_par_dat;
    char fname[GPIV_MAX_CHARS], fname_out[GPIV_MAX_CHARS],
	fname_parameter[GPIV_MAX_CHARS], fname_in_piv[GPIV_MAX_CHARS];
    char d_line[GPIV_MAX_LINES][GPIV_MAX_CHARS], 
        c_line[GPIV_MAX_LINES][GPIV_MAX_CHARS];
    char return_val;
    char title[GPIV_MAX_CHARS];
    int nd_lines = 0, nc_lines = 0, scale = 0;

    GpivImagePar image_par;
    GpivEvalPar piv_eval_par, piv_eval_par_default;
    GpivValidPar piv_valid_par, piv_valid_par_default;
    guint16 **img1 = NULL, **img2 = NULL;

    piv_valid_par.auto_thresh=0;

    gpiv_valid_parameters_logic(&piv_valid_par, FALSE);
    gpiv_valid_default_parameters(&piv_valid_par_default, TRUE);
    gpiv_piv_parameters_logic(&piv_eval_par, FALSE);
    gpiv_piv_default_parameters(&piv_eval_par_default, TRUE);
    klass.nbins = GPIV_NBINS_DEFAULT;

    if (strcmp(argv[0], "medtst") == 0) {
	piv_valid_par.res_stats = 0;
	piv_valid_par.res_stats_logic = TRUE;
	piv_valid_par.residu_max = 0;
	piv_valid_par.residu_max_logic = TRUE;
    }

    command_args(argc, argv, fname, &piv_valid_par);
    if (print_par) {
      printf("# %s\n# Command line options:\n", RCSID);
      gpiv_valid_print_parameters(piv_valid_par);
    }

    if (fname_logic == TRUE) {
/*
 * Generating proper filenames
 */
	if ((return_val = make_fname(fname, fname_in_piv, fname_parameter,
				     fname_out, piv_valid_par)) != 0) {
	    gpiv_error("%s: Failure calling make_fname", RCSID);
	}


/*
 * Prints command line parameters to stdout and data par-file
 */
	if ((fp_par_dat = fopen(fname_parameter, "a")) == NULL) {
	    gpiv_error("%s: failure opening %s for input",
		    RCSID, fname_parameter);
	}
	fprintf(fp_par_dat, "\n\n # %s\n# Command line options:\n", RCSID);
	gpiv_valid_fprint_parameters(fp_par_dat, piv_valid_par);


/*
 * Reading parametes from PARFILE (writing to data par-file will be done 
 * after eventually calculating the threshold from the PIV dataset)
 */
        gpiv_scan_parameter(GPIV_VALID_PAR_KEY, PARFILE, &piv_valid_par, 
                            print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_VALID_PAR_KEY, &piv_valid_par, 
                                     print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);


        if (piv_valid_par.res_stats == 1
            || piv_valid_par.auto_thresh == 1) {
/*
 * We also need image evaluation parameters for piv now
 */
            gpiv_scan_parameter(GPIV_EVAL_PAR_KEY, PARFILE, &piv_eval_par, 
                                print_par);
            if ((err_msg =
                 gpiv_scan_resourcefiles(GPIV_EVAL_PAR_KEY, &piv_eval_par, 
                                         print_par))
                != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
            gpiv_piv_fprint_parameters(fp_par_dat, piv_eval_par);
        }
	fclose(fp_par_dat);


    } else {
/*
 * use stdin, stdout
 */
        gpiv_scan_parameter(GPIV_VALID_PAR_KEY, PARFILE, &piv_valid_par, 
                            print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_VALID_PAR_KEY, &piv_valid_par, 
                                     print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

        if (piv_valid_par.res_stats == 1
            || piv_valid_par.auto_thresh == 1) {
            gpiv_scan_parameter(GPIV_EVAL_PAR_KEY, PARFILE, &piv_eval_par, 
                                print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_EVAL_PAR_KEY, &piv_eval_par, 
                                     print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
        }
   }


/*     if ((err_msg =  */
    gpiv_valid_check_parameters_read(&piv_valid_par, piv_valid_par_default);
/*         != NULL) gpiv_error ("%s: %s", RCSID, err_msg); */

    if (piv_valid_par.res_stats == 1
        || piv_valid_par.auto_thresh == 1) {
        gpiv_piv_check_parameters_read(&piv_eval_par, piv_eval_par_default);
    }
/*
 * Check parameters on correct values and adjust belonging variables
 */
    if (fname_logic == FALSE && piv_valid_par.subst_type == 2) {
	gpiv_error("%s: -f \"filename\" has to be used in combination with "
		   "'peak'", RCSID);
    }

    if (fname_logic == FALSE && gnuplot == 1) {
	gpiv_error("%s: -f \"filename\" has to be used in combination with "
		   "'gnuplot'", RCSID);
    }


/*
 * As in_data.nx or in_data.ny are not known, the input data file will 
 * first be read
 */

    if (fname_logic == TRUE) {
	if ((scale = gpiv_fcount_pivdata(fname_in_piv, &in_data)) == -1) {
	    gpiv_error("%s: Failure calling gpiv_count_pivdata", RCSID);
	}
    } else {
	if ((scale =
	     gpiv_count_pivdata(&in_data, d_line, &nd_lines)) == -1) {
	    gpiv_error("%s: Failure calling gpiv_count_pivdata", RCSID);
	}
    }


/*
 * Now the parameters are known, the data matrices can be defined
 *  by memory allocation
 */
    gpiv_alloc_pivdata(&in_data);
    out_data.nx = in_data.nx;
    out_data.ny = in_data.ny;
    gpiv_alloc_pivdata(&out_data);

/*
 * estimation of critical residu by means of fitting a straight line
 * through inverse cumulative histogram data. 
 * Fit depends on number and accuracy of bins. As a first approximation, 
 * a relative good fit will be found for 10 data (residus) 
 * per bin, defined by GPIV_NBINS_DEFAULT. 
 * BUGFIX: To be improved by minimizing linreg.sumsqr(klass.nbins).
 */
    if (piv_valid_par.res_stats == 1
        || piv_valid_par.auto_thresh == 1) {
    klass.nbins = in_data.nx * in_data.ny / GPIV_NBINS_DEFAULT;
    gpiv_alloc_bindata(&klass);
    }

/*
 * Reading input data
 */
    if (fname_logic == TRUE) {
	if ((err_msg =
	     gpiv_fread_pivdata(fname_in_piv, &in_data, c_line,
				&nc_lines))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
    } else {
	gpiv_read_pivdata(&in_data, d_line, nd_lines, c_line, &nc_lines);
    }

/*
 * Calculates threshold value if automatic threshold has been set
 */
    if (piv_valid_par.auto_thresh) {
        gpiv_valid_residu(in_data, &out_data, piv_valid_par, TRUE);
        if ((err_msg = gpiv_valid_residu_stats(&linreg, &out_data, &klass)) 
            != NULL) {
            err_msg = "GPIV_VALID_RESIDU: error from residu_stats";
            g_warning("%s", err_msg);
            exit(1);
        }
/* BUGFIX: differs from gpiv */
        piv_valid_par.residu_max = 
            gpiv_valid_threshold(piv_eval_par, piv_valid_par, linreg);
    }

/*
 * Validating PIV data on outliers and substituting
 * or calculating residu histogram
 */
    if ((err_msg = 
         gpiv_valid_errvec(image_par, piv_eval_par, piv_valid_par, in_data, 
                           &out_data, &klass, &linreg, img1, img2, FALSE))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

/*
 * Writing PIV data (and valid parameters) to output
 */
    if (piv_valid_par.res_stats == 0) {
	if (fname_logic == TRUE) {
            if ((fp_par_dat = fopen(fname_parameter, "a")) == NULL) {
                gpiv_error("%s: failure opening %s for input",
                           RCSID, fname_parameter);
            }
            gpiv_valid_fprint_parameters(fp_par_dat, piv_valid_par);
            fclose(fp_par_dat);


            if ((err_msg =
                 gpiv_fwrite_pivdata(fname_out, &out_data, c_line, nc_lines, 
                                     scale, RCSID))
                != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
	} else {
	    gpiv_write_pivdata(&out_data, c_line, nc_lines, scale, RCSID);
	}

/*
 * print median residus as a (CUMULATIVE!!) histogram
 */
    } else if (piv_valid_par.res_stats == 1) {
        float residu_max = 0.0;
        if (piv_eval_par.int_size_1_logic == FALSE) {
            gpiv_error("\n%s error: Int_Size_1 not set", GPIV_EVAL_PAR_KEY);
        }
        snprintf(c_line[nc_lines], GPIV_MAX_CHARS,
                 " straight line fitting results:");
        nc_lines++;
        snprintf(c_line[nc_lines], GPIV_MAX_CHARS,
                 " c0 = %f c1 = %f (= <residu>)", linreg.c0, linreg.c1);
        nc_lines++;
        snprintf(c_line[nc_lines], GPIV_MAX_CHARS,
                 " cov00 = %f cov01 = %f", linreg.cov00, linreg.cov01);
        nc_lines++;
        snprintf(c_line[nc_lines], GPIV_MAX_CHARS,
                 " cov11 = %f sumsq = %f", linreg.cov11, linreg.sumsq);
        nc_lines++;
        snprintf(c_line[nc_lines], GPIV_MAX_CHARS,
                 " residu_max = %f",
                 residu_max = gpiv_valid_threshold(piv_eval_par, piv_valid_par,
                                                   linreg));
        nc_lines++;

	if (fname_logic == TRUE) {
	    gpiv_fprint_cumhisto_eqdatbin(fname_out, &klass, c_line, nc_lines,
			      scale, RCSID);
	} else {
	    gpiv_print_cumhisto_eqdatbin(&klass, c_line, nc_lines, scale, 
                                         RCSID);
	}

/*
 * Plots histogram with gnuplot
 */
	if (gnuplot) {
	    snprintf(title, GPIV_MAX_CHARS, "residu statistics of %s", fname);
	    gpiv_cumhisto_eqdatbin_gnuplot(fname_out, title, 
                                           GNUPLOT_DISPLAY_COLOR, 
                                           GNUPLOT_DISPLAY_SIZE,
                                           &linreg);
	}
    } else {
        gpiv_error("errvec: main: should not arrive here; res_stats");
    }


/*
 * Freeing allocated memory of matrices
 */
    gpiv_free_pivdata(&in_data);
    gpiv_free_pivdata(&out_data);
    if (piv_valid_par.res_stats == 1
        || piv_valid_par.auto_thresh == 1) {
        gpiv_free_bindata(&klass);
    }
    if (print_par)
	printf("\n");

    exit(0);
}
