/*
  Gnome-o-Phone - A program for internet telephony
  Copyright (C) 1999  Roland Dreier
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  $Id: ui.c 1.17 Sat, 11 Dec 1999 23:53:26 -0600 dreier $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ui.h"

#include <gtk/gtk.h>

#ifdef HAVE_LIBSLANG
#if defined (HAVE_SLANG_H)
#include <slang.h>
#elif defined (HAVE_SLANG_SLANG_H)
#include <slang/slang.h>
#else
#error "Can't find slang.h."
#endif
#endif

#include "config-ui.h"
#include "call-ui.h"
#include "thread-queue.h"
#include "request.h"
#include "talk.h"
#include "listen.h"
#include "connection.h"
#include "thread.h"
#include "sound.h"

#include "gphone.h"
#include "gphone-lib.h"

static Thread_Queue UI_Queue;

extern int r, w;
extern char *conn_name;

static GtkWidget *Mic_Widget, *Spk_Widget;
static GtkWidget *Stat_Label, *Conn_Label;
static GtkWidget *Switch_Button;

void
ui_init(void)
{
  UI_Queue = thread_queue_new();
}

void
ui_add_request(Request req)
{
  thread_queue_add(UI_Queue, req);
}

static void
quit_command(void)
{
  Request req;

  req = g_malloc(sizeof *req);
  
  req->type = REQUEST_QUIT;
  req->data_len = 0;
  req->data = NULL;

  talk_add_request(req);

  req = g_malloc(sizeof *req);
  
  req->type = REQUEST_QUIT;
  req->data_len = 0;
  req->data = NULL;

  listen_add_request(req);
}

static void
destroy(GtkWidget *widget, gpointer data)
{
  quit_command();

  gtk_main_quit();
}

static void
call_button(GtkWidget *widget, gpointer data)
{
  call_open_window();
}

static void
config_button(GtkWidget *widget, gpointer data)
{
  config_open_window();
}

static void
quit_button(GtkWidget *widget, gpointer data)
{
  g_log("GPHONE", G_LOG_LEVEL_INFO,
        "Quitting by user request");

  quit_command();

  gtk_main_quit();
}

static void
switch_button_clicked(GtkWidget *widget, gpointer data)
{
  Request req;

  req = g_malloc(sizeof *req);
  
  req->type = REQUEST_SWITCH;
  req->data_len = 0;
  req->data = NULL;

  talk_add_request(req);
}

static void
mute_button_clicked(GtkWidget *widget, gpointer data)
{
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
    sound_mute_mic(1);
  } else {
    sound_mute_mic(0);
  }
}

static gint
update_status_display(gpointer data)
{
  switch (get_status()) {
  case STAT_IDLE:
    gtk_label_set_text(GTK_LABEL(Stat_Label), "IDLE");
    break;
  case STAT_WAIT:
    gtk_label_set_text(GTK_LABEL(Stat_Label), "Waiting for connection...");
    break;
  case STAT_CALL:
    gtk_label_set_text(GTK_LABEL(Stat_Label), "Calling out...");
    break;
  case STAT_NEGOTIATE:
    gtk_label_set_text(GTK_LABEL(Stat_Label), "Negotiating...");
    break;
  case STAT_TALK:
    gtk_label_set_text(GTK_LABEL(Stat_Label), "TALK");
    break;
  case STAT_LISTEN:
    gtk_label_set_text(GTK_LABEL(Stat_Label), "LISTEN");
    break;
  case STAT_TALK_FD:
    gtk_label_set_text(GTK_LABEL(Stat_Label), "FULL DUPLEX");
    break;
  case STAT_RINGING:
    gtk_label_set_text(GTK_LABEL(Stat_Label), "Ringing...");
    break;
  case STAT_NO_ANSWER:
    gtk_label_set_text(GTK_LABEL(Stat_Label), "NO ANSWER");
    break;
  case STAT_QUIT:
    break;
  }

  if (connection_connected()) {
    gchar *hostname = connection_hostname();

    gtk_label_set_text(GTK_LABEL(Conn_Label), hostname);

    g_free(hostname);
  } else {
    gtk_label_set_text(GTK_LABEL(Conn_Label), "<nobody>");
  }    

  if (get_sound_duplex() == FULL_DUPLEX || get_status() != STAT_TALK) {
    gtk_widget_set_sensitive(Switch_Button, FALSE);
  } else {
    gtk_widget_set_sensitive(Switch_Button, TRUE);
  }

  gtk_progress_bar_update(GTK_PROGRESS_BAR(Mic_Widget),
                          ((gfloat) get_mic_level()) / G_MAXSHORT);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(Spk_Widget),
                          ((gfloat) get_spk_level()) / G_MAXSHORT);

  return(TRUE);
}

void
build_main_window(void)
{
  GtkWidget *window;
  GtkWidget *label;
  GtkWidget *big_vbox, *vbox, *hbox;
  GtkWidget *button;
  GtkWidget *separator;
  GtkWidget *frame;

                                /* Make the main window */

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), "gphone");
  gtk_container_set_border_width(GTK_CONTAINER(window), 10);

  gtk_signal_connect(GTK_OBJECT(window), "destroy",
                     GTK_SIGNAL_FUNC(destroy), NULL);

                                /* pack our window into a big vbox */

  big_vbox = gtk_vbox_new(FALSE, 5);

                                /* display the status inside a frame */

  frame = gtk_frame_new("status");
  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(frame), vbox);

  Stat_Label = gtk_label_new("<initialisation>");
  gtk_widget_show(Stat_Label);
  gtk_box_pack_start(GTK_BOX(vbox), Stat_Label, TRUE, TRUE, 0);

  hbox = gtk_hbox_new(FALSE, 5);
  label = gtk_label_new("Connected to:");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);
  Conn_Label = gtk_label_new("<nobody>");
  gtk_box_pack_start(GTK_BOX(hbox), Conn_Label, FALSE, FALSE, 0);
  gtk_widget_show(Conn_Label);
  gtk_widget_show(hbox);

  gtk_widget_show(vbox);
  gtk_widget_show(frame);

  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(big_vbox), frame, TRUE, TRUE, 0);

  separator = gtk_hseparator_new();
  gtk_widget_show(separator);

  gtk_box_pack_start(GTK_BOX(big_vbox), separator, FALSE, FALSE, 0);

                                /* use progress bar to display the mic 
                                 and speaker levels */

  Mic_Widget = gtk_progress_bar_new();
  label = gtk_label_new("mic");
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), Mic_Widget, TRUE, TRUE, 0);
  gtk_widget_show(Mic_Widget);
  gtk_widget_show(label);
  gtk_widget_show(hbox);

  gtk_box_pack_start(GTK_BOX(big_vbox), hbox, TRUE, TRUE, 0);

  Spk_Widget = gtk_progress_bar_new();
  label = gtk_label_new("spk");
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), Spk_Widget, TRUE, TRUE, 0);
  gtk_widget_show(Spk_Widget);
  gtk_widget_show(label);
  gtk_widget_show(hbox);

  gtk_box_pack_start(GTK_BOX(big_vbox), hbox, TRUE, TRUE, 0);

                                /* put some buttons on the bottom */

  separator = gtk_hseparator_new();
  gtk_widget_show(separator);

  gtk_box_pack_start(GTK_BOX(big_vbox), separator, FALSE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 5);
  button = gtk_button_new_with_label("Switch");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(switch_button_clicked), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  gtk_widget_show(button);
  Switch_Button = button;
  button = gtk_toggle_button_new_with_label("Mute");
  gtk_signal_connect(GTK_OBJECT(button), "toggled",
                     GTK_SIGNAL_FUNC(mute_button_clicked), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  gtk_widget_show(button);

  gtk_widget_show(hbox);

  gtk_box_pack_start(GTK_BOX(big_vbox), hbox, TRUE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 5);
  button = gtk_button_new_with_label("Call");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(call_button), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  gtk_widget_show(button);

  button = gtk_button_new_with_label("Config");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(config_button), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  gtk_widget_show(button);

  button = gtk_button_new_with_label("Quit");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(quit_button), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  gtk_widget_show(button);

  gtk_widget_show(hbox);

  gtk_box_pack_start(GTK_BOX(big_vbox), hbox, TRUE, FALSE, 0);

                                /* display everything and get it on */

  gtk_container_add(GTK_CONTAINER(window), big_vbox);
  gtk_widget_show(big_vbox);
  gtk_widget_show(window);
}

void *
run_ui(void *arg)
{
  build_main_window();

  if (get_sound_duplex() == FULL_DUPLEX) {
    gtk_widget_set_sensitive(Switch_Button, FALSE);
  } else {
    gtk_widget_set_sensitive(Switch_Button, TRUE);
  }

  gtk_timeout_add(100, update_status_display, NULL);

  gtk_main();

  return(NULL);
}

#ifdef HAVE_LIBSLANG
void
display_text_status(void)
{
  int level;
  
  SLsmg_gotorc(0, 0);
  
  switch (get_status()) {
  case STAT_IDLE:
    SLsmg_write_string("<IDLE>");
    break;
  case STAT_WAIT:
    SLsmg_write_string("Waiting for connection...");
    break;
  case STAT_CALL:
    SLsmg_write_string("Calling out...");
    break;
  case STAT_NEGOTIATE:
    SLsmg_write_string("Negotiating...");
    break;
  case STAT_TALK:
    SLsmg_write_string("<TALK>");
    break;
  case STAT_LISTEN:
    SLsmg_write_string("<LISTEN>");
    break;
  case STAT_TALK_FD:
    SLsmg_write_string("<FULL DUPLEX>");
    break;
  case STAT_RINGING:
    SLsmg_write_string("Ringing...");
    break;
  case STAT_NO_ANSWER:
    SLsmg_write_string("NO ANSWER");
    break;
  case STAT_QUIT:
    break;
  }
  SLsmg_erase_eol();
  
  if (connection_connected()) {
    gchar *hostname = connection_hostname();

    SLsmg_gotorc(4, 0);
    SLsmg_printf("connected to: %s", hostname);

    g_free(hostname);
  }

  SLsmg_gotorc(1, 0);
  SLsmg_write_string("mic: ");
  level = get_mic_level() * 10 / G_MAXSHORT;
  for (; level > 0; level--) {
    SLsmg_write_char('=');
  }
  SLsmg_erase_eol();

  SLsmg_gotorc(2, 0);
  SLsmg_write_string("spk: ");
  level = get_spk_level();
  for (; level > 0; level--) {
    SLsmg_write_char('=');
  }
  SLsmg_erase_eol();
  
  SLsmg_gotorc(SLtt_Screen_Rows - 1, 0);

  SLsmg_refresh();
}

void *
run_text_ui(void *arg)
{
  char c;

  term_setup();

  while (1) {
    
    if (SLang_input_pending(1)) {
      c = SLang_getkey();
      switch (c) {
      case 'q':
        {
          Request req;
          
          req = g_malloc(sizeof *req);
          
          req->type = REQUEST_QUIT;
          req->data_len = 0;
          req->data = NULL;
          
          talk_add_request(req);
        }
        break;
      default:
        {
          Request req;
          
          req = g_malloc(sizeof *req);
          
          req->type = REQUEST_SWITCH;
          req->data_len = 0;
          req->data = NULL;
          
          talk_add_request(req);
        }
        break;
      }
    }
    
    if (get_status() == STAT_QUIT) {
      break;
    } else {
      display_text_status();
    }
  }

  term_cleanup();
  return(NULL);
}
#endif



/*
 * Local variables:
 *  compile-command: "make -k gphone"
 * End:
 */
