/**
 * output.c -- output to the standard output stream
 * Copyright (C) 2009-2012 by Gerardo Orellana <goaccess@prosoftcorp.com>
 * GoAccess - An Ncurses apache weblog analyzer & interactive viewer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * A copy of the GNU General Public License is attached to this
 * source distribution for its full text.
 *
 * Visit http://goaccess.prosoftcorp.com for new releases.
 */

#define _LARGEFILE_SOURCE
#define _LARGEFILE64_SOURCE
#define _FILE_OFFSET_BITS 64

#include <glib.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <ctype.h>
#include <stdio.h>

#include "commons.h"
#include "error.h"
#include "output.h"
#include "parser.h"
#include "ui.h"
#include "util.h"

static GOutput **
new_goutput (int n)
{
   GOutput **output = malloc (sizeof (GOutput *) * n);
   if (output == NULL)
      error_handler (__PRETTY_FUNCTION__, __FILE__, __LINE__,
                     "Unable to allocate memory for new GOutput.");
   memset (output, 0, sizeof *output);

   return output;
}

static void
print_html_header (FILE *fp, char *now)
{
   fprintf (fp, "<html>\n");
   fprintf (fp, "<head>\n");
   fprintf (fp, "<title>Server Statistics - %s</title>\n", now);
   fprintf (fp, "<script type=\"text/javascript\">\n");

   fprintf (fp,
   "function t(c){for(var "
   "b=c.parentNode.parentNode.parentNode.getElementsByTagName('tr'),"
   "a=0;a<b.length;a++)'hide'==b[a].className?(b[a].className='show',"
   "c.innerHTML='[-] Collapse'):'show'=="
   "b[a].className&&(b[a].className='hide',c.innerHTML='[+] Expand')};");

   fprintf (fp,
   "function a(c){var b=c.parentNode.parentNode.nextElementSibling;"
   "'a-hide'==b.className?(b.className='a-show',c.innerHTML='[-]'):"
   "'a-show'==b.className&&(b.className='a-hide',c.innerHTML='[+]')};");

   fprintf (fp, "</script>\n");
   fprintf (fp, "<style type=\"text/css\">\n");

   fprintf (fp,
   "body {"
   "   font-family: Verdana;"
   "   font-size: 11px;"
   "}"
   "table.a1,"
   "table.a2 {"
   "   border-spacing: 0;"
   "   font-size: 11px;"
   "   margin: 5px;"
   "   table-layout: fixed;"
   "   white-space: nowrap;"
   "}"
   "table.a1 { width: 600px }"
   "table.a2 {"
   "   background-color: #EEE;"
   "   padding: 0 5px;"
   "   width: 590px;"
   "}"
   ".head {"
   "   background-color: #222;"
   "   color: #FFF;"
   "   padding: 5px;"
   "}"
   ".head span,"
   ".s {"
   "   cursor: pointer;"
   "   font-size: 9px;"
   "}"
   ".r { text-align: right }"
   ".red {"
   "   color: #D20B2C;"
   "   font-weight: 700;"
   "}"
   ".lnk {"
   "   font-weight:bold;"
   "   font-size:10px;"
   "}"
   "a { color: #222 }"
   ".desc {"
   "   background-color: #EEE;"
   "   color: #222;"
   "   padding: 5px;"
   "}"
   ".d1,"
   ".d2 {"
   "   overflow: hidden;"
   "   text-overflow: ellipsis;"
   "}"
   ".d1 { border-bottom: 1px dotted #eee }"
   ".d2 { border-bottom: 1px dotted #ccc }"
   ".bar {"
   "   background-color: #777;"
   "   border-right: 1px #FFF solid;"
   "   height: 10px;"
   "}"
   ".a-hide,"
   ".hide { display: none }");

   fprintf (fp, "</style>\n");
   fprintf (fp, "</head>\n");
   fprintf (fp, "<body>\n");
}

static void
print_html_footer (FILE * fp, char *now)
{
   fprintf (fp, "<p class=\"lnk\">Generated by: ");
   fprintf (fp, "<a href=\"http://goaccess.prosoftcorp.com/\">GoAccess</a> ");
   fprintf (fp, "%s on %s</p>", GO_VERSION, now);
   fprintf (fp, "</body>\n");
   fprintf (fp, "</html>");
}

static void
print_html_begin_table (FILE * fp)
{
   fprintf (fp, "<table class=\"a1\">\n");
}

static void
print_html_end_table (FILE * fp)
{
   fprintf (fp, "</table>\n");
}

static void
print_html_head_top (FILE * fp, const char *s, int span, int exp)
{
   fprintf (fp, "<tr>");
   if (exp) {
      span--;
      fprintf (fp, "<td class=\"head\" colspan=\"%d\">%s</td>", span, s);
      fprintf (fp, "<td class=\"head r\">");
      fprintf (fp, "<span onclick=\"t(this)\">Expand [+]</span>");
      fprintf (fp, "</td>");
   } else
      fprintf (fp, "<td class=\"head\" colspan=\"%d\">%s</td>", span, s);
   fprintf (fp, "</tr>\n");
}

static void
print_html_head_bottom (FILE * fp, const char *s, int colspan)
{
   fprintf (fp, "<tr>");
   fprintf (fp, "<td class=\"desc\" colspan=\"%d\">%s</td>", colspan, s);
   fprintf (fp, "</tr>\n");
}

static void
print_html_col (FILE * fp, int w)
{
   fprintf (fp, "<col style=\"width:%dpx\">\n", w);
}

static void
print_html_begin_tr (FILE * fp, int hide)
{
   if (hide)
      fprintf (fp, "<tr class=\"hide\">");
   else
      fprintf (fp, "<tr>");
}

static void
print_html_end_tr (FILE * fp)
{
   fprintf (fp, "</tr>");
}

static void
report_iter (gpointer k, gpointer v, GOutput ** output)
{
   output[iter_ctr] = malloc (sizeof (GOutput));
   output[iter_ctr]->data = (gchar *) k;
   output[iter_ctr++]->hits = GPOINTER_TO_INT (v);
}

static void
print_html_status (FILE * fp, struct logger *logger)
{
   GHashTable *ht = ht_status_code;

   char *v = NULL;
   float t;
   int n = g_hash_table_size (ht), i, k;
   if (n == 0)
      return;

   iter_ctr = 0;
   GOutput **output = new_goutput (n);
   g_hash_table_foreach (ht, (GHFunc) report_iter, output);

   qsort (output, iter_ctr, sizeof (GOutput *), struct_cmp_by_hits);

   print_html_begin_table (fp);

   print_html_col (fp, 60);
   print_html_col (fp, 80);
   print_html_col (fp, 60);
   print_html_col (fp, 400);

   print_html_head_top (fp, status_head, 4, 0);
   print_html_head_bottom (fp, status_desc, 4);

   for (i = 0; i < n; i++) {
      k = output[i]->hits;
      v = output[i]->data;

      t = ((float) (k * 100) / logger->total_process);

      print_html_begin_tr (fp, 0);
      fprintf (fp, "<td class=\"d1\">%d</td>", k);
      fprintf (fp, "<td class=\"d1\">%4.2f%%</td>", t);
      fprintf (fp, "<td class=\"d1\">%s</td>", v);
      fprintf (fp, "<td class=\"d1\">%s</td>", verify_status_code (v));
      print_html_end_tr (fp);

      free (output[i]);
   }
   free (output);
   print_html_end_table (fp);
}

static void
print_html_hosts (FILE * fp, struct logger *logger)
{
   GHashTable *ht = ht_hosts;
   char *bw, *ag, *v = NULL, *ptr_value;
   float t, l;
   int n = g_hash_table_size (ht), i, k, max, j, until = 0, delims = 0;
   size_t alloc = 0;
   struct struct_agents *s_agents;

   iter_ctr = 0;
   GOutput **output = new_goutput (n);
   g_hash_table_foreach (ht, (GHFunc) report_iter, output);

   qsort (output, iter_ctr, sizeof (GOutput *), struct_cmp_by_hits);

   print_html_begin_table (fp);

   print_html_col (fp, 20);
   print_html_col (fp, 60);
   print_html_col (fp, 80);
   print_html_col (fp, 120);
   print_html_col (fp, 80);
   print_html_col (fp, 240);

   print_html_head_top (fp, host_head, 6, n > OUTPUT_N ? 1 : 0);
   print_html_head_bottom (fp, host_desc, 6);

   until = n < MAX_CHOICES ? n : MAX_CHOICES;
   max = 0;
   for (i = 0; i < until; i++) {
      if (output[i]->hits > max)
         max = output[i]->hits;
   }
   for (i = 0; i < until; i++) {
      k = output[i]->hits;
      v = output[i]->data;

      bw = ht_bw_str (ht_host_bw, v);
      t = ((float) (k * 100) / logger->total_process);
      l = ((float) (k * 100) / max);
      l = l < 1 ? 1 : l;

      ag = g_hash_table_lookup (ht_hosts_agents, v);

      print_html_begin_tr (fp, i > OUTPUT_N ? 1 : 0);
      fprintf (fp, "<td>");
      if (ag != NULL)
         fprintf (fp, "<span class=\"s\" onclick=\"a(this)\">[+]</span>");
      else
         fprintf (fp, "<span class=\"s\">-</span>");
      fprintf (fp, "</td>");

      fprintf (fp, "<td class=\"d1\">%d</td>", k);
      fprintf (fp, "<td class=\"d1\">%4.2f%%</td>", t);
      fprintf (fp, "<td class=\"d1\">%s</td>", v);
      fprintf (fp, "<td class=\"d1\">%s</td>", bw);

      fprintf (fp, "<td class=\"d1\">");
      fprintf (fp, "<div class=\"bar\" style=\"width:%f%%\"></div>", l);
      fprintf (fp, "</td>");
      print_html_end_tr (fp);

      if (ag != NULL) {
         ptr_value = (char *) ag;

         delims = count_occurrences (ptr_value, '|');
         /* round-up + padding */
         alloc = ((strlen (ptr_value) + 200 - 1) / 200) + delims + 1;
         s_agents = malloc (alloc * sizeof (s_agents));
         if (s_agents == NULL)
            error_handler (__PRETTY_FUNCTION__, __FILE__, __LINE__,
                           "Unable to allocate memory.");
         memset (s_agents, 0, alloc * sizeof (s_agents));

         split_agent_str (ptr_value, s_agents, 200);
         fprintf (fp, "<tr class=\"a-hide\">\n");
         fprintf (fp, "<td colspan=\"5\">\n");
         fprintf (fp, "<div style=\"padding:10px 0;\">");
         fprintf (fp, "<table class=\"a2\">");

         for (j = 0; (j < 10) && (s_agents[j].agents != NULL); j++) {
            print_html_begin_tr (fp, 0);
            fprintf (fp, "<td class=\"d2\">%s</td>", s_agents[j].agents);
            print_html_end_tr (fp);
         }

         fprintf (fp, "</table>\n");
         fprintf (fp, "</div>\n");
         fprintf (fp, "</td>\n");
         print_html_end_tr (fp);

         for (j = 0; (s_agents[j].agents != NULL); j++)
            free (s_agents[j].agents);
         free (s_agents);
      }
      free (bw);
   }

   for (i = 0; i < n; i++)
      free (output[i]);
   free (output);
   print_html_end_table (fp);
}

static void
print_html_os_browser (FILE * fp, GHashTable * ht, struct logger *logger)
{
   char *val = NULL, *v = NULL;
   float t, l;
   int n = g_hash_table_size (ht), i, k, max, hide = 0;
   int total_uniq = g_hash_table_size (ht_unique_visitors);
   if (n == 0)
      return;

   iter_ctr = 0;
   GOutput **output = new_goutput (n);
   g_hash_table_foreach (ht, (GHFunc) report_iter, output);

   qsort (output, iter_ctr, sizeof (GOutput *), struct_cmp_asc);

   print_html_begin_table (fp);

   print_html_col (fp, 60);
   print_html_col (fp, 80);
   print_html_col (fp, 120);
   print_html_col (fp, 340);

   if (ht == ht_os) {
      print_html_head_top (fp, os_head, 4, 1);
      print_html_head_bottom (fp, os_desc, 4);
   } else if (ht == ht_browsers) {
      print_html_head_top (fp, browser_head, 4, 1);
      print_html_head_bottom (fp, browser_desc, 4);
   }

   max = 0;
   for (i = 0; i < n; i++) {
      if (output[i]->hits > max)
         max = output[i]->hits;
   }
   for (i = 0; i < n; i++) {
      k = output[i]->hits;
      v = output[i]->data;

      t = ((float) (k * 100) / total_uniq);
      l = ((float) (k * 100) / max);
      l = l < 1 ? 1 : l;

      hide = 0;
      if (strchr (v, '|') != NULL) {
         val = malloc (snprintf (NULL, 0, "|`- %s", v) + 1);
         if (val == NULL)
            error_handler (__PRETTY_FUNCTION__, __FILE__, __LINE__,
                           "Unable to allocate memory");
         sprintf (val, "|`- %s", get_browser_type (v));
         hide = 1;
      }

      print_html_begin_tr (fp, hide ? 1 : 0);
      fprintf (fp, "<td class=\"d1\">%d</td>", k);
      if (k == max)
         fprintf (fp, "<td class=\"d1 red\">%4.2f%%</td>", t);
      else
         fprintf (fp, "<td class=\"d1\">%4.2f%%</td>", t);

      if (hide) {
         fprintf (fp, "<td class=\"d1\">%s</td>", val);
         free (val);
      } else
         fprintf (fp, "<td class=\"d1\">%s</td>", v);

      fprintf (fp, "<td class=\"d1\">");
      fprintf (fp, "<div class=\"bar\" style=\"width:%f%%\"></div>", l);
      fprintf (fp, "</td>");
      print_html_end_tr (fp);
   }
   for (i = 0; i < n; i++)
      free (output[i]);

   free (output);
   print_html_end_table (fp);
}

static void
print_html_generic (FILE * fp, GHashTable * ht, struct logger *logger,
                    int module)
{
   char *v = NULL;
   float t;
   int n = g_hash_table_size (ht), i, k, until = 0;

   if (n == 0)
      return;

   iter_ctr = 0;
   GOutput **output = new_goutput (n);
   g_hash_table_foreach (ht, (GHFunc) report_iter, output);

   qsort (output, iter_ctr, sizeof (GOutput *), struct_cmp_by_hits);

   print_html_begin_table (fp);

   print_html_col (fp, 60);
   print_html_col (fp, 80);
   print_html_col (fp, 460);

   switch (module) {
    case REFERRERS:
       print_html_head_top (fp, ref_head, 3, n > OUTPUT_N ? 1 : 0);
       print_html_head_bottom (fp, ref_desc, 3);
       break;
    case NOT_FOUND:
       print_html_head_top (fp, not_found_head, 3, n > OUTPUT_N ? 1 : 0);
       print_html_head_bottom (fp, not_found_desc, 3);
       break;
    case REFERRING_SITES:
       print_html_head_top (fp, sites_head, 3, n > OUTPUT_N ? 1 : 0);
       print_html_head_bottom (fp, sites_desc, 3);
       break;
    case KEYPHRASES:
       print_html_head_top (fp, key_head, 3, n > OUTPUT_N ? 1 : 0);
       print_html_head_bottom (fp, key_desc, 3);
       break;
   }

   until = n < MAX_CHOICES ? n : MAX_CHOICES;
   for (i = 0; i < until; i++) {
      k = output[i]->hits;
      v = output[i]->data;

      t = ((float) (k * 100) / logger->total_process);

      print_html_begin_tr (fp, i > OUTPUT_N ? 1 : 0);
      fprintf (fp, "<td class=\"d1\">%d</td>", k);
      fprintf (fp, "<td class=\"d1\">%4.2f%%</td>", t);
      if (ht == ht_referrers) {
         fprintf (fp, "<td class=\"d1\">");
         fprintf (fp, "<a href=\"%s\" rel=\"nofollow\">%s</a>", v, v);
         fprintf (fp, "</td>");
      } else
         fprintf (fp, "<td class=\"d1\">%s</td>", v);
      print_html_end_tr (fp);
   }
   for (i = 0; i < n; i++)
      free (output[i]);

   free (output);
   print_html_end_table (fp);
}

static void
print_html_request_report (FILE * fp, GHashTable * ht, struct logger *logger)
{
   char *bw, *v = NULL;
   float t;
   int n = g_hash_table_size (ht), i, k, until = 0;

   if (n == 0)
      return;

   iter_ctr = 0;
   GOutput **output = new_goutput (n);
   g_hash_table_foreach (ht, (GHFunc) report_iter, output);

   qsort (output, iter_ctr, sizeof (GOutput *), struct_cmp_by_hits);

   print_html_begin_table (fp);

   print_html_col (fp, 60);
   print_html_col (fp, 80);
   print_html_col (fp, 100);
   print_html_col (fp, 360);

   if (ht == ht_requests) {
      print_html_head_top (fp, req_head, 4, n > OUTPUT_N ? 1 : 0);
      print_html_head_bottom (fp, req_desc, 4);
   } else if (ht == ht_requests_static) {
      print_html_head_top (fp, static_head, 4, n > OUTPUT_N ? 1 : 0);
      print_html_head_bottom (fp, static_desc, 4);
   }

   until = n < MAX_CHOICES ? n : MAX_CHOICES;
   for (i = 0; i < until; i++) {
      k = output[i]->hits;
      v = output[i]->data;

      bw = ht_bw_str (ht_file_bw, v);
      t = ((float) (k * 100) / n);

      print_html_begin_tr (fp, i > OUTPUT_N ? 1 : 0);
      fprintf (fp, "<td class=\"d1\">%d</td>", k);
      fprintf (fp, "<td class=\"d1\">%4.2f%%</td>", t);
      fprintf (fp, "<td class=\"d1\">%s</td>", bw);
      fprintf (fp, "<td class=\"d1\">%s</td>", v);
      print_html_end_tr (fp);

      free (bw);
   }
   for (i = 0; i < n; i++)
      free (output[i]);

   free (output);
   print_html_end_table (fp);
}

static void
print_html_visitors_report (FILE * fp, struct logger *logger)
{
   GHashTable *ht = ht_unique_vis;
   char *bw, *v = NULL, buf[DATELEN] = "";
   float t, l;
   int n = g_hash_table_size (ht), i, k, max;
   int total_uniq = g_hash_table_size (ht_unique_visitors);

   iter_ctr = 0;
   GOutput **output = new_goutput (n);
   g_hash_table_foreach (ht, (GHFunc) report_iter, output);

   qsort (output, iter_ctr, sizeof (GOutput *), struct_cmp_desc);

   print_html_begin_table (fp);

   print_html_col (fp, 60);
   print_html_col (fp, 80);
   print_html_col (fp, 100);
   print_html_col (fp, 100);
   print_html_col (fp, 260);

   print_html_head_top (fp, vis_head, 5, n > OUTPUT_N ? 1 : 0);
   print_html_head_bottom (fp, vis_desc, 5);

   max = 0;
   for (i = 0; i < n; i++) {
      if (output[i]->hits > max)
         max = output[i]->hits;
   }
   for (i = 0; i < n; i++) {
      k = output[i]->hits;
      v = output[i]->data;

      bw = ht_bw_str (ht_date_bw, v);
      convert_date (buf, v, DATELEN);
      t = ((float) (k * 100) / total_uniq);
      l = ((float) (k * 100) / max);
      l = l < 1 ? 1 : l;

      print_html_begin_tr (fp, i > OUTPUT_N ? 1 : 0);
      fprintf (fp, "<td class=\"d1\">%d</td>", k);
      if (k == max)
         fprintf (fp, "<td class=\"d1 red\">%4.2f%%</td>", t);
      else
         fprintf (fp, "<td class=\"d1\">%4.2f%%</td>", t);
      fprintf (fp, "<td class=\"d1\">%s</td>", buf);
      fprintf (fp, "<td class=\"d1\">%s</td>", bw);
      fprintf (fp, "<td class=\"d1\">");
      fprintf (fp, "<div class=\"bar\" style=\"width:%f%%\"></div>", l);
      fprintf (fp, "</td>");
      print_html_end_tr (fp);

      free (bw);
      free (output[i]);
   }
   free (output);
   print_html_end_table (fp);
}

static void
print_html_summary_field (FILE * fp, int hits, char *field)
{
   fprintf (fp, "<td class=\"d1\">%s</td>", field);
   fprintf (fp, "<td class=\"d1\">%d</td>", hits);
}

static void
print_html_summary (FILE * fp, struct logger *logger)
{
   char *bw, *size;
   off_t log_size;
   print_html_begin_table (fp);

   print_html_col (fp, 100);
   print_html_col (fp, 60);
   print_html_col (fp, 100);
   print_html_col (fp, 60);
   print_html_col (fp, 80);
   print_html_col (fp, 60);
   print_html_col (fp, 60);
   print_html_col (fp, 60);

   print_html_head_top (fp, T_HEAD, 8, 0);

   print_html_begin_tr (fp, 0);
   print_html_summary_field (fp, logger->total_process, T_REQUESTS);
   print_html_summary_field (fp, g_hash_table_size (ht_unique_visitors),
                             T_UNIQUE_VIS);
   print_html_summary_field (fp, g_hash_table_size (ht_referrers),
                             T_REFERRER);

   if (!piping) {
      log_size = file_size (ifile);
      size = filesize_str (log_size);
   } else
      size = alloc_string ("N/A");

   bw = filesize_str ((float) req_size);
   if (ifile == NULL)
      ifile = "STDIN";
   fprintf (fp, "<td class=\"d1\">%s</td>", T_LOG);
   fprintf (fp, "<td class=\"d1\">%s</td>", size);
   print_html_end_tr (fp);

   print_html_begin_tr (fp, 0);
   print_html_summary_field (fp, logger->total_invalid, T_F_REQUESTS);
   print_html_summary_field (fp, g_hash_table_size (ht_requests),
                             T_UNIQUE_FIL);
   print_html_summary_field (fp, g_hash_table_size (ht_not_found_requests),
                             T_UNIQUE404);
   fprintf (fp, "<td class=\"d1\">%s</td>", T_BW);
   fprintf (fp, "<td class=\"d1\">%s</td>", bw);
   print_html_end_tr (fp);

   print_html_begin_tr (fp, 0);
   fprintf (fp, "<td class=\"d1\">%s</td>", T_GEN_TIME);
   fprintf (fp, "<td class=\"d1\">%lu</td>", ((int) end_proc - start_proc));
   print_html_summary_field (fp, g_hash_table_size (ht_requests_static),
                             T_STATIC_FIL);
   fprintf (fp, "<td colspan=\"4\" class=\"d1\">%s</td>", ifile);

   print_html_end_tr (fp);
   print_html_end_table (fp);
   free (bw);
   free (size);
}

void
output_html (struct logger *logger)
{
   generate_time ();

   FILE *fp = stdout;

   print_html_header (fp, asctime (now_tm));
   print_html_summary (fp, logger);
   print_html_visitors_report (fp, logger);
   print_html_request_report (fp, ht_requests, logger);
   print_html_request_report (fp, ht_requests_static, logger);
   print_html_generic (fp, ht_referrers, logger, REFERRERS);
   print_html_generic (fp, ht_not_found_requests, logger, NOT_FOUND);
   print_html_os_browser (fp, ht_os, logger);
   print_html_os_browser (fp, ht_browsers, logger);
   print_html_hosts (fp, logger);
   print_html_status (fp, logger);
   print_html_generic (fp, ht_referring_sites, logger, REFERRING_SITES);
   print_html_generic (fp, ht_keyphrases, logger, KEYPHRASES);

   print_html_footer (fp, asctime (now_tm));
}
