/* Josh Pieper, (c) 2000
   Robert Munafo, (c) 2001

   This file is distributed under the GPL, see file COPYING for details */

#include <stdio.h>
#include <stdlib.h>

#ifndef WIN32
# include <unistd.h>
# include <sys/file.h>
#else
# include <io.h>
#endif

#if defined(HAVE_FCNTL)
# include <fcntl.h>
#endif

#include <errno.h>
#include <pthread.h>

#include "conf.h"
#include "connection.h"
#include "gnut.h"
#include "http.h"
#include "lib.h"
#include "list.h"
#include "net.h"
#include "threads.h"
#include "transfer.h"
#include "ui.h"

/* 0.4.28.c01 Moved mutex to gnut.c */

Gnut_List *transfer_list = 0;

void fre_gt(gnut_transfer **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

int gnut_xfer_num()
{
  return gnut_list_size(transfer_list);
}

/* Decay the rate if it's time to do so */
void decay_rate(gnut_transfer *gt)
{
  float64 temp;
  
  while (gt->rate_incr_time < time(0)) {
    temp = ((float64) gt->rate_bytes);
    temp = (temp * 15.0) / 16.0;
    gt->rate_bytes = ((uint32) temp);
    (gt->rate_incr_time)++;
  }
}

/* this is called after the file has been opened and the socket
 * set up, it either sends or receives the entire content of the
 * file, updating the gnut_transfer struc along the way...
 * It leaves the structure afterwards though, so that the user
 * can see downloads that occurred while he/she was away. */
int gnut_xfer_loop(gnut_transfer *gt, int overlap, char *overlap_buf,
		       int dest_cache)
{
  char *netbuf;
  int readfd, writefd;
  int ret = 0;
  int ret2;
  uint32 left;
  uint32 sent;
  uint32 rate = 0;
  int download;
#if !defined(HAVE_FLOCK) && defined(HAVE_FCNTL)
  struct flock lock;
#endif
  int msg_flag;

  dqi(0x0062);
  
  GD_S(3, "gnut.xfer_loop entering gt="); GD_P(3, gt); GD_S(3, "\n");
  
  msg_flag = 0;

  if ( (gt->type & 1) == 0 ) {
    /* types 2 and 4 are downloads (from network to local file) */
    GD_S(3, "download!\n");
    readfd = gt->sock;
    writefd = gt->fsock;
    download = 1;
  } else {
    /* types 1 and 3 are uploads (from local file to network) */
    GD_S(3, "upload!\n");
    readfd = gt->fsock;
    writefd = gt->sock;
    download = 0;
    overlap = 0;
  }

  if (overlap) {
    /* Move write file pointer back, and read in the overlap buffer */
  }

  netbuf = (char *) ymaloc(4096, 310);
  left = gt->gt_total - gt->gt_bytes;
  GD_S(3, "gnut_xfer_loop entering loop\n");
#ifdef WIN32
#undef close
#undef read
#undef write
#endif
  while ((gt->gt_state == STATE_CONNECTED) && (left)) {
    do {
      /* this recv crap is here because win32 doesn't treat
       * socks and fd's the same.... :( */
      if (download) {
	/* Read from network connection */
	ret = timed_recv(readfd, netbuf, MIN(left, 4096), 0, 300); /* tagok */
	dqi(0x0006);
      } else {
	/* Read from file */
	ret = read(readfd, netbuf, MIN(left, 4096));
	dqi(0x0007);
      }
    } while (errno==EINTR && ret<0);

    if (ret<0) {
      GD_PERROR(3,"gnut_xfer_loop, read");
    }

    if (ret==0) {
      break;
    }

    /* Decay the rate before adding credit for bytes just completed */
    decay_rate(gt);

    if (ret>0) {
      char *nb2;
      int going;

      sent = 0;
      nb2 = netbuf;
      going = 1;
      do {
        if (download) {
	  /* Write to file */
	  ret2 = write(writefd, nb2, ret);
  	  dqi(0x0008);
	} else {
	  /* Write to network connection */
	  ret2 = send(writefd, nb2, ret, 0);
  	  dqi(0x0009);
	}

	if ((errno == EINTR) && (ret2 < 0)) {
	  /* keep trying... */
	} else if ((ret2 > 0) && (ret2 < ret)) {
	  /* It didn't send all of it. */
	  nb2 += ret2; /* skip the part we've sent */
	  sent += ret2;
	  ret -= ret2; /* that many fewer bytes to send */ 
        } else {
	  /* we be done! */
	  going = 0;
	}
      } while (going);

      /* If there was an error... */
      if (ret2 <= 0) {
        dqi(0x0045);
	/* ... print any appropriate error messages... */
	if (errno == EINTR) {
	  /* already handled */
	} else if (errno == EPIPE) {
	  if (gc_verbose & 2) {
	    fprintf(stdout, UI_TR_UPLOAD_ABORT, gt->fname, gt->gt_bytes + ret);
	  }
	} else {
	  if (gc_verbose & 2) {
	    fprintf(stdout, UI_TR_UPLOAD_ABRT2, gt->fname, gt->gt_bytes + ret);
	    GD_PERROR(0, "The error was");
	  }
	}

	/* ... and exit the transfer */
	break;
      } else {
        dqi(0x0046);
      }

      sent += ret2;
      GD_S(5, "copied "); GD_U(5, sent); GD_S(5, " bytes from sock to sock\n");
      gt->gt_bytes += sent;
      gt->rate_bytes += sent;

      /* Now that file offsets are unsigned, the math has to be done
         very carefully to avoid underflow */
      if (left >= sent) {
        left -= sent;
      } else {
	left = 0;
      }

      if (sent > 512) { /* tagok */
	if (msg_flag == 0) {
          if (dest_cache) {
	    /* Cache download */
	    if (gc_verbose & 4) {
	      printf("\nDownloading to cache: %s\n", gt->fname);
	    }
	  } else if (gt->type == TYPE_RECV_PUSH) {
	    /* Push download */
	    if (gc_verbose & 1) {
	      printf(UI_TH_PDL_STARTED, gt->fname);
	    }
	  } else if (gt->type == TYPE_DOWNLOAD) {
	    /* Normal download */
	    if (gc_verbose & 1) {
	      printf(UI_TH_GDL_STARTED, gt->fname);
	    }
	  } else {
	    /* Upload */

            /* Since we have accomplished a partial upload, that proves
               we can upload (it doesn't matter if the upload later gets
               cancelled by the remote user or stopped by our user, we're
               just interested in whether uploads are possible. */
            gh_did_upload = 1; /* 0.4.28.c03 */

	    if (gc_verbose & 2) {
	      printf(UI_TR_UPLOAD_STARTED, gt->fname,
		     (unsigned int)(gt->ip[0]), (unsigned int)(gt->ip[1]),
                     (unsigned int)(gt->ip[2]), (unsigned int)(gt->ip[3]));
	    }
	  }
	  msg_flag = 1;
	}
      }
    } else {
      break;
    }

    if ((gt->rate_limit)
	|| get_rate_limit(download)
	|| conf_get_int("global_bandwidth_cap")) {
      int wait_done;

      wait_done = 0;
      /* we have a rate limit, and we need to see if we're
       * exceeding it */
      GD_S(3, "gnut.xfer_loop entering rate limiting code\n");
      do {
	GD_S(3, "rate limiting loop.. rate="); GD_I(3, rate); GD_S(3, "\n");
	GD_S(3, "gnut_global_rate: "); GD_I(3, gnut_global_rate());
	GD_S(3, "\n");

  	dqi(0x000a);
	decay_rate(gt);
  	dqi(0x000b);

	rate = gt->rate_bytes >> 4L;

	/* printf("transferred %i, decayed rate %i\n", gt->gt_bytes, rate); */

	if ((gt->rate_limit) && (rate > gt->rate_limit)) {
	  sleep(1);
  	  dqi(0x000c);
	} else if (get_rate_limit(download)
		   && (rate > get_rate_limit(download))) {
	  sleep(1);
  	  dqi(0x000d);
	} else if (conf_get_int("global_bandwidth_cap") &&
		   (gnut_global_rate() > conf_get_int("global_bandwidth_cap"))) {
	  sleep(1);
  	  dqi(0x000e);
	} else {
	  wait_done = 1;
	}
      } while (wait_done == 0);
      GD_S(3, "my rate is: "); GD_LI(3, rate); GD_S(3, " and I want to be: ");
      GD_LI(3, gt->rate_limit); GD_S(3, "\n");
    }
  }

  dqi(0x0063);

  if (left > 0) {
    dqi(0x0032);
  } else {
    dqi(0x0033);
  }

  GD_S(3, "closing connection on ret: "); GD_I(3, ret); GD_S(3, "\n");
  fre_str(&netbuf, 264);

  if (download) {
    dqi(0x000f);
    shutdown(readfd, 2);
    dqi(0x0010);
#ifdef HAVE_FLOCK
    flock(writefd, LOCK_UN);
#elif defined(HAVE_FCNTL)
    lock.l_type=F_UNLCK;
    lock.l_whence=SEEK_SET;
    lock.l_start=0;
    lock.l_len=0;
    fcntl(writefd, F_SETLK, &lock );
#endif

    dqi(0x0011);
    close_s(readfd);

    dqi(0x0012);
    close_f(writefd);
  } else {
    dqi(0x0013);
    shutdown(writefd,2);

    /* Don't have to unlock read file because it was never locked */

    dqi(0x0014);
    close_s(writefd);

    dqi(0x0015);
    close_f(readfd);
  }

  dqi(0x0016);

  gt->sock = -1;
  gt->fsock = -1;

  if (left == 0) {
    /* Moved the other "complete" messages here, they used
       to be in threads.c  Also fixed bug: It used to print the
       "upload complete" message when a download completed. */
    if (msg_flag) {
      if (dest_cache) {
	/* Cache download */
	if (gc_verbose & 4) {
	  printf(UI_TH_CA_SUCCEED, gt->fname);
	}
      } else if (gt->type == TYPE_RECV_PUSH) {
	/* Normal download */
	if (gc_verbose & 1) {
	  printf(UI_TH_PDL_SUCCEED, gt->fname);
	}
      } else if (gt->type == TYPE_DOWNLOAD) {
	/* Push download */
	if (gc_verbose & 1) {
	  printf(UI_TH_GDL_SUCCEED, gt->fname);
	}
      } else {
	/* Upload */
	if (gc_verbose & 2) {
	  fprintf(stdout, UI_TR_UPLOAD_COMPLETE, gt->fname,
		  (unsigned int)(gt->ip[0]), (unsigned int)(gt->ip[1]),
		  (unsigned int)(gt->ip[2]), (unsigned int)(gt->ip[3]),
		  gt->gt_bytes);
	}
      }
    }

    GD_S(3,"gnut.xfer_loop returning success\n");
    return 0;
  } else {
    if (msg_flag) {
      if (dest_cache) {
	/* Cache download */
	if (gc_verbose & 4) {
	  printf(UI_TH_CDL_FAILED, gt->fname, gt->gt_bytes);
	}
      } else if (gt->type == TYPE_RECV_PUSH) {
	/* Normal download */
	if (gc_verbose & 1) {
	  printf(UI_TH_PDL_FAILED, gt->fname, gt->gt_bytes);
	}
      } else if (gt->type == TYPE_DOWNLOAD) {
	/* Push download */
	if (gc_verbose & 1) {
	  printf(UI_TH_GDL_FAILED, gt->fname, gt->gt_bytes);
	}
      } else {
	/* Upload */
	/* No message here, because a more specific message will have already
	   printed when we got the error */
      }
    }

    GD_S(3,"gnut.xfer_loop returning partial\n");
    return -1;
  }
}

int32 g_xfer_newindices = 1000;

gnut_transfer * gnut_xfer_new()
{
  gnut_transfer *gt;
  int i;

  GD_S(3, "gnut_xfer.new entering\n");

  gt = (gnut_transfer *) ymaloc(sizeof(gnut_transfer), 311);

  gt->sock = -1;
  for(i=0; i<4; i++) {
    gt->ip[i] = 0;
  }
  gt->gt_port = 0;
  gt->fsock = -1;
  gt->fname = 0;
  gt->type = TYPE_UNSPEC;
  gt->gt_bytes = 0;
  gt->gt_total = 0;
  gt->rate_bytes = 0;
  gt->rate_incr_time = time(0);
  gt->rate_limit = 0;
  gt->gt_state = STATE_UNCON;
  gt->substate = 0;
  gt->gt_uindex = g_xfer_newindices++;
  gt->tid = pthread_self();
  gt->gt_dest_cache = 0;

  pthread_mutex_lock(&transfer_mutex);

  transfer_list = gnut_list_prepend(transfer_list, gt);

  pthread_mutex_unlock(&transfer_mutex);

  GD_S(3,"gnut_xfer.new returning success\n");

  return gt;
}
  
int gnut_xfer_delete(gnut_transfer *gt)
{
  GD_S(3, "gnut_xfer_delete entering\n");

  pthread_mutex_lock(&transfer_mutex);

  transfer_list = gnut_list_remove(transfer_list, gt, 10);

  pthread_mutex_unlock(&transfer_mutex);

  if (gt->fname) {
    fre_str(&(gt->fname), 265);
  }

  fre_gt(&gt, 266);

  GD_S(3, "gnut_xfer_delete returning success\n");

  return 0;
}

int gnut_xfer_enumerate(int(*a)(gnut_transfer *))
{
  Gnut_List *tmp;

  pthread_mutex_lock(&transfer_mutex);
  
  for (tmp = transfer_list; tmp; tmp=tmp->next) {
    if ((*a)(tmp->data) == -1) {
      break;
    }
  }
  
  pthread_mutex_unlock(&transfer_mutex);

  return 0;
}
