/* gnut   - Command-line Gnutella client for POSIX-compliant OS's
            Copyright (c) 2000, Josh Pieper
            Copyright (c) 2000-2001, Robert Munafo

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    To contact one of the authors, send electronic mail to:
    gnut@mrob.com or paper mail to: Gnut, c/o Robert Munafo, P.O.
    Box 0513, Scituate, MA 02066-0513, USA.
*/

/* The actial main() function is in cli.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <signal.h>
#include <time.h>

/* 0.4.28.c17 Many #includes of system headers have been replaced with
   includes of the local header sh_xxx.h, which contains the test for the
   existence of the system header. I have not placed 0.4.28.c17 tags on
   all of these because it's easy to find them all with
      egrep 'include.*sh_' *.[ch]
   */
#include "sh_sys_types.h"

#include <pthread.h>

#include "cache.h"
#include "cli.h"
#include "connection.h"
#include "gnut.h"
#include "threads.h"
#include "net.h"
#include "conf.h"
#include "host.h"
#include "transfer.h"
#include "qry.h"
#include "share.h"
#include "monitor.h"
#include "route.h"

/* Note on "NULL" versus "0":
 *
 * Read the following:
 *
 *    http://www.cs.mcgill.ca/faqpages/cfaq.html#null
 *
 * for a description of NULL macro and related issues. I happen to believe
 * that NULL hides the truth and doesn't provide any advantages (as
 * explained best by questions 1.5 through 1.7), so I've been eliminating
 * them.
 */

/* 0.4.28.c01 */
#ifndef PTHREADS_DRAFT4
  pthread_mutex_t ctfu_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t gc_list_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t host_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t lruh_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t make_conn_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t monitor_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t push_crc_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t push_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t qrepl_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t query_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t query_packet_mutex = PTHREAD_MUTEX_INITIALIZER;

  /* this mutex protects all finds, adds, and removes to the routing table */
  pthread_mutex_t route_mutex = PTHREAD_MUTEX_INITIALIZER;

  pthread_mutex_t send_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t share_hash_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t share_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t transfer_mutex = PTHREAD_MUTEX_INITIALIZER;
#else
  pthread_mutex_t ctfu_mutex;
  pthread_mutex_t gc_list_mutex;
  pthread_mutex_t host_mutex;
  pthread_mutex_t lruh_mutex;
  pthread_mutex_t make_conn_mutex;
  pthread_mutex_t monitor_mutex;
  pthread_mutex_t push_crc_mutex;
  pthread_mutex_t push_mutex;
  pthread_mutex_t qrepl_mutex;
  pthread_mutex_t query_mutex;
  pthread_mutex_t query_packet_mutex;
  pthread_mutex_t route_mutex;
  pthread_mutex_t send_mutex;
  pthread_mutex_t share_hash_mutex;
  pthread_mutex_t share_mutex;
  pthread_mutex_t transfer_mutex;
#endif

/* 0.4.28.c01 Moved initializers for mutexes to this file */
#ifdef PTHREADS_DRAFT4
# include <pthread.h>
/* Pthreads draft 4 implements a number of things differently, one of which
 * is that you can't statically initalise mutexes. kxp@atl.hp.com  */
 
void gnut_mutexes_init()
{
  pthread_mutex_init(&cache_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&ctfu_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&gc_list_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&_g_debug_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&host_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&lruh_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&make_conn_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&monitor_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&mutex_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&push_crc_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&push_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&qrepl_mutex,pthread_mutexattr_default);
  pthread_mutex_init(&query_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&query_packet_mutex,pthread_mutexattr_default);
  pthread_mutex_init(&route_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&send_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&share_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&share_hash_mutex, pthread_mutexattr_default);
  pthread_mutex_init(&transfer_mutex, pthread_mutexattr_default);
}
#endif


int global_rate=0;
int new_global_rate;
int gh_error;
uint32 gnut_heartbeat;

#ifndef WIN32
pthread_t segv_pt;

int segv_callback2(gnut_transfer *gt)
{
  if (pthread_equal(gt->tid, segv_pt)) {
    GD_S(1, "The following transfer was the culprit.\n"); GD_I(1, gt->ip[0]);
    GD_S(1, "."); GD_I(1, gt->ip[1]); GD_S(1, "."); GD_I(1, gt->ip[2]);
    GD_S(1, "."); GD_I(1, gt->ip[3]); GD_S(1, ":"); GD_I(1, gt->gt_port);
    GD_S(1, " fsock: "); GD_I(1, gt->fsock); GD_S(1, "  fname: ");
    GD_S(1, gt->fname); GD_S(1, "\ntype: "); GD_I(1, gt->type);
    GD_S(1, "  bytes: "); GD_I(1, gt->gt_bytes); GD_S(1, "  total: ");
    GD_U(1, gt->gt_total); GD_S(1, "\nrateb: "); GD_U(1, gt->rate_bytes);
    GD_S(1, "  rate_lim: "); GD_I(1, gt->rate_limit); GD_S(1, "\nstate: ");
    GD_I(1, gt->gt_state); GD_S(1, "  index: "); GD_I(1, gt->gt_uindex);
    GD_S(1, "\n");
  }
  
  return 0;
}

int segv_callback(gcs *gc)
{
  if (readline_crashed && readline_firsttime) {
    printf("\nIt is possible that readline caused the crash. Sometimes an incompatible\nreadline library causes gnut to crash at startup. You should try recompiling\ngnut without readline support. To do this:\n  1. Place '#undef USE_READLINE' at the beginning of src/cli.c\n  2. Repeat the 'make' and/or 'make install' steps.\n");
    exit(0);
  }

  if (pthread_equal(gc->tid, segv_pt) ) {
    /* we found it! */
    GD_S(1, "The following connection was the culprit.\n");
    GD_I(1, gc->ip.b[0]); GD_S(1, "."); GD_I(1, gc->ip.b[1]); GD_S(1, ".");
    GD_I(1, gc->ip.b[2]); GD_S(1, "."); GD_I(1, gc->ip.b[3]); GD_S(1, ":");
    GD_I(1, gc->port); GD_S(1, " pack: "); GD_I(1, gc->sent_packets);
    GD_S(1, "/"); GD_I(1, gc->received_packets); GD_S(1, "/");
    GD_I(1, gc->dropped_packets); GD_S(1, " bytes: ");
    GD_F64(1, gc->snt_bytes); GD_S(1, "/"); GD_F64(1, gc->rcv_bytes);
    GD_S(1, "\nrate: "); GD_F64(1, gc->rate_snt_bytes); GD_S(1, "/");
    GD_F64(1, gc->rate_rcv_bytes); GD_S(1, "/");
    GD_I(1, (int) gc->rate_start_time); GD_S(1, " err: ");
    GD_I(1, gc->routing_errors); GD_S(1, " type: "); GD_I(1, gc->ctype);
    GD_S(1, "\nstate: "); GD_I(1, gc->cstate); GD_S(1, "  index: ");
    GD_I(1, gc->uindex); GD_S(1, "\n");
  } else {
    GD_S(1, "tid: ");
#if defined(PTHREADS_DRAFT4) && defined(hpux)
    GD_LI(1, ((uint32) ((HPUX_hack)gc->tid).longpart)); /* 0.4.28.c12 */
#else
    GD_LI(1, ((uint32) gc->tid));
#endif
    GD_S(1," wasn't a prob\n");
  }
  return 0;
}    


void segv_handler(int a)
{
  dq_dump("SEGV");

  segv_pt = pthread_self();

  if (gc_debug_opts) {
    fprintf(stderr,"Segfault!!! TID=%u  PID=%u\n",
#if defined(PTHREADS_DRAFT4) && defined(hpux)
	    ((uint32) ((HPUX_hack)pthread_self()).longpart),
#else
	    (uint32) pthread_self(),
#endif
	    ((uint32) getpid()));
  }

  fflush(stderr);
  sleep(1);
  fflush(stderr);
  /* now I'm going to go through first the connection array and see if
   * this ID exists anywhere... */
  
  gnut_connection_enumerate(segv_callback);
  gnut_xfer_enumerate(segv_callback2);  
  
#ifdef DMALLOC
  dmalloc_shutdown();
#endif
  
  while(1) {
    sleep(1);
  }
}

void reaper(int a)
{
  (void) wait(0);
}
#endif /* ndef WIN32 */

int gnut_global_rate()
{
  return global_rate;
}

int get_rate_limit(int download)
{
  if (download) {
    return(conf_get_int("default_download_cap"));
  }
  return(conf_get_int("default_upload_cap"));
}

#ifdef PTHREADS_DRAFT4
  pthread_once_t gnut_mutexes_once = pthread_once_init;
#endif

int gnut_init()
{
  char mac[6];
  char guid[16];
  char *d;
  int i;
#ifdef WIN32
  WORD wVersionRequested;
  WSADATA wsaData;
  int err;
  
  wVersionRequested = MAKEWORD( 2, 2);
  
  err=WSAStartup( wVersionRequested, &wsaData);
  if (err != 0) {
    return 0;
  }
#else  

  struct sigaction sa;

#ifdef PTHREADS_DRAFT4
  pthread_once(&gnut_mutexes_once, gnut_mutexes_init);
#endif

  memset(&sa,0,sizeof(sa));  

  sa.sa_handler=SIG_IGN;
  sa.sa_flags=SA_RESTART;

  /*  sigaction(2,&sa,0); */
  
  sigaction(13,&sa,0);
  sigaction(28,&sa,0);
  sigaction(14,&sa,0);
  sigaction(15,&sa,0);
  sigaction(18,&sa,0);
  sigaction(19,&sa,0);
  /*  sigaction(20,&sa,0); */
  /*  sigaction(21,&sa,0); */
  /*  sigaction(22,&sa,0); */
  sigaction(23,&sa,0);
  
  /*  sigaction(10,&sa,0); */
  /*  sigaction(12,&sa,0); */
  sigaction(29,&sa,0);
  sigaction(30,&sa,0);
  sigaction(6,&sa,0);
  
  signal(11,segv_handler);
  signal(17,reaper);
#endif
  
  srand(time(0));
  
  for (i=0;i<6;i++) {
    mac[i]=rand();
  }
  conf_set_str_len("mac", mac, 6);
  
  for (i=0;i<6;i++) {
    guid[i]=mac[i];
  }
  for (i=7;i<16;i++) {
    guid[i]=rand();
  }
  conf_set_str_len("guid", guid, 16);

  current_query_packet = 0;

  route_init();
  monitor_init();
  query_init();
  share_init();
  host_init();

  d = expand_path("~/.gnut_hosts");
  GD_S(2, "gnut.init trying to restore hosts\n");
  gh_error = 0;
  if (host_restore(d)) {
    gh_error = 1;
  }
  
  fre_str(&d, 384);
  
  return 0;
}

/* The query response record passed to this routine will be a COPY, al.located
 * just for calling gnut_xfer.start. In other words, it's ok to delete
 * when done. */
int gnut_xfer_start(query_resp *a_qrp4)
{
  pthread_t nt;

#ifndef PTHREADS_DRAFT4
  pthread_create(&nt, 0, gnut_threads_dl, a_qrp4); /* qr-del */
  pthread_detach(nt);
#else  
  pthread_create(&nt, pthread_attr_default, gnut_threads_dl, a_qrp4); /* qr-del */
  pthread_detach(&nt);
#endif

  return 0;
}  

/* sets up up a new push connection... */
int gnut_push_new(gnutella_push_a *gp_a)
{
  pthread_t np;
  push_arg *pa;
  share_item *si;
  uint32 ref;

  ref = GET_LEU32((*gp_a) + PROT_PUSH_REF, 18);

  si = share_find(ref);

  if (si==0) {
    return -1;
  }

  pa = (push_arg *) ymaloc(sizeof(push_arg), 331);

  pa->si = si;
  memcpy(pa->ip, (*gp_a) + PROT_PUSH_IP, 4);
  memcpy(pa->port_le, (*gp_a)  + PROT_PUSH_PORT, 2);

#ifndef PTHREADS_DRAFT4
  pthread_create(&np,0,gnut_threads_push,pa);
  pthread_detach(np);
#else  
  pthread_create(&np,pthread_attr_default,gnut_threads_push,pa);
  pthread_detach(&np);
#endif

  return 0;
}

/* gnut_try_host tests a host entry to see if it's a good candidate for
   making a new GnutellaNet connection.

   First parameter is pointer to element to "try"; second parameter
   is a pointer to a pointer, this gets set to say that this element
   has "succeeded". Also on "success" a -1 result is returned. */
int gnut_try_host(void *a, void *b)
{
  host_entry *he;
  host_entry **he2;
  uint16 port;
  int flag;
  uchar tip[4];
  
  he = (host_entry *) a;
  he2 = (host_entry **) b;
  
  port = GET_LEU16(he->port_le, 19);

  /* First, check if it's already in the connections list */
  flag = gnut_connection_find(he->ip, port);

  memcpy(tip, he->ip, 4);
  
  /* Then eliminate various illegal addresses */
  if (memcmp(he->ip, net_local_ip(), 4) == 0) { /* our own IP address */
    flag = 0;
  } else if (!(host_ok_for_gnet(he->ip))) {
    /* It's at an unreachable address */
    flag = 0;
  }
  
  if (flag) {
    /* Okay, it's valid */
    *he2 = he;
    return -1;
  }  
  return 0;
}

/* int gnut_outgoing_new(char *host)
 *
 * sets up a new outgoing connection. Uses gnut_hash_foreach to find
 * new host */
int gnut_outgoing_new(char *host)
{
  char *mhost;
  pthread_t no;
  Gnut_Hash *gl;
  host_entry *he;
  uchar myip[4];
  uint16 mport;
  host_entry lruhost;

  if (host) {
    mhost = ystdup(host, 465);
  } else {
    /* Get a host from the LRU cache */
    host_get_lru(&lruhost);

    /* If none (this happens if gnut has just started up and not many
       packets have come in yet), then get one off the entire host table
       hash */
    if (lruhost.ip[0] == 0) {
      host_lock();
      gl = host_retrieve();

      he = 0;
      gnut_hash_foreach(gl, gnut_try_host, &he);

      if (!he) {
        GD_S(4, "gnut_outgoing_new no eligible hosts in catcher\n");
        host_unlock();
        return 0;
      }
      lruhost = *he;
      host_unlock();
    }
    memcpy(myip, lruhost.ip, 4);
    mport = GET_LEU16(lruhost.port_le, 20);
    mhost = (char *) ymaloc(100, 269);
    sprintf(mhost, "%i.%i.%i.%i:%i", myip[0], myip[1], myip[2], myip[3],
	    mport);
  }

#ifndef PTHREADS_DRAFT4
  pthread_create(&no, 0, gnut_threads_outgoing, mhost);
  pthread_detach(no);
#else
  pthread_create(&no, pthread_attr_default, gnut_threads_outgoing, mhost);
  pthread_detach(&no);
#endif

  return 0;
}

int gnut_incoming_new(int sock)
{
  int newsock;
  struct sockaddr_in sin;
  int len;
  pthread_t ni;
  incoming_arg *ia;
  
  GD_S(2, "gnut_incoming_new entering\n");
  
  len = sizeof(sin);
  newsock = accept(sock, (struct sockaddr *) &sin, &len);
  
  if (newsock < 0) {
    GD_PERROR(0, "gnut_incoming_new, accept");
    return newsock;
  }

  ia=(incoming_arg *) ymaloc(sizeof(incoming_arg), 332);
  ia->s=newsock;
  memcpy(&ia->sin,&sin,sizeof(sin));

#ifndef PTHREADS_DRAFT4
  pthread_create(&ni,0,gnut_threads_incoming,ia);
  pthread_detach(ni);
#else
  pthread_create(&ni,pthread_attr_default,gnut_threads_incoming,ia);
  pthread_detach(&ni);
#endif

  GD_S(2,"gnut_incoming_new leaving\n");
  return 0;
}

int count_rate(gnut_transfer *gt)
{
  uint32 tr_rate;

  if ((gt->gt_state & 1)==0 || gt->gt_state==STATE_DEAD || 
      gt->gt_state==STATE_ERROR || gt->gt_state==STATE_CLEANABLE) {
    return 0;
  }
  
  tr_rate = gt->rate_bytes >> 4L;
  GD_S(3, "count.rate tr_rate="); GD_LI(3, tr_rate);
  GD_S(3, " new_global_rate="); GD_I(3, new_global_rate); GD_S(3, "\n");
  new_global_rate += tr_rate;
  return 0;
}

int trigger_bug1 = 0;
  
/* This is the main loop for maintaining the Gnutellanet connections.
 * The main loop that takes commands is command_loop() in cli_input.c.
 * The actial main() function is in cli_interface.c */
#define KILLW_INTERVAL (4L * 4L * 60L * 4L)
void *main_loop(void *arg)
{
  int listen_socket;
  int port,ret;
  fd_set fsr;
  struct timeval tv;
  long killw_time;
  int min_conn, total_conn;
  int nn, i;

  dq_start("MA", 0x0051);

  GD_S(3, "main_loop entering\n");

  /* Create the socket for taking new GnutellaNet connections */
  port = conf_get_int("local_port");
  listen_socket = create_listen_socket(&port);
  if (listen_socket < 0) {
    printf("Couldn't create listen socket\n");

    dq_end(0);
  }
  /* If port changed, update the variable */
  conf_set_int("local_port",port);

  killw_time = 0;
  while (1) {
    min_conn = conf_get_int("min_connections");
    total_conn = gnut_threads_num_incoming() + gnut_threads_num_outgoing();

    /* Check for incoming connections, timeout 0.25 seconds */
    FD_ZERO(&fsr);
    FD_SET(listen_socket, &fsr);

    tv.tv_sec = 0;
    tv.tv_usec = 250000;
    ret = select(listen_socket+1, &fsr, 0, 0, &tv);
    GD_S(5, "main_loop select returned: "); GD_I(5, ret); GD_S(5, "\n");

    if (FD_ISSET(listen_socket, &fsr)) {
      GD_S(2,"main_loop incoming connection\n");
      gnut_incoming_new(listen_socket);
    } else {
      /* We got a timeout */

      /* This is a good place to put any periodic tasks that should run
	 at set time intervals (like once every minute, etc.) */

      if (conf_get_int("autokill_thres")) {
	/* At periodic intervals (normalized to once per 16 minutes per
	   connection) we kill the connection with the worst duplicate-packets
	   statistic. */
	killw_time ++;
	if (total_conn > 4) {
	  if (killw_time * min_conn > KILLW_INTERVAL) {
	    killw_time = 0;
	    gnut_connection_kill_worst(0);
	  }
	}
      }
    }

    /* Check to see if we should initiate another outgoing connection,
       based on the min_connections configuration variable. */
    nn = min_conn - total_conn;
    for (i=0; (i<3) && (i<nn); i++) {  
      GD_S(4, "main_loop initiating outgoing\n");
      gnut_outgoing_new(0);
    }

    /* Update the global transfer rate total */
    new_global_rate = 0;
    gnut_xfer_enumerate(count_rate);
    global_rate = new_global_rate;

    cache_fixlock();

    gnut_heartbeat++;

    if(trigger_bug1) {
      printf("trigger_bug1\n");
      while(1) {
	sleep(10);
      }
      break;
    }
  }

  dq_end(0);

  return 0;
}

int gnut_start_main_loop()
{
  pthread_t ml;

#ifndef PTHREADS_DRAFT4
  pthread_create(&ml, 0, main_loop, 0);
  pthread_detach(ml);
#else  
  pthread_create(&ml, pthread_attr_default, main_loop, 0);
  pthread_detach(&ml);
#endif

  return 0;
}
