/*
**  PreferencesWindowController.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "PreferencesWindowController.h"

#import "GNUMail.h"
#import "GNUMailConstants.h"
#import "MailWindowController.h"

#import <GNUMail/PreferencesModule.h>

#ifndef MACOSX
#import "PreferencesWindow.h"
#endif

static PreferencesWindowController *singleInstance = nil;

@implementation PreferencesWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
  
#else
  PreferencesWindow *preferencesWindow;

  preferencesWindow = [[PreferencesWindow alloc] initWithContentRect: NSMakeRect(250,250,472,525)
						 styleMask: NSTitledWindowMask
						 backing: NSBackingStoreRetained
						 defer: NO];

  self = [super initWithWindow: preferencesWindow];
  
  [preferencesWindow layoutWindow];
  [preferencesWindow setDelegate: self];
 
  // We link our outlets
  matrix = [preferencesWindow matrix];
  box = [preferencesWindow box];

  RELEASE(preferencesWindow);
#endif

  // We set our window title
  [[self window] setTitle: _(@"Preferences Panel")];
  
  // We initialize our matrix with the standard modules
  [self initializeWithStandardModules];

  return self;
}


//
//
//
- (void) dealloc
{
  NSLog(@"PreferencesWindowController: -dealloc");
  
  RELEASE(allModules);

  [super dealloc];
}


//
// delegate methods
//

- (void) windowDidLoad
{
  // We maintain an array of opened modules
  allModules = [[NSMutableDictionary alloc] initWithCapacity: 9];
}

- (void) windowWillClose: (NSNotification *) theNotification
{
  AUTORELEASE(self);

  singleInstance = nil;
}

- (void) handleCellAction: (id) sender
{  
  id aModule;
  
  switch ( [[sender selectedCell] tag] )
    {
    case 0:
      {
	aModule = [allModules objectForKey: @"Personal"];
	
	if ( aModule )
	  {
	    [self addModuleToView: aModule];
	  }
	else
	  {
	    NSLog(@"Unable to load the Personal bundle.");
	  }
      }
      break;
      
    case 1:
      { 
	aModule = [allModules objectForKey: @"Viewing"];
	
	if ( aModule )
	  {
	    [self addModuleToView: aModule];
	  }
	else
	  {
	    NSLog(@"Unable to load the Viewing bundle.");
	  }
      }
      break;
      
    case 2: 
      {
	aModule = [allModules objectForKey: @"Sending"];
	
	if ( aModule )
	  {
	    [self addModuleToView: aModule];
	  }
	else
	  {
	    NSLog(@"Unable to load the Sending bundle.");
	  }
      }
      break;
	
    case 3: 
      {
	aModule = [allModules objectForKey: @"Receiving"];
	
	if ( aModule )
	  {
	    [self addModuleToView: aModule];
	  }
	else
	  {
	    NSLog(@"Unable to load the Receiving bundle.");
	  }
      }
      break;
      
  case 4: 
    {
      aModule = [allModules objectForKey: @"Compose"];
      
      if ( aModule )
	{
	    [self addModuleToView: aModule];
	}
      else
	{
	  NSLog(@"Unable to load the Compose bundle.");
	}
    }
    break;
    
  case 5: 
    {
      aModule = [allModules objectForKey: @"Fonts"];
      
      if ( aModule )
	{
	  [self addModuleToView: aModule];
	}
      else
	{
	  NSLog(@"Unable to load the Quoting bundle.");
	}
    }
    break;

  case 6: 
    {
      aModule = [allModules objectForKey: @"MIME"];
      
      if ( aModule )
	{
	  [self addModuleToView: aModule];
	}
      else
	{
	  NSLog(@"Unable to load the MIME bundle.");
	}
    }
    break;

  case 7:
    {
      aModule = [allModules objectForKey: @"Filtering"];
      
      if ( aModule )
	{
	  [self addModuleToView: aModule];
	}
      else
	{
	  NSLog(@"Unable to load the Filtering bundle.");
	}
    }
    break;
    
  default: 
    NSLog(@"huh?");
    break;
  }
}


//
// action methods
//

- (IBAction) cancelClicked: (id) sender
{
  [NSApp stopModal];
  [self close];
}

- (IBAction) saveAndClose: (id) sender
{
  [self savePreferences: nil];
  
  [NSApp stopModal];
  [self close];
}

- (IBAction) savePreferences: (id) sender
{
  NSArray *allNames;
  id<PreferencesModule> aModule;
  int i;

  allNames = [allModules allKeys];

  for (i = 0; i < [allNames count]; i++)
    {
      aModule = [allModules objectForKey: [allNames objectAtIndex: i]];
      [aModule saveChanges];
    }
  
  [[NSUserDefaults standardUserDefaults] synchronize];
  
  for (i = 0; i < [[GNUMail allMailWindows] count]; i++)
    {
      MailWindowController *aWindowController;
      
      aWindowController = (MailWindowController *)[[[GNUMail allMailWindows] 
						     objectAtIndex: i] windowController];
      [aWindowController reloadTableColumns];
    }
}


//
// other methods
//

- (void) addModuleToView: (id) aModule
{  
  if ( aModule == nil )
    {
      return;
    }

  if ( [box contentView] != [aModule view] )
    {
      [box setContentView: [aModule view]];
      [box setTitle: [aModule name]];
    }
}

- (void) initializeWithStandardModules
{
  [self _initializeModuleWithName: @"Personal" atIndex: 0];
  [self _initializeModuleWithName: @"Viewing" atIndex: 1];
  [self _initializeModuleWithName: @"Sending" atIndex: 2];
  [self _initializeModuleWithName: @"Receiving" atIndex: 3];
  [self _initializeModuleWithName: @"Compose" atIndex: 4];
  [self _initializeModuleWithName: @"Fonts" atIndex: 5];
  [self _initializeModuleWithName: @"MIME" atIndex: 6];
  [self _initializeModuleWithName: @"Filtering" atIndex: 7];
}

//
// access/mutation methods
//

- (NSMatrix *) matrix
{
  return matrix;
}


//
// class methods
//
+ (id) singleInstance
{
  if ( !singleInstance )
    {
      singleInstance = [[PreferencesWindowController alloc] initWithWindowNibName: @"PreferencesWindow"];

      // We select the first cell in our matrix
      [[singleInstance matrix] selectCellAtRow:0 column:0];
      [singleInstance handleCellAction: [singleInstance matrix]];
    }
  else
    {
      return nil;
    }

  return singleInstance;
}

@end


//
// Private interface
//
@implementation PreferencesWindowController (Private)

- (void) _initializeModuleWithName: (NSString *) theName
			   atIndex: (int) theIndex
{
  id<PreferencesModule> aModule;
  NSButtonCell *aButtonCell;

  aModule = [self _instanceForBundleWithName: theName];

  if (! aModule )
    {
      NSLog(@"Unable to initialize module %@", theName);
      return;
    }

  [allModules setObject: aModule
	      forKey: theName];

  
  aButtonCell = [matrix cellAtRow: 0
			column: theIndex];
      
  [aButtonCell setTag: theIndex];
  [aButtonCell setTitle: [aModule name]];
#ifdef MACOSX
  [aButtonCell setFont: [NSFont systemFontOfSize: 10]];
#else
  [aButtonCell setFont: [NSFont systemFontOfSize: 8]];
#endif
  [aButtonCell setImage: [aModule image]];
}


//
// Can be:  $GNUSTEP_INSTALATION_DIR/{Local,Network,System}/Apps/GNUMail.app/Resources or
//     or:  ANY_OTHER_DIRECTORY/GNUMail/GNUMail.app/Resources
//
- (id) _instanceForBundleWithName: (NSString *) theName
{
  NSString *aString;
  NSBundle *aBundle;
  Class aClass;

#ifdef MACOSX
  NSLog(@"bundle path = %@", [[NSBundle mainBundle] bundlePath]);
  aString = [[NSBundle mainBundle] pathForResource: theName 
				   ofType: @"prefs"
				   inDirectory: @"Bundles"];
  if (aString == nil)
    {
      aString = [NSHomeDirectory() stringByAppendingPathComponent: @"Library/GNUMail/Bundles/"];
      aString = [NSString stringWithFormat: @"%@/%@.prefs", aString, theName];
    }
#else // MACOSX  

  aString = [[[NSBundle mainBundle] resourcePath] stringByDeletingLastPathComponent];

  // We verify if we must load the bundles in the GNUstep's Local, Network or System dir.
  // We must also be careful in case poeple are using GNUstep with --enable-flattened
  if ( [aString hasSuffix: @"/Applications/GNUMail.app"] )
    {
      aString = [NSString stringWithFormat: @"%@/Library/GNUMail/%@.prefs",
			  [[aString stringByDeletingLastPathComponent] 
			    stringByDeletingLastPathComponent],
			  theName];
    }
  else
    {
      aString = [NSString stringWithFormat: @"%@/Bundles/%@/%@.prefs",
			  [aString stringByDeletingLastPathComponent],
			  theName, theName];
    }
#endif // MACOSX
  
  NSLog(@"Loading bundle at path %@", aString);

  aBundle = [NSBundle bundleWithPath: aString];

  aClass = [aBundle principalClass];
  
  if ( [aClass conformsToProtocol: @protocol(PreferencesModule)] )
    {
      return AUTORELEASE([aClass singleInstance]);
    }

  // Reached in case of an error.
  return nil;
}

@end
