/*
**  EditWindowController.m
**
**  Copyright (c) 2001, 2002
**
**  Ludovic Marcotte <ludovic@Sophos.ca>
**  Jonathan B. Leffert <jonathan@leffert.net>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "EditWindowController.h"

#import "Address.h"
#import "AddressBookController.h"

#ifndef MACOSX
#import "EditWindow.h"
#endif

#import "ExtendedAttachmentCell.h"
#import "GNUMail.h"
#import "GNUMailConstants.h"
#import "LabelWidget.h"
#import "MailWindowController.h"
#import "MimeTypeManager.h"
#import "MimeType.h"
#import "NSStringExtensions.h"
#import "PasswordPanelController.h"
#import "Utilities.h"

#import <Pantomime/Charset.h>
#import <Pantomime/Constants.h>
#import <Pantomime/LocalFolder.h>
#import <Pantomime/LocalStore.h>
#import <Pantomime/Message.h>
#import <Pantomime/MimeUtility.h>
#import <Pantomime/Part.h>
#import <Pantomime/Sendmail.h>
#import <Pantomime/SMTP.h>

@implementation EditWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifndef MACOSX
  EditWindow *editWindow;
#endif

  // We first verify if we have at least one transport agent defined
  if (! [[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"] ||
      ! [(NSArray *)[[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"] count] )
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"You must have at least one transport agent defined.\nSee Preferences -> Sending."),
		      _(@"OK"), // default
		      NULL,     // alternate
		      NULL);
      
      AUTORELEASE(self);
      return nil;
    }
  

#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
  
#else
  
  editWindow = [[EditWindow alloc] initWithContentRect: NSMakeRect(50,75,650,520)
				   styleMask: (NSClosableWindowMask|NSTitledWindowMask |
					       NSMiniaturizableWindowMask|NSResizableWindowMask)
				   backing: NSBackingStoreRetained
				   defer: NO];
    
  self = [super initWithWindow: editWindow];
  [editWindow layoutWindow];
  [editWindow setDelegate: self];
  //[editWindow setFrame:[Utilities rectForEditWindow] display: NO];
  
  // We link our outlets
  subjectText = [editWindow subjectText];
  toText = [editWindow toText];
  ccText = [editWindow ccText];
  bccText = [editWindow bccText];

  subjectLabel = [editWindow subjectLabel];
  toLabel = [editWindow toLabel];
  ccLabel = [editWindow ccLabel];
  bccLabel = [editWindow bccLabel];

  scrollView = [editWindow scrollView];
  textView = [editWindow textView];

  icon = [editWindow icon];
  send = [editWindow send];
  insert = [editWindow insert];
  addBcc = [editWindow addBcc];
  addCc = [editWindow addCc];
  addresses = [editWindow addresses];
  cancel = [editWindow cancel];
  
  personalProfilePopUpButton = [editWindow personalProfilePopUpButton];
  transportMethodPopUpButton = [editWindow transportMethodPopUpButton];

  RELEASE(editWindow);
#endif

  // We set our window title and edited attribute
  [[self window] setTitle: @""];
  [[self window] setDocumentEdited: NO];
  
  // We load our personal profiles and our transport methods
  [Utilities loadPersonalProfilesInPopUpButton: personalProfilePopUpButton];
  [Utilities loadTransportMethodsInPopUpButton: transportMethodPopUpButton];

  [self personalProfilesSelectionHasChanged: nil];

  return self;
}


//
//
//
- (void) dealloc
{
  NSLog(@"EditWindowController: -dealloc");
  
  // We remove our observer for our two notifications
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: PersonalProfilesHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: TransportMethodsHaveChanged
    object: nil];

  TEST_RELEASE(mimeTypeManager);

  TEST_RELEASE(message);
  TEST_RELEASE(unmodifiedMessage);

  // We release our NSConnection object and our ports
  RELEASE(connection);
  RELEASE(ports);

  // We release our passwordCache
  RELEASE(passwordCache);

  [super dealloc];
}


//
// Implementation of the AddressTaker protocol.
//
- (void) takeToAddress: (Address *) theAddress
{
  [Utilities appendAddress: theAddress
	     toTextField: toText];
}


//
//
//
- (void) takeCcAddress: (Address *) theAddress
{
  if ( ! [self showCc] )
    {
      [self showCc: self];
    }
  
  [Utilities appendAddress: theAddress
	     toTextField: ccText];
}


//
//
//
- (void) takeBccAddress: (Address *) theAddress
{
  if ( ! [self showBcc] )
    {
      [self showBcc: self];
    }
  
  [Utilities appendAddress: theAddress
	     toTextField: bccText];
}


//
// action methods
//


//
//
//
- (IBAction) insertFile: (id) sender
{
  NSOpenPanel *oPanel;
  int result;
  
  oPanel = [NSOpenPanel openPanel];
  [oPanel setAllowsMultipleSelection:YES];
  result = [oPanel runModalForDirectory: [GNUMail currentWorkingPath] file:nil types:nil];
  
  if (result == NSOKButton)
    {
      ExtendedAttachmentCell *cell;
      NSTextAttachment *attachment;
      NSFileWrapper *filewrapper;
      NSEnumerator *filesToOpenEnumerator = [[oPanel filenames] objectEnumerator];
      NSString *theFilename;
      id encodedAttatchment;
      
      while ( (theFilename = [filesToOpenEnumerator nextObject]) )
	{
	  filewrapper = [[NSFileWrapper alloc] initWithPath: theFilename];
	  AUTORELEASE(filewrapper);

          // We save the path of the last file
          [GNUMail setCurrentWorkingPath: [theFilename stringByDeletingLastPathComponent]];
	  
	  // We set the icon of the attachment if the mime-type is found
	  if ( mimeTypeManager )
	    {
	      MimeType *aMimeType;

	      aMimeType = [mimeTypeManager mimeTypeForFileExtension:
					     [[theFilename lastPathComponent]
					       pathExtension]];
	      
	      // If we found a MIME-type for this attachment...
	      if ( aMimeType )
		{
		  [filewrapper setIcon: [aMimeType icon]];
		}
	    }
	  
	  // We now create our attachment from our filewrapper
	  attachment = [[NSTextAttachment alloc] initWithFileWrapper: filewrapper];

	  cell = [[ExtendedAttachmentCell alloc] initWithFilename: [theFilename lastPathComponent]
						 size: [[filewrapper regularFileContents] length]];
	  
	  [attachment setAttachmentCell: cell];
          
          // Cocoa bug
#ifdef MACOSX
          [cell setAttachment: attachment];
          [cell setImage: [filewrapper icon]];
#endif
      	  RELEASE(cell);

	  encodedAttatchment = [NSAttributedString attributedStringWithAttachment: attachment];
	  RELEASE(attachment);
	  
	  if ( encodedAttatchment )
	    {
	      [textView insertText: encodedAttatchment];
	    }
	}
      
      [[self window] makeFirstResponder: textView ];
    }
}


//
//
//
- (IBAction) showBcc: (id) sender
{    
  [self setShowBcc: ![self showBcc]];
  [[[self window] contentView] setNeedsDisplay: YES];
}


//
//
//
- (IBAction) showCc: (id) sender
{ 
  [self setShowCc: ![self showCc]];
  [[[self window] contentView] setNeedsDisplay: YES];
}


//
//
//
- (IBAction) personalProfilesSelectionHasChanged: (id) sender
{
  NSArray *anArray;
  int i;

  // We synchronize our from selection from both popups
  [personalProfilePopUpButton synchronizeTitleAndSelectedItem];
  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];

  anArray = [[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"];
  
  for (i = 0; i < [anArray count]; i++)
    {
      NSDictionary *aDictionary;
      
      aDictionary = [anArray objectAtIndex: i];
      
      if ( [[aDictionary objectForKey: @"PERSONAL_PROFILE"] isEqualToString: 
							      [Utilities profileNameForItemTitle: 
									   [personalProfilePopUpButton titleOfSelectedItem]]] )
	{
	  [transportMethodPopUpButton selectItemAtIndex: i];
	  return;
	}
    }
  
  [transportMethodPopUpButton selectItemAtIndex: 0];
}


//
//
//
- (IBAction) updateAnimatedIcon: (id) sender
{
  if (animation_index == 9)
    {
      animation_index = 1;
    }
  
  [icon setImage: [NSImage imageNamed: [NSString stringWithFormat: @"anim-logo-%d.tiff", animation_index]]];
  [[self window] update];
  
  animation_index += 1;
}



//
//
//
- (IBAction) sendMessage : (id) sender
{
  NSDictionary *aDictionary;

  // We first try to initialize our message with all the information 
  if ( ![self updateMessageContentFromTextView: YES] )
    {
      return;
    }

  // We sync our popup
  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];

  // We first ask for our password and we cache it (if we need too)
  aDictionary = [[[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"]
		  objectAtIndex: [transportMethodPopUpButton indexOfSelectedItem]];
  
  if ( [aDictionary objectForKey: @"SMTP_AUTH"] &&
       [[aDictionary objectForKey: @"SMTP_AUTH"] intValue] == NSOnState &&
       [[aDictionary objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_SMTP )
    {
      [self _passwordForServerName: [aDictionary objectForKey: @"SMTP_HOST"]
	    prompt: YES];
    }


  // We disable our buttons
  [send setEnabled: NO];
  [insert setEnabled: NO];
  [addBcc setEnabled: NO];
  [addCc setEnabled: NO];
  [addresses setEnabled: NO];
  
  // We start our animation timer
  animation_index = 1;
  animation =  [NSTimer scheduledTimerWithTimeInterval: 0.1
			target: self
			selector: @selector(updateAnimatedIcon:)
			userInfo: nil
			repeats: YES];
  RETAIN(animation);

  //
  // We detach our thread that will send the message using a SMTP
  // connection of a mailer.
  //
  [NSThread detachNewThreadSelector: @selector(_sendMessageUsingPorts:)
	    toTarget: self
	    withObject: ports];
}


//
// delegate methods
//

- (void) controlTextDidChange: (NSNotification *) aNotification
{
  if ( [aNotification object] == subjectText )
    {
      [[self window] setTitle: [subjectText stringValue]];
    }
  
  [[self window] setDocumentEdited: YES];
}


- (void) textDidChange: (NSNotification *) aNotification
{
  [[self window] setDocumentEdited: YES];
}


- (BOOL) windowShouldClose: (id) sender
{
  // We first verify if the animation timer is running, if it is,
  // we are currently sending a mail so we inform the user about this.
  if ( animation && [animation isValid] )
    {
      NSRunInformationalAlertPanel(_(@"Closing..."),
				   _(@"A mail is currently being sent. Please wait."),
				   _(@"OK"), 
				   NULL, 
				   NULL);
      return NO;
    }


  if ( [[self window] isDocumentEdited] )
    {
      int choice;
      
      choice = NSRunAlertPanel(_(@"Closing..."),
			       _(@"Would you like to save this message in the Drafts folder?"),
			       _(@"Cancel"), // default
			       _(@"Yes"),    // alternate
			       _(@"No"));    // other return
      
      // We don't want to close the window
      if ( choice == NSAlertDefaultReturn )
	{
	  return NO;
	}
      // Yes we want to close it, and we also want to save the message to the Drafts folder.
      else if ( choice == NSAlertAlternateReturn )
	{
	  LocalStore *aLocalStore;
	  LocalFolder *draftsFolder;
	  id aWindow;
	  
	  aWindow = nil;
	  aLocalStore = [(GNUMail *)[NSApp delegate] localStore];
	  draftsFolder = (LocalFolder *)[aLocalStore folderForName:
						       [[NSUserDefaults standardUserDefaults] 
							 objectForKey: @"DRAFTSFOLDERNAME"]];
	  
	  // Then, we try to open our Outbox maibox, if it's already open, we get the window used by this folder
	  if (! draftsFolder )
	    {
	      aWindow = [Utilities windowForFolderName: [[NSUserDefaults standardUserDefaults] 
							  objectForKey: @"DRAFTSFOLDERNAME"]];
	      draftsFolder = (LocalFolder *)[[aWindow windowController] folder];
	      
	    } // if (! draftsFolder )
	  
	  
	  [self updateMessageContentFromTextView: NO];
	  
	  // We create a copy of the message in Drafts and we close this folder after
	  [draftsFolder appendMessageFromRawSource: [[self message] dataUsingSendingMode: SEND_TO_FOLDER]];
	  
	  
	  if ( aWindow )
	    {
	      [[aWindow windowController] tableViewShouldReloadData];
	    }
	  else
	    {
	      [draftsFolder close];
	    }
	}
    }
  
  return YES;
}


- (void) windowWillClose: (NSNotification *) theNotification
{ 
  if ( [GNUMail lastAddressTakerWindowOnTop] == self )
    {
      [GNUMail setLastAddressTakerWindowOnTop: nil];
    }

  // We save our frame rect
  [Utilities saveRect: [[self window] frame]
	     forWindowName: @"RECT_EDITWINDOW"];
      
  [[NSApp delegate] setEnableSaveInDraftsMenuItem: NO];

  AUTORELEASE(self);
}

- (void) windowDidBecomeMain: (NSNotification *) theNotification
{
  [GNUMail setLastAddressTakerWindowOnTop: self];

  [[NSApp delegate] setEnableSaveInDraftsMenuItem: YES];
}

- (void) windowDidLoad
{
  NSPort *port1, *port2;
  
#ifdef MACOSX
  [addresses setTarget: [NSApp delegate]];
  [addresses setAction: @selector(showAddressBook:)];
#endif

  // We initialize our password cache
  passwordCache = [[NSMutableDictionary alloc] init];

  // We initialize our NSConnection object and our ports
  port1 = [[NSPort alloc] init];
  port2 = [[NSPort alloc] init];
  
  connection = [[NSConnection alloc] initWithReceivePort: port1
				     sendPort: port2];
  [connection setRootObject: self];

  ports = [NSArray arrayWithObjects: port2, port1, nil];

  RETAIN(ports);
  RELEASE(port1);
  RELEASE(port2);

  // We initialize our variables
  [self setMessageWasSent: NO];
  [self setShowCc: NO];
  [self setShowBcc: NO];

  [self _loadMimeTypes];
  [self setUnmodifiedMessage: nil];

  [self setSignaturePosition: SIGNATURE_END];

  // We add our observer for our two notifications
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_loadPersonalProfiles)
    name: PersonalProfilesHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_loadTransportMethods)
    name: TransportMethodsHaveChanged
    object: nil];
}


//
// access/mutation methods
//

//
//
//
- (BOOL) messageWasSent
{
  return messageWasSent;
}


//
//
//
- (void) setMessageWasSent: (BOOL) theBOOL
{
  messageWasSent = theBOOL;
}


//
//
//
- (Message *) message
{
  return message;
}

//
//
//
- (void) setMessage: (Message *) theMessage
{  
  if (theMessage)
    {
      RETAIN(theMessage);
      RELEASE(message);
      message = theMessage;
      
      [self _updateViewWithMessage: message
	    appendSignature: YES];
    }
  else
    {
      RELEASE(message);
      message = nil;
    }
}


- (Message *) unmodifiedMessage
{
  return unmodifiedMessage;
}


- (void) setUnmodifiedMessage: (Message *) theUnmodifiedMessage
{
  if ( theUnmodifiedMessage )
    {
      RETAIN(theUnmodifiedMessage);
      RELEASE(unmodifiedMessage);
      unmodifiedMessage = theUnmodifiedMessage;
    }
  else
    {
      RELEASE(unmodifiedMessage);
      unmodifiedMessage = nil;
    }
}


- (void) setMessageFromDraftsFolder: (Message *) theMessage
{
  if (theMessage)
    {
      RETAIN(theMessage);
      RELEASE(message);
      message = theMessage;
      
      [self _updateViewWithMessage: message
	    appendSignature: NO];
    }
  else
    {
      RELEASE(message);
      message = nil;
    }
}

//
//
//
- (BOOL) showCc
{
  return showCc;
}


//
//
//
- (void) setShowCc: (BOOL) theBOOL
{
  showCc = theBOOL;
  
  if (showCc)
    {
#ifdef MACOSX
      [ccText setEnabled: YES];
#else
      [[[self window] contentView] addSubview: ccLabel];
      [[[self window] contentView] addSubview: ccText];
#endif
      [addCc setTitle: _(@"Remove Cc")];
      [addCc setImage: [NSImage imageNamed: @"MailIcon_cc_remove.tiff"]];
    }
  else
    {
#ifdef MACOSX
      [ccText setEnabled: NO];
#else
      [ccLabel removeFromSuperviewWithoutNeedingDisplay];
      [ccText removeFromSuperviewWithoutNeedingDisplay];
#endif
      [addCc setTitle: _(@"Add Cc")];
      [addCc setImage: [NSImage imageNamed: @"MailIcon_cc.tiff"]];
    }
    
#ifndef MACOSX
  [self _adjustWidgetsPosition];
#endif
}

//
//
//
- (BOOL) showBcc
{
  return showBcc;
}


//
//
//
- (void) setShowBcc: (BOOL) theBOOL
{
  showBcc = theBOOL;
 
  if (showBcc)
    {
#ifdef MACOSX
      [bccText setEnabled: YES];
#else
      [[[self window] contentView] addSubview: bccLabel];
      [[[self window] contentView] addSubview: bccText];
#endif
      [addBcc setTitle: _(@"Remove Bcc")];
      [addBcc setImage: [NSImage imageNamed: @"MailIcon_bcc_remove.tiff"]];
    }
  else
    {
#ifdef MACOSX
      [bccText setEnabled: NO];
#else
      [bccLabel removeFromSuperviewWithoutNeedingDisplay];
      [bccText removeFromSuperviewWithoutNeedingDisplay];
#endif
      [addBcc setTitle: _(@"Add Bcc")];
      [addBcc setImage: [NSImage imageNamed: @"MailIcon_bcc.tiff"]];
    }

#ifndef MACOSX  
  [self _adjustWidgetsPosition];
#endif
}


//
//
//
- (int) signaturePosition
{
  return signaturePosition;
}

//
//
//
- (void) setSignaturePosition: (int) thePosition
{
  signaturePosition = thePosition;
}


//
// other methods
//

- (void) authenticationFailedForServer: (NSString *) theServer
{
  NSRunAlertPanel(_(@"Error!"),
		  _(@"An error occured during the authentication with the SMTP server (%@)."),
		  _(@"OK"),
		  NULL,
		  NULL,
		  theServer);
  
  // We re-enable our buttons
  [send setEnabled: YES];
  [insert setEnabled: YES];
  [addBcc setEnabled: YES];
  [addCc setEnabled: YES];
  [addresses setEnabled: YES];

  // We stop our animation timer
  [animation invalidate];
  DESTROY(animation);
}

- (void) errorOccuredWhenSendingMessage
{
  NSRunAlertPanel(_(@"Error!"),
		  _(@"An error occured while sending the E-Mail. It might be a network\nproblem or an error in your sending preferences."),
		  _(@"OK"),
		  NULL,
		  NULL);
  
  // We re-enable our buttons
  [send setEnabled: YES];
  [insert setEnabled: YES];
  [addBcc setEnabled: YES];
  [addCc setEnabled: YES];
  [addresses setEnabled: YES];

  // We stop our animation timer
  [animation invalidate];
  DESTROY(animation);
}


//
//
//
- (void) messageWasSentFromRAWSource: (NSData *) theRAWSource
			      window: (id) theWindow
                        outboxFolder: (Folder *) theFolder
{
  // We create a copy of the message in Outbox and we close this folder after
  [theFolder appendMessageFromRawSource: theRAWSource];
  
  if ( theWindow )
    {
      [[theWindow windowController] tableViewShouldReloadData];
    }

  // We send the notification to inform our MailWindowController that the replied message has been
  // sucessfully sent so we can set the flag to 'ANSWERED'.
  if ( [self unmodifiedMessage] )
    {
      [[NSNotificationCenter defaultCenter]
	postNotificationName: ReplyToMessageWasSuccessful
	object: nil
	userInfo: [NSDictionary dictionaryWithObject: [self unmodifiedMessage] 
				forKey: @"Message"]];
    }

  // We stop our animation timer
  [animation invalidate];
  DESTROY(animation);

  [self close];
}

//
//
//
- (BOOL) updateMessageContentFromTextView: (BOOL) verifyRecipients
{

  NSArray *to_array, *cc_array, *bcc_array;
  NSTextStorage *textStorage;
 
  NSDictionary *allAdditionalHeaders, *aProfile;
 
  InternetAddress *anInternetAddress;
  BOOL hasFoundXMailerHeader;
  NSString *aString;
  int i;
  
  if ( verifyRecipients )
    {
      // We verify that at least one recipient has been defined in To:
      if ( [[[toText stringValue] stringByTrimmingWhiteSpaces] length] == 0 )
	{
	  NSRunAlertPanel(_(@"Error!"),
			  _(@"You must define a recipient in the field \"To\"."),
			  _(@"OK"), // default
			  NULL,     // alternate
			  NULL);
	  
	  return NO;
	}
    }  
  
  // We initialize our boolean value to false
  hasFoundXMailerHeader = NO;
  
  // We get the current selected profile when sending this email
  [personalProfilePopUpButton synchronizeTitleAndSelectedItem];
  
  // Then, we get our profile from our user defaults
  aProfile = [[[NSUserDefaults standardUserDefaults] objectForKey: @"PERSONAL"]
	       objectForKey: [Utilities profileNameForItemTitle: [personalProfilePopUpButton titleOfSelectedItem]] ];
  
  // We set the From header value
  anInternetAddress = [[InternetAddress alloc] initWithPersonal: [aProfile objectForKey: @"NAME"]
					       andAddress: [aProfile objectForKey: @"EMAILADDR"]];
  [message setFrom: anInternetAddress];
  RELEASE(anInternetAddress);
  
  // We set the Reply-To address (if we need too)
  aString = [aProfile objectForKey: @"REPLYTOADDR"];
  
  if (aString && [[aString stringByTrimmingWhiteSpaces] length] > 0)
    {
      anInternetAddress = [[InternetAddress alloc] initWithString: aString];
      
      if (anInternetAddress)
	{
	  [message setReplyTo: anInternetAddress];
	}
    }
  else
    {
      [message setReplyTo: nil];
    }
  
  // We set the Organization header value (if we need to)
  aString = [aProfile objectForKey: @"ORGANIZATION"];
  
  if (aString && [[aString stringByTrimmingWhiteSpaces] length] > 0)
    {
      [message setOrganization: aString];
    }

  // For now, if there are recipients, we remove them in case a user add or changed them manually
  if ([message recipientsCount] > 0) 
    {
      [message removeAllRecipients];
    }

  // We decode our recipients, let's begin with the To field.
  if ( [[[toText stringValue] stringByTrimmingWhiteSpaces] length] > 0 )
    {
      to_array = [self _recipientsFromString: [toText stringValue] ];
      
      for (i = 0; i < [to_array count]; i++)
	{
	  anInternetAddress = [[InternetAddress alloc] initWithString: [to_array objectAtIndex:i]];
	  
	  if (! anInternetAddress )
	    {
	      [self _showAlertPanelWithString: [to_array objectAtIndex: i] ];
	      return NO;
	    }
	  
	  [anInternetAddress setType: TO];
	  [message addToRecipients: anInternetAddress];
	  RELEASE(anInternetAddress);
	}
    }

  // We decode our Cc field, if we need to
  if ( showCc )
    {
      cc_array = [self _recipientsFromString: [ccText stringValue]];
      
      for (i = 0; i < [cc_array count]; i++)
	{
	  anInternetAddress = [[InternetAddress alloc] initWithString: [cc_array objectAtIndex:i]];

	  if (! anInternetAddress )
	    {
	      [self _showAlertPanelWithString: [cc_array objectAtIndex: i] ];
	      return NO;
	    }

	  [anInternetAddress setType: CC];
	  [message addToRecipients: anInternetAddress];
	  RELEASE(anInternetAddress);
	}
      
    }

  // We decode our Bcc field, if we need to
  if ( showBcc )
    {
      bcc_array = [self _recipientsFromString: [bccText stringValue]];
      
      for (i = 0; i < [bcc_array count]; i++)
	{
	  anInternetAddress = [[InternetAddress alloc] initWithString: [bcc_array objectAtIndex:i]];
	  
	  if (! anInternetAddress )
	    {
	      [self _showAlertPanelWithString: [bcc_array objectAtIndex: i] ];
	      return NO;
	    }

	  [anInternetAddress setType: BCC];
	  [message addToRecipients: anInternetAddress];
	  RELEASE(anInternetAddress);
	}
    }

  
  // We decode our Subject field
  [message setSubject: [subjectText stringValue]];
  
  // We finally add all our addition headers from the user defaults
  allAdditionalHeaders = [[NSUserDefaults standardUserDefaults] 
			   objectForKey: @"ADDITIONALOUTGOINGHEADERS"];
  
  if ( allAdditionalHeaders )
    {
      NSEnumerator *anEnumerator;
      NSString *aKey, *aValue;

      anEnumerator = [allAdditionalHeaders keyEnumerator];
      
      while ( (aKey = [anEnumerator nextObject]) )
	{
	  // We get our value
	  aValue = [allAdditionalHeaders objectForKey: aKey];
	  
	  // We add our X- if we don't have it
	  if (! [aKey hasPrefix: @"X-"] )
	    {
	      aKey = [NSString stringWithFormat: @"X-%@", aKey];
	    }

	  if ( [aKey compare:@"X-Mailer" options:NSCaseInsensitiveSearch] == NSOrderedSame )
	    {
	      hasFoundXMailerHeader = YES;
	    }
	  
	  [message addHeader: aKey
		   withValue: aValue];
	}
    }

  // We add our X-Mailer: GNUMail.app (Version ...) header if the user hasn't defined one
  if ( !hasFoundXMailerHeader )
    {
      [message addHeader: @"X-Mailer"
	       withValue: [NSString stringWithFormat: @"GNUMail.app (Version %@)", GNUMAIL_VERSION] ];
    }

  /// We get our text storage
  textStorage = [textView textStorage];
  
  // We now build our message body - considering all the attachments
  if ( ! [textStorage containsAttachments] )
    {
      [self _setPlainTextContentFromString: [textView string]
	    inPart: message];
    }
  else
    {   
      MimeMultipart *aMimeMultipart;
      MimeBodyPart *aMimeBodyPart;      
      
      NSTextAttachment *aTextAttachment;
      NSAutoreleasePool *pool;

      aMimeMultipart = [[MimeMultipart alloc] init];
      
      // We first decode our text/plain body
      // FIXME - reuse the code when there's no attachment
      aMimeBodyPart = [[MimeBodyPart alloc] init];

      [self _setPlainTextContentFromString: [self _plainTextContentFromTextView]
	    inPart: aMimeBodyPart];

      // We add our new part
      [aMimeMultipart addBodyPart: aMimeBodyPart];
      RELEASE(aMimeBodyPart);
      
      pool = nil;

      // We finally add all our attachments
      for (i = 0; i < [textStorage length]; i++)
	{
	  if ( (i % 100) == 0 )
	    {
	      TEST_RELEASE(pool);
	      pool = [[NSAutoreleasePool alloc] init];
	    } 

	  aTextAttachment = [textStorage attribute: NSAttachmentAttributeName		      
					 atIndex: i 
					 effectiveRange: NULL];
	  if ( aTextAttachment ) 
	    {
	      Part *aPart;
	      
	      aPart = [(ExtendedAttachmentCell *)[aTextAttachment attachmentCell] part];
	      
	      if ( aPart )
		{
		  [aMimeMultipart addBodyPart: (MimeBodyPart *)aPart];
		}
	      else
		{
		  NSFileWrapper *aFileWrapper;
		  NSData *aData;
		  
		  aMimeBodyPart = [[MimeBodyPart alloc] init];
		  
		  aFileWrapper = [aTextAttachment fileWrapper];
		  [aMimeBodyPart setFilename: [[aFileWrapper filename] lastPathComponent]];
		  
		  // We search for the content-type to use 
		  if ( mimeTypeManager )
		    {
		      MimeType *aMimeType;

		      aMimeType = [mimeTypeManager bestMimeTypeForFileExtension:
						     [[[aFileWrapper filename] lastPathComponent]
						       pathExtension]];
		      if ( aMimeType )
			{
			  [aMimeBodyPart setContentType: [aMimeType mimeType]];
			}
		      else
			{
			  [aMimeBodyPart setContentType: @"application/octet-stream"];
			}
		    }
		  else
		    {
		      [aMimeBodyPart setContentType: @"application/octet-stream"];
		    }
		  
		  [aMimeBodyPart setContentTransferEncoding: BASE64]; // always base64 encoding for now
		  aData = [aFileWrapper regularFileContents];
		  
		  [aMimeBodyPart setContent: aData];
		  [aMimeMultipart addBodyPart: aMimeBodyPart];
		  RELEASE(aMimeBodyPart);
		}

	    } // if ( aTextAttachment ) 

	} // for (...)
      
      [message setContentType: @"multipart/mixed"];     
      [message setContent: aMimeMultipart];

      RELEASE(aMimeMultipart);
      TEST_RELEASE(pool);
    }

  return YES;
}

@end

//
// private methods
//
@implementation EditWindowController (Private)

- (void) _adjustWidgetsPosition
{
  if (showCc && showBcc)
    {
      //NSLog(@"showCc is YES showBcc is YES");
      [ccLabel setFrame: NSMakeRect(5,360,55,21)];
      [ccText setFrame: NSMakeRect(65,360,565,21)];
      
      [bccLabel setFrame: NSMakeRect(5,335,55,21)];
      [bccText setFrame: NSMakeRect(65,335,565,21)];
      
      [subjectLabel setFrame: NSMakeRect(5,310,55,21)];
      [subjectText setFrame: NSMakeRect(65,310,565,21)];
      
      [scrollView setFrame: NSMakeRect(5,5,640,285)];
      [[textView textContainer] setContainerSize: NSMakeSize([[scrollView contentView] frame].size.width, 
							     1E7)];
      
      [toText setNextKeyView: ccText];
      [ccText setNextKeyView: bccText];
      [bccText setNextKeyView: subjectText];
    }	 
  else if (showCc && !showBcc)
    {
      //NSLog(@"showCc is YES showBcc is NO");
      [ccLabel setFrame: NSMakeRect(5,360,55,21)];
      [ccText setFrame: NSMakeRect(65,360,565,21)];
      
      [subjectLabel setFrame: NSMakeRect(5,335,55,21)];
      [subjectText setFrame: NSMakeRect(65,335,565,21)];
      
      [scrollView setFrame: NSMakeRect(5,5,640,310)];
      [[textView textContainer] setContainerSize: NSMakeSize([[scrollView contentView] frame].size.width, 
							     1E7)];
      
      [toText setNextKeyView: ccText];
      [ccText setNextKeyView: subjectText];
    }
  else if (!showCc && showBcc)
    {
      //NSLog(@"showCc is NO showBcc is YES");
      [bccLabel setFrame: NSMakeRect(5,360,55,21)];
      [bccText setFrame: NSMakeRect(65,360,565,21)];
      
      [subjectLabel setFrame: NSMakeRect(5,335,55,21)];
      [subjectText setFrame: NSMakeRect(65,335,565,21)];
      
      [scrollView setFrame: NSMakeRect(5,5,640,310)];
      [[textView textContainer] setContainerSize: NSMakeSize([[scrollView contentView] frame].size.width, 
							     1E7)];

      [toText setNextKeyView: bccText];
      [bccText setNextKeyView: subjectText];
    }
  else
    {
      //NSLog(@"showCc is NO showBcc is NO");
      [subjectLabel setFrame: NSMakeRect(5,360,55,21)];
      [subjectText setFrame: NSMakeRect(65,360,565,21)];
     
      [scrollView setFrame: NSMakeRect(5,5,640,335)];
      [[textView textContainer] setContainerSize: NSMakeSize([[scrollView contentView] frame].size.width, 
							     1E7)];
      
      [toText setNextKeyView: subjectText];
    }
}

- (void) _loadPersonalProfiles
{
  [Utilities loadPersonalProfilesInPopUpButton: personalProfilePopUpButton];
}

- (void) _loadTransportMethods
{
  [Utilities loadTransportMethodsInPopUpButton: transportMethodPopUpButton];
}

- (void) _loadSignature
{
  NSString *aSignature;
  NSMutableDictionary *tAttr;       // text attributes

  tAttr = [[NSMutableDictionary alloc] init];
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"MESSAGE_FONT_NAME"] )
    {
      [tAttr setObject: [NSFont fontWithName: [[NSUserDefaults standardUserDefaults] stringForKey: @"MESSAGE_FONT_NAME"]
                                size: [[NSUserDefaults standardUserDefaults] floatForKey: @"MESSAGE_FONT_SIZE"]]
             forKey: NSFontAttributeName];
    }
  else
    {
      [tAttr setObject: [NSFont userFixedPitchFontOfSize: 0] forKey:NSFontAttributeName];
    }

  // If we don't want a signature...
  if ( [self signaturePosition] == SIGNATURE_HIDDEN )
    {
      return;
    }

  if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"SIGNATURE_SOURCE"] == 0 )
    {
      aSignature = [NSString stringWithContentsOfFile:
			       [[NSUserDefaults standardUserDefaults] objectForKey: @"SIGNATURE"]];
    }
  else
    {
      NSFileHandle *aFileHandle;
      NSTask *aTask;
      NSPipe *aPipe;
      NSData *aData;
      NSRange aRange;
      NSString *aString;
      
      // We get our program's name (and arguments, if any)
      aString = [[NSUserDefaults standardUserDefaults] objectForKey: @"SIGNATURE"];

      // If a signature hasn't been set, let's return.
      if (! aString )
	{
	  return;
	}
            
      aPipe = [NSPipe pipe];
      aFileHandle = [aPipe fileHandleForReading];
      
      aTask = [[NSTask alloc] init];
      [aTask setStandardOutput: aPipe];
      
      // We trim our string from any whitespaces
      aString = [aString stringByTrimmingWhiteSpaces];

      // We verify if our program to lauch has any arguments
      aRange = [aString rangeOfString: @" "];
      
      if (aRange.length)
	{
	  [aTask setLaunchPath: [aString substringToIndex: aRange.location]];
	  [aTask setArguments: [NSArray arrayWithObjects: [aString substringFromIndex: (aRange.location + 1)],
					nil]];
	}
      else
	{
	  [aTask setLaunchPath: aString];
	}
      
      // We verify if our launch path points to an executable file
      if ( ![[NSFileManager defaultManager] isExecutableFileAtPath: [aTask launchPath]] )
	{
	  NSLog(@"The signature's path doesn't point to an executable! Ignored.");
	  RELEASE(aTask);
	  return;
	}
      
      
      // We launch our task
      [aTask launch];
      
      while ( [aTask isRunning] )
	{
	  [[NSRunLoop currentRunLoop] runMode: NSDefaultRunLoopMode
				      beforeDate: [NSDate distantFuture]];
	}
      
      aData = [aFileHandle readDataToEndOfFile];
      
      aSignature = [[NSString alloc] initWithData: aData
				  encoding: NSASCIIStringEncoding];
      
      RELEASE(aTask);
    }
  
  if ( aSignature )
    {
      if ( [self signaturePosition] == SIGNATURE_BEGINNING )
	{
	  NSAttributedString *anAttributedString;

	  anAttributedString = [[NSAttributedString alloc] initWithString: 
							     [NSString stringWithFormat: @"\n\n-- \n%@\n", aSignature]
							   attributes: tAttr];
	  
	  [[textView textStorage] insertAttributedString: anAttributedString
				  atIndex: 0];
	  
	  RELEASE(anAttributedString);
	}
      else if ( [self signaturePosition] == SIGNATURE_END )
	{
	  NSAttributedString *anAttributedString;

	  anAttributedString = [[NSAttributedString alloc] initWithString:
							     [NSString stringWithFormat: @"\n\n-- \n%@", aSignature]
							   attributes: tAttr];
							     
	  [[textView textStorage] appendAttributedString: anAttributedString];
	  
	  RELEASE(anAttributedString);
	}
    }
  RELEASE(tAttr);
}
  
- (void) _loadMimeTypes
{
  RELEASE(mimeTypeManager);

  mimeTypeManager = [MimeTypeManager mimeTypesFromDisk];
  
  if ( mimeTypeManager )
    {
      RETAIN(mimeTypeManager);
    }
}


//
//
//
- (NSString *) _passwordForServerName: (NSString *) theName
			       prompt: (BOOL) aBOOL
{
  NSDictionary *aDictionary;
  NSString *password;
  
  //
  // We get our right transport method
  // We don't need to verify if it's a SMTP transport because this method isn't called
  // if it's NOT SMTP.
  //
  aDictionary = [[[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"]
		  objectAtIndex: [transportMethodPopUpButton indexOfSelectedItem]];

  if ( aDictionary )
    {    
      password = nil;
      
      // We verify in the user defaults
      if ( [[aDictionary objectForKey: @"REMEMBERPASSWORD"] intValue] == NSOnState )
	{
	  password = [Utilities decryptPassword: [aDictionary objectForKey: @"SMTP_PASSWORD"] 
				withKey: theName];
	}

      // We verify in our cache
      if (! password )
	{
	  password = [passwordCache objectForKey: theName];
	}
      
      // If we must prompt for the password
      if (! password && aBOOL )
	{
	  PasswordPanelController *theController;
	  int result;
	  
	  theController = [[PasswordPanelController alloc] initWithWindowNibName: @"PasswordPanel"];
	  [[theController window] setTitle: theName];
      
	  result = [NSApp runModalForWindow: [theController window]];
	  
	  // If the user has entered a password...
	  if (result == NSRunStoppedResponse)
	    {
	      password = [theController password];
	      
	      // Let's cache this password...
	      [passwordCache setObject: password
			     forKey: theName];
	    }
	  else
	    {
	      password = nil;
	    }
	  
	  RELEASE(theController);
	}
    }
  else
    {
      password = nil;
    }
  
  return password;
}

//
//
//
- (NSString *) _plainTextContentFromTextView
{
  NSAttributedString *anAttributedString;
  NSTextStorage *textStorage;
  NSTextAttachment *attachment;
  NSMutableString *aMutableString;

  NSAutoreleasePool *pool;
  int i;
  
  textStorage = [textView textStorage];

  aMutableString = [[NSMutableString alloc] init];
  pool = nil;

  for (i = 0; i < [textStorage length]; i++)
    {
      if ( (i % 100) == 0 )
	{
	  TEST_RELEASE(pool);
	  pool = [[NSAutoreleasePool alloc] init];
	} 
      
      attachment = [textStorage attribute: NSAttachmentAttributeName
				atIndex: i
				effectiveRange: NULL];
      
      if ( ! attachment )
	{
	  anAttributedString = [textStorage attributedSubstringFromRange: NSMakeRange(i,1)];
	  [aMutableString appendString: [anAttributedString string]];
	}
      else 
	{
	  ExtendedAttachmentCell *cell;
	  
	  cell = (ExtendedAttachmentCell *)[attachment attachmentCell];
	  
	  if ( [cell part] && [[cell part] filename] )
	    {
	      [aMutableString appendFormat: @"<%@>", 
			      [[cell part] filename]];
	    }
	  else if ( [cell part] )
	    {
	      [aMutableString appendString: @"<unknown>"];
	    }
	  else
	    {
	      [aMutableString appendFormat: @"<%@>", 
			      [[[attachment fileWrapper] filename] lastPathComponent]];
	    }
	}
    }
  
  RELEASE(pool);
  
  return AUTORELEASE(aMutableString);
}

//
//
//
- (NSArray *) _recipientsFromString: (NSString *) theString
{
  NSMutableArray *returnArray;

  returnArray = [NSMutableArray array];

  if ( [[theString stringByTrimmingWhiteSpaces] length] > 0 )
    {
      NSArray *components = [theString componentsSeparatedByString: @","];
      NSEnumerator *componentsEnumerator;
      NSString *theComponent;

      componentsEnumerator = [components objectEnumerator];
      while ( (theComponent = [componentsEnumerator nextObject]) &&
	      ([theComponent length]) )
	{
	  theComponent = [theComponent stringByTrimmingWhiteSpaces];
	  if ( [theComponent length] )
	    {
	      [returnArray addObject: theComponent];
	    }
	}
    }

  return returnArray;
}


//
//
//
- (void) _sendMessageUsingPorts: (NSArray *) thePorts
{
  NSAutoreleasePool *pool;

  LocalStore *aLocalStore;
  LocalFolder *outboxFolder;
  id aWindow = nil;
  
  NSConnection *serverConnection;
  
  NSDictionary *aDictionary;

  // We first create our autorealease pool for our thread
  pool = [[NSAutoreleasePool alloc] init];

  // We first get our connection to our main thread
  serverConnection = [[NSConnection alloc] initWithReceivePort: [thePorts objectAtIndex: 0]
					   sendPort: [thePorts objectAtIndex: 1]];


  aLocalStore = [(GNUMail *)[NSApp delegate] localStore];
  outboxFolder = (LocalFolder *)[aLocalStore folderForName:
					       [[NSUserDefaults standardUserDefaults] 
						 objectForKey: @"OUTBOXFOLDERNAME"]];

  // Then, we try to open our Outbox maibox, if it's already open, we get the window used by this folder
  if (! outboxFolder )
    {
      aWindow = [Utilities windowForFolderName: [[NSUserDefaults standardUserDefaults] 
						  objectForKey: @"OUTBOXFOLDERNAME"]];
      outboxFolder = (LocalFolder *)[[aWindow windowController] folder];
    }
 
  // Then we really send our message using the right delivery method!
  aDictionary = [[[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"]
		  objectAtIndex: [transportMethodPopUpButton indexOfSelectedItem]];

  if ( [[aDictionary objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_SMTP )
    {
      SMTP *aSMTP;
      
      aSMTP = [[SMTP alloc] initWithName: [aDictionary objectForKey: @"SMTP_HOST"]
			    port: 25];

      // We verify if we need to authenticate ourself to the SMTP sever
      if ( [aDictionary objectForKey: @"SMTP_AUTH"] &&
	   [[aDictionary objectForKey: @"SMTP_AUTH"] intValue] == NSOnState )
	{
	  NSString *password;
	  BOOL aBOOL;
	 
	  password = [self _passwordForServerName: [aDictionary objectForKey: @"SMTP_HOST"]
			   prompt: NO];
	  aBOOL = NO;

	  if ( password )
	    {
	      aBOOL = [aSMTP authenticateWithUsername: [aDictionary objectForKey: @"SMTP_USERNAME"]
			     password: password
			     mechanism: [aDictionary objectForKey: @"SMTP_AUTH_MECHANISM"]];
	    }
	  
	  if ( !aBOOL )
	    {
	      NSLog(@"Failed to authenticate...");
	      [aSMTP close];
	      RELEASE(aSMTP);

	      [(id)[serverConnection rootProxy] authenticationFailedForServer: [aDictionary objectForKey: @"SMTP_HOST"]];
	      goto done;
	    }
	}

      if ( aSMTP )
	{
	  messageWasSent = [aSMTP sendMessage: message];
	  [aSMTP close];
	}
      else
	{
	  messageWasSent = NO;
	}
      
      RELEASE(aSMTP);
    }
  else 
    {
      Sendmail *aSendmail;

      aSendmail = [[Sendmail alloc] initWithPathToSendmail: [aDictionary objectForKey: @"MAILER_PATH"]];
      messageWasSent = [aSendmail sendMessage: message];

      RELEASE(aSendmail);
    }

  if ( messageWasSent )
    {
      [(id)[serverConnection rootProxy] messageWasSentFromRAWSource:
	     [message dataUsingSendingMode: SEND_TO_FOLDER]
	   window: aWindow
	   outboxFolder: outboxFolder];
    }
  else 
    {
      [(id)[serverConnection rootProxy] errorOccuredWhenSendingMessage];
    }

 done:

  if ( !aWindow )
    {
      [outboxFolder close];
    }

  RELEASE(serverConnection);
  RELEASE(pool);

  NSLog(@"Exiting the thread...");
  [NSThread exit];
}

//
//
//
- (void) _setPlainTextContentFromString: (NSString *) theString
                                 inPart: (Part *) thePart;
{
  // FIXME - We must handle properly format=flowed here...
  // We must first verify if our string contains any non-ascii character
  if ( [MimeUtility isASCIIString: theString] )
    {
      int wrapLimit;
      
      wrapLimit = [[NSUserDefaults standardUserDefaults] integerForKey: @"LINE_WRAP_LIMIT"];
      
      if (! wrapLimit)
	{
	  wrapLimit = 998;
	}
      
      [thePart setContentType: @"text/plain"];                // content-type
      [thePart setContentTransferEncoding: NONE];             // encoding -> none
      [thePart setCharset: @"us-ascii"];
      [thePart setFormat: FORMAT_FLOWED];
      [thePart setLineLength: wrapLimit];
      [thePart setContent: theString];
    }
  else
    {
      NSString *aCharset;

      // We get our charset currently used
      aCharset = [MimeUtility charsetForString: theString];
      
      [thePart setContentType: @"text/plain"];               // Content-Type
      [thePart setContentTransferEncoding: QUOTEDPRINTABLE]; // encoding -> quoted-printable
      [thePart setFormat: FORMAT_UNKNOWN];
      [thePart setCharset: aCharset];
      [thePart setContent: theString];
    }
}


//
//
//
- (void) _showAlertPanelWithString: (NSString *) theString
{
  NSRunInformationalAlertPanel(_(@"Error!"),
			       _(@"An error occured while decoding %@. Please fix this address."),
			       _(@"OK"),
			       NULL,
			       NULL,
			       theString);
}


//
//
//
- (void) _updateViewWithMessage: (Message *) theMessage
		appendSignature: (BOOL) aBOOL
{
  NSMutableString *aMutableString;
  InternetAddress *theRecipient;
  NSEnumerator *recipientsEnumerator;

  // We get all recipients and we set our To/Cc fields.
  aMutableString = [[NSMutableString alloc] init];
  recipientsEnumerator = [[theMessage recipients] objectEnumerator];
  
  while ( (theRecipient = [recipientsEnumerator nextObject]) )
    {  
      if ( [theRecipient type] == TO )
	{
	  [toText setStringValue:[theRecipient unicodeStringValue]];
	}
      else if ([theRecipient type] == CC) 
	{
	  [aMutableString appendString:[NSString stringWithFormat:@"%@, ", [theRecipient unicodeStringValue]]];
	}
    }
  
  if ([aMutableString length] > 0) 
    {
      [ccText setStringValue: [aMutableString substringToIndex:([aMutableString length] - 2)]];
    }

  // We are done with our mutable string, let's release it.
  RELEASE(aMutableString);

  // We set our subject
  if ( [theMessage subject] )
    {
      [subjectText setStringValue: [theMessage subject]];
    }
  else
    {
      [subjectText setStringValue: @""];
    }

  // We now set the content of the message (considering all attachments)
  // in case of a forward / reply
  if ( [theMessage content] )
    {
      [[textView textStorage]
	appendAttributedString: [Utilities formattedAttributedStringFromAttributedString:
					     [Utilities attributedStringFromContentForPart: theMessage
							mimeTypeManager: mimeTypeManager]] ];
    }

  // We finally set the signature, if we need to
  if ( aBOOL )
    {
      [self _loadSignature];
    }
}

@end
