/*
**  PersonalViewController.m
**
**  Copyright (c) 2001
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "PersonalViewController.h"

#import "GNUMailConstants.h"
#import "NewPersonalProfilePanelController.h"

#ifndef MACOSX
#import "PersonalView.h"
#endif

static PersonalViewController *singleInstance = nil;

@implementation PersonalViewController

- (id) initWithNibName: (NSString *) theName
{
  self = [super init];

#ifdef MACOSX
  
  if (! [NSBundle loadNibNamed: theName
		  owner: self] )
    {
      NSLog(@"Fatal error occured while loading the PersonalView nib file");
      AUTORELEASE(self);
      return nil;
    }

  RETAIN(view);

#else
  // We link our view
  view = [[PersonalView alloc] initWithParent: self];
  [view layoutView];

  // We link our outlets
  box = [view box];
  popUpButton = [view popUpButton];

  nameField = [view nameField];
  emailField = [view emailField];
  replytoField = [view replytoField];
  organizationField = [view organizationField];
  signaturePopUp = [view signaturePopUp];
  signatureField = [view signatureField];
  showWaitPanels = [view showWaitPanels];

  defaultProfileButton = [view defaultProfileButton];
#endif
    
  // We get our defaults for this panel
  [self initializeFromDefaults];

  return self;
}

- (void) dealloc
{
  singleInstance = nil;

  TEST_RELEASE(defaultPersonal);
  
  RELEASE(allPersonalProfiles);
  RELEASE(allKeys);

  RELEASE(view);

  [super dealloc];
}



//
// action methods
//

- (IBAction) addClicked: (id) sender
{
  NewPersonalProfilePanelController *aNewPersonalProfilePanelController;
  int result;

  aNewPersonalProfilePanelController = [[NewPersonalProfilePanelController alloc]
					 initWithWindowNibName: @"NewPersonalProfilePanel"];
  
  result = [NSApp runModalForWindow: [aNewPersonalProfilePanelController window]];
  
  // We must add...
  if (result == NSRunStoppedResponse)
    {
      NSMutableDictionary *aMutableDictionary;
      NSString *aString;
      int i;

      aString = [aNewPersonalProfilePanelController name];
      
      // We first make sure that this profile name is unique
      if ( [allKeys containsObject: aString] )
	{
	  NSRunAlertPanel(_(@"Error!"),
			  _(@"This profile name already exists. Please choose another name."),
			  _(@"OK"),
			  NULL,
			  NULL);
	  return;
	}
      
      // The new profile name is unique, let's create it.
      aMutableDictionary = [[NSMutableDictionary alloc] init];
      [aMutableDictionary setObject: @"" forKey: @"NAME"];
      [aMutableDictionary setObject: @"" forKey: @"EMAILADDR"];
      [aMutableDictionary setObject: @"" forKey: @"ORGANIZATION"];
      [aMutableDictionary setObject: @"" forKey: @"REPLYTOADDR"];

      [allPersonalProfiles setObject: aMutableDictionary
			   forKey: aString];

      // We must now re-initialize our allKeys array and our popup button
      RELEASE(allKeys);
      allKeys = [allPersonalProfiles allKeys];
      RETAIN(allKeys);
      
      [popUpButton removeAllItems];
      
      for (i = 0; i < [allKeys count]; i++)
	{
	  [popUpButton insertItemWithTitle: [allKeys objectAtIndex: i]
		   atIndex: i];
	}
      
      // We finally select our new item
      [popUpButton selectItemWithTitle: aString];
      [self changePersonalSelection: nil];
    }

  // We release our controller
  RELEASE(aNewPersonalProfilePanelController);
}


//
//
//
- (IBAction) changePersonalSelection: (id) sender
{
  NSDictionary *aDictionary;

  [popUpButton synchronizeTitleAndSelectedItem];

  [box setTitle: [popUpButton titleOfSelectedItem]];

  aDictionary = [allPersonalProfiles objectForKey: [popUpButton titleOfSelectedItem]];
  
  [nameField setStringValue: [aDictionary objectForKey: @"NAME"]];
  [emailField setStringValue: [aDictionary objectForKey: @"EMAILADDR"]];
  [replytoField setStringValue: [aDictionary objectForKey: @"REPLYTOADDR"]];
  [organizationField setStringValue: [aDictionary objectForKey: @"ORGANIZATION"]];

  // We verify if we've selected the default profile
  if ( defaultPersonal &&
       [defaultPersonal isEqualToString: [popUpButton titleOfSelectedItem]] )
    {
      [defaultProfileButton setState: NSOnState];
    }
  else
    {
      [defaultProfileButton setState: NSOffState];
    }
}


//
//
//
- (IBAction) defaultProfileButtonClicked: (id) sender
{
  if ( [defaultProfileButton state] == NSOnState )
    {
      defaultPersonal = [popUpButton titleOfSelectedItem];
      RETAIN(defaultPersonal);
    }
  else
    {
      TEST_RELEASE(defaultPersonal);
      defaultPersonal = nil;
    }
}


//
//
//
- (IBAction) deleteClicked: (id) sender
{
  if ( [allPersonalProfiles count] <= 1 )
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"You must have at least one defined personal profile at any time."),
		      _(@"OK"),
		      NULL,
		      NULL);
      return;
    }
  else
    {
      int i, choice;
      
      choice = NSRunAlertPanel(_(@"Delete..."),
			       _(@"Are you sure you want to delete the %@ personal profile?"),
			       _(@"Cancel"), // default
			       _(@"Yes"),    // alternate
			       NULL,
			       [popUpButton titleOfSelectedItem]);
      
      // If we must delete it...
      if (choice == NSAlertAlternateReturn)
	{
	  // We remove our key
	  [allPersonalProfiles removeObjectForKey: [popUpButton titleOfSelectedItem]];

	  // We must now re-initialize our allKeys array and our popup button
	  RELEASE(allKeys);
	  allKeys = [allPersonalProfiles allKeys];
	  RETAIN(allKeys);
	  
	  [popUpButton removeAllItems];
	  
	  for (i = 0; i < [allKeys count]; i++)
	    {
	      [popUpButton insertItemWithTitle: [allKeys objectAtIndex: i]
			   atIndex: i];
	    }
	  
	  // We finally select our new item
	  [popUpButton selectItemAtIndex: 0];
	  [self changePersonalSelection: nil];
	}
    }
}


- (IBAction) getSignature: (id) sender
{
  NSArray *fileToOpen;
  NSOpenPanel *oPanel;
  NSString *fileName;
  int count, result;
  
  oPanel = [NSOpenPanel openPanel];
  [oPanel setAllowsMultipleSelection:NO];
  result = [oPanel runModalForDirectory:NSHomeDirectory() file:nil types:nil];
  
  if (result == NSOKButton) {
    
    fileToOpen = [oPanel filenames];
    count = [fileToOpen count];
    
    if (count > 0) {
      fileName = [fileToOpen objectAtIndex:0];
      [signatureField setStringValue:fileName];
    }
  }
}


//
// access methods
//

- (NSImage *) image
{
  NSBundle *aBundle;
  
  aBundle = [NSBundle bundleForClass: [self class]];

  NSLog(@"%@", [aBundle pathForResource: @"personal" ofType: @"tiff"]);
  return AUTORELEASE([[NSImage alloc] initWithContentsOfFile:
					[aBundle pathForResource: @"personal" ofType: @"tiff"]]);
}

- (NSString *) name
{
  return _(@"Personal");
}

- (NSView *) view
{
  return view;
}

- (BOOL) hasChangesPending
{
  return YES;
}


//
// This method is used to initialize the fields in this view to
// the values from the user defaults database. We must take care
// of nil values since MacOS-X doesn't like that.
//
- (void) initializeFromDefaults
{
  NSDictionary *aDictionary;
  int i;

  allPersonalProfiles = [NSMutableDictionary dictionaryWithDictionary: [[NSUserDefaults standardUserDefaults]
									 objectForKey: @"PERSONAL"]];
  RETAIN(allPersonalProfiles);

  allKeys = [[allPersonalProfiles allKeys] sortedArrayUsingSelector: @selector(caseInsensitiveCompare:)];
  RETAIN(allKeys);

  // We initialize our popup button
  [popUpButton removeAllItems];
  
  for (i = 0; i < [allKeys count]; i++)
    {
      [popUpButton insertItemWithTitle: [allKeys objectAtIndex: i]
		   atIndex: i];
    }
  
  [popUpButton selectItemAtIndex: 0];
  [popUpButton synchronizeTitleAndSelectedItem];

  // We initialize our fields related to the personal informations
  // with the values of the first dictionary.
  [box setTitle: [allKeys objectAtIndex: 0]];
  
  aDictionary = [allPersonalProfiles objectForKey: [allKeys objectAtIndex: 0]];
  
  // NAME
  if ( [aDictionary objectForKey:@"NAME"] )
    {
      [nameField setStringValue: [aDictionary objectForKey:@"NAME"] ];
    }
  else
    {
      [nameField setStringValue: @""];
    }
 
  // EMAILADDR
  if ( [aDictionary objectForKey:@"EMAILADDR"] )
    {
      [emailField setStringValue: [aDictionary objectForKey:@"EMAILADDR"] ];
    }
  else
    {
      [emailField setStringValue: @""];
    }

  // REPLYTOADDR
  if ( [aDictionary objectForKey:@"REPLYTOADDR"] )
    {
      [replytoField setStringValue: [aDictionary objectForKey:@"REPLYTOADDR"] ];
    }
  else
    {
      [replytoField setStringValue: @""];
    }
  
  // ORGANIZATION
  if ( [aDictionary objectForKey:@"ORGANIZATION"] )
    {
      [organizationField setStringValue: [aDictionary objectForKey:@"ORGANIZATION"] ];
    }
  else
    {
      [organizationField setStringValue: @""];
    }
  
  // We finally load the defaults for our signature and our 'wait panels'
  // SIGNATURE_SOURCE
  [signaturePopUp selectItemAtIndex: [[NSUserDefaults standardUserDefaults]
				       integerForKey: @"SIGNATURE_SOURCE"] ];  
  // SIGNATURE
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"SIGNATURE"] )
    {
      [signatureField setStringValue: [[NSUserDefaults standardUserDefaults] objectForKey:@"SIGNATURE"] ];
    }
  else
    {
      [signatureField setStringValue: @""];
    }

  // SHOWWAITPANELS
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWWAITPANELS"] )
    {
      [showWaitPanels setState: [[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWWAITPANELS"] intValue] ];
    }
  else
    {
      [showWaitPanels setState: NSOffState];
    }

  // DEFAULT_PERSONAL
  defaultPersonal = [[NSUserDefaults standardUserDefaults] objectForKey: @"DEFAULT_PERSONAL"];
  
  if ( defaultPersonal )
    {
      RETAIN(defaultPersonal);
    }
}


- (void) saveChanges
{

  // We first synchronize our personal values with the one in our dictionary and
  // we than save this dictionary
  [self _synchronizePersonalPreferencesWithCurrentSelection];

  [[NSUserDefaults standardUserDefaults] setObject: allPersonalProfiles
					 forKey: @"PERSONAL"];
 
  // We finally save the defaults for our signature and our 'wait panels'
  [[NSUserDefaults standardUserDefaults] setInteger: [signaturePopUp indexOfSelectedItem]
					 forKey: @"SIGNATURE_SOURCE"];
  
  [[NSUserDefaults standardUserDefaults] setObject: [signatureField stringValue]
					 forKey: @"SIGNATURE"];

  [[NSUserDefaults standardUserDefaults] setInteger: [showWaitPanels state]
					 forKey: @"SHOWWAITPANELS"];

  if ( defaultPersonal )
    {
      [[NSUserDefaults standardUserDefaults] setObject: defaultPersonal
					     forKey: @"DEFAULT_PERSONAL"];
    }
  else
    {
      [[NSUserDefaults standardUserDefaults] removeObjectForKey: @"DEFAULT_PERSONAL"];
    }
  
  // We inform that our personal profiles have changed
  [[NSNotificationCenter defaultCenter]
    postNotificationName: PersonalProfilesHaveChanged
    object: nil
    userInfo: nil];
}


// 
// private method
//
- (void) _synchronizePersonalPreferencesWithCurrentSelection
{
  NSMutableDictionary *aMutableDictionary;
  
  if ( [popUpButton indexOfSelectedItem] < 0 )
    {
      return;
    }
  
  aMutableDictionary = [NSMutableDictionary dictionaryWithDictionary:
					      [allPersonalProfiles objectForKey:
								     [popUpButton titleOfSelectedItem]]];
  
  
  [aMutableDictionary setObject: [nameField stringValue]
		      forKey: @"NAME"];
  
  [aMutableDictionary setObject: [emailField stringValue]
		      forKey: @"EMAILADDR"];
  
  [aMutableDictionary setObject: [replytoField stringValue]
		      forKey: @"REPLYTOADDR"];
  
  [aMutableDictionary setObject: [organizationField stringValue]
		      forKey: @"ORGANIZATION"];

  [allPersonalProfiles setObject: aMutableDictionary
		       forKey: [popUpButton titleOfSelectedItem]];
}

//
// class methods
//
+ (id) singleInstance
{
  if (! singleInstance)
    {
      singleInstance = [[PersonalViewController alloc] initWithNibName: @"PersonalView"];
    }

  return singleInstance;
}

@end
