/*  GnomeKiss - A KiSS viewer for the GNOME desktop
    Copyright (C) 2000-2002  Nick Lamb <njl195@zepler.org.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <gnome.h>
#include "support.h"
#include "kiss.h"

KissActionList fkiss_errors;

static int random_value(int from, int to);
static KissActionList label_by_id(gint id);
static void object_unmap(KissObject *object);
static void object_map(KissObject *object);
static void object_altmap(KissObject *object);
static void object_ghost(KissObject *object, int value);
static void object_transparent(KissObject *object, int amount);
static void object_move_relative(KissObject *object, int x, int y);
static void play_midi(gchar *song);
static void stop_midi(void);

static unsigned int depth= 0, cycles = 0;
static unsigned int control_flag[MAX_DEPTH+1];

static pid_t midi_child= 0; /* MIDI child process */

/* returns a random integer between "from" and "to" inclusive */
static int random_value(int from, int to) {
  int range;

  if (to < from) {
    range= from - to;
    from= to;
    to= from + range;
  } else if (to == from) {
    return from;
  } else {
    range= to - from;
  }

  return from + (rand() % (range + 1));
}

static KissActionList label_by_id(gint id) {
  KissActionList *event;

  event = event_find(config.labels, id);

  if (event == NULL) {
    log_error(_("label(%d) is undefined"), id);
    return NULL;
  } else {
    return *event;
  }
}

static void object_unmap(KissObject *object) {
  GSList *list;

  for (list= object->cells; list != NULL; list= g_slist_next(list)) {
    ((KissCell *) list->data)->mapped= 0;
  }
  render_object(object);
}

static void object_map(KissObject *object) {
  GSList *list;

  for (list= object->cells; list != NULL; list= g_slist_next(list)) {
    ((KissCell *) list->data)->mapped= 1;
  }
  render_object(object);
}

static void object_altmap(KissObject *object) {
  GSList *list;

  for (list= object->cells; list != NULL; list= g_slist_next(list)) {
    ((KissCell *) list->data)->mapped^= 1;
  }
  render_object(object);
}

static void object_ghost(KissObject *object, int value) {
  GSList *list;
  KissCell *cell;

  for (list= object->cells; list != NULL; list= g_slist_next(list)) {
    cell= (KissCell *) list->data;
    cell->ghosted= value;
  }
}

static void object_transparent(KissObject *object, int amount) {
  GSList *list;
  KissCell *cell;

  for (list= object->cells; list != NULL; list= g_slist_next(list)) {
    cell= (KissCell *) list->data;
    cell->alpha= CLAMP(cell->alpha - amount, 0, 255);
  }
  render_object(object);
}

static void object_move_relative(KissObject *object, int x, int y) {
  object_move(object, object->x[view] + x, object->y[view] + y);
}

static void stop_midi(void) {

  if (!prefs.midi) return; /* user doesn't want MIDI */

  if (midi_child) {
    kill(midi_child, SIGTERM);
    waitpid(midi_child, NULL, 0);
    midi_child= 0;
  }
}

static void play_midi(gchar *song) {
  int k;
  char *string, *args[13]; /* 1 command, 10 parameters, 1 song, 1 NULL */

  if (!prefs.midi) return; /* user doesn't want MIDI */

  string = strdup(prefs.midi_player);
  args[0]= strtok(string, " ");
  for (k= 1; k < 11; ++k) {
    args[k]= strtok(NULL, " ");
    if (args[k] == NULL) break;
  }
  args[k]= song;
  args[k+1]= NULL;

  stop_midi(); /* just in case */

  midi_child = fork();
  if (midi_child == 0) {
    execvp(args[0], args);
    _exit(182);
  } else if (midi_child == -1) {
    midi_child = 0;
  }

  free(string);
}

static void trace(KissAction *action, int depth) {
  int k;

  if (action == NULL) return;

  if (!prefs.trace) return;

  fprintf(stderr, "%4u ", action->line);
  for(k= 1; k < depth; ++k) {
    fputc(' ', stderr);
  }

  if (action->type < CONTROL_CONDITIONAL && control_flag[depth] == 0) {
    fprintf(stderr, _("##skipped## "));
  }

  switch (action->type) {
  case ACTION_ALTMAP_CELL:
    fprintf(stderr, "altmap(\"%s\")\n", KissCell(action->args[0])->name);
    break;
  case ACTION_ALTMAP_OBJECT:
    fprintf(stderr, "altmap(#%d)\n", KissObject(action->args[0])->id);
    break;
  case ACTION_CHANGESET:
    fprintf(stderr, "changeset(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_CHANGECOL:
    fprintf(stderr, "changecol(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_DEBUG:
    fprintf(stderr, "debug(\"%s\")\n", KissString(action->args[0]));
    break;
  case ACTION_MAP_CELL:
    fprintf(stderr, "map(\"%s\")\n", KissCell(action->args[0])->name);
    break;
  case ACTION_MAP_OBJECT:
    fprintf(stderr, "map(#%d)\n", KissObject(action->args[0])->id);
    break;
  case ACTION_MOVE_OBJECT:
    fprintf(stderr, "move(#%d,", KissObject(action->args[0])->id);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_NOP:
    fprintf(stderr, "nop()\n");
    break;
  case ACTION_QUIT:
    fprintf(stderr, "quit()\n");
    break;
  case ACTION_RANDOMTIMER:
    fprintf(stderr, "randomtimer(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_SHELL:
    fprintf(stderr, "shell(\"%s\")\n", KissString(action->args[0]));
    break;
  case ACTION_SOUND:
    fprintf(stderr, "sound(\"%s\")\n", KissString(action->args[0]));
    break;
  case ACTION_SOUND_CANCEL:
    fprintf(stderr, "sound(\"\")\n");
    break;
  case ACTION_TIMER:
    fprintf(stderr, "timer(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_TRANSPARENT_CELL:
    fprintf(stderr, "transparent(\"%s\",", KissCell(action->args[0])->name);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_TRANSPARENT_OBJECT:
    fprintf(stderr, "transparent(#%d,", KissObject(action->args[0])->id);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");

    break;
  case ACTION_UNMAP_CELL:
    fprintf(stderr, "unmap(\"%s\")\n", KissCell(action->args[0])->name);
    break;
  case ACTION_UNMAP_OBJECT:
    fprintf(stderr, "unmap(#%d)\n", KissObject(action->args[0])->id);
    break;
  case ACTION_VIEWPORT:
    fprintf(stderr, "viewport(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");

    break;
  case ACTION_WINDOW_SIZE:
    fprintf(stderr, "windowsize(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");
    break;

/* FKiSS 2 onwards */

  case ACTION_MOVE_BYX:
    fprintf(stderr, "movebyx(#%d,#%d,", KissObject(action->args[0])->id,
            KissObject(action->args[1])->id);
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_MOVE_BYY:
    fprintf(stderr, "movebyy(#%d,#%d,", KissObject(action->args[0])->id,
            KissObject(action->args[1])->id);
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");

    break;
  case ACTION_MOVE_TO:
    fprintf(stderr, "moveto(#%d,", KissObject(action->args[0])->id);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");

    break;
  case ACTION_MUSIC:
    fprintf(stderr, "music(\"%s\")\n", KissString(action->args[0]));
    break;
  case ACTION_MUSIC_CANCEL:
    fprintf(stderr, "music(\"\")\n");
    break;
  case ACTION_NOTIFY:
    fprintf(stderr, "notify(\"%s\")\n", KissString(action->args[0]));
    break;

/* FKiSS 2.1 onwards */

  case ACTION_IF_FIXED:
    fprintf(stderr, "iffixed(#%d,", KissObject(action->args[0])->id);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));

    if (KissObject(action->args[0])->fix > 0)
      fprintf(stderr, ") TRUE\n");
    else
      fprintf(stderr, ") FALSE\n");
    break;
  case ACTION_IF_MAPPED:
    fprintf(stderr, "ifmapped(\"%s\",", KissCell(action->args[0])->name);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));

    if (KissCell(action->args[0])->mapped)
      fprintf(stderr, ") TRUE\n");
    else
      fprintf(stderr, ") FALSE\n");
    break;
  case ACTION_IF_MOVED:
    fprintf(stderr, "ifmoved(#%d,", KissObject(action->args[0])->id);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_IF_NOT_FIXED:
    fprintf(stderr, "ifnotfixed(#%d,", KissObject(action->args[0])->id);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));

    if (KissObject(action->args[0])->fix == 0)
      fprintf(stderr, ") TRUE\n");
    else
      fprintf(stderr, ") FALSE\n");
    break;
  case ACTION_IF_NOT_MAPPED:
    fprintf(stderr, "ifnotmapped(\"%s\",", KissCell(action->args[0])->name);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));

    if (!KissCell(action->args[0])->mapped)
      fprintf(stderr, ") TRUE\n");
    else
      fprintf(stderr, ") FALSE\n");
    break;
  case ACTION_IF_NOT_MOVED:
    fprintf(stderr, "ifnotmoved(#%d,", KissObject(action->args[0])->id);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_MOVE_RANDX:
    fprintf(stderr, "moverandx(#%d,", KissObject(action->args[0])->id);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_MOVE_RANDY:
    fprintf(stderr, "moverandy(#%d,", KissObject(action->args[0])->id);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_MOVE_TORAND:
    fprintf(stderr, "movetorand(#%d)\n", KissObject(action->args[0])->id);
    break;
  case ACTION_SETFIX:
    fprintf(stderr, "setfix(#%d,", KissObject(action->args[0])->id);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");
    break;

/* FKiSS 3.0 onwards */

  case ACTION_VAR_LET:
    fprintf(stderr, "let(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_VAR_ADD:
    fprintf(stderr, "add(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_VAR_SUB:
    fprintf(stderr, "sub(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_VAR_MUL:
    fprintf(stderr, "mul(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_VAR_DIV:
    fprintf(stderr, "div(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_VAR_MOD:
    fprintf(stderr, "mod(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_VAR_RANDOM:
    fprintf(stderr, "random(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_VAR_OBJECT_X:
    fprintf(stderr, "letobjectx(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",#%d) = %d\n", KissObject(action->args[1])->id,
                                    KissObject(action->args[1])->x[view]);
    break;
  case ACTION_VAR_OBJECT_Y:
    fprintf(stderr, "letobjecty(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",#%d) = %d\n", KissObject(action->args[1])->id,
                                    KissObject(action->args[1])->y[view]);
    break;
  case ACTION_VAR_OBJECT_FIX:
    fprintf(stderr, "letfix(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",#%d) = %d\n", KissObject(action->args[1])->id,
                                    KissObject(action->args[1])->fix);
    break;
  case ACTION_VAR_CELL_MAPPED:
    fprintf(stderr, "letmapped(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",\"%s\")\n", KissCell(action->args[1])->name);
    break;
  case ACTION_VAR_SET:
    fprintf(stderr, "letset(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ") = %d\n", view);
    break;
  case ACTION_VAR_PAL:
    fprintf(stderr, "letpal(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ") = %d\n", config.pal_set[view]);
    break;
  case ACTION_VAR_MOUSE_X:
    fprintf(stderr, "letmousex(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ") = %d\n", mouse_x);
    break;
  case ACTION_VAR_MOUSE_Y:
    fprintf(stderr, "letmousey(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ") = %d\n", mouse_y);
    break;
  case ACTION_VAR_CATCH:
    fprintf(stderr, "letcatch(");
    var_describe(KissVariable(action->args[0]));
    if (config.target)
      fprintf(stderr, ") = %d\n", config.target->object->id);
    else
      fprintf(stderr, ") = none\n");
    break;
  case ACTION_VAR_COLLIDE:
    fprintf(stderr, "letcollide(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",\"%s\",\"%s\")\n",
            KissCell(action->args[1])->name, KissCell(action->args[2])->name);
    break;
  case ACTION_VAR_INSIDE:
    fprintf(stderr, "letinside(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",#%d,#%d)\n", KissObject(action->args[1])->id,
                                   KissObject(action->args[2])->id);
    break;
  case ACTION_VAR_CELL_TRANSPARENT:
    fprintf(stderr, "lettransparent(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",\"%s\")\n", KissCell(action->args[1])->name);
    break;

  case ACTION_GHOST_CELL:
    fprintf(stderr, "ghost(\"%s\",", KissCell(action->args[0])->name);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");
    break;
  case ACTION_GHOST_OBJECT:
    fprintf(stderr, "ghost(#%d,", KissObject(action->args[0])->id);
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");
    break;

/* FKiSS 3.0 control structures :( */
  case CONTROL_EXIT_EVENT:
    fprintf(stderr, "exitevent()\n");
    break;
    
  case CONTROL_GOSUB:
    fprintf(stderr, "gosub(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ")\n");
    break;
    
  case CONTROL_GOTO:
    fprintf(stderr, "goto(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ")\n");
    break;

  case CONTROL_GOSUB_RANDOM:
    fprintf(stderr, "gosubrandom(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;

  case CONTROL_GOTO_RANDOM:
    fprintf(stderr, "gotorandom(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[2]));
    fprintf(stderr, ")\n");
    break;
    
  case CONTROL_EQUAL:
    fprintf(stderr, "ifequal(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");
    break;
  case CONTROL_NOT_EQUAL:
    fprintf(stderr, "ifnotequal(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");
    break;
  case CONTROL_GREATER_THAN:
    fprintf(stderr, "ifgreaterthan(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");
    break;
  case CONTROL_LESS_THAN:
    fprintf(stderr, "iflessthan(");
    var_describe(KissVariable(action->args[0]));
    fprintf(stderr, ",");
    var_describe(KissVariable(action->args[1]));
    fprintf(stderr, ")\n");
    break;
  case CONTROL_ELSE:
    fprintf(stderr, "else()\n");
    break;
  case CONTROL_ENDIF:
    fprintf(stderr, "endif()\n");
    break;

  default:
    fprintf(stderr, "## Unknown action? ##\n");
  }
}

/* if non-zero is returned no further actions should be processed
   from the current action list */

static KissActionList trigger(KissAction *action, KissActionList next) {
  int k;
  static unsigned int offset_x= 0, offset_y= 0;

  g_assert(action != NULL);

  if (action->type < CONTROL_CONDITIONAL && control_flag[depth] == 0)
    return next; /* FKiSS 3 if { ... } else { ... } endif */

  switch (action->type) {
  case ACTION_ALTMAP_CELL:
    KissCell(action->args[0])->mapped^=1;
    render_cell(KissCell(action->args[0]));
    break;
  case ACTION_ALTMAP_OBJECT:
    object_altmap(KissObject(action->args[0]));
    render_object(KissObject(action->args[0]));
    break;
  case ACTION_CHANGESET:
    switch_view(KissValue(action->args[0])); /* includes render */
    break;
  case ACTION_CHANGECOL:
    switch_color(KissValue(action->args[0])); /* includes render */
    break;
  case ACTION_DEBUG:
    fprintf(stderr, _("DEBUG: %s\n"), KissString(action->args[0]));
    break;
  case ACTION_MAP_CELL:
    KissCell(action->args[0])->mapped= 1;
    render_cell(KissCell(action->args[0]));
    break;
  case ACTION_MAP_OBJECT:
    object_map(KissObject(action->args[0]));
    break;
  case ACTION_MOVE_OBJECT:
    object_move_relative(KissObject(action->args[0]),
          KissValue(action->args[1]), KissValue(action->args[2]));
    break;
  case ACTION_NOP:
    break;
  case ACTION_QUIT:
    events(config.end);
    break;
  case ACTION_RANDOMTIMER:
    if (KissValue(action->args[2]) > 0)
      k= KissValue(action->args[1]) + (rand() % KissValue(action->args[2]));
    else
      k= KissValue(action->args[1]);
    set_timer(KissValue(action->args[0]), k);
    break;
  case ACTION_SHELL:
    fprintf(stderr, _("WARNING: tried to execute \"%s\"\n"),
                                   KissString(action->args[0]));
    break;
  case ACTION_SOUND:
    if (prefs.esound) {
      gnome_sound_play(KissString(action->args[0]));
    }
    break;
  case ACTION_SOUND_CANCEL:
    if (prefs.esound) {
      /* FIXME: it's not clear if even a non-portable cancel exists (!) */
      fprintf(stderr, "Don't know how to cancel sounds yet\n");
    }
    break;
  case ACTION_TIMER:
    set_timer(KissValue(action->args[0]), KissValue(action->args[1]));
    break;
  case ACTION_TRANSPARENT_CELL:
    k = KissCell(action->args[0])->alpha - KissValue(action->args[1]);
    KissCell(action->args[0])->alpha= CLAMP(k, 0, 255);
    render_cell(KissCell(action->args[0]));
    break;
  case ACTION_TRANSPARENT_OBJECT:
    object_transparent(KissObject(action->args[0]), KissValue(action->args[1]));
    break;
  case ACTION_UNMAP_CELL:
    KissCell(action->args[0])->mapped= 0;
    render_cell(KissCell(action->args[0]));
    break;
  case ACTION_UNMAP_OBJECT:
    object_unmap(KissObject(action->args[0]));
    break;
  case ACTION_VIEWPORT:
    /* FIXME */
    fprintf(stderr, "Action viewport(...) ignored\n");
    break;
  case ACTION_WINDOW_SIZE:
    offset_x+= KissValue(action->args[0]);
    offset_y+= KissValue(action->args[1]);
    gtk_widget_set_usize(get_widget(app, "scrolledwindow1"),
                         config.width + offset_x, config.height + offset_y);
    break;

/* FKiSS 2 onwards */

  case ACTION_MOVE_BYX:
    object_move(KissObject(action->args[0]),
                KissObject(action->args[1])->x[view] +
                    KissValue(action->args[2]),
                KissObject(action->args[0])->y[view]);
    break;
  case ACTION_MOVE_BYY:
    object_move(KissObject(action->args[0]),
                KissObject(action->args[0])->x[view],
                KissObject(action->args[1])->y[view] +
                    KissValue(action->args[2]));
    break;
  case ACTION_MOVE_TO:
    object_move(KissObject(action->args[0]),
                KissValue(action->args[1]), KissValue(action->args[2]));
    break;
  case ACTION_MUSIC:
    play_midi(KissString(action->args[0]));
    break;
  case ACTION_MUSIC_CANCEL:
    stop_midi();
    break;
  case ACTION_NOTIFY:
    gnome_app_message(GNOME_APP(app), KissString(action->args[0]));
    break;

/* FKiSS 2.1 onwards */

  case ACTION_IF_FIXED:
    if (KissObject(action->args[0])->fix > 0)
      set_timer(KissValue(action->args[1]), KissValue(action->args[2]));
    break;
  case ACTION_IF_MAPPED:
    if (KissCell(action->args[0])->mapped)
      set_timer(KissValue(action->args[1]), KissValue(action->args[2]));
    break;
  case ACTION_IF_MOVED:
    if (KissObject(action->args[0])->ox[view] !=
        KissObject(action->args[0])->x[view]
     || KissObject(action->args[0])->oy[view] !=
        KissObject(action->args[0])->y[view])
      set_timer(KissValue(action->args[1]), KissValue(action->args[2]));
    break;
  case ACTION_IF_NOT_FIXED:
    if (KissObject(action->args[0])->fix == 0)
      set_timer(KissValue(action->args[1]), KissValue(action->args[2]));
    break;
  case ACTION_IF_NOT_MAPPED:
    if (!KissCell(action->args[0])->mapped)
      set_timer(KissValue(action->args[1]), KissValue(action->args[2]));
    break;
  case ACTION_IF_NOT_MOVED:
    if (KissObject(action->args[0])->ox[view] ==
        KissObject(action->args[0])->x[view] 
     && KissObject(action->args[0])->oy[view] ==
        KissObject(action->args[0])->y[view])
      set_timer(KissValue(action->args[1]), KissValue(action->args[2]));
    break;
  case ACTION_MOVE_RANDX:
    k= random_value(KissValue(action->args[1]), KissValue(action->args[2]));
    object_move_relative(KissObject(action->args[0]), k, 0);
    break;
  case ACTION_MOVE_RANDY:
    k= random_value(KissValue(action->args[1]), KissValue(action->args[2]));
    object_move_relative(KissObject(action->args[0]), 0, k);
    break;
  case ACTION_MOVE_TORAND:
    object_move(KissObject(action->args[0]), rand() % config.width,
                                             rand() % config.height);
    break;
  case ACTION_SETFIX:
    k= KissValue(action->args[1]);
    KissObject(action->args[0])->fix= k;
    if (k == 0)
      events(KissObject(action->args[0])->unfix);
    break;

/* FKiSS 3.0 onwards */

  case ACTION_VAR_LET:
    KissValue(action->args[0]) = KissValue(action->args[1]);
    break;
  case ACTION_VAR_ADD:
    KissValue(action->args[0]) = KissValue(action->args[1])
                               + KissValue(action->args[2]);
    break;
  case ACTION_VAR_SUB:
    KissValue(action->args[0]) = KissValue(action->args[1])
                               - KissValue(action->args[2]);
    break;
  case ACTION_VAR_MUL:
    KissValue(action->args[0]) = KissValue(action->args[1])
                               * KissValue(action->args[2]);
    break;
  case ACTION_VAR_DIV:
    if (KissValue(action->args[2]) > 0) {
      KissValue(action->args[0]) = KissValue(action->args[1])
                                 / KissValue(action->args[2]);
    } else {
      next= config.overflow;
    }
    break;
  case ACTION_VAR_MOD:
    if (KissValue(action->args[2]) > 0) {
      KissValue(action->args[0]) = KissValue(action->args[1])
                                 % KissValue(action->args[2]);
    } else {
      next= config.overflow;
    }
    break;
  case ACTION_VAR_RANDOM:
    KissValue(action->args[0]) = random_value(KissValue(action->args[1]),
                                              KissValue(action->args[2]));
    break;
  case ACTION_VAR_OBJECT_X:
    KissValue(action->args[0]) = KissObject(action->args[1])->x[view];
    break;
  case ACTION_VAR_OBJECT_Y:
    KissValue(action->args[0]) = KissObject(action->args[1])->y[view];
    break;
  case ACTION_VAR_OBJECT_FIX:
    KissValue(action->args[0]) = KissObject(action->args[1])->fix;
    break;
  case ACTION_VAR_CELL_MAPPED:
    KissValue(action->args[0]) = KissCell(action->args[1])->mapped;
    break;
  case ACTION_VAR_SET:
    KissValue(action->args[0]) = view;
    break;
  case ACTION_VAR_PAL:
    KissValue(action->args[0]) = config.pal_set[view];
    break;
  case ACTION_VAR_MOUSE_X:
    KissValue(action->args[0]) = mouse_x;
    break;
  case ACTION_VAR_MOUSE_Y:
    KissValue(action->args[0]) = mouse_y;
    break;
  case ACTION_VAR_CATCH:
    if (config.target)
      KissValue(action->args[0]) = config.target->object->id;
    break;
  case ACTION_VAR_COLLIDE:
    KissValue(action->args[0]) = cell_collision(KissCell(action->args[1]),
                                                KissCell(action->args[2]));
    break;
  case ACTION_VAR_INSIDE:
    KissValue(action->args[0]) = object_collision(KissObject(action->args[1]),
                                                  KissObject(action->args[2]));
    break;
  case ACTION_VAR_CELL_TRANSPARENT:
    KissValue(action->args[0]) = 255 - KissCell(action->args[1])->alpha;
    break;

  case ACTION_GHOST_CELL:
    KissCell(action->args[0])->ghosted= KissValue(action->args[1]);
    break;
  case ACTION_GHOST_OBJECT:
    object_ghost(KissObject(action->args[0]), KissValue(action->args[1]));
    break;

/* FKiSS 3.0 control structures :( */
  case CONTROL_EXIT_EVENT:
    next= NULL;
    break;

  case CONTROL_GOTO:
    next= label_by_id(KissValue(action->args[0]));
    break;
    
  case CONTROL_GOSUB:
    if (next == NULL) /* tail recursion == loop, avoid stack */
      next= label_by_id(KissValue(action->args[0]));
    else
      events(label_by_id(KissValue(action->args[0])));
    break;
    
  case CONTROL_GOTO_RANDOM:
    if (random_value(1, 100) <= KissValue(action->args[0])) {
      next= label_by_id(KissValue(action->args[1]));
    } else {
      next= label_by_id(KissValue(action->args[2]));
    }
    break;

  case CONTROL_GOSUB_RANDOM:
    if (random_value(1, 100) <= KissValue(action->args[0]))
      k = 1;
    else
      k = 2;
    if (next == NULL) /* tail recursion == loop, avoid stack */
      next= label_by_id(KissValue(action->args[k]));
    else
      events(label_by_id(KissValue(action->args[k])));
    break;
    
  case CONTROL_EQUAL:
    if (KissValue(action->args[0]) == KissValue(action->args[1]))
      control_flag[depth]= 1;
    else
      control_flag[depth]= 0;
    break;
  case CONTROL_NOT_EQUAL:
    if (KissValue(action->args[0]) != KissValue(action->args[1]))
      control_flag[depth]= 1;
    else
      control_flag[depth]= 0;
    break;
  case CONTROL_GREATER_THAN:
    if (KissValue(action->args[0]) > KissValue(action->args[1]))
      control_flag[depth]= 1;
    else
      control_flag[depth]= 0;
    break;
  case CONTROL_LESS_THAN:
    if (KissValue(action->args[0]) < KissValue(action->args[1]))
      control_flag[depth]= 1;
    else
      control_flag[depth]= 0;
    break;
  case CONTROL_ELSE:
    control_flag[depth] = (control_flag[depth]) ? 0 : 1;
    break;
  case CONTROL_ENDIF:
    control_flag[depth] = 1;
    break;

  default:
    fprintf(stderr, _("Invalid action %d executed.\n"), action->type);
  }

  return next;
}

void events(KissActionList list) {
  if (list == NULL) return;

  if (depth >= MAX_DEPTH) {
    fprintf(stderr, _("French KiSS stack overflow.\n"));
    return;
  } else {
    depth++;
  }
  control_flag[depth]= 1;
  while (list != NULL) {
    trace(list->data, depth);
    list= trigger(list->data, g_slist_next(list));
    if (++cycles % 4096 == 0 && gtk_main_iteration_do(FALSE))
      exit(0); /* all done */
  }
  depth--;
  if (depth == 0) {
    cycles = 0; /* reset */
  }
}

