/*  GnomeKiss - A KiSS viewer for the GNOME desktop
    Copyright (C) 2000-2001  Nick Lamb <njl195@zepler.org.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include "kiss.h"

#define KISS_IN		1
#define KISS_OUT	0
#define KISS_NEITHER	-1

static int quick_collision(KissCell *cell1, KissCell *cell2);

KissObjectCollision *impact_find(char *one, char *two) {
  KissObjectCollision *collision;
  GSList *list;
  KissObject *obj1, *obj2;

  obj1= object_find(one);
  obj2= object_find(two);

  /* If the objects aren't real they can't very well collide */
  if (obj1 == NULL || obj2 == NULL) return NULL;

  for (list= obj1->obj_collisions; list != NULL; list= g_slist_next(list)) {
    collision= (KissObjectCollision *) list->data;

    if (collision->obj2 == obj2)
      return collision;
  }

  /* Can't find any suitable event, create one */
  
  collision= g_new0(KissObjectCollision, 1);
  collision->obj1= obj1;
  collision->obj2= obj2;
  collision->together= KISS_OUT;
  obj1->obj_collisions= g_slist_prepend(obj1->obj_collisions, collision);
  obj2->obj_collisions= g_slist_prepend(obj2->obj_collisions, collision);

  return collision;
}

KissCollisionEvents *collision_find(char *cell_one, char *cell_two) {
  KissCellCollision *collision;
  KissCollisionEvents *events;
  KissCell *cell1, *cell2 = NULL;
  GSList *list1, *list2;

  cell1= cell_find(cell_one);
  if (cell1 == NULL) {
    log_error(_("cell \"%s\" not loaded"), cell_one);
    return NULL;
  } else {
    list1= cell1->object->cell_collisions;
  }

  for (; list1 != NULL; list1= g_slist_next(list1)) {
    collision= (KissCellCollision *) list1->data;
 
    /* We need to match BOTH cells, since this list is per OBJECT */

    if ((collision->cell1 == cell1 &&
        !g_strcasecmp(collision->cell2->name, cell_two)) ||
        (collision->cell2 == cell1 &&
        !g_strcasecmp(collision->cell1->name, cell_two))) {

      return collision->events;
    }
  }

  events= g_new0(KissCollisionEvents, 1);
  events->together= KISS_OUT;
  collision= NULL;

  for (list2= cells; list2 != NULL; list2= g_slist_next(list2)) {
    cell2= (KissCell *) list2->data;

    if (!g_strcasecmp(cell2->name, cell_two)) {
      if (cell2->duplicate) {
        log_warning(_("cell name \"%s\" is ambiguous"), cell_two);
      }
      for (list1= cells; list1 != NULL; list1= g_slist_next(list1)) {
        cell1= (KissCell *) list1->data;
        if (!g_strcasecmp(cell1->name, cell_one)) {
          collision= g_new(KissCellCollision, 1);
          collision->cell1= cell1;
          collision->cell2= cell2;
          collision->events= events;
          cell1->object->cell_collisions=
            g_slist_prepend(cell1->object->cell_collisions, collision);
          cell2->object->cell_collisions=
            g_slist_prepend(cell2->object->cell_collisions, collision);
        }
      }
    }
  }

  if (collision == NULL) {
    g_free(events);
    log_error(_("cell \"%s\" not loaded"), cell_two);
    return NULL;
  } else {
    return events;
  }
}

int cell_collision(KissCell *cell1, KissCell *cell2) {
  int x, x1, x2, y, y1, y2;
  int width, height;

  if (!cell1->mapped || !(cell1->visible & (1 << view)))
    return KISS_OUT;

  if (!cell2->mapped || !(cell2->visible & (1 << view)))
    return KISS_OUT;

  if (cell1->x[view] >= cell2->x[view] + cell2->width)
    return KISS_OUT;
  if (cell1->x[view] + cell1->width <= cell2->x[view])
    return KISS_OUT;
  if (cell1->y[view] >= cell2->y[view] + cell2->height)
    return KISS_OUT;
  if (cell1->y[view] + cell1->height <= cell2->y[view])
    return KISS_OUT;

  x1= MAX(cell1->x[view], cell2->x[view]) - cell1->x[view];
  x2= MAX(cell1->x[view], cell2->x[view]) - cell2->x[view];
  y1= MAX(cell1->y[view], cell2->y[view]) - cell1->y[view];
  y2= MAX(cell1->y[view], cell2->y[view]) - cell2->y[view];

  width= MIN(cell1->x[view] + cell1->width, cell2->x[view] + cell2->width)
              - MAX(cell1->x[view], cell2->x[view]);
  height= MIN(cell1->y[view] + cell1->height, cell2->y[view] + cell2->height)
              - MAX(cell1->y[view], cell2->y[view]);
  
  for (x = 0; x < width; ++x) {
    for (y = 0; y < height; ++y) {
      if (cell1->index && cell1->index[(y1+y) * cell1->width + x1 + x] == 0)
        continue;
      if (cell2->index && cell2->index[(y2+y) * cell2->width + x2 + x] == 0)
        continue;
      if (cell1->argb && cell1->argb[((y1+y) * cell1->width +x1 +x) * 4] == 0)
        continue;
      if (cell2->argb && cell2->argb[((y2+y) * cell2->width +x2 +x) * 4] == 0)
        continue;
      return KISS_IN;
    }
  }
  return KISS_OUT;

}

static int quick_collision(KissCell *cell1, KissCell *cell2) {
  if (!cell1->mapped || !(cell1->visible & (1 << view)))
    return KISS_OUT;

  if (!cell2->mapped || !(cell2->visible & (1 << view)))
    return KISS_OUT;

  if (cell1->x[view] >= cell2->x[view] + cell2->width)
    return KISS_OUT;
  if (cell1->x[view] + cell1->width <= cell2->x[view])
    return KISS_OUT;
  if (cell1->y[view] >= cell2->y[view] + cell2->height)
    return KISS_OUT;
  if (cell1->y[view] + cell1->height <= cell2->y[view])
    return KISS_OUT;

  return KISS_IN;
}

/* object collision ignores transparent pixels, it is a bbox collision */

int object_collision(KissObject *obj1, KissObject *obj2) {
  GSList *inner, *outer;
  KissCell *cell1, *cell2;

  for (outer = obj1->cells; outer != NULL ; outer= g_slist_next(outer)) {
    cell1= (KissCell *) outer->data;
    for (inner = obj2->cells; inner != NULL ; inner= g_slist_next(inner)) {
      cell2= (KissCell *) inner->data;

      if (quick_collision(cell1, cell2))
        return KISS_IN;
    }
  }

  return KISS_OUT;
}

void check_collisions(KissObject *object, int ignore) {
  GSList *list;
  KissCellCollision *collision;
  KissObjectCollision *impact;

  for (list= object->cell_collisions; list != NULL; list= g_slist_next(list)) {
    collision= (KissCellCollision *) list->data;

    if (cell_collision(collision->cell1, collision->cell2)) {
      if (collision->events->together != KISS_IN) {
        collision->events->together= KISS_IN;
        if (!ignore) events(collision->events->collide);
      }
    } else {
      if (collision->events->together != KISS_OUT) {
        collision->events->together= KISS_OUT;
        if (!ignore) events(collision->events->apart);
      }
    }
  }

  for (list= object->obj_collisions; list != NULL; list= g_slist_next(list)) {
    impact= (KissObjectCollision *) list->data;

    if (object_collision(impact->obj1, impact->obj2)) {
      events(impact->stillin); /* Always */
      if (impact->together != KISS_IN) {
        impact->together= KISS_IN;
        if (!ignore) events(impact->in);
      }
    } else {
      events(impact->stillout); /* Always */
      if (impact->together != KISS_OUT) {
        impact->together= KISS_OUT;
        if (!ignore) events(impact->out);
      }
    }
  }

}

