/* rio500-method.h - VFS modules for Rio500
 *
 *  Copyright (C) 2001,2002 Bastien Nocera
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Author: Bastien Nocera <hadess@hadess.net>
 * Inspired by Alex Larsson's neato SMB vfs method
 */

#include <string.h>
#include <glib.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>

#include <libgnomevfs/gnome-vfs-method.h>
#include <libgnomevfs/gnome-vfs-module.h>
#include <libgnomevfs/gnome-vfs-module-shared.h>
#include <libgnomevfs/gnome-vfs-module-callback-module-api.h>
#include <libgnomevfs/gnome-vfs-standard-callbacks.h>

#include <librio500_api.h>

typedef struct _RioTreeContent RioTreeContent;

struct _RioTreeContent {
	Rio500 *rio_hw;
	gboolean have_flash;
	GList *int_content, *ext_content, *virt_content;
	gulong int_mem_total, ext_mem_total;
	gulong int_mem_free, ext_mem_free;
	gboolean dirty;
};

/* Global vars: */
static GMutex *rio_lock = NULL;
static RioTreeContent *rio = NULL;

#define DEBUG_RIO_ENABLE
#define DEBUG_RIO_LOCKS

#ifdef DEBUG_RIO_ENABLE
#define DEBUG_RIO(x) g_print x
#else
#define DEBUG_RIO(x) 
#endif

#ifdef DEBUG_RIO_LOCKS
#define LOCK_RIO() 	{g_mutex_lock (rio_lock); g_print ("LOCK %s\n", G_GNUC_PRETTY_FUNCTION);}
#define UNLOCK_RIO() 	{g_print ("UNLOCK %s\n", G_GNUC_PRETTY_FUNCTION); g_mutex_unlock (rio_lock);}
#else
#define LOCK_RIO() 	g_mutex_lock (rio_lock)
#define UNLOCK_RIO() 	g_mutex_unlock (rio_lock)
#endif

static RioTreeContent* update_tree_content(RioTreeContent *tmp);
static void destroy_rio(RioTreeContent *tmp);

static GnomeVFSResult
lookup_uri(GnomeVFSURI *uri, gint *card, gint *folder, gint *song)
{
	gchar *cardname;
	const gchar *path;
	GList *content, *list;
	gchar **str;
	RioFolderEntry *folder_entry = NULL;
	RioSongEntry *song_entry;

	*card = *folder = *song = -1;

	DEBUG_RIO(("lookup_uri: %s\n", gnome_vfs_uri_to_string(uri, 0)));

	rio = update_tree_content(rio);

	LOCK_RIO();
	if (rio == NULL)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}

	cardname = (gchar *) gnome_vfs_uri_get_host_name (uri);
	if (cardname != NULL)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_INVALID_URI;
	}

	/* Is this root? */
	if (gnome_vfs_uri_has_parent (uri) == 0)
	{
		UNLOCK_RIO();
		return GNOME_VFS_OK;
	}

	{
		const gchar *tmp;

		tmp = gnome_vfs_uri_get_path(uri);
		path = gnome_vfs_unescape_string(tmp, "/");
		if (path == NULL)
			path = tmp;

		DEBUG_RIO(("lookup_uri: path %s\n", path));
	}

	str = g_strsplit(path, "/", 0);
	/* this is pretty safe i think */
	cardname = g_strdup(str[1]);

	DEBUG_RIO(("lookup_uri: cardname %s\n", cardname));

	/* now we are in a flash card... maybe */
	if (g_strcasecmp(cardname, "Internal") == 0)
	{
		/* internal card */
		DEBUG_RIO(("lookup_uri: internal card detected\n"));
		*card = 0;
		content = rio->int_content;
	} else {
		if (!g_strcasecmp(cardname, "External") &&
				rio->have_flash == TRUE)
		{
			DEBUG_RIO(("lookup_uri: external card detected\n"));
			/* external card */
			*card = 1;
			content = rio->ext_content;
		} else {
			DEBUG_RIO(("lookup_uri: nonexistant card detected\n"));
			/* none existant directory */
			g_free(cardname);
			g_strfreev(str);
			UNLOCK_RIO();
			return GNOME_VFS_ERROR_NOT_FOUND;
		}
	}
	g_free(cardname);

	if (str[2] == NULL)
	{
		g_strfreev(str);
		UNLOCK_RIO();
		return GNOME_VFS_OK;
	}

	/* look for the folder name */
	DEBUG_RIO(("lookup_uri: looking up folder '%s'\n", str[2]));
	for (list = content; list != NULL; list = list->next)
	{
		folder_entry = (RioFolderEntry *) list->data;

		if (folder_entry
				&& (strcmp(str[2], folder_entry->name) == 0))
		{
			DEBUG_RIO(("lookup_uri: found folder %d\n", folder_entry->folder_num));
			/* match the folder */
			*folder = folder_entry->folder_num;
			break;
		}
	}

	/* if the folder didn't match anything on the rio, go "404" */
	if (*folder == -1)
	{
		g_strfreev(str);
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_FOUND;
	}

	/* is it a directory ? */
	if (str[3] == NULL)
	{
		g_strfreev(str);
		UNLOCK_RIO();
		return GNOME_VFS_OK;
	}

	/* look for the song */
	for (list = folder_entry->songs; list != NULL; list = list->next)
	{
		song_entry = (RioSongEntry *) list->data;
		if (song_entry
				&& (strcmp(str[3], song_entry->name) == 0))
		{
			/* match the song */
			*song = song_entry->song_num;
			break;
		}
	}
	/* if the song didn't match anything on the rio, go "404" */
	if (*song == -1)
	{
		g_strfreev(str);
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_FOUND;
	}

	/* oh my ! too much information there */
	if (str[4] != NULL)
	{
		g_strfreev(str);
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_INVALID_URI;
	}

	g_strfreev(str);
	UNLOCK_RIO();
	return GNOME_VFS_OK;
}

static RioTreeContent*
update_tree_content(RioTreeContent *tmp)
{
	RioFolderEntry *folder_entry = NULL;
	gint card_num, major, minor;

	DEBUG_RIO(("update_tree_content start\n"));

	if (tmp != NULL)
	{
		DEBUG_RIO(("update_tree_content: RioTreeContent is not NULL\n"));
		if (tmp->dirty == TRUE)
		{
			DEBUG_RIO(("update_tree_content: RioTreeContent is dirty\n"));
			destroy_rio(tmp);
		} else {
			DEBUG_RIO(("update_tree_content: RioTreeContent is not dirty\n"));
			return rio;
		}
	}

	DEBUG_RIO(("update_tree_content: creating new RioTreeContent\n"));

	LOCK_RIO();

	tmp = g_new(RioTreeContent, 1);
	tmp->rio_hw = NULL;
	tmp->int_content = NULL;
	tmp->ext_content = NULL;
	tmp->virt_content = NULL;
	tmp->int_mem_total = tmp->ext_mem_total = tmp->int_mem_free =
		tmp->ext_mem_free = 0;
	tmp->dirty = FALSE;

	tmp->rio_hw = rio_new();
	if (rio_check(tmp->rio_hw) == FALSE)
	{
		DEBUG_RIO(("update_tree_content: check of rio is FALSE, destroying\n"));
		destroy_rio(tmp);
		UNLOCK_RIO();
		return NULL;
	}

	rio_get_revision(tmp->rio_hw, &major, &minor);

	DEBUG_RIO(("update_tree_content: version %d.%d\n", major, minor));
	
	card_num = rio_get_card_number(tmp->rio_hw);
	if (card_num >= 2)
		tmp->have_flash = TRUE;

	rio_set_card(tmp->rio_hw, 0);
	tmp->int_mem_free = rio_memory_left(tmp->rio_hw);
	tmp->int_mem_total = rio_get_mem_total(tmp->rio_hw);
	tmp->int_content = rio_get_content(tmp->rio_hw);

	DEBUG_RIO(("update_tree_content: got internal card info\n"));

	folder_entry = g_new(RioFolderEntry, 1);
	folder_entry->name = g_strdup("Internal");
	folder_entry->folder_num = 0;
	folder_entry->songs = NULL;
	tmp->virt_content = g_list_append(tmp->virt_content,
			(gpointer) folder_entry);

	DEBUG_RIO(("update_tree_content: created dummy folder for internal\n"));

	if (tmp->have_flash == TRUE)
	{
		rio_set_card(tmp->rio_hw, 1);
		tmp->ext_mem_free = rio_memory_left(tmp->rio_hw);
		tmp->ext_mem_total = rio_get_mem_total(tmp->rio_hw);

		DEBUG_RIO(("update_tree_content: getting external card info\n"));
		
		tmp->ext_content = rio_get_content(tmp->rio_hw);

		DEBUG_RIO(("update_tree_content: got external card info\n"));

		folder_entry = g_new(RioFolderEntry, 1);
		folder_entry->name = g_strdup("External");
		folder_entry->folder_num = 1;
		folder_entry->songs = NULL;
		tmp->virt_content = g_list_append(tmp->virt_content,
				(gpointer) folder_entry);

		DEBUG_RIO(("update_tree_content: created dummy folder for external\n"));
	}
	rio_delete(tmp->rio_hw);
	UNLOCK_RIO();

	return tmp;
}

static void
destroy_rio(RioTreeContent *tmp)
{
	g_return_if_fail(tmp != NULL);

	DEBUG_RIO(("destroy_rio: start\n"));
	if (tmp->int_content != NULL)
		rio_destroy_content(tmp->int_content);
	DEBUG_RIO(("destroy_rio: destroyed int_content\n"));
	if (tmp->have_flash == TRUE
			&& tmp->ext_content != NULL)
		rio_destroy_content(tmp->ext_content);
	DEBUG_RIO(("destroy_rio: destroyed ext_content\n"));
	if (tmp->virt_content != NULL)
		rio_destroy_content(tmp->virt_content);
	DEBUG_RIO(("destroy_rio: destroyed virt_content\n"));

	g_free(tmp);

	DEBUG_RIO(("destroy_rio: end\n"));
}

static GnomeVFSResult
rio_to_vfs_error(gint error)
{
	switch (error)
	{
	case RIO_FILEERR:
		return GNOME_VFS_ERROR_IO;
	case RIO_FORMAT:
		return GNOME_VFS_ERROR_GENERIC;
	case RIO_ENDCOMM:
		return GNOME_VFS_ERROR_INTERNAL;
	case RIO_INITCOMM:
		return GNOME_VFS_ERROR_INTERNAL;
	case RIO_NODIR:
		return GNOME_VFS_ERROR_NOT_FOUND;
	case RIO_NOMEM:
		return GNOME_VFS_ERROR_NO_SPACE;
	case PC_MEMERR:
		return GNOME_VFS_ERROR_NO_MEMORY;
	case RIO_SUCCESS:
		return GNOME_VFS_OK;
	}

	g_warning("Rio generated error %d not found\n", error);
	return GNOME_VFS_ERROR_GENERIC;
}

typedef struct {
	gboolean is_read;
	GnomeVFSFileOffset offset;
	/* For reading */
	GList *folders, *songs;
	folder_entry *f_entry;
	song_entry *s_entry;
	gint old_offset, folder_block_offset;
	gint folder;
	unsigned long size;
	gboolean init_done;
	/* For writing */
	//FIXME
} FileHandle;

static GnomeVFSResult
do_open (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle **method_handle,
	 GnomeVFSURI *uri,
	 GnomeVFSOpenMode mode,
	 GnomeVFSContext *context)
{
	FileHandle *handle = NULL;
	GnomeVFSResult res;
	gint card, folder, song;
	gboolean is_read = FALSE;

	DEBUG_RIO(("do_open() %s mode %d\n",
				gnome_vfs_uri_to_string (uri, 0), mode));
//FIXME
	return GNOME_VFS_ERROR_NOT_SUPPORTED;


	if (mode & GNOME_VFS_OPEN_READ)
		is_read = TRUE;
	else if (mode & GNOME_VFS_OPEN_WRITE)
		is_read = FALSE;
	else
		return GNOME_VFS_ERROR_INVALID_OPEN_MODE;

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_open(): lookup %d %d %d\n",
				card, folder, song));

	LOCK_RIO();

	if ((res != GNOME_VFS_OK) && (is_read == TRUE))
	{
		UNLOCK_RIO();
		return res;
	}

	if ((folder != -1) && (song == -1))
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IS_DIRECTORY;
	}

	if ((song == -1) && (is_read == TRUE))
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_FOUND;
	}

	rio->rio_hw = rio_new();
	if (rio_check(rio->rio_hw) == FALSE)
	{
		DEBUG_RIO(("do_open: check of rio is FALSE, destroying\n"));
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}
	rio_set_card(rio->rio_hw, card);

	if (is_read == TRUE)
	{
		GList *folders, *songs;
		folder_entry *f_entry;
		song_entry *s_entry;
		gint old_offset, folder_block_offset;

		DEBUG_RIO(("do_open: opening read-only\n"));

		/* Init communication with rio */
		start_comm (rio->rio_hw);

		/* Read folder & song block */
		send_command(rio->rio_hw->rio_dev, 0x42, 0, 0);
		folders = read_folder_entries (rio->rio_hw->rio_dev,
				rio->rio_hw->card);
		if ( folder > g_list_length (folders)-1 )
			folder = 0;

		songs = read_song_entries (rio->rio_hw->rio_dev,
				folders, folder, rio->rio_hw->card);

		/* Trick the rio into reading song data */
		f_entry = (folder_entry *) g_list_nth_data (folders, 0);
		s_entry = (song_entry *) g_list_nth_data (songs, song);

		old_offset      = f_entry->offset;
		f_entry->offset = s_entry->offset;

		write_folder_entries(rio->rio_hw->rio_dev,
				folders, rio->rio_hw->card);
		send_command (rio->rio_hw->rio_dev, 0x42, 0, 0);
		send_command (rio->rio_hw->rio_dev, 0x42, 0, 0);
		folder_block_offset = send_command (rio->rio_hw->rio_dev,
				0x43, 0x0, 0x0);

		/* Tell Rio where the root folder block is. */
		send_folder_location(rio->rio_hw->rio_dev,
				folder_block_offset, folder,
				rio->rio_hw->card);

		/* Not really sure what this does */
		send_command (rio->rio_hw->rio_dev, 0x58, 0x0,
				rio->rio_hw->card);

		/* Create the handle */
		handle = g_new(FileHandle, 1);
		/* global */
		handle->is_read = is_read;
		handle->offset = 0;
		/* read specific */
		handle->folders = folders;
		handle->songs = songs;
		handle->f_entry = f_entry;
		handle->s_entry = s_entry;
		handle->old_offset = old_offset;
		handle->folder_block_offset = folder_block_offset;
		handle->folder = folder;
		handle->size = s_entry->length;
		handle->init_done = FALSE;
		handle->offset = 0;
		DEBUG_RIO(("do_open: length: %lu\n", handle->size));
	}
#if 0
	GnomeVFSResult res;
	SmbConnection *connection;
	SmbHandle *handle;
	SmbVirtualFile *file;
	char *path_remainder;
	int fnum;
	char *filename;
	char *dos_filename;
	int flags;
	char *server;
	char *path;
	char *end;
	const char *user;
	const char *password;
	
	LOCK_SAMBA();
	
	DEBUG_SMB (("do_open() %s mode %d\n",
		    gnome_vfs_uri_to_string (uri, 0), mode));
	
	if ((mode & GNOME_VFS_OPEN_READ) &&
	    (mode & GNOME_VFS_OPEN_WRITE))
		flags = O_RDWR;
	else if (mode & GNOME_VFS_OPEN_READ)
		flags = O_RDONLY;
	else if (mode & GNOME_VFS_OPEN_WRITE)
		flags = O_WRONLY;
	else
		return GNOME_VFS_ERROR_INVALID_OPEN_MODE;

	res = lookup_uri (uri, &file, &path_remainder, &connection, TRUE);

	if (res) {
		UNLOCK_SAMBA ();
		return res;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_DIRECTORY) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_IS_DIRECTORY;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_FILE) {
		handle = g_new (SmbHandle, 1);
		handle->is_data = TRUE;
		handle->offset = 0;
		handle->file_data = g_strdup (file->u.file.contents);
	} else { /* SMB_VIRTUAL_TYPE_SHARE */
		dos_filename = unix_filename_to_dos ((path_remainder)?path_remainder:"/");
		fnum = cli_open (connection->cli, dos_filename, flags, DENY_NONE);
		g_free (path_remainder);
		g_free (dos_filename);
		if (fnum == -1) {
			res = gnome_vfs_result_from_cli (connection->cli);
			smb_connection_unref (connection);
			UNLOCK_SAMBA();
			return res;
		}
		
		handle = g_new (SmbHandle, 1);
		handle->is_data = FALSE;
		handle->connection = connection;
		handle->fnum = fnum;
		handle->offset = 0;
	}

	UNLOCK_SAMBA();

	*method_handle = (GnomeVFSMethodHandle *)handle;
#endif
	*method_handle = (GnomeVFSMethodHandle *)handle;
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_close (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  GnomeVFSContext *context)

{
	FileHandle *handle = (FileHandle *)method_handle;

	if (handle->is_read == TRUE)
	{
		/* Restore folder */
		handle->f_entry->offset = handle->old_offset;
		write_folder_entries(rio->rio_hw->rio_dev, handle->folders,
				rio->rio_hw->card);
		send_command (rio->rio_hw->rio_dev, 0x42, 0, 0);
		send_command (rio->rio_hw->rio_dev, 0x42, 0, 0);
		handle->folder_block_offset = send_command
			(rio->rio_hw->rio_dev, 0x43, 0x0, 0x0);

		/* Tell Rio where the root folder block is. */
		send_folder_location(rio->rio_hw->rio_dev,
				handle->folder_block_offset,
				handle->folder, rio->rio_hw->card);

		/* Not really sure what this does */
		send_command (rio->rio_hw->rio_dev, 0x58, 0x0,
				rio->rio_hw->card);
	}

	/* Close communication */
	end_comm (rio->rio_hw);

	rio_delete(rio->rio_hw);

	//FIXME delete the handle's content

	UNLOCK_RIO();

	return GNOME_VFS_OK;
#if 0
	SmbHandle *handle = (SmbHandle *)method_handle;
	size_t n;

	LOCK_SAMBA();
	DEBUG_SMB (("do_close() %p\n", method_handle));

	if (handle->is_data) {
		g_free (handle->file_data);
	} else {
		cli_close  (handle->connection->cli, handle->fnum);
		smb_connection_unref (handle->connection);
	}
	
	g_free (handle);
	
	UNLOCK_SAMBA();
#endif
}

static GnomeVFSResult
do_read (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 gpointer buffer,
	 GnomeVFSFileSize num_bytes,
	 GnomeVFSFileSize *bytes_read,
	 GnomeVFSContext *context)
{
	static BYTE *init_buffer = NULL;
	FileHandle *handle = (FileHandle *)method_handle;
	gulong chunk_size, read = 0;

	DEBUG_RIO(("do_read() %Lu bytes, size %lu\n", num_bytes, handle->size));

	if (handle->init_done == FALSE)
	{
		if (init_buffer == NULL)
		{
			gulong this_read;

			/* Read 0x4000 bytes first */
			this_read = MIN (handle->size, 0x4000);
			init_buffer = (char *)malloc (this_read);

			send_command (rio->rio_hw->rio_dev, 0x4e, 0xff, rio->rio_hw->card);
			send_command (rio->rio_hw->rio_dev, 0x45, 0x0, this_read);

			read = bulk_read (rio->rio_hw->rio_dev, init_buffer, this_read);
			if (read > num_bytes)
				read = num_bytes;

			memcpy (buffer, init_buffer, read);
		} else {
			unsigned long n;

			n = MIN (num_bytes,
				MIN(handle->size, 0x4000) - handle->offset);
			memcpy (buffer, init_buffer, n);
			read = n;
		}
		handle->offset += read;
		DEBUG_RIO(("do_read() init read %lu bytes (offset %Ld)\n", read,
					handle->offset));
		if (handle->offset == MIN(handle->size, 0x4000))
			handle->init_done = TRUE;

		return GNOME_VFS_OK;
	}

	if (num_bytes+handle->offset > handle->size)
		chunk_size = handle->size - handle->offset;
	else
		chunk_size = num_bytes;
	send_command (rio->rio_hw->rio_dev, 0x45, chunk_size, 0x0);
	read = bulk_read (rio->rio_hw->rio_dev, (BYTE *)buffer,
			chunk_size);
	send_command (rio->rio_hw->rio_dev, 0x42, 0, 0);
	send_command (rio->rio_hw->rio_dev, 0x42, 0, 0);
	handle->offset += read;

	*bytes_read = read;

	if (handle->offset >= handle->size)
		return GNOME_VFS_ERROR_EOF;
	
	DEBUG_RIO(("do_read() read %d bytes (offset %Ld)\n", *bytes_read,
				handle->offset));
#if 0
	SmbHandle *handle = (SmbHandle *)method_handle;
	size_t n;
	int link_len;

	LOCK_SAMBA();
	DEBUG_SMB (("do_read() %p\n", method_handle));

	if (handle->is_data) {
		link_len = strlen (handle->file_data);
		if (handle->offset >= link_len)
			n = 0;
		else {
			n = MIN (num_bytes, link_len - handle->offset);
			memcpy (buffer, handle->file_data + handle->offset, n);
		}
	} else {
		n = cli_read (handle->connection->cli,
			      handle->fnum,
			      buffer,
			      handle->offset,
			      num_bytes);
	}

	UNLOCK_SAMBA();

	if (n < 0) {
		*bytes_read = 0;
		return gnome_vfs_result_from_cli (handle->connection->cli);
	}
	
	*bytes_read = n;
	
	if (n == 0)
		return GNOME_VFS_ERROR_EOF;
	
	handle->offset += n;
#endif
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_write (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  gconstpointer buffer,
	  GnomeVFSFileSize num_bytes,
	  GnomeVFSFileSize *bytes_written,
	  GnomeVFSContext *context)


{
#if 0
	SmbHandle *handle = (SmbHandle *)method_handle;
	ssize_t n;
	int link_len;

	LOCK_SAMBA();
	DEBUG_SMB (("do_write() %p\n", method_handle));

	if (handle->is_data) {
	  return GNOME_VFS_ERROR_NOT_SUPPORTED;
	} else {
		n = cli_write (handle->connection->cli,
			       handle->fnum,
			       0,
			       (char *)buffer,
			       handle->offset,
			       num_bytes);
	}

	UNLOCK_SAMBA();

	if (n < 0) {
		*bytes_written = 0;
		return gnome_vfs_result_from_cli (handle->connection->cli);
	}
	
	*bytes_written = n;
	
	if (n == 0)
		return GNOME_VFS_ERROR_EOF;
	
	handle->offset += n;
#endif
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_create (GnomeVFSMethod *method,
	   GnomeVFSMethodHandle **method_handle,
	   GnomeVFSURI *uri,
	   GnomeVFSOpenMode mode,
	   gboolean exclusive,
	   guint perm,
	   GnomeVFSContext *context)

{
#if 0
	GnomeVFSResult res;
	SmbVirtualFile *file;
	SmbConnection *connection;
	SmbHandle *handle;
	int fnum;
	char *path_remainder;
	char *dos_filename;
	int flags;
	char *server;
	char *path;

	LOCK_SAMBA();
	
	DEBUG_SMB (("do_create() %s mode %d\n",
		    gnome_vfs_uri_to_string (uri, 0), mode));
	
	if ((mode & GNOME_VFS_OPEN_READ) &&
	    (mode & GNOME_VFS_OPEN_WRITE))
		flags = O_RDWR;
	else if (mode & GNOME_VFS_OPEN_READ)
		flags = O_RDONLY;
	else if (mode & GNOME_VFS_OPEN_WRITE)
		flags = O_WRONLY;
	else
		return GNOME_VFS_ERROR_INVALID_OPEN_MODE;


	flags |= O_CREAT;
	if (exclusive)
		flags |= O_EXCL;
	
	res = lookup_uri (uri, &file, &path_remainder, &connection, TRUE);

	if (res) {
		UNLOCK_SAMBA ();
		if (res == GNOME_VFS_ERROR_NOT_FOUND)
			res = GNOME_VFS_ERROR_ACCESS_DENIED;
		return res;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_DIRECTORY) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_IS_DIRECTORY;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_FILE) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}
	
	/* SMB_VIRTUAL_TYPE_SHARE */
	dos_filename = unix_filename_to_dos ((path_remainder)?path_remainder:"/");
	DEBUG_SMB (("tryping to cli_open '%s', flags: %d\n", dos_filename, flags));
	fnum = cli_open (connection->cli, dos_filename, flags, DENY_NONE);
	g_free (path_remainder);
	g_free (dos_filename);
	if (fnum == -1) {
		res = gnome_vfs_result_from_cli (connection->cli);
		smb_connection_unref (connection);
		UNLOCK_SAMBA();
		return res;
	}
	
	handle = g_new (SmbHandle, 1);
	handle->is_data = FALSE;
	handle->connection = connection;
	handle->fnum = fnum;
	handle->offset = 0;

	UNLOCK_SAMBA();

	*method_handle = (GnomeVFSMethodHandle *)handle;
#endif
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_get_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  GnomeVFSFileInfo *file_info,
		  GnomeVFSFileInfoOptions options,
		  GnomeVFSContext *context)

{
	GnomeVFSResult res;
	gint card, folder, song;

	DEBUG_RIO (("do_get_file_info() %s\n",
				gnome_vfs_uri_to_string (uri, 0)));

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_get_file_info(): lookup %d %d %d\n",
				card, folder, song));

	LOCK_RIO();
	if (res != GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return res;
	}

	file_info->valid_fields =
		GNOME_VFS_FILE_INFO_FIELDS_TYPE |
		GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

	/* Is this root? */
	if (gnome_vfs_uri_has_parent (uri) == 0) {
		file_info->name = g_strdup ("/");
		file_info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		file_info->mime_type = g_strdup ("x-directory/normal");
		UNLOCK_RIO();
		return GNOME_VFS_OK;
	}

	/* is it a folder */
	if (song == -1)
	{
		/* is it not inside a card ? */
		if (folder == -1)
		{
			file_info->name = g_strdup ((card == 0) ? "Internal" :
					"External");
			file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			file_info->mime_type = g_strdup ("x-directory/normal");
		} else {
			file_info->name = g_strdup
				(gnome_vfs_uri_extract_short_name (uri));
			file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			file_info->mime_type = g_strdup ("x-directory/normal");
		}
	} else {
		file_info->name = g_strdup
			(gnome_vfs_uri_extract_short_name (uri));
		file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		file_info->mime_type = g_strdup ("audio/x-mp3");
	}

	UNLOCK_RIO();

	return GNOME_VFS_OK;
}

static gboolean
do_is_local (GnomeVFSMethod *method,
	     const GnomeVFSURI *uri)
{
	DEBUG_RIO (("do_is_local()\n"));

	return FALSE;
}

typedef struct {
	GList *list;
	gchar *dirname;
	guint pos;
	gboolean list_type_is_folder;
} DirectoryHandle;

static GnomeVFSResult
do_open_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle **method_handle,
		   GnomeVFSURI *uri,
		   GnomeVFSFileInfoOptions options,
		   const GnomeVFSDirectoryFilter *filter,
		   GnomeVFSContext *context)

{
	GnomeVFSResult res;
	DirectoryHandle *directory_handle;
	GList *list = NULL;
	gint card, folder, song;
	const gchar *path; //FIXME
	gboolean list_type_is_folder = TRUE;
	
	DEBUG_RIO(("do_open_directory() %s\n",
		gnome_vfs_uri_to_string (uri, 0)));

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_open_directory(): lookup %d %d %d\n",
				card, folder, song));
	LOCK_RIO();
	if (res != GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return res;
	}
	if (song != -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_A_DIRECTORY;
	}

	if (folder == -1)
	{
		/* can be rio500: or rio500://internal/ */
		if (card == -1)
		{
			list = rio->virt_content;
		} else {
			if (card == 1 && rio->have_flash == FALSE)
			{
				UNLOCK_RIO();
				return GNOME_VFS_ERROR_NOT_FOUND;
			}
			if (card == 0)
				list = rio->int_content;
			else
				list = rio->ext_content;
		}
	} else {
		GList *content = NULL;
		RioFolderEntry *entry = NULL;
		
		list_type_is_folder = FALSE;
		if (card == 1 && rio->have_flash == FALSE)
		{
			UNLOCK_RIO();
			return GNOME_VFS_ERROR_NOT_FOUND;
		}
		if (card == 0)
			content = rio->int_content;
		else
			content = rio->ext_content;

		entry = (RioFolderEntry *) g_list_nth_data(content, folder);
		list = entry->songs;
	}

	/* Construct the handle */
	directory_handle = g_new0 (DirectoryHandle, 1);
	directory_handle->dirname = g_strdup (path);
	directory_handle->pos = 0;
	directory_handle->list = list;
	directory_handle->list_type_is_folder = list_type_is_folder;
	*method_handle = (GnomeVFSMethodHandle *) directory_handle;

	UNLOCK_RIO();

	DEBUG_RIO(("do_open_directory end\n"));
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_close_directory (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSContext *context)
{
	DirectoryHandle *directory_handle = (DirectoryHandle *) method_handle;

	DEBUG_RIO(("do_close_directory: %p\n", directory_handle));

	if (directory_handle == NULL)
		return GNOME_VFS_OK;

	g_free(directory_handle->dirname);
	DEBUG_RIO(("do_close_directory: end\n"));

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_read_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle *method_handle,
		   GnomeVFSFileInfo *file_info,
		   GnomeVFSContext *context)
{
	DirectoryHandle *dh = (DirectoryHandle *) method_handle;
	GList *list;
	RioFolderEntry *folder_entry;
	RioSongEntry *song_entry;

	DEBUG_RIO (("do_read_directory()\n"));

	LOCK_RIO();
	if (dh->pos > (g_list_length(dh->list)+2) )
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_EOF;
	}

	/* . or .. directories */
	if (dh->pos == 0 || dh->pos == 1)
	{
		GNOME_VFS_FILE_INFO_SET_LOCAL (file_info, FALSE);
		file_info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_NONE |
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

		file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		file_info->name = g_strdup ((dh->pos == 0) ? "." : "..");
		file_info->mime_type = g_strdup ("x-directory/normal");
		DEBUG_RIO (("do_read_directory(): read folder %s\n", file_info->name));
		dh->pos++;
		UNLOCK_RIO();
		return GNOME_VFS_OK;
	}

	list = g_list_nth(dh->list, dh->pos - 2);
	if (list == NULL)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_EOF;
	}

	GNOME_VFS_FILE_INFO_SET_LOCAL (file_info, FALSE);

	/* Tell gnome-vfs which fields will be valid */
	file_info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_NONE |
		GNOME_VFS_FILE_INFO_FIELDS_TYPE |
		GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

	/* Fill the fields */
	if (dh->list_type_is_folder == TRUE)
	{
		folder_entry = (RioFolderEntry *)(list->data);
		file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		file_info->name = g_strdup (folder_entry->name);
		file_info->mime_type = g_strdup ("x-directory/normal");
		DEBUG_RIO (("do_read_directory(): read folder %s\n", file_info->name));
	} else {
		file_info->valid_fields |= GNOME_VFS_FILE_INFO_FIELDS_SIZE;
		song_entry = (RioSongEntry *)(list->data);
		file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		file_info->name = g_strdup(song_entry->name);
		file_info->size = song_entry->size;
		file_info->mime_type = g_strdup("audio/x-mp3");
		DEBUG_RIO (("do_read_directory(): read file %s\n", file_info->name));
	}

	dh->pos++;

	DEBUG_RIO (("do_read_directory(): position %d\n", dh->pos));
	UNLOCK_RIO();
	return GNOME_VFS_OK;
}
#if 0
static GnomeVFSResult
do_seek (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 GnomeVFSSeekPosition whence,
	 GnomeVFSFileOffset offset,
	 GnomeVFSContext *context)

{
	SmbHandle *handle = (SmbHandle *)method_handle;

	DEBUG_SMB (("do_seek() %d from %d\n", offset, whence));
	
	switch (whence) {
	case GNOME_VFS_SEEK_START:
		handle->offset = offset;
		break;
	case GNOME_VFS_SEEK_CURRENT:
		handle->offset += offset;
		break;
	case GNOME_VFS_SEEK_END:
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
		break;
	}

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_tell (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 GnomeVFSFileOffset *offset_return)


{
	SmbHandle *handle = (SmbHandle *)method_handle;

	*offset_return = handle->offset;

	return GNOME_VFS_OK;
}
#endif
static GnomeVFSResult
do_unlink (GnomeVFSMethod *method,
	   GnomeVFSURI *uri,
	   GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	gint card, folder, song;

	DEBUG_RIO (("do_unlink() %s\n",
				gnome_vfs_uri_to_string (uri, 0)));

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_unlink(): lookup %d %d %d\n",
				card, folder, song));

	LOCK_RIO();
	if (res != GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return res;
	}

	if (folder == -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_PERMITTED;
	}

	if (song == -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_A_DIRECTORY;
	}

	rio->rio_hw = rio_new();
	if (rio_check(rio->rio_hw) == FALSE)
	{
		DEBUG_RIO(("do_unlink: check of rio is FALSE, destroying\n"));
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}
	rio_set_card(rio->rio_hw, card);
	res = rio_to_vfs_error(rio_del_song(rio->rio_hw,
				folder, song));
	rio_delete(rio->rio_hw);

	if (res == GNOME_VFS_OK)
		rio->dirty = TRUE;

	UNLOCK_RIO();

	return res;
}

static GnomeVFSResult
do_check_same_fs (GnomeVFSMethod *method,
		  GnomeVFSURI *a,
		  GnomeVFSURI *b,
		  gboolean *same_fs_return,
		  GnomeVFSContext *context)
{
	DEBUG_RIO (("do_check_same_fs()\n"));
	
	*same_fs_return = FALSE;

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_move (GnomeVFSMethod *method,
	 GnomeVFSURI *old_uri,
	 GnomeVFSURI *new_uri,
	 gboolean force_replace,
	 GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	gint old_card, old_folder, old_song;
	gint new_card, new_folder, new_song;
	gboolean move_folder = FALSE;
	const gchar *new_name;

	g_warning("This function of the rio500 method is not properly supported\nPlease mail hadess@hadess.net with information about your program\n");

	DEBUG_RIO (("do_move() %s %s\n",
		    gnome_vfs_uri_to_string (old_uri, 0),
		    gnome_vfs_uri_to_string (new_uri, 0)));
	LOCK_RIO();

	res = lookup_uri(old_uri, &old_card, &old_folder, &old_song);
	DEBUG_RIO(("do_move(): lookup (src) %d %d %d\n",
				old_card, old_folder, old_song));
	/* if the original file isn't found, bail */
	if (res != GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return res;
	}

	res = lookup_uri(new_uri, &new_card, &new_folder, &new_song);
	DEBUG_RIO(("do_move(): lookup (dest) %d %d %d\n",
				new_card, new_folder, new_song));

	/* if the dest file is found and force_replace is false, bail */
	if (res == GNOME_VFS_OK && force_replace == FALSE)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_FILE_EXISTS;
	} else {
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
	}

	/* if the dest uri is invalid, bail */
	if (res == GNOME_VFS_ERROR_INVALID_URI)
	{
		UNLOCK_RIO();
		return res;
	}

	/* if the move crosses disks, or tries to the rename the cards, bail */
	if (new_card != old_card || new_card == -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
	}

	/* Moving songs */
	if (old_song != -1)
	{
		/* are they in the same folder ? */
		if (old_folder != new_folder)
		{
			UNLOCK_RIO();
			return GNOME_VFS_ERROR_NOT_SUPPORTED;
		}
		move_folder = FALSE;
	}

	rio->rio_hw = rio_new();
	if (rio_check(rio->rio_hw) == FALSE)
	{
		DEBUG_RIO(("do_move: check of rio is FALSE, destroying\n"));
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}
	rio_set_card(rio->rio_hw, old_card);

	/* Getting the name of the file/folder */
	{
		const gchar *tmp;

		tmp = gnome_vfs_uri_get_basename(new_uri);
		if (tmp == NULL)
			new_name = tmp;
		else
			new_name = gnome_vfs_unescape_string(tmp, "/");
	}

	if (move_folder == TRUE)
	{
		res = rio_to_vfs_error(rio_rename_folder(rio->rio_hw,
					old_folder, (gchar *)new_name));
	} else {
		res = rio_to_vfs_error(rio_rename_song(rio->rio_hw,
					old_folder, old_song,
					(gchar *)new_name));
	}
	rio_delete(rio->rio_hw);

	if (res == GNOME_VFS_OK)
		rio->dirty = TRUE;

	UNLOCK_RIO();

	return res;
}


static GnomeVFSResult
do_truncate_handle (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSFileSize where,
		    GnomeVFSContext *context)

{
	DEBUG_RIO(("do_truncate_handle\n"));
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_get_file_info_from_handle (GnomeVFSMethod *method,
			      GnomeVFSMethodHandle *method_handle,
			      GnomeVFSFileInfo *file_info,
			      GnomeVFSFileInfoOptions options,
			      GnomeVFSContext *context)


{
	DEBUG_RIO(("do_get_file_info_from_handle\n"));
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_make_directory (GnomeVFSMethod *method,
		   GnomeVFSURI *uri,
		   guint perm,
		   GnomeVFSContext *context)
{
	GnomeVFSResult res;
	gint card, folder, song;
	const gchar *folder_name = NULL;

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_make_directory(): lookup %d %d %d\n",
				card, folder, song));
	LOCK_RIO();
	if (res == GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_FILE_EXISTS;
	}
	if (res == GNOME_VFS_ERROR_IO)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}

	if (res == GNOME_VFS_ERROR_INVALID_URI
			|| card == -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_PERMITTED;
	}

	if (folder != -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_PERMITTED;
	}

	rio->rio_hw = rio_new();
	if (rio_check(rio->rio_hw) == FALSE)
	{
		DEBUG_RIO(("do_make_directory: check of rio is FALSE, destroying\n"));
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}
	rio_set_card(rio->rio_hw, card);
	{
		const gchar *tmp;

		tmp = gnome_vfs_uri_get_basename(uri);
		if (tmp == NULL)
			folder_name = tmp;
		else
			folder_name = gnome_vfs_unescape_string(tmp, "/");
	}
	res = rio_to_vfs_error(rio_add_folder(rio->rio_hw,
			(gchar *)folder_name));
	rio_delete(rio->rio_hw);

	if (res == GNOME_VFS_OK)
		rio->dirty = TRUE;

	UNLOCK_RIO();

	return res;
}

static GnomeVFSResult
do_remove_directory (GnomeVFSMethod *method,
		     GnomeVFSURI *uri,
		     GnomeVFSContext *context)
{
	GnomeVFSResult res;
	gint card, folder, song;

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_remove_directory(): lookup %d %d %d\n",
				card, folder, song));

	LOCK_RIO();
	if (res != GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return res;
	}

	if (folder == -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	if (song != -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_A_DIRECTORY;
	}

	rio->rio_hw = rio_new();
	if (rio_check(rio->rio_hw) == FALSE)
	{
		DEBUG_RIO(("do_remove_directory: check of rio is FALSE, destroying\n"));
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}
	rio_set_card(rio->rio_hw, card);
	res = rio_to_vfs_error(rio_del_folder(rio->rio_hw, folder));
	rio_delete(rio->rio_hw);

	if (res == GNOME_VFS_OK)
		rio->dirty = TRUE;

	UNLOCK_RIO();

	return res;
}

static GnomeVFSResult
do_set_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  const GnomeVFSFileInfo *info,
		  GnomeVFSSetFileInfoMask mask,
		  GnomeVFSContext *context)
{
	GnomeVFSResult res = GNOME_VFS_OK;
	gint card, folder, song;
	gboolean rename_folder = FALSE;
	const gchar *new_name;

	DEBUG_RIO (("do_set_file_info: mask %d\n", mask));

	if (mask &
		(GNOME_VFS_SET_FILE_INFO_PERMISSIONS |
		GNOME_VFS_SET_FILE_INFO_OWNER |
		GNOME_VFS_SET_FILE_INFO_TIME))
			return GNOME_VFS_ERROR_NOT_SUPPORTED;

	if (mask & GNOME_VFS_SET_FILE_INFO_NAME) {
		new_name = gnome_vfs_unescape_string(info->name, "/");
		DEBUG_RIO (("set_info: set new name: %s\n", new_name));

		res = lookup_uri(uri, &card, &folder, &song);
		DEBUG_RIO(("do_set_file_info(): lookup %d %d %d\n",
					card, folder, song));

		LOCK_RIO();
		if (res != GNOME_VFS_OK)
		{
			UNLOCK_RIO();
			return res;
		}

		if (folder == -1)
		{
			UNLOCK_RIO();
			return GNOME_VFS_ERROR_ACCESS_DENIED;
		}

		if (song == -1)
			rename_folder = TRUE;

		{
			GnomeVFSURI *new_uri;
			gchar *tmp, *new_filename;
			gint new_card, new_folder, new_song;

			tmp = gnome_vfs_uri_extract_dirname(uri);
			DEBUG_RIO(("do_set_file_info(): dirname %s\n",
						tmp));
			new_filename = g_strdup_printf("%s%s",
					tmp, new_name);
			DEBUG_RIO(("do_set_file_info(): filename %s\n",
						new_filename));
			new_uri = gnome_vfs_uri_new(new_filename);
			
			UNLOCK_RIO();
			res = lookup_uri(new_uri, &new_card, &new_folder,
					&new_song);
			LOCK_RIO();

			g_free(new_filename);
			g_free(tmp);
			gnome_vfs_uri_unref(new_uri);
			
			if (res != GNOME_VFS_ERROR_NOT_FOUND)
			{
				UNLOCK_RIO();
				return GNOME_VFS_ERROR_FILE_EXISTS;
			}
		}

		rio->rio_hw = rio_new();
		if (rio_check(rio->rio_hw) == FALSE)
		{
			DEBUG_RIO(("do_set_file_info: check of rio is FALSE, destroying\n"));
			UNLOCK_RIO();
			return GNOME_VFS_ERROR_IO;
		}
		rio_set_card(rio->rio_hw, card);

		if (rename_folder == TRUE)
		{
			res = rio_to_vfs_error(rio_rename_folder(rio->rio_hw,
						folder, (gchar *)new_name));
		} else {
			res = rio_to_vfs_error(rio_rename_song(rio->rio_hw,
						folder, song,
						(gchar *)new_name));
		}
		rio_delete(rio->rio_hw);

		if (res == GNOME_VFS_OK)
			rio->dirty = TRUE;

		UNLOCK_RIO();
	}

	return res;
}

static GnomeVFSResult
do_truncate (GnomeVFSMethod *method,
	     GnomeVFSURI *uri,
	     GnomeVFSFileSize where,
	     GnomeVFSContext *context)

{
	DEBUG_RIO(("do_truncate\n"));
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_find_directory (GnomeVFSMethod *method,
		   GnomeVFSURI *near_uri,
		   GnomeVFSFindDirectoryKind kind,
		   GnomeVFSURI **result_uri,
		   gboolean create_if_needed,
		   gboolean find_if_needed,
		   guint permissions,
		   GnomeVFSContext *context)
{
	DEBUG_RIO(("do_find_directory\n"));
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_create_symbolic_link (GnomeVFSMethod *method,
			 GnomeVFSURI *uri,
			 const char *target_reference,
			 GnomeVFSContext *context)
{
	DEBUG_RIO(("do_create_symbolic_link\n"));
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}


static GnomeVFSMethod method = {
	sizeof (GnomeVFSMethod),
	do_open, //FIXME
	NULL, //do_create,
	NULL, //do_close, //FIXME
	NULL, //do_read, //FIXME
	NULL, //do_write,
	NULL, //do_seek,
	NULL, //do_tell,
	do_truncate_handle,
	do_open_directory,
	do_close_directory,
	do_read_directory,
	do_get_file_info,
	NULL, //do_get_file_info_from_handle,
	do_is_local,
	do_make_directory,
	do_remove_directory,
	do_move,
	do_unlink,
	do_check_same_fs,
	do_set_file_info,
	do_truncate,
	do_find_directory,
	do_create_symbolic_link
};

GnomeVFSMethod *
vfs_module_init (const char *method_name, const char *args)
{
	rio_lock = g_mutex_new();

	DEBUG_RIO (("<-- rio500 module init called -->\n"));
	
	return &method;
}

void
vfs_module_shutdown (GnomeVFSMethod *method)
{
	DEBUG_RIO (("<-- rio500 module shutdown called -->\n"));

	destroy_rio(rio);
	g_mutex_free(rio_lock);
}

