/* gnome-gpg.c - wrapper around GNUPG to use GNOME Keyring

   Copyright (C) 2004 Colin Walters <walters@verbum.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <glib.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkmain.h>
#include <libgnomeui/gnome-password-dialog.h>
#include <libgnomeui/gnome-ui-init.h>
#include <gnome-keyring.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/wait.h>
#include <fcntl.h>

/* Temporary */
#define _(STR) (STR)

static void
error_dialog (const char *fmt, ...)
{
  GtkWidget *dialog;
  char buffer[1025];
  va_list args;

  va_start (args, fmt);
  
  g_vsnprintf (buffer, 1024, fmt, args);
  fprintf (stderr, "%s\n", buffer);

  va_end (args);
  
  dialog = gtk_message_dialog_new (NULL,
				   0,
				   GTK_MESSAGE_ERROR,
				   GTK_BUTTONS_OK,
				   "%s",
				   buffer);
  
/*   gtk_widget_show (GTK_WIDGET (dialog)); */
  gtk_dialog_run (GTK_DIALOG (dialog));
}

static char *
grovel_args_for_id (int argc, char **argv)
{
  int i, exit_status;
  GPtrArray *args = NULL;
  char *child_stdout= NULL;
  char *child_stderr = NULL;
  char **parsed_stdout = NULL;
  char **ptr = NULL;
  char *p = NULL;
  GError *error = NULL;
  char *id = NULL;

  for (i = 1; i < argc; i++) {
    gboolean short_found = FALSE;
    if (!strcmp ("--local-user", argv[i])
	|| (short_found = (!strcmp ("-u", argv[i])))) {
      if (short_found && i < argc-1)
	return g_strdup (argv[i+1]);
      else if (*(argv[i] + strlen ("--local-user")) == '=')
	return g_strdup (argv[i] + strlen ("--local-user") + 1);
      else if (i < argc-1)
	return g_strdup (argv[i+1]);
    }
  }

  args = g_ptr_array_new ();
  g_ptr_array_add (args, "gpg");
  g_ptr_array_add (args, "--list-secret-keys");
  g_ptr_array_add (args, NULL);
  g_spawn_sync (".",
		(char**) g_ptr_array_free (args, FALSE),
		NULL,
		G_SPAWN_SEARCH_PATH,
		NULL,
		NULL,
		&child_stdout,
		&child_stderr,
		&exit_status,
		&error);
  if (error) {
    fprintf (stderr, "%s", error->message);
    return NULL;
  }
  if (exit_status) {
    fprintf (stderr, "gpg exited abnormally");
    return NULL;
  }
  parsed_stdout = g_strsplit (child_stdout, "\n", 0);
  for (ptr = parsed_stdout; *ptr; ptr++) {
    if (!strncmp (*ptr, "sec ", 4)
	&& (*ptr)[10] == '/'
	&& (p = strchr (*ptr + 11, ' '))) {
      *p = '\0';
      id = g_strdup (*ptr + 11);
      break;
    }
  }
  g_strfreev (parsed_stdout);
  return id;
}

int
main (int argc, char **argv)
{
  pid_t pid;
  int pipefds[2];
  char *id = NULL;
  gboolean force_passphrase = FALSE;
  int i;

#if 0
  if (!gnome_keyring_is_available ()) {
    fprintf (stderr, "gnome-keyring is not available\n");
    exit (execvp ("gpg", argv));
  }
#endif

  id = grovel_args_for_id (argc, argv);
  if (!id) {
    fprintf (stderr, "Couldn't determine key id to use!");
    exit (execvp ("gpg", argv));
  }

  for (i = 1; i < argc; i++) {
    if (!strcmp ("--force-passphrase", argv[i]))
      force_passphrase = TRUE;
  }

  if (pipe (pipefds) < 0) {
    g_error ("Couldn't pipe: %s", g_strerror (errno));
    exit (1);
  }

  if ((pid = fork()) < 0) {
    g_error ("Couldn't fork: %s", g_strerror (errno));
    exit (1);
  } else if (pid == 0) {
    GPtrArray *args;
    char *fd_str;

    /* child */

    close (pipefds[1]);

    args = g_ptr_array_new ();

    g_ptr_array_add (args, "gpg");
    g_ptr_array_add (args, "--passphrase-fd");
    fd_str = g_strdup_printf ("%d", pipefds[0]);
    g_ptr_array_add (args, fd_str);
    for (i = 1; i < argc; i++) {
      if (strcmp ("--force-passphrase", argv[i]))
	g_ptr_array_add (args, argv[i]);
    }
    g_ptr_array_add (args, NULL);

    if (execvp ("gpg", (char **) g_ptr_array_free (args, FALSE)) < 0) {
      fprintf (stderr, "Couldn't exec gpg: %s", g_strerror (errno));
    }
    exit (1);
  } else {
    GnomeKeyringResult keyring_result;
    GnomeKeyringAttributeList *attributes = NULL;
    GList *found_items = NULL;
    char *passphrase = NULL;
    char *keyring = NULL;
    int status, ret;
    gboolean save_password = FALSE;
    int item_id;

    /* parent */

    close (pipefds[0]);
    gtk_set_locale ();
    gnome_program_init (PACKAGE_NAME, PACKAGE_VERSION,
			LIBGNOMEUI_MODULE, argc, argv,
			GNOME_PARAM_HUMAN_READABLE_NAME,
			_("GNU Privacy Guard"),
			NULL);

    while (gtk_events_pending ())
      gtk_main_iteration ();

    if (!force_passphrase) {
      attributes = gnome_keyring_attribute_list_new ();
      gnome_keyring_attribute_list_append_string (attributes,
						  "gnome-gpg-keyid",
						  id);
      keyring_result = gnome_keyring_find_items_sync (GNOME_KEYRING_ITEM_GENERIC_SECRET,
						      attributes,
						      &found_items);
      if (keyring_result != GNOME_KEYRING_RESULT_OK
	  && keyring_result != GNOME_KEYRING_RESULT_DENIED) {
	error_dialog (_("Couldn't search keyring (code %d)"), (int) keyring_result);
	kill (pid, SIGTERM);
	exit (1);
      }
      gnome_keyring_attribute_list_free (attributes);
    }
    if (found_items == NULL || force_passphrase) {
      GnomePasswordDialogRemember remember;
      GnomePasswordDialog *dialog;

      dialog = GNOME_PASSWORD_DIALOG (gnome_password_dialog_new (_("GNU Privacy Guard passphrase"),
								 _("Please enter your GNU Privacy Guard passphrase."),
								 NULL,
								 NULL,
								 TRUE));
      gnome_password_dialog_set_show_username (dialog, FALSE);
      gnome_password_dialog_set_show_domain (dialog, FALSE);
      gnome_password_dialog_set_show_remember (dialog, TRUE);

      if (!gnome_password_dialog_run_and_block (dialog)) {
	kill (pid, SIGTERM);
	exit (1);
      }
      
      passphrase = gnome_password_dialog_get_password (dialog);

      remember = gnome_password_dialog_get_remember (dialog);
      if (remember == GNOME_PASSWORD_DIALOG_REMEMBER_SESSION) {
	save_password = TRUE;
	keyring = "session";
      } else if (remember == GNOME_PASSWORD_DIALOG_REMEMBER_FOREVER) {
	save_password = TRUE;
	keyring = NULL;
      } else {
	save_password = FALSE;
      }
    } else {
      passphrase = ((GnomeKeyringFound *)found_items->data)->secret;
      save_password = FALSE;
    }
    
    write (pipefds[1], passphrase, strlen (passphrase));
    close (pipefds[1]);
    while ((ret = waitpid (pid, &status, 0)) < 0
	   && errno == EINTR)
      ;
    if (ret > 0 &&
	WIFEXITED (status) && WEXITSTATUS (status) == 0) {
      if (save_password) {
	gchar *nice_name = g_strdup_printf (_("GNU Privacy Guard passphrase for key ID: %s"),
					    id);
	attributes = gnome_keyring_attribute_list_new ();
	gnome_keyring_attribute_list_append_string (attributes,
						    "gnome-gpg-keyid",
						    id);
	keyring_result = gnome_keyring_item_create_sync (keyring,
							 GNOME_KEYRING_ITEM_GENERIC_SECRET,
							 nice_name,
							 attributes,
							 passphrase,
							 TRUE,
							 &item_id);
	gnome_keyring_attribute_list_free (attributes);
	g_free (nice_name);
	gnome_keyring_free_password (passphrase);
	if (keyring_result != GNOME_KEYRING_RESULT_OK) {
	  error_dialog (_("Couldn't store passphrase in keyring (code %d)"),
			(int) keyring_result);
	}
      }
      exit (0); 
    } else {
      gnome_keyring_free_password (passphrase);
      if (ret < 0) {
	error_dialog (_("Couldn't wait for gpg to exit: %s"),
		      strerror (errno));
	exit (1);
      } else {
	error_dialog (_("gpg exited abnormally"));
	if (WIFEXITED (status))
	  exit (WEXITSTATUS (status));
	else
	  exit (1);
      }
    }
  }
}
