/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* gdm-info-provider-usb.c
 *
 * Copyright (C) 2007 David Zeuthen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <config.h>
#include <glib/gi18n.h>
#include "gdm-info-provider.h"

/* can't make this const due to i18n */
struct usb_class_id {
        int class_id;
        int subclass_id;
        int protocol_id;
        char *short_name;
        char *long_name;
} static usb_class_names[] = {

        {0x01,   -1,   -1, N_("Audio"), N_("Audio")},
        {0x01, 0x01,   -1, N_("Audio Control"), N_("Audio Control")},
        {0x01, 0x02,   -1, N_("Audio Streaming"), N_("Audio Streaming")},
        {0x01, 0x03,   -1, N_("Audio MIDI Streaming"), N_("Audio MIDI Streaming")},

        {0x02,   -1,   -1, N_("Communications"), N_("Communications")},
        {0x02, 0x01,   -1, N_("Direct Line"), N_("Direct Line Communications")},
        {0x02, 0x02,   -1, N_("Modem"), N_("Modem Communications")},
        {0x02, 0x02, 0x01, N_("Modem (AT v.25ter)"), N_("Modem (AT v.25ter) Communications")},
        {0x02, 0x02, 0x02, N_("Modem (PCCA101)"), N_("Modem (PCCA101) Communications")},
        {0x02, 0x02, 0x03, N_("Modem (PCCA101)"), N_("Modem (PCCA101 + wakeup) Communications")},
        {0x02, 0x02, 0x04, N_("Modem (GSM)"), N_("Modem (GSM) Communications")},
        {0x02, 0x02, 0x05, N_("Modem (3G)"), N_("Modem (3G) Communications")},
        {0x02, 0x02, 0x06, N_("Modem (CDMA)"), N_("Modem (CDMA) Communications")},
        {0x02, 0x02, 0xfe, N_("Modem"), N_("Modem (Defined by command set descriptor) Communications")},
        {0x02, 0x02, 0xff, N_("Modem (Vendor Specific)"), N_("Modem (Vendor Specific) Communications")},
        {0x02, 0x03,   -1, N_("Telephone"), N_("Telephone Communications")},
        {0x02, 0x04,   -1, N_("Multi-Channel"), N_("Multi-Channel Communications")},
        {0x02, 0x05,   -1, N_("CAPI Control"), N_("CAPI Control")},
        {0x02, 0x06,   -1, N_("Ethernet Networking"), N_("Ethernet Networking")},
        {0x02, 0x07,   -1, N_("ATM Networking"), N_("ATM Networking")},
        {0x02, 0x08,   -1, N_("Wireless Handset Control"), N_("Wireless Handset Control")},
        {0x02, 0x09,   -1, N_("Device Management"), N_("Device Management")},
        {0x02, 0x0a,   -1, N_("Mobile Direct Line"), N_("Mobile Direct Line")},
        {0x02, 0x0b,   -1, N_("OBEX"), N_("OBEX")},
        {0x02, 0x0c,   -1, N_("Ethernet Emulation"), N_("Ethernet Emulation")},
        {0x02, 0x0c, 0x07, N_("Ethernet Emulation"), N_("Ethernet Emulation (EEM)")},

        {0x03,   -1,   -1, N_("HID Device"), N_("HID Device")},
        {0x03,   -1, 0x00, N_("HID Device"), N_("HID Device")},
        {0x03,   -1, 0x01, N_("Keyboard HID Device"), N_("Keyboard HID Device")},
        {0x03,   -1, 0x02, N_("Mouse HID Device"), N_("Mouse HID Device ")},
        {0x03, 0x01, 0x00, N_("HID Device"), N_("HID Device Interface (Boot)")},
        {0x03, 0x01, 0x01, N_("Keyboard HID Device"), N_("Keyboard HID Device Interface (Boot)")},
        {0x03, 0x01, 0x02, N_("Mouse HID Device"), N_("Mouse HID Device Interface (Boot)")},

        {0x06,   -1,   -1, N_("Imaging Device"), N_("Imaging Device")},
        {0x06, 0x01,   -1, N_("Still Image Capture"), N_("Still Image Capture")},
        {0x06, 0x01, 0x01, N_("PTP Imaging Device"), N_("PTP Imaging Device")},

        {0x07,   -1,   -1, N_("Printer"), N_("Printing")},
        {0x07, 0x01, 0x01, N_("Printer"), N_("Printing Interface (Unidirectional)")},
        {0x07, 0x01, 0x01, N_("Printer"), N_("Printing Interface (Bidirectional)")},
        {0x07, 0x01, 0x01, N_("Printer"), N_("Printing Interface (IEEE 1284.4 Compatible Bidirectional)")},
        {0x07, 0x01, 0xff, N_("Printer"), N_("Printing Interface (Vendor Specific)")},

        {0x08,   -1,   -1, N_("USB Mass Storage"), N_("USB Mass Storage")},
        {0x08, 0x01,   -1, N_("USB Mass Storage"), N_("USB Mass Storage (Flash)")},
        {0x08, 0x02,   -1, N_("USB Mass Storage"), N_("USB Mass Storage (SFF-8020i, MMC-2 (ATAPI))")},
        {0x08, 0x03,   -1, N_("USB Mass Storage"), N_("USB Mass Storage (QIC-157)")},
        {0x08, 0x04,   -1, N_("USB Mass Storage"), N_("USB Mass Storage (Floppy (UFI))")},
        {0x08, 0x05,   -1, N_("USB Mass Storage"), N_("USB Mass Storage (SFF-8070i)")},
        {0x08, 0x06,   -1, N_("USB Mass Storage"), N_("USB Mass Storage (SCSI)")},

        {0x09,   -1,   -1, N_("Hub"), N_("Hub")},
        {0x09, 0x00, 0x00, N_("Hub"), N_("Hub")},
        {0x09, 0x00, 0x01, N_("Hub"), N_("Hub Interface (Single TT)")},
        {0x09, 0x00, 0x02, N_("Hub"), N_("Hub Interface (TT per port)")},

        {0x0a,   -1,   -1, N_("CDC Data"), N_("CDC Data")},
        {0x0a, 0x00, 0x30, N_("I.430 ISDN BRI Data"), N_("I.430 ISDN BRI Data")},
        {0x0a,   -1, 0x31, N_("HDLC Data"), N_("HDLC Data")},
        {0x0a,   -1, 0x32, N_("Transparent Data"), N_("Transparent Data")},
        {0x0a,   -1, 0x50, N_("Q.921M Data"), N_("Q.921M Data")},
        {0x0a,   -1, 0x51, N_("Q.921 Data"), N_("Q.921 Data")},
        {0x0a,   -1, 0x52, N_("Q.921TM Data"), N_("Q.921TM Data")},
        {0x0a,   -1, 0x90, N_("V.42bis Data"), N_("V.42bis Data")},
        {0x0a,   -1, 0x91, N_("Q.932 EuroISDN Data"), N_("Q.932 EuroISDN Data")},
        {0x0a,   -1, 0x92, N_("V.120 V.24 rate ISDN Data"), N_("V.120 V.24 rate ISDN Data")},
        {0x0a,   -1, 0x93, N_("CAPI 2.0 Data"), N_("CAPI 2.0 Data")},
        {0x0a,   -1, 0xfd, N_("Host Based Data Driver"), N_("Host Based Driver Data")},
        {0x0a,   -1, 0xfe, N_("CDC PUF Data"), N_("CDC PUF Data")},
        {0x0a,   -1, 0xff, N_("Vendor Specific Data"), N_("Vendor Specific Data")},

        {0x0b,   -1,   -1, N_("Chip / Smart Card"), N_("Chip / Smart Card")},

        {0x0d,   -1,   -1, N_("Content Security"), N_("Content Security")},

        {0x0e,   -1,   -1, N_("Video"), N_("Video")},
        {0x0e, 0x01,   -1, N_("Video Control"), N_("Video Control")},
        {0x0e, 0x02,   -1, N_("Video Streaming"), N_("Video Streaming")},
        {0x0e, 0x03,   -1, N_("Video Interface Collection"), N_("Video Interface Collection")},

        {0xdc,   -1,   -1, N_("Diagnostics"), N_("Diagnostics")},

        {0xe0,   -1,   -1, N_("Wireless Adapter"), N_("Wireless Adapter")},
        {0xe0, 0x01,   -1, N_("Wireless Radio"), N_("Wireless Radio")},
        {0xe0, 0x01, 0x01, N_("Bluetooth Adapter"), N_("Bluetooth Adapter")},
        {0xe0, 0x01, 0x02, N_("Ultra Wideband Radio Control"), N_("Ultra Wideband Radio Control")},
        {0xe0, 0x01, 0x03, N_("RNDIS"), N_("RNDIS")},
        {0xe0, 0x02,   -1, N_("Wireless USB Wire Adapter"), N_("Wireless USB Wire Adapter")},
        {0xe0, 0x02, 0x01, N_("Wireless USB Wire Adapter"), N_("Host Wire Adapter Control/Data Streaming")},
        {0xe0, 0x02, 0x02, N_("Wireless USB Wire Adapter"), N_("Device Wire Adapter Control/Data Streaming")},
        {0xe0, 0x02, 0x03, N_("Wireless USB Wire Adapter"), N_("Device Wire Adapter Isochronous Streaming")},

        {0xef,   -1,   -1, N_("Miscellanous"), N_("Miscellanous")},
        {0xef, 0x01, 0x01, N_("MS ActiveSync"), N_("MS ActiveSync")},
        {0xef, 0x01, 0x02, N_("Palm Sync"), N_("Palm Sync")},
        {0xef, 0x02,   -1, N_("Miscellanous Common"), N_("Miscellanous Common")},
        {0xef, 0x02, 0x01, N_("Interface Association"), N_("Interface Association")},
        {0xef, 0x02, 0x02, N_("Wire Adapter Multifunction"), N_("Wire Adapter Multifunction Peripheral")},
        {0xef, 0x03, 0x01, N_("Cable Based Association"), N_("Cable Based Association")},

        {0xfe,   -1,   -1, N_("Application Specific"), N_("Application Specific")},
        {0xfe, 0x01,   -1, N_("Device Firmware Update"), N_("Device Firmware Update")},
        {0xfe, 0x02,   -1, N_("IRDA Bridge"), N_("IRDA Bridge")},
        {0xfe, 0x03,   -1, N_("Test and Measurement"), N_("Test and Measurement")},
        {0xfe, 0x03, 0x01, N_("TMC Test and Measurement"), N_("TMC Test and Measurement")},
        {0xfe, 0x03, 0x02, N_("USB488 Test and Measurement"), N_("USB488 Test and Measurement")},
};


static const char *
get_class_name (int dc, int ds, int dp, gboolean get_short_name)
{
        int i;
        for (i = sizeof (usb_class_names) / sizeof (struct usb_class_id) - 1; i >= 0; i--) {
                struct usb_class_id *cid = usb_class_names + i;
                if ((cid->class_id == -1    || cid->class_id == dc) &&
                    (cid->subclass_id == -1 || cid->subclass_id == ds) &&
                    (cid->protocol_id == -1 || cid->protocol_id == dp)) {
                        if (get_short_name)
                                return cid->short_name;
                        else
                                return cid->long_name;
                }
        }
        return NULL;
}

static char *
get_name (GdmDevice *device, gboolean get_short_name, gboolean is_if)
{
        const char *s;
        char *name;
        int dc, ds, dp;

        name = NULL;
        if (is_if) {
                dc = gdm_device_get_property_int (device, "usb.interface.class");
                ds = gdm_device_get_property_int (device, "usb.interface.subclass");
                dp = gdm_device_get_property_int (device, "usb.interface.protocol");
                s = get_class_name (dc, ds, dp, get_short_name);
                if (s == NULL)
                        name = g_strdup (_("USB Interface"));
                else
                        name = g_strdup_printf (_("%s Interface"), s);
        } else {
                dc = gdm_device_get_property_int (device, "usb_device.device_class");
                ds = gdm_device_get_property_int (device, "usb_device.device_subclass");
                dp = gdm_device_get_property_int (device, "usb_device.device_protocol");
                s = get_class_name (dc, ds, dp, get_short_name);
                if (s == NULL)
                        name = g_strdup (_("USB Device"));
                else
                        name = g_strdup (s);
        }

        return name;
}


static const char *
bcd2str (int bcd)
{
        int i, j;
        static char buf[10];

        j = 0;
        for (i = 7; i >= 0; i--) {
                int digit;

                if (i == 1)
                        buf[j++] = '.';

                digit = (bcd >> (i * 4)) & 0x0f;
                if (digit == 0 && j == 0)
                        continue;

                buf[j++] = '0' + digit;
        }
        buf[j] = '\0';

#if 0
        j--;
        while (j >= 0 && (buf[j] == '0' || buf[j] == '.')) {
                if (buf[j] == '.') {
                        buf[j--] = '\0';
                        break;
                }
                
                buf[j--] = '\0';
        }
#endif

        return buf;
}

/***************************************** USB device *****************************************/

static gboolean 
get_provider_matches (GdmDevice *device)
{
        const char *subsys;

        subsys = gdm_device_get_property_string (device, "info.subsystem");
        if (subsys != NULL && g_ascii_strcasecmp (subsys, "usb_device") == 0)
                return TRUE;

        return FALSE;
}

static char *
get_icon_name (GdmDevice *device)
{
        return g_strdup ("gnome-device-manager-device-usb");
}

static char *
get_short_name (GdmDevice *device)
{
        return get_name (device, TRUE, FALSE);
}

static char *
get_long_name (GdmDevice *device)
{
        return get_name (device, FALSE, FALSE);
}

static char *
get_vendor (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "usb_device.vendor"));
}

static char *
get_product (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "usb_device.product"));
}

#define ADD_SUM(p, key, value)                                                       \
        do {                                                                         \
                if (value != NULL) {                                                 \
                        p = g_slist_append (p, g_strdup (key));                      \
                        p = g_slist_append (p, value);                               \
                }                                                                    \
        } while (FALSE)

static GSList *
get_summary (GdmDevice *device)
{
        GSList *p;
        int vid;
        int pid;
        int num_ports;
        int num_if;
        int num_cfg;
        int cur_cfg;
        int speed_bcd;
        int version_bcd;
        int device_rev_bcd;
        int max_power;
        gboolean is_self_powered;
        gboolean can_wake_up;
        const char *vendor;
        const char *product;
        const char *serial;

        p = NULL;

        vid = gdm_device_get_property_int (device, "usb_device.vendor_id");
        pid = gdm_device_get_property_int (device, "usb_device.product_id");
        num_ports = gdm_device_get_property_int (device, "usb_device.num_ports");
        num_if = gdm_device_get_property_int (device, "usb_device.num_interfaces");
        num_cfg = gdm_device_get_property_int (device, "usb_device.num_configurations");
        cur_cfg = gdm_device_get_property_int (device, "usb_device.configuration_value");
        speed_bcd = gdm_device_get_property_int (device, "usb_device.speed_bcd");
        version_bcd = gdm_device_get_property_int (device, "usb_device.version_bcd");
        device_rev_bcd = gdm_device_get_property_int (device, "usb_device.device_revision_bcd");
        max_power = gdm_device_get_property_int (device, "usb_device.max_power");
        
        is_self_powered = gdm_device_get_property_bool (device, "usb_device.is_self_powered");
        can_wake_up = gdm_device_get_property_bool (device, "usb_device.can_wake_up");
        
        vendor = gdm_device_get_property_string (device, "usb_device.vendor");
        product = gdm_device_get_property_string (device, "usb_device.product");
        serial = gdm_device_get_property_string (device, "usb_device.serial");
        
        if (product != NULL)
                ADD_SUM (p, _("Model"), g_strdup (product));
        else
                ADD_SUM (p, _("Model"), g_strdup_printf (_("Unknown Model (id = 0x%04x)"), pid));
        
        if (vendor != NULL)
                ADD_SUM (p, _("Vendor"), g_strdup (vendor));
        else
                ADD_SUM (p, _("Vendor"), g_strdup_printf (_("Unknown Vendor (id = 0x%04x)"), vid));
        
        if (device_rev_bcd > 0)
                ADD_SUM (p, _("Revision"), g_strdup (bcd2str (device_rev_bcd)));
        
        ADD_SUM (p, _("Serial Number"), g_strdup (serial));
        ADD_SUM (p, _("Connection"), g_strdup (_("USB (Universal Serial Bus)")));
        ADD_SUM (p, _("USB Version"), g_strdup (bcd2str (version_bcd)));
        ADD_SUM (p, _("Connected at"), g_strdup_printf (_("%s Mbit/s"), bcd2str (speed_bcd)));
        if (num_ports > 0)
                ADD_SUM (p, _("Number of ports"), g_strdup_printf ("%d", num_ports));
        ADD_SUM (p, _("Remote Wakeup"), g_strdup (can_wake_up ? _("Yes") : _("No")));
        ADD_SUM (p, _("Bus Powered"), g_strdup (is_self_powered ? _("No") : _("Yes")));
        if (max_power > 0) {
                ADD_SUM (p, _("Max. Power"), g_strdup_printf (_("%d mA"), max_power));
        }
        if (num_cfg > 1)
                ADD_SUM (p, _("Current Configuration"), g_strdup_printf (_("%d (%d possible)"), 
                                                                                   cur_cfg, num_cfg));
        return p;
}


static GSList *
get_warnings (GdmDevice *device)
{
        int speed_bcd;
        int version_bcd;
        const char *sysfs_path;
        gboolean can_do_hispeed;
        GSList *p = NULL;

        sysfs_path = gdm_device_get_property_string (device, "usb_device.linux.sysfs_path");
        speed_bcd = gdm_device_get_property_int (device, "usb_device.speed_bcd");
        version_bcd = gdm_device_get_property_int (device, "usb_device.version_bcd");

        /* TODO: abstract this in HAL
         * TODO: this is a heuristic - verify it's correct 
         */
        can_do_hispeed = FALSE;
        if (sysfs_path != NULL && version_bcd >= 0x0200) {
                char *filename;
                char *buf;
                filename = g_strdup_printf ("%s/bMaxPacketSize0", sysfs_path);
                if (g_file_get_contents (filename, &buf, NULL, NULL)) {
                        if (g_ascii_strncasecmp (buf, "64", 2) == 0) {
                                can_do_hispeed = TRUE;
                        }
                        g_free (buf);
                }
                g_free (filename);
        }
        if (can_do_hispeed && speed_bcd < 0x48000) {
                p = g_slist_append (p, gdm_info_provider_tip_new (
                                            -1, 
                                            _("Hi-Speed USB device is connected to a slow port."),
                                            "Resolve..."));
        }
        


        return p;
}


static GSList *
get_errors (GdmDevice *device)
{
        int cur_cfg;
        gboolean insufficient_power;
        GSList *p = NULL;

        cur_cfg = gdm_device_get_property_int (device, "usb_device.configuration_value");

        /* TODO: this is a heuristic - verify it's correct */
        insufficient_power = FALSE;
        if (cur_cfg < 1) {
                insufficient_power = TRUE;
        }
        if (insufficient_power) {
                p = g_slist_append (p, gdm_info_provider_tip_new (
                                            -1, 
                                            _("Insufficient power to operate USB device."),
                                            NULL));
        }

        return p;
}


static GSList *
get_notices (GdmDevice *device)
{
        GSList *p = NULL;

        return p;
}

GdmInfoProviderIface gdm_info_provider_usb =
{
        .get_provider_matches = get_provider_matches,
        .get_icon_name        = get_icon_name,
        .get_short_name       = get_short_name,
        .get_long_name        = get_long_name,
        .get_vendor           = get_vendor,
        .get_product          = get_product,
        .get_summary          = get_summary,
        .get_errors           = get_errors,
        .get_warnings         = get_warnings,
        .get_notices          = get_notices
};


/***************************************** USB interface *****************************************/

static gboolean 
get_provider_matches_if (GdmDevice *device)
{
        const char *subsys;

        subsys = gdm_device_get_property_string (device, "info.subsystem");
        if (subsys != NULL && g_ascii_strcasecmp (subsys, "usb") == 0)
                return TRUE;

        return FALSE;
}

static char *
get_icon_name_if (GdmDevice *device)
{
        return g_strdup ("gnome-device-manager-device-usb-interface");
}

static char *
get_short_name_if (GdmDevice *device)
{
        return get_name (device, TRUE, TRUE);
}

static char *
get_long_name_if (GdmDevice *device)
{
        return get_name (device, FALSE, TRUE);
}

static char *
get_vendor_if (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "usb.vendor"));
}

static char *
get_product_if (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "usb.product"));
}

static GSList *
get_summary_if (GdmDevice *device)
{
        int if_num;
        int num_if;
        int dc, ds, dp;
        const char *description;
        GSList *p = NULL;
                
        dc = gdm_device_get_property_int (device, "usb.interface.class");
        ds = gdm_device_get_property_int (device, "usb.interface.subclass");
        dp = gdm_device_get_property_int (device, "usb.interface.protocol");
        if_num = gdm_device_get_property_int (device, "usb.interface.number");
        num_if = gdm_device_get_property_int (device, "usb.num_interfaces");
        description = gdm_device_get_property_string (device, "usb.interface.description");
        
        if (description != NULL)
                ADD_SUM (p, _("Description"), g_strdup (description));
        ADD_SUM (p, _("USB Interface Number"), g_strdup_printf (_("%d (of %d)"), if_num + 1, num_if));
        ADD_SUM (p, _("Class/Subclass/Protocol"), g_strdup_printf ("%02x/%02x/%02x", dc, ds, dp));

        return p;
}


GdmInfoProviderIface gdm_info_provider_usb_if =
{
        .get_provider_matches = get_provider_matches_if,
        .get_icon_name        = get_icon_name_if,
        .get_short_name       = get_short_name_if,
        .get_long_name        = get_long_name_if,
        .get_vendor           = get_vendor_if,
        .get_product          = get_product_if,
        .get_summary          = get_summary_if,
};

/***************************************** USB raw *****************************************/

static gboolean 
get_provider_matches_raw (GdmDevice *device)
{
        return gdm_device_test_capability (device, "usbraw");
}

static char *
get_icon_name_raw (GdmDevice *device)
{
        return g_strdup ("gnome-device-manager-device-usb-interface");
}

static char *
get_short_name_raw (GdmDevice *device)
{
        return g_strdup (_("USB Raw Device"));
}

static char *
get_long_name_raw (GdmDevice *device)
{
        return g_strdup (_("USB Raw Device Access"));
}

static GSList *
get_summary_raw (GdmDevice *device)
{
        const char *device_file;
        GSList *p = NULL;

        device_file = gdm_device_get_property_string (device, "usbraw.device");
        
        ADD_SUM (p, _("Device File"), g_strdup (device_file));

        return p;
}


GdmInfoProviderIface gdm_info_provider_usb_raw =
{
        .get_provider_matches = get_provider_matches_raw,
        .get_icon_name        = get_icon_name_raw,
        .get_short_name       = get_short_name_raw,
        .get_long_name        = get_long_name_raw,
        .get_summary          = get_summary_raw,
};
