/* gnome-drud.c
 * Copyright (C) 1999 Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gnome.h>
#include "gnome-drud.h"
enum {
	CANCEL,
	LAST_SIGNAL
};
static void gnome_drud_init		(GnomeDrud		 *drud);
static void gnome_drud_class_init	(GnomeDrudClass	 *klass);
static void gnome_drud_destroy         (GtkObject               *object);
static void gnome_drud_size_request    (GtkWidget               *widget,
					 GtkRequisition          *requisition);
static void gnome_drud_size_allocate   (GtkWidget               *widget,
					 GtkAllocation           *allocation);
static void gnome_drud_draw            (GtkWidget               *widget,
					 GdkRectangle            *area);
static gint gnome_drud_expose          (GtkWidget               *widget,
					 GdkEventExpose          *event);
static void gnome_drud_map             (GtkWidget               *widget);
static void gnome_drud_unmap           (GtkWidget               *widget);
static GtkType gnome_drud_child_type   (GtkContainer            *container);
static void gnome_drud_add             (GtkContainer            *widget,
					 GtkWidget               *page);
static void gnome_drud_remove          (GtkContainer            *widget,
					 GtkWidget               *child);
static void gnome_drud_forall          (GtkContainer            *container,
					 gboolean                include_internals,
					 GtkCallback             callback,
					 gpointer                callback_data);
static void gnome_drud_back_callback   (GtkWidget               *button,
					 GnomeDrud              *drud);
static void gnome_drud_next_callback   (GtkWidget               *button,
					 GnomeDrud              *drud);
static void gnome_drud_cancel_callback (GtkWidget               *button,
					 GtkWidget               *drud);
static GtkContainerClass *parent_class = NULL;
static guint drud_signals[LAST_SIGNAL] = { 0 };


GtkType
gnome_drud_get_type (void)
{
  static GtkType drud_type = 0;

  if (!drud_type)
    {
      static const GtkTypeInfo drud_info =
      {
        "GnomeDrud",
        sizeof (GnomeDrud),
        sizeof (GnomeDrudClass),
        (GtkClassInitFunc) gnome_drud_class_init,
        (GtkObjectInitFunc) gnome_drud_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      drud_type = gtk_type_unique (gtk_container_get_type (), &drud_info);
    }

  return drud_type;
}

static void
gnome_drud_class_init (GnomeDrudClass *klass)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;
	GtkContainerClass *container_class;

	object_class = (GtkObjectClass*) klass;
	widget_class = (GtkWidgetClass*) klass;
	container_class = (GtkContainerClass*) klass;
	parent_class = gtk_type_class (gtk_container_get_type ());

	drud_signals[CANCEL] = 
		gtk_signal_new ("cancel",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GnomeDrudClass, cancel),
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);
	
	gtk_object_class_add_signals (object_class, drud_signals, LAST_SIGNAL);
	
	object_class->destroy = gnome_drud_destroy;
	widget_class->size_request = gnome_drud_size_request;
	widget_class->size_allocate = gnome_drud_size_allocate;
	widget_class->map = gnome_drud_map;
	widget_class->unmap = gnome_drud_unmap;
	widget_class->draw = gnome_drud_draw;
	widget_class->expose_event = gnome_drud_expose;

	container_class->forall = gnome_drud_forall;
	container_class->add = gnome_drud_add;
	container_class->remove = gnome_drud_remove;
	container_class->child_type = gnome_drud_child_type;
}

static void
gnome_drud_init (GnomeDrud *drud)
{
	GtkWidget *pixmap;

	/* set up the buttons */
	GTK_WIDGET_SET_FLAGS (GTK_WIDGET (drud), GTK_NO_WINDOW);
	pixmap =  gnome_stock_pixmap_widget(NULL, GNOME_STOCK_BUTTON_PREV);
	drud->back = gnome_pixmap_button (pixmap, _("Back"));
	GTK_WIDGET_SET_FLAGS (drud->back, GTK_CAN_DEFAULT);
	drud->next = gnome_stock_or_ordinary_button (GNOME_STOCK_BUTTON_NEXT);
	GTK_WIDGET_SET_FLAGS (drud->next, GTK_CAN_DEFAULT);
	drud->cancel = gnome_stock_or_ordinary_button (GNOME_STOCK_BUTTON_CANCEL);
	GTK_WIDGET_SET_FLAGS (drud->cancel, GTK_CAN_DEFAULT);
	pixmap =  gnome_stock_pixmap_widget(NULL, GNOME_STOCK_BUTTON_APPLY);
	drud->finish = gnome_pixmap_button (pixmap, _("Finish"));
	GTK_WIDGET_SET_FLAGS (drud->finish, GTK_CAN_DEFAULT);
	gtk_widget_set_parent (drud->back, GTK_WIDGET (drud));
	gtk_widget_set_parent (drud->next, GTK_WIDGET (drud));
	gtk_widget_set_parent (drud->cancel, GTK_WIDGET (drud));
	gtk_widget_set_parent (drud->finish, GTK_WIDGET (drud));
	gtk_widget_show (drud->back);
	gtk_widget_show (drud->next);
	gtk_widget_show (drud->cancel);
	gtk_widget_show (drud->finish);

	/* other flags */
	drud->current = NULL;
	drud->children = NULL;
	drud->show_finish = FALSE;
	gtk_signal_connect (GTK_OBJECT (drud->back),
			    "clicked",
			    gnome_drud_back_callback,
			    drud);
	gtk_signal_connect (GTK_OBJECT (drud->next),
			    "clicked",
			    gnome_drud_next_callback,
			    drud);
	gtk_signal_connect (GTK_OBJECT (drud->cancel),
			    "clicked",
			    gnome_drud_cancel_callback,
			    drud);
	gtk_signal_connect (GTK_OBJECT (drud->finish),
			    "clicked",
			    gnome_drud_next_callback,
			    drud);
}



static void
gnome_drud_destroy (GtkObject *object)
{
	GnomeDrud *drud;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_IS_DRUD (object));

	drud = GNOME_DRUD (object);
        GTK_OBJECT_CLASS(parent_class)->destroy(object);

	gtk_widget_destroy (drud->back);
	gtk_widget_destroy (drud->next);
	gtk_widget_destroy (drud->cancel);
	gtk_widget_destroy (drud->finish);
	g_list_free (drud->children);
        drud->children = NULL;

}

static void
gnome_drud_size_request (GtkWidget *widget,
			  GtkRequisition *requisition)
{
	guint16 temp_width, temp_height;
	GList *list;
	GnomeDrud *drud;
	GtkRequisition child_requisition;
	GnomeDrudPage *child;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GNOME_IS_DRUD (widget));

	drud = GNOME_DRUD (widget);
	temp_height = temp_width = 0;

	/* We find the maximum size of all children widgets */
	for (list = drud->children; list; list = list->next) {
		child = GNOME_DRUD_PAGE (list->data);
		if (GTK_WIDGET_VISIBLE (child)) {
			gtk_widget_size_request (GTK_WIDGET (child), &child_requisition);
			temp_width = MAX (temp_width, child_requisition.width);
			temp_height = MAX (temp_height, child_requisition.height);
		}
	}
	
        requisition->width = temp_width + 2 * GNOME_PAD_SMALL;
        requisition->height = temp_height + 2 * GNOME_PAD_SMALL;

	/* In an Attempt to show how the widgets are packed,
	 * here's a little diagram.
	 * 
	 * ------------- [  Back  ] [  Next  ]    [ Cancel ]
	 *    \
	 *     This part needs to be at least 1 button width.
	 *     In addition, there is 1/4 X Button width between Cancel and Next,
	 *     and a GNOME_PAD_SMALL between Next and Back.
	 */
	/* our_button width is temp_width and temp_height */
	temp_height = 0;
	temp_width = 0;

	gtk_widget_size_request (drud->back, &child_requisition);
	temp_width = MAX (temp_width, child_requisition.width);
	temp_height = MAX (temp_height, child_requisition.height);

	gtk_widget_size_request (drud->next, &child_requisition);
	temp_width = MAX (temp_width, child_requisition.width);
	temp_height = MAX (temp_height, child_requisition.height);

	gtk_widget_size_request (drud->cancel, &child_requisition);
	temp_width = MAX (temp_width, child_requisition.width);
	temp_height = MAX (temp_height, child_requisition.height);

	gtk_widget_size_request (drud->finish, &child_requisition);
	temp_width = MAX (temp_width, child_requisition.width);
	temp_height = MAX (temp_height, child_requisition.height);

	temp_width += GNOME_PAD_SMALL * 2;
	temp_height += GNOME_PAD_SMALL;
	/* FIXME. do we need to do something with the buttons requisition? */
	temp_width = temp_width * 17/4  + GNOME_PAD_SMALL * 3;

	/* pick which is bigger, the buttons, or the GnomeDrudPages */
	requisition->width = MAX (temp_width, requisition->width);
	requisition->height += temp_height + GNOME_PAD_SMALL * 2;
	/* And finally, put the side padding in */
	requisition->width += GNOME_PAD_SMALL *2;
}
static void
gnome_drud_size_allocate (GtkWidget *widget,
			   GtkAllocation *allocation)
{
	GnomeDrud *drud;
	GtkAllocation child_allocation;
	gint button_height;
	GList *list;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GNOME_IS_DRUD (widget));

	drud = GNOME_DRUD (widget);
	widget->allocation = *allocation;

		
		

	/* deal with the buttons */
	child_allocation.width = child_allocation.height = 0;
	child_allocation.width = drud->back->requisition.width;
	child_allocation.height = drud->back->requisition.height;
	child_allocation.width = MAX (child_allocation.width,
			    drud->next->requisition.width);
	child_allocation.height = MAX (child_allocation.height,
			    drud->next->requisition.height);
	child_allocation.width = MAX (child_allocation.width,
			    drud->cancel->requisition.width);
	child_allocation.height = MAX (child_allocation.height,
			    drud->cancel->requisition.height);

	child_allocation.height += GNOME_PAD_SMALL;
	button_height = child_allocation.height;
	child_allocation.width += 2 * GNOME_PAD_SMALL;
	child_allocation.x = allocation->x + allocation->width - GNOME_PAD_SMALL - child_allocation.width;
	child_allocation.y = allocation->y + allocation->height - GNOME_PAD_SMALL - child_allocation.height;
	gtk_widget_size_allocate (drud->cancel, &child_allocation);
	child_allocation.x -= (child_allocation.width * 5 / 4);
	gtk_widget_size_allocate (drud->next, &child_allocation);
	gtk_widget_size_allocate (drud->finish, &child_allocation);
	child_allocation.x -= (GNOME_PAD_SMALL + child_allocation.width);
	gtk_widget_size_allocate (drud->back, &child_allocation);

	/* Put up the GnomeDrudPage */
	child_allocation.x = allocation->x + GNOME_PAD_SMALL;
	child_allocation.y = allocation->y + GNOME_PAD_SMALL;
	child_allocation.width =
		((allocation->width - 2* GNOME_PAD_SMALL) > 0) ?
		(allocation->width - 2* GNOME_PAD_SMALL):0;
	child_allocation.height =
		((allocation->height - 3 * GNOME_PAD_SMALL - button_height) > 0) ?
		(allocation->height - 3 * GNOME_PAD_SMALL - button_height):0;
	for (list = drud->children; list; list=list->next) {
		if (GTK_WIDGET_VISIBLE (list->data)) {
			gtk_widget_size_allocate (GTK_WIDGET (list->data), &child_allocation);
		}
	}
}

static GtkType
gnome_drud_child_type (GtkContainer *container)
{
	return gnome_drud_page_get_type ();
}

static void
gnome_drud_map (GtkWidget *widget)
{
	GnomeDrud *drud;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GNOME_IS_DRUD (widget));

	drud = GNOME_DRUD (widget);
	GTK_WIDGET_SET_FLAGS (drud, GTK_MAPPED);

	gtk_widget_map (drud->back);
	if (drud->show_finish)
		gtk_widget_map (drud->finish);
	else
		gtk_widget_map (drud->next);
	gtk_widget_map (drud->cancel);
	if (drud->current &&
	    GTK_WIDGET_VISIBLE (drud->current) &&
	    !GTK_WIDGET_MAPPED (drud->current)) {
		gtk_widget_map (GTK_WIDGET (drud->current));
	}
}

static void
gnome_drud_unmap (GtkWidget *widget)
{
	GnomeDrud *drud;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GNOME_IS_DRUD (widget));

	drud = GNOME_DRUD (widget);
	GTK_WIDGET_UNSET_FLAGS (drud, GTK_MAPPED);

	gtk_widget_unmap (drud->back);
	if (drud->show_finish)
		gtk_widget_unmap (drud->finish);
	else
		gtk_widget_unmap (drud->next);
	gtk_widget_unmap (drud->cancel);
	if (drud->current &&
	    GTK_WIDGET_VISIBLE (drud->current) &&
	    GTK_WIDGET_MAPPED (drud->current))
		gtk_widget_unmap (GTK_WIDGET (drud->current));
}
static void
gnome_drud_add (GtkContainer *widget,
		 GtkWidget *page)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GNOME_IS_DRUD (widget));
	g_return_if_fail (page != NULL);
	g_return_if_fail (GNOME_IS_DRUD_PAGE (page));

	gnome_drud_append_page (GNOME_DRUD (widget), GNOME_DRUD_PAGE (page));
}
static void
gnome_drud_remove (GtkContainer *widget,
		    GtkWidget *child)
{
	GnomeDrud *drud = GNOME_DRUD (widget);
	drud->children = g_list_remove (drud->children, child);
	gtk_widget_unparent (child);
}

static void
gnome_drud_forall (GtkContainer *container,
		    gboolean      include_internals,
		    GtkCallback   callback,
		    gpointer      callback_data)
{
	GnomeDrud *drud;
	GnomeDrudPage *child;
	GList *children;

	g_return_if_fail (container != NULL);
	g_return_if_fail (GNOME_IS_DRUD (container));
	g_return_if_fail (callback != NULL);

	drud = GNOME_DRUD (container);

	children = drud->children;
	while (children) {
		child = children->data;
		children = children->next;

		(* callback) (GTK_WIDGET (child), callback_data);
	}
	if (include_internals) {
		(* callback) (drud->back, callback_data);
		(* callback) (drud->next, callback_data);
		(* callback) (drud->cancel, callback_data);
		(* callback) (drud->finish, callback_data);
	}
}

static void
gnome_drud_draw (GtkWidget    *widget,
		  GdkRectangle *area)
{
	GnomeDrud *drud;
	GdkRectangle child_area;
	GtkWidget *child;
	GList *children;
  
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GNOME_IS_DRUD (widget));

	if (GTK_WIDGET_DRAWABLE (widget)) {
		drud = GNOME_DRUD (widget);
		children = drud->children;

		while (children) {
			child = GTK_WIDGET (children->data);
			children = children->next;
	     
			if (GTK_WIDGET_DRAWABLE (child) && gtk_widget_intersect (child, area, &child_area))
				gtk_widget_draw (child, &child_area);
		}
		child = drud->back;
		if (GTK_WIDGET_DRAWABLE (child) && gtk_widget_intersect (child, area, &child_area))
			gtk_widget_draw (child, &child_area);
		child = drud->next;
		if (GTK_WIDGET_DRAWABLE (child) && gtk_widget_intersect (child, area, &child_area))
			gtk_widget_draw (child, &child_area);
		child = drud->cancel;
		if (GTK_WIDGET_DRAWABLE (child) && gtk_widget_intersect (child, area, &child_area))
			gtk_widget_draw (child, &child_area);
		child = drud->finish;
		if (GTK_WIDGET_DRAWABLE (child) && gtk_widget_intersect (child, area, &child_area))
			gtk_widget_draw (child, &child_area);
	}
}

static gint
gnome_drud_expose (GtkWidget      *widget,
		    GdkEventExpose *event)
{
	GnomeDrud *drud;
	GtkWidget *child;
	GdkEventExpose child_event;
	GList *children;

	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GNOME_IS_DRUD (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	if (GTK_WIDGET_DRAWABLE (widget)) {
		drud = GNOME_DRUD (widget);
		child_event = *event;
		children = drud->children;

		while (children) {
			child = GTK_WIDGET (children->data);
			children = children->next;

			if (GTK_WIDGET_DRAWABLE (child) &&
			    GTK_WIDGET_NO_WINDOW (child) &&
			    gtk_widget_intersect (child, &event->area, &child_event.area)) {
				gtk_widget_event (child, (GdkEvent*) &child_event);
			}
		}
		child = drud->back;
		if (GTK_WIDGET_DRAWABLE (child) &&
		    GTK_WIDGET_NO_WINDOW (child) &&
		    gtk_widget_intersect (child, &event->area, &child_event.area))
			gtk_widget_event (child, (GdkEvent*) &child_event);
		child = drud->next;
		if (GTK_WIDGET_DRAWABLE (child) &&
		    GTK_WIDGET_NO_WINDOW (child) &&
		    gtk_widget_intersect (child, &event->area, &child_event.area))
			gtk_widget_event (child, (GdkEvent*) &child_event);
		child = drud->cancel;
		if (GTK_WIDGET_DRAWABLE (child) &&
		    GTK_WIDGET_NO_WINDOW (child) &&
		    gtk_widget_intersect (child, &event->area, &child_event.area))
			gtk_widget_event (child, (GdkEvent*) &child_event);
		child = drud->finish;
		if (GTK_WIDGET_DRAWABLE (child) &&
		    GTK_WIDGET_NO_WINDOW (child) &&
		    gtk_widget_intersect (child, &event->area, &child_event.area))
			gtk_widget_event (child, (GdkEvent*) &child_event);
	}
	return FALSE;
}

static void
gnome_drud_back_callback (GtkWidget *button, GnomeDrud *drud)
{
	GList *list;
	g_return_if_fail (drud->current != NULL);

	if (gnome_drud_page_back (drud->current))
		return;

	/* Make sure that we have a next list item */
	list = g_list_find (drud->children, drud->current);
	g_return_if_fail (list->prev != NULL);
	gnome_drud_set_page (drud, GNOME_DRUD_PAGE (list->prev->data));
}
static void
gnome_drud_next_callback (GtkWidget *button, GnomeDrud *drud)
{
	GList *list;
	g_return_if_fail (drud->current != NULL);

	if (drud->show_finish == FALSE) {
		if (gnome_drud_page_next (drud->current))
			return;

		/* Make sure that we have a next list item */
		/* FIXME: we want to find the next VISIBLE one... */
		list = g_list_find (drud->children, drud->current);
		g_return_if_fail (list->next != NULL);
		gnome_drud_set_page (drud, GNOME_DRUD_PAGE (list->next->data));
	} else {
		gnome_drud_page_finish (drud->current);
	}
}
static void
gnome_drud_cancel_callback (GtkWidget *button, GtkWidget *drud)
{
     if (GNOME_DRUD (drud)->current) {
	     if (gnome_drud_page_cancel (GNOME_DRUD (drud)->current))
		     return;

	     gtk_signal_emit (GTK_OBJECT (drud), drud_signals [CANCEL]);
     }
}

/* Public Functions */
GtkWidget *
gnome_drud_new (void)
{
	return GTK_WIDGET (gtk_type_new (gnome_drud_get_type ()));
}

/**
 * gnome_drud_set_buttons_sensitive
 * @drud: A Drud.
 * @back_sensitive: The sensitivity of the back button.
 * @next_sensitive: The sensitivity of the next button.
 * @cancel_sensitive: The sensitivity of the cancel button.
 *
 * Description: Sets the sensitivity of the @drud's control-buttons.  If the
 * variables are TRUE, then they will be clickable.  This function is used
 * primarily by the actual GnomeDrudPage widgets.
 **/

void
gnome_drud_set_buttons_sensitive (GnomeDrud *drud,
				   gboolean back_sensitive,
				   gboolean next_sensitive,
				   gboolean cancel_sensitive)
{
	g_return_if_fail (drud != NULL);
	g_return_if_fail (GNOME_IS_DRUD (drud));

	gtk_widget_set_sensitive (drud->back, back_sensitive);
	gtk_widget_set_sensitive (drud->next, next_sensitive);
	gtk_widget_set_sensitive (drud->cancel, cancel_sensitive);
}
/**
 * gnome_drud_set_show_finish
 * @drud: A Drud widget.
 # @show_finish: If TRUE, then the "Cancel" button is changed to be "Finish"
 *
 * Description: Sets the text on the last button on the @drud.  If @show_finish
 * is TRUE, then the text becomes "Finish".  If @show_finish is FALSE, then the
 * text becomes "Cancel".
 **/
void
gnome_drud_set_show_finish (GnomeDrud *drud,
			     gboolean show_finish)
{
	g_return_if_fail (drud != NULL);
	g_return_if_fail (GNOME_IS_DRUD (drud));

	if (show_finish) {
		if (GTK_WIDGET_MAPPED (drud->next)) {
			gtk_widget_unmap (drud->next);
			gtk_widget_map (drud->finish);
		}
	} else {
		if (GTK_WIDGET_MAPPED (drud->finish)) {
			gtk_widget_unmap (drud->finish);
			gtk_widget_map (drud->next);
		}
	}
	drud->show_finish = show_finish;
}
/**
 * gnome_drud_prepend_page:
 * @drud: A Drud widget.
 * @page: The page to be inserted.
 * 
 * Description: This will prepend a GnomeDrudPage into the internal list of
 * pages that the @drud has.
 **/
void
gnome_drud_prepend_page (GnomeDrud *drud,
			  GnomeDrudPage *page)
{
	g_return_if_fail (drud != NULL);
	g_return_if_fail (GNOME_IS_DRUD (drud));
	g_return_if_fail (page != NULL);
	g_return_if_fail (GNOME_IS_DRUD_PAGE (page));

	gnome_drud_insert_page (drud, NULL, page);
}
/**
 * gnome_drud_insert_page:
 * @drud: A Drud widget.
 * @back_page: The page prior to the page to be inserted.
 * @page: The page to insert.
 * 
 * Description: This will insert @page after @back_page into the list of
 * internal pages that the @drud has.  If @back_page is not present in the list
 * or NULL, @page will be prepended to the list.
 **/
void
gnome_drud_insert_page (GnomeDrud *drud,
			 GnomeDrudPage *back_page,
			 GnomeDrudPage *page)
{
	GList *list;

	g_return_if_fail (drud != NULL);
	g_return_if_fail (GNOME_IS_DRUD (drud));
	g_return_if_fail (page != NULL);
	g_return_if_fail (GNOME_IS_DRUD_PAGE (page));

	list = g_list_find (drud->children, back_page);
	if (list == NULL)
		drud->children = g_list_prepend (drud->children, page);
	else {
		GList *new_el = g_list_alloc ();
		new_el->next = list->next;
		new_el->prev = list;
		if (new_el->next) 
			new_el->next->prev = new_el;
		new_el->prev->next = new_el;
		new_el->data = (gpointer) page;
	}
	gtk_widget_set_parent (GTK_WIDGET (page), GTK_WIDGET (drud));
}

/**
 * gnome_drud_append_page: 
 * @drud: A Drud widget.
 * @page: The page to be appended.
 * 
 * Description: This will append @page onto the end of the internal list.  
 **/
void gnome_drud_append_page (GnomeDrud *drud,
			      GnomeDrudPage *page)
{
	GList *list;
	g_return_if_fail (drud != NULL);
	g_return_if_fail (GNOME_IS_DRUD (drud));
	g_return_if_fail (page != NULL);
	g_return_if_fail (GNOME_IS_DRUD_PAGE (page));

	list = g_list_last (drud->children);
	if (list) {
		gnome_drud_insert_page (drud, GNOME_DRUD_PAGE (list->data), page);
	} else {
		gnome_drud_insert_page (drud, NULL, page);
	}	
}
/**
 * gnome_drud_set_page:
 * @drud: A Drud widget.
 * @page: The page to be brought to the foreground.
 * 
 * Description: This will make @page the currently showing page in the drud.
 * @page must already be in the drud.
 **/
void
gnome_drud_set_page (GnomeDrud *drud,
		      GnomeDrudPage *page)
{
	GList *list;
	GtkWidget *old = NULL;
	g_return_if_fail (drud != NULL);
	g_return_if_fail (GNOME_IS_DRUD (drud));
	g_return_if_fail (page != NULL);
	g_return_if_fail (GNOME_IS_DRUD_PAGE (page));

	if (drud->current == page)
	     return;
	list = g_list_find (drud->children, page);
	g_return_if_fail (list != NULL);

	if ((drud->current) && (GTK_WIDGET_VISIBLE (drud->current)) && (GTK_WIDGET_MAPPED (drud))) {
		old = GTK_WIDGET (drud->current);
	}
	drud->current = GNOME_DRUD_PAGE (list->data);
	gnome_drud_page_prepare (drud->current);
	if (GTK_WIDGET_VISIBLE (drud->current) && (GTK_WIDGET_MAPPED (drud))) {
		gtk_widget_map (GTK_WIDGET (drud->current));
		if (old)
			gtk_widget_unmap (old);
	}
}
