/*
 * GNoise
 *
 * Copyright (C) 1999-2001 Dwight Engen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: gtkwavedisplay.c,v 1.8 2002/01/13 02:51:16 dengen Exp $
 *
 */

#include <stdlib.h>
#include <string.h>
#include <gdk/gdkx.h>
#include <gtk/gtksignal.h>
#include "gtkwavedisplay.h"
#include "display_cache.h"

#define SAMPLES_PER_PIXEL                   (1<<wd->zoom)

static void gtk_wave_display_class_init     (GtkWaveDisplayClass *klass);
static void gtk_wave_display_init           (GtkWaveDisplay *wd);
static void gtk_wave_display_realize        (GtkWidget *widget);
static void gtk_wave_display_unrealize      (GtkWidget *widget);
static void gtk_wave_display_size_allocate  (GtkWidget *widget, GtkAllocation *allocation);
static void gtk_wave_display_draw           (GtkWidget *widget, GdkRectangle *area);
static void gtk_wave_display_draw_data      (GtkWaveDisplay *wd, int color, int x, int width, GdkDrawable *drawable);
static void gtk_wave_display_draw_main      (GtkWidget *widget, GdkRectangle *area, GdkDrawable *drawable, gboolean playline_update);
static gint gtk_wave_display_expose         (GtkWidget *widget, GdkEventExpose *event);
static void gtk_wave_display_expose_area    (GtkWidget *widget, GdkRectangle *area);
static void gtk_wave_display_style_set      (GtkWidget *widget, GtkStyle *previous_style);
static void gtk_wave_display_colors_set     (GtkWidget *widget);

static gint32    __inline__ smpl_to_xpos    (GtkWaveDisplay *wd, smpl_indx smp);
static smpl_indx __inline__ xpos_to_smpl    (GtkWaveDisplay *wd, guint32 xpos);

static GtkWidgetClass *parent_class = NULL;


/* if you define USE_GTK_COLORS, the only thing used from the following
 * table is the color for the playline, all other colors will be deduced
 * the GTK theme you're using's colors
 */
#define USE_GTK_COLORS
static const int default_colors[] =
{
    230, 0,   0,	/* play line */
    37,  132, 221,	/* marker, background */
    10,  114, 109,	/* marker, wave */
    234, 198, 35,	/* marker, selected */
    244, 244, 35,	/* marker, selected wave */
    197, 194, 197,	/* background normal */
    4,   68,  132,	/* foreground normal (wave data) */
    164, 161, 164,	/* background selected */
    49,  111, 165,	/* foreground selected (wave data) */
};


GtkType
gtk_wave_display_get_type (void)
{
    static GtkType wave_display_type = 0;

    if (!wave_display_type)
    {
	static const GtkTypeInfo wave_display_info =
	{
	    "GtkWaveDisplay",
	    sizeof(GtkWaveDisplay),
	    sizeof(GtkWaveDisplayClass),
	    (GtkClassInitFunc) gtk_wave_display_class_init,
	    (GtkObjectInitFunc) gtk_wave_display_init,
	    (GtkArgSetFunc) NULL,
	    (GtkArgGetFunc) NULL,
	};

	wave_display_type = gtk_type_unique(GTK_TYPE_WIDGET, &wave_display_info);
    }

    return wave_display_type;
}


static void
gtk_wave_display_class_init (GtkWaveDisplayClass *klass)
{
    GtkObjectClass *object_class;
    GtkWidgetClass *widget_class;
    int n;
    const int *p;
    GdkColor *c;

    object_class = (GtkObjectClass*) klass;
    widget_class = (GtkWidgetClass*) klass;
    parent_class = gtk_type_class (GTK_TYPE_WIDGET);

    widget_class->draw = gtk_wave_display_draw;
    widget_class->expose_event = gtk_wave_display_expose;
    widget_class->realize = gtk_wave_display_realize;
    widget_class->unrealize = gtk_wave_display_unrealize;
    widget_class->size_allocate = gtk_wave_display_size_allocate;
    widget_class->style_set = gtk_wave_display_style_set;

#   ifdef USE_GTK_COLORS
    /* only allocate new colors for playline and markers */
    klass->color_cnt = 5;
#   else
    klass->color_cnt = WD_COLOR_LAST;
#   endif

    for(n = 0, p = default_colors, c = klass->colors;
	n < klass->color_cnt;
	n++, c++)
    {
	c->red = *p++ * 65535 / 255;
	c->green = *p++ * 65535 / 255;
	c->blue = *p++ * 65535 / 255;
        c->pixel = (gulong)((c->red & 0xff00)*256 + (c->green & 0xff00) + (c->blue & 0xff00)/256);
        gdk_color_alloc(gdk_colormap_get_system(), c);
    }
}


static void
gtk_wave_display_init (GtkWaveDisplay *wd)
{
    wd->snd_buf = NULL;
    wd->play_smpl = 0;
    wd->play_xpix = -1;
    wd->start_smpl = 0;
    wd->selection = FALSE;
    wd->selection_start = -1;
    wd->selection_end = -1;
    wd->selection_start_old = -1;
    wd->selection_end_old = -1;
    wd->backing_store = FALSE;
}


GtkWidget*
gtk_wave_display_new (void)
{
    return GTK_WIDGET (gtk_type_new(gtk_wave_display_get_type()));
}


static void
gtk_wave_display_realize (GtkWidget *widget)
{
    GdkWindowAttr		attributes;
    gint			attributes_mask;
    GtkWaveDisplay		*wd;
    GtkWaveDisplayClass		*wd_class;
    int				i;

    /* raw X stuff */
    XSetWindowAttributes	xwinattr;
    Display			*Xdisplay;
    Window			Xwindow;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_WAVE_DISPLAY(widget));

    wd = GTK_WAVE_DISPLAY(widget);
    GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);

    attributes.x = widget->allocation.x;
    attributes.y = widget->allocation.y;
    attributes.width = widget->allocation.width;
    attributes.height = widget->allocation.height;
    attributes.wclass = GDK_INPUT_OUTPUT;
    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.event_mask = gtk_widget_get_events(widget)
			| GDK_EXPOSURE_MASK
			| GDK_BUTTON_PRESS_MASK
			| GDK_BUTTON_RELEASE_MASK
			| GDK_POINTER_MOTION_MASK;
    attributes.visual = gtk_widget_get_visual (widget);
    attributes.colormap = gtk_widget_get_colormap (widget);

    attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

    widget->window = gdk_window_new (gtk_widget_get_parent_window(widget),
				     &attributes, attributes_mask);
    widget->style = gtk_style_attach(widget->style, widget->window);
    gdk_window_set_user_data(widget->window, widget);

    /* create playline window */
    wd->play_win = gdk_window_new (widget->window, &attributes, attributes_mask);
    gdk_window_set_user_data(wd->play_win, widget);

    /* set colors */
    wd_class = GTK_WAVE_DISPLAY_CLASS(GTK_OBJECT(widget)->klass);
    for(i = 0; i < wd_class->color_cnt; i++)
    {
	wd->color[i] = gdk_gc_new(widget->window);
	gdk_gc_set_foreground(wd->color[i],
		&GTK_WAVE_DISPLAY_CLASS(GTK_OBJECT(widget)->klass)->colors[i]);
	gdk_gc_set_exposures(wd->color[i], 1);
    }
    gtk_wave_display_colors_set(widget);

    gtk_style_set_background(widget->style, widget->window, GTK_STATE_NORMAL);
    gtk_style_set_background(widget->style, wd->play_win, GTK_STATE_NORMAL);
    gdk_window_show(wd->play_win);


    Xdisplay = GDK_WINDOW_XDISPLAY(widget->window);
    Xwindow  = GDK_WINDOW_XWINDOW(widget->window);

    if (DoesBackingStore(DefaultScreenOfDisplay(Xdisplay)) != NotUseful)
    {
	xwinattr.backing_store = Always;
	XChangeWindowAttributes(Xdisplay, Xwindow, CWBackingStore, &xwinattr);
	wd->backing_store = TRUE;
    }
}

static void
gtk_wave_display_unrealize (GtkWidget *widget)
{
    GtkWaveDisplay *wd;

    g_return_if_fail (widget != NULL);
    g_return_if_fail (GTK_IS_WAVE_DISPLAY (widget));

    wd = GTK_WAVE_DISPLAY(widget);
    if (wd->play_win)
    {
	gdk_window_destroy (wd->play_win);
	wd->play_win = NULL;
    }
    if (GTK_WIDGET_CLASS (parent_class)->unrealize)
	(* GTK_WIDGET_CLASS (parent_class)->unrealize) (widget);
}

static void
gtk_wave_display_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
    GtkWaveDisplay *wd;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_WAVE_DISPLAY (widget));
    g_return_if_fail(allocation != NULL);

    widget->allocation = *allocation;
    if (GTK_WIDGET_REALIZED(widget))
    {
	wd = GTK_WAVE_DISPLAY(widget);

	gdk_window_move_resize (widget->window,
				allocation->x, allocation->y,
				allocation->width, allocation->height);

	gdk_window_resize (wd->play_win, 1, allocation->height);

	/* free backing pixmap we already had */
	if (wd->pixmap)
	    gdk_pixmap_unref(wd->pixmap);

	/* get backing pixmap to draw into */
	wd->pixmap = gdk_pixmap_new(widget->window, allocation->width,
				    allocation->height, -1);
    }
}


static void
gtk_wave_display_draw(GtkWidget *widget, GdkRectangle *area)
{
    gtk_wave_display_draw_main(widget, area, NULL, TRUE);
}


static void
gtk_wave_display_draw_main(GtkWidget *widget, GdkRectangle *area,
			   GdkDrawable *callers_drawable,
			   gboolean playline_update)
{
    GtkWaveDisplay *wd = GTK_WAVE_DISPLAY(widget);
    GdkDrawable *drawable;
    int x, x2;
    const int areax = area->x;
    const int areaxmax = area->x + area->width;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(area != NULL);
    g_return_if_fail(GTK_IS_WAVE_DISPLAY(widget));


    if (callers_drawable)
	drawable = callers_drawable;
    else
	drawable = wd->pixmap;

    if (playline_update)
	gtk_wave_display_play_set(wd, wd->play_smpl, TRUE);

    //log("DRAWMAIN", "area x:%d y:%d width:%d height:%d\n", area->x, area->y,
    //		area->width, area->height);

    if (area->width == 0)
    {
	log("DRAWMAIN", "asked to draw 0 width region?\n");
	return;
    }

    if (area->x + area->width > widget->allocation.width)
    {
	log("DRAWMAIN", "asked to draw too wide region?\n");
	return;
    }


    if (!wd->snd_buf)
    {
	gdk_draw_rectangle(widget->window,
			   wd->color[WD_COLOR_FG],
			   TRUE, area->x, area->y, area->width, area->height);

	/* dE: don't draw middle line
	 *	gdk_draw_line(widget->window,
	 *	      wd->color[WD_COLOR_FG],
	 *	      area->x, s->height / 2,
	 *	      area->x + area->width - 1, s->height / 2);
	 */
	return;
    }


    if (wd->selection_start != -1)
    {
	/* draw the part to the left of the selection */
	x = smpl_to_xpos(wd, wd->selection_start);
	x = MIN(areaxmax, MAX(areax, x));
	gtk_wave_display_draw_data(wd, WD_COLOR_FG, areax, x - areax, drawable);

	/* draw the selection */
	x2 = smpl_to_xpos(wd, wd->selection_end);
	x2 = MIN(areaxmax, MAX(areax, x2));
	gtk_wave_display_draw_data(wd, WD_COLOR_SEL_FG, x, x2 - x, drawable);
    } else {
	/* we don't have a selection */
	x2 = areax;
    }

    /* draw the part to the right of the selection */
    gtk_wave_display_draw_data(wd, WD_COLOR_FG, x2, areaxmax - x2, drawable);

    /* sync the screen from the double buffer that we just drew into */
    if (!callers_drawable)
	gtk_wave_display_expose_area(widget, area);
}


static gboolean
is_marker(GtkWaveDisplay *wd, smpl_indx start, smpl_indx end, int *color_indx)
{
    int i;

    for(i = 0; i < wd->snd_buf->markers; i++)
    {
	if (wd->snd_buf->marker[i] > end)
	    return FALSE;

	if (wd->snd_buf->marker[i] >= start && wd->snd_buf->marker[i] < end)
	{
	    if (i == wd->snd_buf->marker_sel)
		*color_indx = WD_COLOR_MARKER_SEL_BG;
	    else
		*color_indx = WD_COLOR_MARKER_BG;
	    return TRUE;
	}

    }
    return FALSE;
}


static void
gtk_wave_display_draw_data(GtkWaveDisplay *wd, int color, int x, int width, GdkDrawable *drawable)
{
    const guint16 height = wd->widget.allocation.height;
    GdkGC      *gc;
    gint16	smp;
    gint16	max;
    gint16	min;
    int		i;
    dcache_t   *dcache;
    guint32	cache_indx;
    int		color_indx;

    if ((width == 0) || (!wd->snd_buf))
	return;

    g_return_if_fail(x >= 0);
    g_return_if_fail(x + width <= wd->widget.allocation.width);

    //log("WAVEDISPLAY", "draw_data:area x:%d width:%d\n", x, width);

    /* clear area behind wave data */
    gdk_draw_rectangle(drawable, wd->color[color-1], TRUE, x, 0,
		       width, height);
    dcache = wd->snd_buf->dcache;

    /* use display cache for this zoom level if available */
    if (dcache && dcache->data[wd->zoom] != NULL)
    {
	guint32 zoom_ratio = 1<<wd->zoom;

	for(cache_indx = ((wd->start_smpl/zoom_ratio)+x)*2*wd->snd_buf->info.channels;
	    width > 0; cache_indx+=2*wd->snd_buf->info.channels, x++, width--)
	{
	    //log("DC", "cache indx:%d entr:%d\n", cache_indx, dcache->size[wd->zoom]);
	    gc = wd->color[color];
	    if (xpos_to_smpl(wd,x) + (1<<wd->zoom) >= wd->snd_buf->info.samples)
	    {
		/* fill after wave samples with SEL_BG */
		gdk_draw_rectangle(drawable, wd->color[WD_COLOR_SEL_BG],
				   TRUE, x, 0, 1, height);
		continue;
	    }

	    if (cache_indx + 2*wd->channel >= dcache->size[wd->zoom])
	    {
		log("ASSERT", "shouldn't draw...\n");
	    }

	    if (is_marker(wd, xpos_to_smpl(wd, x), xpos_to_smpl(wd, x+1), &color_indx))
	    {
		gdk_draw_line(drawable, wd->color[color_indx], x, 0, x, height);
		gc = wd->color[color_indx+1];
	    }

	    max = dcache->data[wd->zoom][cache_indx + 2*wd->channel];
	    min = dcache->data[wd->zoom][cache_indx + 2*wd->channel+1];
	    gdk_draw_line(drawable, gc,
			  x, ((max) * height) >> 8,
			  x, ((min) * height) >> 8);

	    //gdk_draw_point(wd->pixmap, gc, x, 0);
	    //gdk_draw_point(wd->pixmap, gc, x, height-1);
	}
    }
    else
    {
	/* there is no display cache for this level, display data will
	 * have to be generated by finding min/max now...
	 */
	if (wd->snd_buf->info.sample_bits == 16)
	{
	    while(width >= 0)
	    {
		guint32 start_indx;
		guint32 last_indx;

		start_indx = xpos_to_smpl(wd, x) * wd->snd_buf->info.channels;
		last_indx = start_indx + SAMPLES_PER_PIXEL * wd->snd_buf->info.channels;
		gc = wd->color[color];

		if (xpos_to_smpl(wd, x) + (1<<wd->zoom) >= wd->snd_buf->info.samples)
		{
		    gdk_draw_rectangle(drawable, wd->color[WD_COLOR_SEL_BG],
				TRUE, x, 0, 1, height);
		} else {
		    if (is_marker(wd, xpos_to_smpl(wd, x), xpos_to_smpl(wd, x+1), &color_indx))
		    {
			gdk_draw_line(drawable, wd->color[color_indx], x, 0, x, height);
			gc = wd->color[color_indx+1];
		    }

		    for(i = start_indx, max = min = 0;
			i <= last_indx;
			i += wd->snd_buf->info.channels)
		    {
			smp = -1 * (((gint16*)wd->snd_buf->data)[i+wd->channel]);
			max = (smp > max) ? smp : max;
			min = (smp < min) ? smp : min;
		    }

		    gdk_draw_line(drawable, gc,
			x, ((max + 32768) * height) >> 16,
			x, ((min + 32768) * height) >> 16);
		}
		x++;
		width--;
	    }
	} else {
	    while(width >= 0)
	    {
		guint32 start_indx;
		guint32 last_indx;

		start_indx = xpos_to_smpl(wd, x) * wd->snd_buf->info.channels;
		last_indx = start_indx + SAMPLES_PER_PIXEL * wd->snd_buf->info.channels;
		gc = wd->color[color];

		if (xpos_to_smpl(wd, x) + (1<<wd->zoom) >= wd->snd_buf->info.samples)
		{
		    gdk_draw_rectangle(drawable, wd->color[WD_COLOR_SEL_BG],
				TRUE, x, 0, 1, height);
		} else {
		    if (is_marker(wd, xpos_to_smpl(wd, x), xpos_to_smpl(wd, x+1), &color_indx))
		    {
			gdk_draw_line(drawable, wd->color[color_indx], x, 0, x, height);
			gc = wd->color[color_indx+1];
		    }

		    for(i = start_indx, max = min = 0;
			i <= last_indx;
			i += wd->snd_buf->info.channels)
		    {
			smp = -1 * (((guint8 *)wd->snd_buf->data)[i+wd->channel] - 128);
			max = (smp > max) ? smp : max;
			min = (smp < min) ? smp : min;
		    }

		    gdk_draw_line(drawable, gc,
			x, ((max + 128) * height) >> 8,
			x, ((min + 128) * height) >> 8);
		}
		x++;
		width--;
	    }
	} /* else 8 bit */
    } /* else no display cache */
}


void
gtk_wave_display_win_set (GtkWaveDisplay *wd, smpl_indx smpl, gboolean redraw)
{
    g_return_if_fail(wd != NULL);
    g_return_if_fail(GTK_IS_WAVE_DISPLAY(wd));

    if (!wd->snd_buf)
	return;

    g_return_if_fail(smpl >= 0 && smpl < wd->snd_buf->info.samples);

    wd->start_smpl = smpl;
    if (redraw)
	gtk_widget_queue_draw(GTK_WIDGET(wd));
}



void
gtk_wave_display_play_set (GtkWaveDisplay *wd, smpl_indx smpl, gboolean redraw)
{
    gint new_xpix;

    g_return_if_fail(wd != NULL);
    g_return_if_fail(GTK_IS_WAVE_DISPLAY(wd));

    wd->play_smpl = smpl;
    if (!redraw)
	return;

    new_xpix = smpl_to_xpos(wd, smpl);

    if (new_xpix >= 0 && new_xpix < wd->widget.allocation.width)
    {
	if (wd->play_xpix < 0)
	    gdk_window_show(wd->play_win);

	if (new_xpix != wd->play_xpix)
	    gdk_window_move(wd->play_win, new_xpix, 0);
    }
    else
    {
	/* new playline posisition is not visible */
	if (wd->play_xpix >= 0)
	    gdk_window_hide(wd->play_win);
	new_xpix = -1;
    }

    wd->play_xpix = new_xpix;
    return;
}


static gint
gtk_wave_display_expose(GtkWidget *widget, GdkEventExpose *event)
{
    GtkWaveDisplay *wd = GTK_WAVE_DISPLAY(widget);

    if (event->window == widget->window)
    {
	/* normally exposes can just be satisfied by blitting from the
	 * backing buffer but if we're scrolling, the backing buffer hasn't
	 * been kept up to date (we scroll the screen directly) so we'll
	 * need to draw from the data into the window
	 */
	if (scrolling && !wd->backing_store)
	    gtk_wave_display_draw_main(widget, &event->area, widget->window,
				       FALSE);
	else
	    gdk_draw_pixmap(widget->window,
			    widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
			    wd->pixmap, event->area.x, event->area.y,
			    event->area.x, event->area.y,
			    event->area.width, event->area.height);
    }
    else if (event->window == wd->play_win)
    {
	gdk_draw_line(wd->play_win, wd->color[WD_COLOR_PLAYLINE], 0, 0,
	    0, widget->allocation.height);
    }

    return FALSE;
}



void
gtk_wave_display_scroll (GtkWaveDisplay *wd, gint pix)
{
    GtkWidget *widget = GTK_WIDGET(wd);
    GdkRectangle area;

    g_return_if_fail (GTK_IS_WAVE_DISPLAY(wd));

    /* bitblt screen region */
    gdk_window_copy_area (widget->window, wd->color[1], 0, 0,
			  NULL, pix, 0,
			  widget->allocation.width-pix,
			  widget->allocation.height);

    /* draw directly into window */
    area.x = widget->allocation.width-pix;
    area.y = 0;
    area.width = pix;
    area.height = widget->allocation.height;
    gtk_wave_display_draw_main(GTK_WIDGET(wd), &area, widget->window, FALSE);
}



static void
gtk_wave_display_expose_area(GtkWidget *widget, GdkRectangle *area)
{
    GdkEventExpose event;

    event.type = GDK_EXPOSE;
    event.window = widget->window;
    event.send_event = TRUE;
    event.area = *area;
    event.count = 0;

    gtk_wave_display_expose(widget, &event);
}


static smpl_indx __inline__
xpos_to_smpl (GtkWaveDisplay *wd, guint32 xpos)
{
    return wd->start_smpl + xpos * SAMPLES_PER_PIXEL;
}

static gint32 __inline__
smpl_to_xpos (GtkWaveDisplay *wd, smpl_indx smp)
{
    if (smp < wd->start_smpl)
	return -1;
    return (smp - wd->start_smpl) / SAMPLES_PER_PIXEL;
}


static void
gtk_wave_display_style_set (GtkWidget *widget, GtkStyle *previous_style)
{
    GtkWaveDisplay *wd;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_WAVE_DISPLAY(widget));

    wd = GTK_WAVE_DISPLAY(widget);
    if (GTK_WIDGET_REALIZED(widget))
    {
	gtk_wave_display_colors_set(widget);

	gdk_gc_destroy(wd->color[0]);
	wd->color[0] = gdk_gc_new(widget->window);
	gdk_gc_set_foreground(wd->color[0],
		&GTK_WAVE_DISPLAY_CLASS(GTK_OBJECT(widget)->klass)->colors[0]);

	/* force playline to be redrawn */
	scrolling = FALSE;
    }
}


static void
gtk_wave_display_colors_set (GtkWidget *widget)
{
    int i;
    GtkWaveDisplay *wd;

    wd = GTK_WAVE_DISPLAY(widget);
    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_WAVE_DISPLAY(widget));

#   ifdef USE_GTK_COLORS
    wd->color[WD_COLOR_BG]	= widget->style->bg_gc[GTK_STATE_NORMAL];
    wd->color[WD_COLOR_FG]	= widget->style->bg_gc[GTK_STATE_SELECTED];
    wd->color[WD_COLOR_SEL_BG]	= widget->style->bg_gc[GTK_STATE_ACTIVE];
    wd->color[WD_COLOR_SEL_FG]	= widget->style->fg_gc[GTK_STATE_INSENSITIVE];
#   endif /* USE_GTK_COLORS */

    for (i=0; i < WD_COLOR_LAST; i++)
    {
	gdk_gc_set_clip_rectangle(wd->color[i], NULL);
	gdk_gc_set_exposures(wd->color[i], 1);
    }
}



void
gtk_wave_display_snd_set(GtkWaveDisplay *wd, snd_buf_t *snd, chnl_indx chan)
{
    wd->snd_buf = snd;
    wd->channel = chan;
}

void
gtk_wave_display_zoom_set(GtkWaveDisplay *wd, guint zoom)
{
    wd->zoom = zoom;
    gtk_widget_queue_draw(GTK_WIDGET(wd));
}

void
gtk_wave_display_cursor_set(GtkWaveDisplay *wd, GdkCursor *cursor)
{
    if (wd->cursor != cursor)
    {
	gdk_window_set_cursor(GTK_WIDGET(wd)->window, cursor);
	wd->cursor = cursor;
    }
}
