-------------------------------------------------------------------------------
--                                                                           --
--                     GNADE  : GNat Ada Database Environment                --
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/esql/esql_parser.adb,v $
--  Description     : ESQL Parser                                            --
--  Author          : Michael Erdmann <Michael.Erdmann@snafu.de>             --
--  Created On      : 22-Dec-2000                                            --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2003/12/26 16:18:41 $                           --
--  Status          : $State: Exp $                                          --
--                                                                           --
--  Copyright (C) 2000-2002 Michael Erdmann                                  --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  This package contains the esql parser. The result of the                 --
--  parse process is stored in the ESQL_Reader data. As a consequence        --
--  the code generation is a part of this package as well.                   --
--                                                                           --
--  The Syntax of Embedded SQL is based upon the //1// plus some             --
--  extensions.                                                              --
--                                                                           --
--  For more details on the implemented syntax please refere to the          --
--  statements of complience.                                                --
--                                                                           --
--  ESQL - Syntax                                                            --
--  =============                                                            --
--  For the syntax diagram please check the documentation of the gnade       --
--  project.                                                                 --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  R.1 - Character clause not implementend.                                 --
--  R.2 - Host variable constraints not implemented.                         --
--                                                                           --
--  References                                                               --
--  ==========                                                               --
--  //1// - (Second Informal Review Draft) ISO/IEC 9075:1992, Database       --
--  Language SQL- July 30, 1992                                              --
--                                                                           --
-------------------------------------------------------------------------------

--* Ada
with Ada.Exceptions;                    use Ada.Exceptions;
with Ada.Strings;                       use Ada.Strings;
with Ada.Strings.Unbounded;             use Ada.Strings.Unbounded;
with Ada.Unchecked_Conversion;
with Ada.Text_IO;                       use Ada.Text_IO;
with Ada.Characters.Handling;           use Ada.Characters.Handling;
with Ada.Characters.Latin_1;            use Ada.Characters;
with Ada.Strings.Fixed;                 use Ada.Strings.Fixed;

--* Translator packages
with Esql_Variables;                    use ESQL_Variables;

with Scanner;                           use Scanner;
with Options;                           use Options;

with GNU.DB.ESQL_ISO92_Types;           use GNU.DB.ESQL_ISO92_Types;
with GNU.DB.Support.Tables;
use  GNU.DB.Support;

pragma Elaborate_All( GNU.DB.Support.Tables );

package body ESQL_Parser is

   Version : constant String :=
         "$Id: esql_parser.adb,v 1.92 2003/12/26 16:18:41 merdmann Exp $";

   -- This is the list of keyword. The list is larger then realy
   -- needed.
   type ESQL_Reserved_Words is (
        ESQL_Declare,
        ESQL_Select,
        ESQL_Fetch,
        ESQL_Cursor,
        ESQL_For,
        ESQL_Open,
        ESQL_Close,
        ESQL_Into,
        ESQL_Delete,
        ESQL_From,
        ESQL_Update,
        ESQL_Set,
        ESQL_Where,
        ESQL_Current,
        ESQL_Of,
        ESQL_SQL,
        ESQL_Identifier,
        ESQL_End_Of_File,
        ESQL_Colon,
        ESQL_Semicolon,
        ESQL_Comma,
        ESQL_Range,
        ESQL_Equal,
        ESQL_Open_Bracket,
        ESQL_Close_Bracket,
        ESQL_AT,
        ESQL_Begin,
        ESQL_End,
        ESQL_EXEC,
        ESQL_Section,
        ESQL_STAR,
        ESQL_Ampersand,
        ESQL_Whenever,
        ESQL_GO,
        ESQL_TO,
        ESQL_GOTO,
        ESQL_Continue,
        ESQL_STOP,
        ESQL_SQL_Error,
        ESQL_SQL_Warning,
        ESQL_Not,
        ESQL_Found,
        ESQL_Dash,
        ESQL_Character,
        ESQL_Names,
        ESQL_ARE,
        ESQL_Connect,
        ESQL_Connection,
        ESQL_Disconnect,
        ESQL_As,
        ESQL_Database,
        ESQL_Table,
        ESQL_Indicator,
        ESQL_Temporary,
        ESQL_Temp,
        ESQL_VAR,
        ESQL_Identified,
        ESQL_Include,
        ESQL_By,
        ESQL_Named,
        ESQL_Reset,
        ESQL_DEFAULT,
        ESQL_SQLCA,
        ESQL_Port,
        ESQL_NULL,
        ESQL_Reopenable,
        ESQL_Final,
        ESQL_Local,
        ESQL_ALL,
        ESQL_Commit,
        ESQL_On,
        ESQL_Error,
                                             -- dynamic sql support
        ESQL_Statement,
        ESQL_Prepare,
        ESQL_Execute,
        ESQL_Immediate,
        ESQL_Using,

        ESQL_Dot,
        ESQL_String,
                                             -- odbc / esql interface
        ESQL_Handle,

        ADA_PACKAGE,                         -- used Ada keywords
        ADA_BODY,
        ADA_IS,
        ADA_RAISE,
        ADA_BEGIN,
        ADA_IF,
        ADA_CASE,
        ADA_RECORD,
        ADA_LOOP,
        ADA_DO,
        ADA_PROCEDURE,
        ADA_FUNCTION,
        ADA_NEW,
        ADA_RETURN
   );

   type Keyword_List is array( ESQL_Reserved_Words ) of String(1..10);

   Keywords : constant Keyword_List := (
      ESQL_Declare      => "DECLARE   ",
      ESQL_Select       => "SELECT    ",
      ESQL_Fetch        => "FETCH     ",
      ESQL_Cursor       => "CURSOR    ",
      ESQL_For          => "FOR       ",
      ESQL_Open         => "OPEN      ",
      ESQL_Close        => "CLOSE     ",
      ESQL_Into         => "INTO      ",
      ESQL_Delete       => "DELETE    ",
      ESQL_From         => "FROM      ",
      ESQL_Update       => "UPDATE    ",
      ESQL_Set          => "SET       ",
      ESQL_Where        => "WHERE     ",
      ESQL_Current      => "CURRENT   ",
      ESQL_SQL          => "SQL       ",
      ESQL_Of           => "OF        ",
      ESQL_Begin        => "BEGIN     ",
      ESQL_END          => "END       ",
      ESQL_Section      => "SECTION   ",
      ESQL_EXEC         => "EXEC      ",
      ESQL_AT           => "AT        ",
      ESQL_Whenever     => "WHENEVER  ",
      ESQL_GO           => "GO        ",
      ESQL_TO           => "TO        ",
      ESQL_GOTO         => "GOTO      ",
      ESQL_Continue     => "CONTINUE  ",
      ESQL_STOP         => "STOP      ",
      ESQL_NOT          => "NOT       ",
      ESQL_Found        => "FOUND     ",
      ESQL_SQL_Error    => "SQLERROR  ",
      ESQL_SQL_Warning  => "SQLWARNING",

      ESQL_Character    => "CHARACTER ",
      ESQL_Names        => "NAMES     ",
      ESQL_Are          => "ARE       ",
      ESQL_CONNECT      => "CONNECT   ",
      ESQL_CONNECTION   => "CONNECTION",
      ESQL_DISCONNECT   => "DISCONNECT",
      ESQL_IDENTIFIED   => "IDENTIFIED",
      ESQL_BY           => "BY        ",
      ESQL_AS           => "AS        ",
      ESQL_DATABASE     => "DATABASE  ",
      ESQL_TABLE        => "TABLE     ",
      ESQL_Indicator    => "INDICATOR ",
      ESQL_Temporary    => "TEMPORARY ",
      ESQL_Temp         => "TEMP      ",
      ESQL_VAR          => "VAR       ",
      ESQL_Include      => "INCLUDE   ",
      ESQL_Named        => "NAMED     ",
      ESQL_Reset        => "RESET     ",
      ESQL_SQLCA        => "SQLCA     ",
      ESQL_Port         => "PORT      ",
      ESQL_Default      => "DEFAULT   ",
      ESQL_Reopenable   => "REOPENABLE",
      ESQL_Final        => "FINAL     ",
      ESQL_Local        => "LOCAL     ",
      ESQL_ALL          => "ALL       ",
      ESQL_Commit       => "COMMIT    ",
      ESQL_On           => "ON        ",
      ESQL_Error        => "ERROR     ",

      ESQL_Statement    => "STATEMENT ",
      ESQL_Prepare      => "PREPARE   ",
      ESQL_Execute      => "EXECUTE   ",
      ESQL_Using        => "USING     ",
      ESQL_Immediate    => "IMMEDIATE ",

      ESQL_Handle       => "HANDLE    ",

      ADA_PACKAGE       => "PACKAGE   ",
      ADA_BODY          => "BODY      ",
      ADA_IS            => "IS        ",
      ADA_RAISE         => "RAISE     ",
      ADA_BEGIN         => "BEGIN     ",
      ADA_IF            => "IF        ",
      ADA_RECORD        => "RECORD    ",
      ADA_CASE          => "CASE      ",
      ADA_LOOP          => "LOOP      ",
      ADA_DO            => "DO        ",
      ADA_Procedure     => "PROCEDURE ",
      ADA_Function      => "FUNCTION  ",
      ADA_New           => "NEW       ",
      ADA_Return        => "RETURN    ",


      ESQL_STAR         => "*         ",

      ESQL_Colon        => ":         ",
      ESQL_String       => """         ",
      ESQL_Identifier   => "<undef>   ",
      ESQL_End_Of_File  => "<undef>   ",
      ESQL_Semicolon    => ";         ",
      ESQL_Comma        => ",         ",
      ESQL_Equal        => "=         ",
      ESQL_Dash         => "-         ",
      ESQL_Open_Bracket => "(         ",
      ESQL_Close_Bracket=> ")         ",
      ESQL_Ampersand    => "&         ",
      ESQL_Range        => "..        ",
      ESQL_Dot          => ".         ",
      ESQL_NULL         => "          "
   );

   type Identifier_Array is
      array( Natural Range <> ) of Identifier_String;

   Default_SQLCA_Name : constant Identifier_String :=
                            "SQLCA                                           ";

   -- constants for the connection error handling;
   type Connection_Error_Type is (
         Connection_Failure,
         Authorization_Failure,
         Other_Fault
   );


   type Connection_Error_Record is record
         Target : Identifier_String   := Blank_Identifier;
         Action : ESQL_Reserved_Words := ESQL_Continue;
      end record;

   Null_Connection_Error : constant Connection_Error_Record := (
         Target => Blank_Identifier,
         Action => ESQL_Continue
      );

   type Connection_Error_Array is array(
      Connection_Error_Type ) of Connection_Error_Record;

   --|
   --| This is the main data structure where the parser leaves
   --| his data and the code generator takes his data from.
   --|
   type ESQL_Reader_Type is record
         --- Data of the parser
         Current              : Identifier_String    := Blank_Identifier;
         ESQL_Terminator      : ESQL_Reserved_Words  := ESQL_Semicolon;

         -- currently processed tokens
         Current_Token        : Token_Type;
         Current_Keyword      : ESQL_Reserved_Words;
         -- push back storage
         Push_Back            : Token_Type;
         Push_Back_Keyword    : ESQL_Reserved_Words;

         --- Data created by the parser
         Connection           : Identifier_String    := Blank_Identifier;
         DB_Name              : Identifier_String    := Blank_Identifier;
         Host                 : Identifier_String    := Blank_Identifier;
         Port                 : Identifier_String    := Blank_Identifier;
         Login_Name           : Identifier_String    := Blank_Identifier;
         Password             : Identifier_String    := Blank_Identifier;
         Connection_Errors    : Connection_Error_Array := (
                                             others => Null_Connection_Error );

         Package_Name         : Unbounded_String     := Null_Unbounded_String;
         Max_Count            : Natural              := 0;

         Columns              : Identifier_Array( 1..50 );
         Nbr_Of_Columns       : Natural              := 0;
         Indicators           : Identifier_Array( 1..50 );
         Parameters           : Identifier_Array( 1..50 );
         Nbr_Of_Parameters    : Natural              := 0;
         Parameter_Ind        : Identifier_Array( 1..50 );

         Query_String         : Unbounded_String     := Null_Unbounded_String;
         Query_Variable       : Unbounded_String     := Null_Unbounded_String;
         Query_Line_Number    : Natural              := 0;

         Cursor_Is_Reopenable : Boolean              := False;
         Current_Cursor       : Identifier_String    := Blank_Identifier;
         Cursor_To_Declare    : Identifier_String    := Blank_Identifier;
         Cursor_To_Close      : Identifier_String    := Blank_Identifier;
         Cursor_Dynamic_SQL   : Identifier_String    := Blank_Identifier;
         Finalize_Cursor      : Boolean              := False;
         Cursor_To_Open       : Identifier_String    := Blank_Identifier;
         Cursor_Is_Local      : Boolean              := False;

         Current_Statement    : Identifier_String    := Blank_Identifier;
         Perform_Commit       : Boolean              := False;
         Connection_To_Close  : Identifier_String    := Blank_Identifier;

         Statement_Name       : Identifier_String    := Blank_Identifier;

         Ada_Declare          : Boolean              := False;
         Packages_Inserted    : Boolean              := False;
         Global_SQL_Env_Set   : Boolean              := False;
         Inject_Fetch         : Boolean              := False;
         SQL_Context_At_IS    : Boolean              := True;
         Eval_Result          : Boolean              := False;

         --- Persistent data
         Reader               : Scanner.File_Reader ;
         Nesting_Level        : Integer              := 0;
         Nbr_Of_Syntax_Errors : Natural              := 0;
         Nbr_Of_Warnings      : Natural              := 0;
         SQLCA_Name           : Identifier_String    := DEFAULT_SQLCA_Name;

         --- Exception handling data
         Not_Found_Action     : ESQL_Reserved_Words  := ESQL_Continue;
         SQL_Error_Action     : ESQL_Reserved_Words  := ESQL_Continue;
         SQL_Warning_Action   : ESQL_Reserved_Words  := ESQL_Continue;

         Not_Found_Target     : Identifier_String    := Blank_Identifier;
         SQL_Error_Target     : Identifier_String    := Blank_Identifier;
         SQL_Warning_Target   : Identifier_String    := Blank_Identifier;

      end record;

   Default_Connection : constant Identifier_String :=
                            "DEFAULT                                         ";

   -- control the scanner
   type Get_Mode is (
      Suppress_Token,          -- dont copy into output file
      Accept_Token,            -- copy into output file
      Defere_Copy              -- keep the token in the output buffer, until
                               -- either a Accept or a Discard Symbol is
                               -- executed.
      );

   -- ******************** HOST VARIABLE TABLE ****************************
   -- Symbol table for all host variables. there is only one global
   -- table for all variables.
   type Host_Variable_Record is record
         Name      : Identifier_String := Blank_Identifier;
         Base_Type : Identifier_String := Blank_Identifier;
      end record;

   package Host_Var_Table is new Tables( Host_Variable_Record );
   use Host_Var_Table;

   Host_Vars : Host_Var_Table.Tree_Node_Access := Create_Tree;

   type Type_Identifier_Array is array(ISO92_Host_Var_Type) of Identifier_String;

   Ada_SQL92_Types : constant Type_Identifier_Array := (
      ISO92_CHAR_TYPE             => "CHAR                                            ",
      ISO92_BIT_TYPE              => "BIT                                             ",
      ISO92_SMALLINT_TYPE         => "SMALLINT                                        ",
      ISO92_INT_TYPE              => "INT                                             ",
      ISO92_REAL_TYPE             => "REAL                                            ",
      ISO92_DOUBLE_PRECISION_TYPE => "DOUBLE_PRECISION                                ",
      ISO92_SQLCODE_TYPE          => "SQLCODE_TYPE                                    ",
      ISO92_SQLSTATE_TYPE         => "SQLSTATE_TYPE                                   ",
      ISO92_INDICATOR_TYPE        => "INDICATOR_TYPE                                  ",
      GNADE_VARCHAR_TYPE          => "GNADE.VARCHAR                                   ",
      GNADE_BINARY_TYPE           => "GNADE.BINARY                                    ",
      GNADE_VARBINARY_TYPE        => "GNADE.VARBINARY                                 ",
      ISO92_Unknown_Type          =>  Blank_Identifier                                );

   type SQL_Type_Mapping_Array is array( ISO92_Host_Var_Type ) of Identifier_String;

   SQL_Type_Mapping : constant SQL_Type_Mapping_Array  := (
      ISO92_CHAR_TYPE             => "CHARACTER                                       ",
      ISO92_BIT_TYPE              => "BIT                                             ",
      ISO92_SMALLINT_TYPE         => "INTEGER                                         ",
      ISO92_INT_TYPE              => "INTEGER                                         ",
      ISO92_REAL_TYPE             => "DECIMAL                                         ",
      ISO92_DOUBLE_PRECISION_TYPE => "DECIMAL                                         ",
      ISO92_SQLCODE_TYPE          =>  Blank_Identifier                                 ,
      ISO92_SQLSTATE_TYPE         =>  Blank_Identifier                                 ,
      ISO92_INDICATOR_TYPE        =>  Blank_Identifier                                 ,
      ISO92_Unknown_Type          =>  Blank_Identifier                                 ,
      GNADE_VARCHAR_TYPE          => "GNADE.VARCHAR                                   ",
      GNADE_BINARY_TYPE           => "GNADE.BINARY                                    ",
      GNADE_VARBINARY_TYPE        => "GNADE.VARBINARY                                 "
   );

   -- ********************** DATA BASE TABLE *******************************
   type Database_Table_Record is record
         Name      : Identifier_String := Blank_Identifier;
      end record;

   package DB_Table is new Tables( Database_Table_Record );

   Databases : DB_Table.Tree_Node_Access := DB_Table.Create_Tree;

   -- ********************** STATEMENT TABLE *******************************
   type Statement_Table_Record is record
         Name      : Identifier_String := Blank_Identifier;
      end record;

   package Statement_Table is new Tables( Statement_Table_Record );

   Statements : Statement_Table.Tree_Node_Access := Statement_Table.Create_Tree;

   ---========================================================================---
   ---===             S U P P O R T    F U N C T I O N S                   ===---
   ---========================================================================---

   ------------------
   -- Syntax_Error --
   ------------------
   procedure Syntax_Error(
      E    : in ESQL_Reader;
      Text : in String ) is
      -- Indicate a syntax error
   begin
      E.Nbr_Of_Syntax_Errors := E.Nbr_Of_Syntax_Errors + 1;
      Message( e.Reader, " error : " & text );
      raise Syntax_Exception;
   end Syntax_Error;

   -------------
   -- Warning --
   -------------

   procedure Warning(
      e    : ESQL_Reader;
      text : in String ) is
      -- Indicate a warining to the user.
   begin
      E.Nbr_Of_Warnings := E.Nbr_Of_Warnings + 1;
      Message( e.Reader, " warning : " & text );
   end Warning;

   --------------
   -- Pedantic --
   --------------
   procedure Pedantic(
      e    : ESQL_Reader;
      text : in String ) is
      -- indicate a pedantic warning to the user
   begin
      if Option_Pedantic then
         Warning( E, Text );
      end if;
   end Pedantic;

   ----------------
   -- Add_Column --
   ----------------
   procedure Add_Column(
      e    : in ESQL_reader;
      name : in Identifier_String ) is
      -- Add an affected columns to the list of affected columns.
      Nbr_Of_Columns : Natural renames e.Nbr_Of_Columns;
   begin
      Nbr_Of_Columns := Nbr_Of_Columns + 1;
      e.Columns( Nbr_Of_Columns ) := name;
   end Add_Column;

   -------------------
   -- Add_Parameter --
   -------------------
   procedure Add_Parameter(
      e        : in ESQL_reader;
      name     : in Identifier_String;
      Ind_Name : in Identifier_String := Blank_Identifier ) is
      -- Add an affected columns to the list of affected columns.
      Nbr_Of_Parameters : Natural renames e.Nbr_Of_Parameters;
   begin
      Nbr_Of_Parameters := Nbr_Of_Parameters + 1;
      E.Parameters( Nbr_Of_Parameters ) := Name;
      E.Parameter_Ind( Nbr_Of_Parameters ) := Ind_Name;
   end Add_Parameter;

   ----------------
   -- Upper_Case --
   ----------------
   function Upper_Case(
      s      : in Identifier_String ) return Identifier_String is
      ---
      Result : Identifier_String;
      ---
   begin
      for i in Identifier_String'Range loop
         if Is_Lower(s(I)) then
            Result(I) := To_Upper(s(I));
         else
            Result(I) := S(I);
         end if;
      end loop;

      return Result;
   end Upper_Case;

   -----------
   -- Quote --
   -----------
   function Quote (
      ID : String) return String is
      ---
      QT : constant Character := Character'Val (34);
      ---
   begin
      return QT & ID & QT;
   end Quote;

   ----------
   -- Name --
   ----------
   function Name(
      s : in String ) return String is
      -- Return the name part by trimming of the blanks
   begin
      return Trim( s, Right );
   end Name;

   -------------------
   -- To_Identifier --
   -------------------
   function To_Identifier(
      s : String ) return Identifier_String is
      -- Convert to identifier
      Result : Identifier_String := Blank_Identifier;
   begin
      for i in Identifier_String'Range loop
         exit when i > s'Length;

         Result(i) := s(i);
      end loop;

      return Result;
   end To_Identifier;

   ------------
   -- Append --
   ------------
   function Append(
      A : in String;
      B : in String ) return Identifier_String is
   begin
      return To_Identifier( Name(A) & Name(B) );
   end Append;


   -----------------
   -- Next_Symbol --
   -----------------
   function Next_Symbol(
      e        : ESQL_Reader;
      mode     : Get_Mode := Suppress_Token ) return ESQL_Reserved_Words is
      -- Read in the next symbol. It is checked against the
      -- list of reserved words. If it is not a reserved word
      -- we assume an identifier.
      T        : Token_Type renames e.Current_Token;
      Temp     : Identifier_String;
   begin
      if E.Push_Back /= Null_Token then
          T                 := E.Push_Back;
          E.Current_Keyword := E.Push_Back_Keyword;
          E.Push_Back       := Null_Token;
      else
          -- read from file and strip of any comment token.
          Get_Token( E.Reader, T );
          while T.Lexical_Unit = Comment_Lex and
                T.Lexical_Unit /= End_Of_File_Lex
          loop
             Get_Token( e.Reader, T );
          end loop;
      end if;

      case mode is
         when Suppress_Token =>
            Ignore_Input( E.Reader );
         when Accept_Token =>
            Accept_Input( E.Reader );
         when Others =>
            null;
      end case;

      if T.Lexical_Unit = End_Of_File_Lex then
         if Option_Debug then
            Put_Line("Next_Symbol: end of file");
         end if;
         raise End_of_File_Exception;
      end if;

      -- match keywords in uper cases only
      Temp              := Upper_Case( T.Lexicon );

      E.Current         := T.Lexicon;
      E.Current_Keyword := ESQL_Identifier;

      for i in ESQL_Reserved_Words loop
         if Name( Keywords(I) ) = Name( Temp ) then
            E.Current         := Temp;
            E.Current_Keyword := I;
            exit;
         end if;
      end loop;

      pragma Debug( Put_Line( "Next_Symbol: " & E.Current) );
      return E.Current_Keyword;

   exception
      when ESQL_Variables.Not_Existing =>
         Syntax_Error( E, "Substituation variable '" &
                       To_String( Scanner.Last_Variable_Name) &
                          "' not defined" );
         raise;
   end Next_Symbol;

   -----------------------------
   -- Push_Back_Current_Token --
   -----------------------------
   procedure Push_Back_Current_Token(
      -- This allows to push back one token which will
      -- read in again upon invoktion of the Next_Symbol
      -- procedure.
      E : in ESQL_Reader ) is
   begin
      E.Push_Back         := E.Current_Token;
      E.Push_Back_Keyword := E.Current_Keyword;

   end Push_Back_Current_Token;

   -------------------
   -- Accept_Symbol --
   -------------------
   procedure Accept_Symbol(
      E : in ESQL_Reader ) is
      -- Accept the symbol for copying into the processing result.
   begin
      pragma Debug( Put_Line( "Accept_Symbol" ) );
      Accept_Input( E.Reader );
   end Accept_Symbol;

   --------------------
   -- Discard_Symbol --
   --------------------
   procedure Discard_Symbol(
      E  : in ESQL_Reader ) is
   begin
      pragma Debug( Put_Line( "Discard_Symbol" ) );
      Ignore_Input( E.Reader );
   end Discard_Symbol;

   ------------
   -- Expect --
   ------------
   procedure Expect(
      e       : ESQL_Reader;
      what    : ESQL_Reserved_Words;
      where   : String := "";
      mode    : Get_Mode := Suppress_Token ) is
      -- Check for the expected token type. If the item is not found,
      -- an error message will be generated.
      Current : ESQL_Reserved_Words;
      --
   begin
      Current := Next_Symbol(e, mode);
      if Current /= what then
         if where = ""  then
            Syntax_Error( e,
               " unexpected token " & ESQL_Reserved_Words'Image(Current) &
               " found, expected " &  ESQL_Reserved_Words'Image(what));
         else
            Syntax_Error( e,
               " unexpected token " & ESQL_Reserved_Words'Image(Current) &
               " found in " & where &
               ", expected " &  ESQL_Reserved_Words'Image(what));
         end if;
      end if;
   end Expect;

   -----------------
   -- Is_Terminal --
   -----------------
   function Is_Terminal(
      S : in ESQL_Reserved_Words ) return Boolean is
      -- this returns true for those keyword which are terminals, where
      -- no need is to insert blanks afterwards.
   begin
      -- Put_Line( ESQL_Reserved_Words'Image(S) );

      return  S = ESQL_Open_Bracket or S = ESQL_Close_Bracket or
              S = ESQL_Equal or
              S = ESQL_Dot or
              S = ESQL_Comma or
              S = ESQL_String ;
   end Is_Terminal;

   ---------------
   -- Skip_Till --
   ---------------
   procedure Skip_Till(
      e        : ESQL_Reader;
      what     : ESQL_Reserved_Words;
      mode     : Get_Mode := Suppress_Token ) is
      -- Read all tokens till the given token is found.
      Current  : ESQL_Reserved_Words;
      ---
   begin
      Current := Next_Symbol(e, mode);
      while Current /= what loop
         Current := Next_Symbol(e, mode);

         if Current = ESQL_End_Of_File then
            Syntax_Error( e, "missing " & ESQL_Reserved_Words'Image(what) );
            exit;
         end if;

      end loop;
   end Skip_Till;

   ---------------------
   -- Identifier_List --
   ---------------------
   procedure Identifier_List(
      e        : in  ESQL_Reader;
      ids      : out Identifier_Array;
      length   : out Natural;
      sep      : in  ESQL_Reserved_Words := ESQL_Comma;
      mode     : in  Get_Mode := Suppress_Token ) is
      -- copy a list of identifiers into the array.
      Current  : ESQL_Reserved_Words;
      --
   begin
      length := 0;
      for i in ids'Range loop
         Current := Next_Symbol( e, mode );
         if Current = ESQL_Colon then             -- host variable
            Current := Next_Symbol( e, mode );
         end if;

         if Current /= ESQL_identifier then
            Syntax_Error( e, "exptected identifier in list" );
         end if;
         ids(i) := e.Current;
         length := length + 1;

         Current := Next_Symbol( e, mode );

         exit when Current /= ESQL_Comma;
      end loop;

      Push_Back_Current_Token(e);

   end Identifier_List;

   --------------------
   -- SQL_Terminator --
   --------------------
   procedure SQL_Terminator(
      E       : ESQL_Reader ) is
      -- scan the SQL terminator if there. If it is
      -- not there, then copy all tokens into the result file.
      -- In all other cases donot copy.
      Current : ESQL_Reserved_Words;
      Query   : Unbounded_String := Null_Unbounded_String;
      --
   begin
      Current := Next_Symbol(e, mode => Defere_Copy );
      case Current is
         when ESQL_Semicolon =>
            Discard_Symbol(e);
         when ESQL_END =>
            Current := Next_Symbol(e, mode => Defere_Copy );
            if Current /= ESQL_Dash then
               Accept_Symbol(e);
            else                        -- END - is no valid ada
               Expect( e, ESQL_Exec );
            end if;
         when Others =>
            Accept_Symbol(e);           -- write out the unknown token
            Push_Back_Current_Token(e); -- push back for later processing
      end case;

   end SQL_Terminator;

   ----------------
   -- Identifier --
   ----------------
   function Identifier(
      E : in ESQL_Reader ) return String is
      -- Get the current identifier
   begin
      return E.Current;
   end Identifier;

   ---========================================================================---
   ---===              D A T A    B A S E  H A N D L I N G                 ===---
   ---========================================================================---

   --------------------------
   -- Insert_DB_Table_Name --
   --------------------------
   procedure Insert_DB_Table_Name(
      E    : ESQL_Reader;
      Name : in Identifier_String ) is
      ---
      Info : Database_Table_Record;
      ---
   begin
      Info.Name := Name;
      DB_Table.Insert( Databases, Trim(Name, Right), Info );

   exception
      when DB_Table.Already_Stored =>
         Warning( E, Trim(Name, Right ) & " multiple declared ");
   end Insert_DB_Table_Name;

   ----------------------
   -- Is_Database_Name --
   ----------------------
   function Is_Database_Name(
     Name : in Identifier_String ) return Boolean is
     ---
     Db_Info : Database_Table_Record;
     ---
   begin
     DB_Table.Fetch( Databases, Trim(Name,Right), DB_Info );
     return True;

   exception
     when DB_Table.Entry_Not_Found =>
        return False;
   end Is_Database_Name;

   ---------------------------
   -- Insert_Statement_Name --
   ---------------------------
   procedure Insert_STATEMENT_Name(
      E    : in ESQL_Reader;
      Name : in Identifier_String ) is
      ---
      Info : Statement_Table_Record;
   begin
      Info.Name := Upper_Case(Name);
      Statement_Table.Insert(
         Statements,
         Trim(Upper_Case(Name), Right),
         Info );

   exception
      when Statement_Table.Already_Stored =>
         Warning( E, Trim(Name, Right ) & " multiple declared ");
   end Insert_STATEMENT_Name;

   ----------------------
   -- Is_Statement_Name --
   ----------------------
   function Is_Statement_Name(
     Name    : in Identifier_String ) return Boolean is
     ---
     Stmt_Info : Statement_Table_Record;
   begin
     Statement_Table.Fetch(
        Statements,
        Trim(Upper_Case(Name),Right),
        Stmt_Info );
     return True;

   exception
     when Statement_Table.Entry_Not_Found =>
        return False;
   end Is_Statement_Name;

   ---========================================================================---
   ---===               T Y P E     H A N D L I N G                        ===---
   ---========================================================================---

   -------------------
   -- Is_SQL92_Type --
   -------------------

   function Is_SQL92_Type (
      name : Identifier_String ) return Boolean is
      -- check for an allowed SQL 92 Ada type
      Tmp  : Identifier_String := Upper_Case(Name);
      --
   begin
      for i in ISO92_Host_Var_Type'Range loop
         if Ada_SQL92_Types(I) = Tmp or
            To_Identifier("SQL_STANDARD." & Trim(Ada_SQL92_Types(I), Right)) = Tmp
         then
            return True;
         end if;
      end loop;

      return False;
   end Is_SQL92_Type;

   ----------------
   -- SQL92_Type --
   ----------------

   function SQL92_Type(
      Name : Identifier_String ) return ISO92_Host_Var_Type is
      --
      Tmp  : Identifier_String := Upper_Case( Name );
      --
   begin
      for I in ISO92_Host_Var_Type'Range loop
         if Ada_SQL92_Types(I) = Tmp or
            To_Identifier("SQL_STANDARD." & Trim(Ada_SQL92_Types(I), Right)) = Tmp
         then
            return I;
         end if;
      end loop;

      return ISO92_Unknown_Type ;
   end SQL92_Type;


   ---------------------------
   -- Type_Of_Host_Variable --
   ---------------------------

   function Type_Of_Host_Variable(
      Name       : in Identifier_String ) return ISO92_Host_Var_Type is
      ---
      Variable   : Host_Variable_Record ;
      Cap_Name   : Identifier_String := Upper_Case(Name);
      ---
   begin
      Fetch( Host_Vars, Trim(Cap_Name,Right), Variable );

      return SQL92_Type( Variable.Base_Type );

   exception
      when others =>
         -- Put_Line("Name not found : '" & Cap_Name & "'");
         return ISO92_Unknown_Type;

   end Type_Of_Host_Variable;

   --------------------------
   -- Host_Var_type_String --
   --------------------------
   function Host_Var_Type_String(
      S        : in Identifier_String ) return String is
      -- return the type string of a host variable
      Var_Type : ISO92_Host_Var_Type := Type_Of_Host_Variable(S);
   begin
      if Var_Type = GNADE_VARCHAR_TYPE then
         Var_Type := ISO92_CHAR_TYPE;
      end if;

      return ISO92_Host_Var_Type'Image(Var_Type);
   end Host_Var_Type_String;

   -----------------------
   -- Add_Host_Variable --
   -----------------------
   procedure Add_Host_Variable(
      Name      : in String;
      Base_Type : in String ) is
      -- Add a symbol and the data type. If the value already exists
      -- simply update with the base type.
      Variable  : Host_Variable_Record;
      Cap_Name  : Identifier_String := Upper_Case( Name );
      ---
   begin
      Variable.Name      := Cap_Name;
      Variable.Base_Type := base_type;
      Insert( Host_Vars, Trim(Cap_Name,Right), Variable);

   exception
      when Already_Stored =>
         Update( Host_Vars, Trim(Cap_Name,Right), Variable ); -- Bug fix: 1601.4

      when others =>
         raise;
   end Add_Host_Variable;

   ---========================================================================---
   ---===             S O U R C E  C O D E  G E N E R A T O R              ===---
   ---========================================================================---

   ---------
   -- Ada --
   ---------
   procedure Ada(
      e             : in ESQL_reader;
      statement     : in String;
      comment       : in String := "" ) is
      -- This procedure inserts a nice Ada statement into the result
      -- file applying the correct itendation level.
      Nesting_Level : Integer renames e.Nesting_Level;
      ---
   begin
      if comment = "" then
         Insert( e.Reader, (
             (Nesting_Level*3) * " " ) & statement );
      else
         Insert( e.Reader,
             ((Nesting_Level*3) * " " ) & statement &
             "-- " & comment
         );
      end if;
   end Ada;

   -----------------
   -- Ada_Declare --
   -----------------

   procedure Ada_Declare(
      E             : ESQL_Reader ) is
      -- This outputs a declare statement. Applying this procedure
      -- will yield the correct layout for the variable declarations.
      Nesting_Level : Integer renames e.Nesting_Level;
      Ada_Declare   : Boolean renames e.Ada_Declare;
      --
   begin
      Ada( e, "declare" );
      Ada_Declare   := True;
      Nesting_Level := Nesting_Level + 1;
   end Ada_Declare;

   ---------------
   -- Ada_Begin --
   ---------------

   procedure Ada_Begin(
      e             : ESQL_Reader ) is
      -- Refere to Ada_Declare
      Nesting_Level : Integer renames e.Nesting_Level;
      Ada_Declare   : Boolean renames e.Ada_Declare;
      --
   begin
      if not Ada_Declare then
         Ada(e, "begin");
         Nesting_Level := Nesting_Level + 1;
      else
         Ada_Declare := False;
         Nesting_Level := Nesting_Level - 1;
         Ada_Begin( e );
      end if;
   end Ada_Begin;

   -------------
   -- Ada_End --
   -------------
   procedure Ada_End(
      e    : in ESQL_reader ) is
      -- Close a text block. For details refer to Ada_Declare.
      Nesting_Level   : Integer renames e.Nesting_Level;
   begin
      Nesting_Level := Nesting_Level - 1;
      Ada(e, "end;");
   end Ada_End;

   -------------------
   -- Ada_Exception --
   -------------------
   procedure Ada_Exception(
      E  : in ESQL_Reader ) is
      -- Insert an exception clause
      Nesting_Level   : Integer renames E.Nesting_Level;
   begin
      Nesting_Level := Nesting_Level - 1;
      Ada( e, "exception" );
      Nesting_Level := Nesting_Level + 1;
   end Ada_Exception;

   ------------------------
   -- Indicator_Variable --
   ------------------------
   function Indicator_Variable(
      host_var : Identifier_String ) return String is
      -- This creates the name of an indicator variable
      -- for a host variable.
   begin
      return Name(host_var) & "_Indicator" ;
   end Indicator_Variable;

   ------------------
   -- GNADE_Handle --
   ------------------
   function GNADE_Handle (
      Name : in String ) return String is
      -- Generate the name if a DB connection mentioned in the
      -- AT clause.
   begin
      return "GNADE_DB_" & Trim(Name, Right) ;
   end GNADE_Handle;

   ------------------
   -- GNADE_Handle --
   ------------------
   function GNADE_Handle (
      e : in ESQL_Reader ) return String is
      -- Generate the name of the current GNADE handle
   begin
      return GNADE_Handle( E.Connection );
   end GNADE_Handle;

   ------------
   -- Invoke --
   ------------
   function Invoke(
      S : in String ) return String is
      -- invoke a procedure from the embedded SQL package
   begin
      return To_String(Option_DBCS) & "." & S;
   end Invoke;

   ----------------------
   -- Invoke_Connector --
   ----------------------
   function Invoke_Connector(
      S : in String ) return String is
      -- invoke the correct connector module. If no external impl.
      -- is given use the default connection module of the esql translator
   begin
      if Option_Connect_Pkg = Null_Unbounded_String then
         return Invoke(S);
      end if;

      return To_String(Option_Connect_Pkg) & "." & S;
   end;

   ------------------
   -- SQLCA_Access --
   ------------------
   function SQLCA_Access(
      E : ESQL_Reader ) return String is
     -- This returns the correct way of accessing the sql
     -- communication area.
   begin
      return Trim(E.SQLCA_Name, Right) & "'Address";
   end SQLCA_Access;

   ---------------
   -- SQLCA_Var --
   ---------------

   function SQLCA_Var(
      E   : in ESQL_Reader;
      Var : in String  ) return String is
   begin
      return Trim(E.SQLCA_Name, Right) & "." & Var;
   end SQLCA_Var;

   -------------------
   -- SQLCA_Declare --
   -------------------

   procedure SQLCA_Ada_Declare(
      E : in ESQL_Reader ) is
   begin
      Ada( E, "SQLCA            : aliased ESQL_SUPPORT.SQLCA_TYPE;"              );
   end SQLCA_Ada_Declare;

   ----------------
   -- String_Arg --
   ----------------
   function String_Arg(
      s : Identifier_String ) return String is
      -- Generate a string argument. If the variable is
      -- not defined (blank) and empty string will be
      -- returned.
   begin
      if s = Blank_Identifier then
         return Quote("");
      end if;

      return Trim( s, Right );
   end String_Arg;

   -----------------
   -- Make_Access --
   -----------------
   function Make_Access(
      E  : in ESQL_Reader;
      N  : in String ) return String is
   begin

      return Trim(N, Right) & "'Access";
   end Make_Access;

   -----------------
   -- Cursor_Name --
   -----------------
   function Cursor_Name(
      E : ESQL_Reader;
      S : String ) return String is
   begin
      return To_String(E.Package_Name) & "." & Trim(S, Right);
   end Cursor_Name;

   ----------------
   -- Cursor_Arg --
   ----------------

   function Cursor_Arg(
      e : ESQL_reader ) return String is
      -- This returns the currently declared cursor argument
   begin
      if e.Current_Cursor /= Blank_Identifier then          -- Bug fix 0701.2
        return ", Cursor => " &
           Quote( Cursor_Name(E, e.Current_Cursor) ) ;
      else
        return "";
      end if;
   end Cursor_Arg;

   --------------------
   -- Context_Object --
   --------------------

   function Context_Object(
      E : ESQL_Reader ) return String is
      -- return the correct context object
   begin
      if E.Cursor_Is_Local then
         return "GNADE_L_CONTEXT";
      else
         return "GNADE_G_CONTEXT";
      end if;
   end Context_Object;

   -----------------
   -- Long_String --
   -----------------
   function Long_String(
      E             : in ESQL_Reader;
      Query         : in Unbounded_String ) return String is
      -- This function returns a SQL string in a Ada 95 representation
      -- by appling the following rules:
      --
      -- R.1 - Output a long string in chunks of 60 characters
      --       per line by rewriting:
      --       ...."   ->   ...." &  <LF> ".......
      --
      -- R.2 - SQL Strings are translated into a Ada 95 strings
      --       X = "RFI" ->  ...X = " & """" & "RFI" & """" ...
      Temp          : String(1..60) := (others=>' ');
      Next          : Positive := Temp'First;
      Result        : Unbounded_String := Null_Unbounded_String;
      Intend        : constant String  := ((E.Nesting_Level + 1) * 3) * " " ;
      S             : constant String  := To_String(Query);

      function Add_Line (
         Previous_Text : in Unbounded_String;
         New_Text      : in String) return Unbounded_String is
      begin
         if Previous_Text = Null_Unbounded_String then
            return To_Unbounded_String (Quote (New_Text));
         else
            return
              Previous_Text & " &" & Latin_1.LF &
              Intend & Quote (New_Text);
         end if;
      end Add_Line;

   begin
      for I in 1..S'Length loop
         if not (Next in Temp'Range) then              -- R.1
            Result := Add_Line (Result, Temp);
            Next   := Temp'First;
         end if;

         if S(I) = '"' then                            -- R.2
            Result := Add_Line (Result, Temp(1..Next-1));
            Next   := Temp'First;

            Result := Add_Line (Result, S(I) & S(I) );
         else
            Temp(Next) := S(I);
            Next       := Next + 1;
         end if;
      end loop;

      if Next > 1 then
         Result := Add_Line (Result, Temp(1 .. Next -1 ));
      end if;
      return To_String( Result );
   end Long_String;

   ----------------
   -- Ada_Action --
   ----------------
   procedure Ada_Action(
      E       : in ESQL_Reader;
      Action  : in ESQL_Reserved_Words;
      Target  : in Identifier_String;
      Comment : in String := "" )  is

      Info    : constant String :=
               "Line" & Natural'Image(E.Query_Line_Number) &
               " in package " & To_String( E.Package_Name ) &
               " : " & Comment ;
   begin
      case  Action is
         when ESQL_GOTO =>
            Ada( e, "   GOTO "  & Name( target ) & ";" );

         when Ada_DO =>
            Ada( e, "   " & Name( target ) & ";"       );    -- Pro*Ada

         when ADA_RAISE =>
            Ada( E, "   Raise_Exception(" & Name( Target ) & "'Identity,");
            Ada( E, "     " & Quote( Info & " failed " )
                 & " &  Latin_1.LF & "
                 &  SQLCA_Var(E, "message" ) & ");"
                 );
         when Others=>
           null;
      end case;
   end Ada_Action;

   ----------------
   -- ODBC_Debug --
   ----------------
   procedure ODBC_Debug(
      E        : in ESQL_Reader;
      Query    : in Unbounded_String := Null_Unbounded_String;
      Comments : in String := "";
      Reason   : in String := "";
      Force    : in Boolean := False ) is
      -- insert the debug code if requiered by setting up the query info
      -- and some additional comments
      Info     : constant String :=
               To_String( E.Package_Name ) & " at" & Natural'Image(E.Query_Line_Number) &
               ", " & Comments ;
      Q        : Unbounded_String := To_Unbounded_String(Quote(""));
   begin
      if Query /= Null_Unbounded_String then
         Q := Query;
       end if;

      if Option_Debug_Code or Force then
         Ada (E, "ESQL_SUPPORT.DEBUG("             );
	 if Reason = "" then
            Ada (E, "   " & Quote( Info )           & "," );
	 else
            Ada (E, "   " & Quote( Info ) & "& " & Reason & "," );
	 end if;
         Ada (E, "   " & To_String(Q)            & "," );
         Ada (E, "   " & "SQLCA'Address);"         );
         Ada (E, "");
      end if;
   end ODBC_Debug;

   ----------------
   -- ODBC_Debug --
   ----------------
   procedure ODBC_Debug(
      E        : in ESQL_Reader;
      Query    : in String  := "";
      Comments : in String  := "";
      Reason   : in String  := "";
      Force    : in Boolean := False) is
   begin
      ODBC_Debug( E, To_Unbounded_String(Query), Comments, Reason, Force );
   end ODBC_Debug;

   -------------------
   -- ODBC_packages --
   -------------------
   procedure ODBC_Packages(
      -- Insert the code to invoke the support packages used by
      -- the ODBC package if it not already inserted.
      e : in ESQL_Reader ) is
   begin
      if e.Packages_Inserted then
         return;
      else
         ADA( E, "with Ada.Exceptions;            use Ada.Exceptions;"         );
         Ada( E, "with Ada.Characters.Latin_1;    use Ada.Characters;"         );

         Ada( e, "WITH GNU.DB.ESQL_ISO92_TYPES;   USE GNU.DB.ESQL_ISO92_TYPES;");
         Ada( e, "WITH GNU.DB.ESQL_SUPPORT;       USE GNU.DB.ESQL_SUPPORT;"    );
         Ada( E, "WITH GNU.DB.ESQL_SUPPORT." & To_String(Option_DBCS) & "; "   );
         Ada( E, "USE  GNU.DB.ESQL_SUPPORT." & To_String(Option_DBCS) & "; "   );

         Ada( E, "USE  GNU.DB.ESQL_SUPPORT;"                                   );
         Ada( E, "WITH GNU.DB.SQLCLI;"                                         );
         Ada( E, "USE  GNU.DB;"                                                );

         if Option_Connect_Pkg /= Null_Unbounded_String then
            Ada( E, "WITH " & To_String(Option_Connect_Pkg) & "; ");
         end if;

         Ada( E, ""                                                            );
         Ada( E, "pragma Elaborate_All(GNU.DB.ESQL_SUPPORT."
                                  & To_String(Option_DBCS) & ");"              );
         Ada( E, ""  );
         Mark_Source_Line( E.Reader );

         E.Packages_Inserted := True;
      end if;

      Ada( e, "" );
      Ada( e, "" );
   end ODBC_Packages;

   ------------------
   -- ODBC_Prelude --
   ------------------
   procedure ODBC_Prelude(
      e                 : in ESQL_Reader ) is
      -- This code we be written before every query, specialy the
      -- length indicators for each host variable used here are made
      -- available.
      Nbr_Of_Parameters : Natural           renames e.Nbr_Of_Parameters;
      Parameters        : Identifier_Array  renames e.Parameters;
      Columns           : Identifier_Array  renames e.Columns;
      Nbr_Of_Columns    : Natural           renames e.Nbr_Of_Columns;
      DB_Name           : Identifier_String renames e.DB_Name;
      Indicators        : Identifier_Array  renames e.Indicators;
      Parameter_Ind     : Identifier_Array  renames e.Parameter_Ind;
      --
   begin
      Comment(E.Reader, "*** Query ***" );

      if (Nbr_Of_Columns > 0 or Nbr_Of_Parameters > 0) then
         Ada_Declare(e);
         for i in 1..Nbr_Of_Columns loop
            if Indicators(i) = Blank_Identifier then
               Ada( e,
                  Indicator_Variable(Columns(i)) &
                  " : aliased SQL_STANDARD.INDICATOR_TYPE := 0;" );
            end if;
         end loop;

         for i in 1..Nbr_Of_Parameters loop
            if Parameter_Ind(I) = Blank_Identifier then
               Ada( e,
                  Indicator_Variable(Parameters(i)) &
                  " : aliased SQL_STANDARD.INDICATOR_TYPE := 0;" );
            end if;
         end loop;
      end if;

      Ada_Begin(e);

   end ODBC_Prelude;

   ------------------------
   -- ODBC_Prepare_Query --
   ------------------------
   procedure ODBC_Prepare_Query(
      E                    :  in ESQL_Reader ) is
      -- Prepare the query. The prepare is only used if there are
      -- columns of parameters to bind.
      --
      -- The following rules do apply:
      --
      -- R.2 - Insert a PREPARE if the query contains parameters and
      --       result columns.
      Query                : Unbounded_String  renames e.Query_String;
      Query_Variable       : Unbounded_String  renames e.Query_Variable;
      Nbr_Of_Parameters    : Natural           renames e.Nbr_Of_Parameters;
      Nbr_Of_Columns       : Natural           renames e.Nbr_Of_Columns;
      Cursor_To_Declare    : Identifier_String renames e.Cursor_To_Declare;
      Current_Cursor       : Identifier_String renames e.Current_Cursor;
      Is_Reopenable        : Boolean           renames E.Cursor_Is_Reopenable;
      Q                    : Unbounded_String  := Null_Unbounded_String;
      --
   begin
      Ada( E, "-- Prepare");
      if Query /= Null_Unbounded_String then
         Q := To_Unbounded_String( Long_String(E, Query) );
      else
         Q := Query_Variable;
      end if;

      if Cursor_To_Declare /= Blank_Identifier then               -- R.1
         Current_Cursor := Cursor_To_Declare;
      end if;

      if ( Nbr_of_Parameters > 0 or Nbr_Of_Columns > 0 or         -- R.2
           Cursor_To_Declare /= Blank_Identifier )                -- R.1
           and Q /= Null_Unbounded_String
      then
         Ada( E, Invoke("PREPARE(") & GNADE_Handle(E) & ","           );
         Ada( E, "   " & Context_Object(E)            & ","           );
         Ada( E, "   " & To_String(Q)                 & ","           );
         Ada( E, "   " & SQLCA_Access(E)              & ","           );
         Ada( E, "   " & "Is_Reopenable =>" & Boolean'Image(Is_Reopenable) &
                          Cursor_Arg(e) & ");"                              );

         E.Eval_Result := True;
      else
         E.Eval_Result := False;
      end if;

   end ODBC_Prepare_Query;

   ------------------
   -- ODBC_Execute --
   ------------------
   procedure ODBC_Execute(
      e                 :  in ESQL_Reader ) is
      -- Execute the command. If there are columns exptected to
      -- be returned by the DBCS a fetch is executed.
      --
      -- The following rules do apply:
      --  R.1 - If there is a cursor to be closed, only a close cursor
      --        statement is generated
      --  R.2 - If the query is not part of an declare cursor caluse
      --        an execute is directly inserted. Otherwise it is left
      --        to open cursor evaluation.
      --  R.3 - In case of an FETCH operation we include only the
      --        fetch.
      --  R.4 - In case of an dynamic SQL statement use the cursor
      --        as derived from the statement name.
      --  R.5 - In case of an open cursor, simply insert an execute
      --        on the cursor, which creates the result set.
      Query             : Unbounded_String  renames e.Query_String;
      Nbr_Of_Parameters : Natural           renames e.Nbr_Of_Parameters;
      Nbr_Of_Columns    : Natural           renames e.Nbr_Of_Columns;
      Columns           : Identifier_Array  renames e.Columns;
      Inject_Fetch      : Boolean           renames e.Inject_Fetch;
      Cursor_To_Close   : Identifier_String renames e.Cursor_To_Close;
      Cursor_To_Open    : Identifier_String renames e.Cursor_To_Open;
      Cursor_To_Declare : Identifier_String renames e.Cursor_To_Declare;
      Current_Cursor    : Identifier_String renames e.Current_Cursor;
      Cursor_Dynamic_SQL: Identifier_String renames E.Cursor_Dynamic_SQL;
      Finalize_Cursor   : Boolean           renames e.Finalize_Cursor;

   begin
      Ada( E, "-- Execute");
      if Query /= Null_Unbounded_String then
         if Cursor_To_Declare = Blank_Identifier then                   -- R.2
            if Nbr_of_Parameters > 0 or Nbr_Of_Columns > 0 then
               -- the query has been prepared ealier.
               Ada( e, Invoke("EXECUTE(") &
                        GNADE_Handle(E)  & "," &
                        SQLCA_Access(E)  &
                        Cursor_Arg(E)    & ");");
            else -- no preparation has been done, do an direct execute.
               Ada( e, Invoke("EXECUTE(")  & GNADE_Handle(E) & "," &
                        SQLCA_Access(E) & ", " );
               Ada( e, "   Statement => " & Long_String(E, Query) &
                        Cursor_Arg(E) & ");");
            end if;
            ODBC_Debug(E, Comments => "Execute", Query => Long_String(E, Query));
         end if;
      elsif Cursor_Dynamic_SQL /= Blank_Identifier  then               -- R.4
         Current_Cursor := Cursor_Dynamic_SQL;

         Ada( E, Invoke("EXECUTE(") &
                 GNADE_Handle(E)  & "," &
                 SQLCA_Access(E)  &
                 Cursor_Arg(E)    & ");");
         ODBC_Debug( E, Query => "", Comments => "Execute");
      elsif Cursor_To_Close /= Blank_Identifier then                    -- R.1
            Ada( e, Invoke("CLOSE_CURSOR(") &
                            GNADE_Handle(E) & ","   );
            Ada( e, "   " & Quote( Cursor_Name( E, Cursor_To_Close ) ) & "," );
            Ada( E, "   Finalize=>" & Boolean'Image(Finalize_Cursor) & ");" );
            ODBC_Debug(E, Query => "", Comments => "CLOSE_CURSOR" );
      elsif Cursor_To_Open /= Blank_Identifier  then
         Ada( e, Invoke("EXECUTE(") &
                     GNADE_Handle(E)  & "," &
                     SQLCA_Access(E)  & "," &
                     "CURSOR =>" & Quote( Cursor_Name( E, Cursor_To_Open ) ) & ");");
         ODBC_Debug(E, Query => "", Comments => "EXECUTE " );
      end if;


      --- Bug fix 0701.3: In case of fetch we only insert the fetch command but
      --- the query is Null_Unbounded_String.
      if Inject_Fetch then
         Ada( e, Invoke("FETCH(") &
                            GNADE_Handle(E) & "," &
                            SQLCA_Access(E)       &
                            Cursor_Arg(E) & ");" );
      end if;

      E.Eval_Result := True;
   end ODBC_Execute;

   --------------------------
   -- ODBC_Bind_Parameters --
   --------------------------
   procedure ODBC_Bind_Parameters(
      e                 :  in ESQL_Reader ) is
      -- Bind the parameters if the query
      Nbr_Of_Parameters : Natural renames e.Nbr_Of_Parameters;
      Parameters        : Identifier_Array renames E.Parameters;
      Parameter_Ind     : Identifier_Array renames E.Parameter_Ind;

      procedure Bindparameter(
         No        : in Natural;
         Variable  : in String;
         Size      : in String;
         Ctype     : in String;
         Indicator : in String ) is
         -- create the ESQL_Support.Bindparamter call
      begin
         ADA( E, Invoke("BINDPARAMETER(")                               );
         Ada( E, "       " & GNADE_Handle(E)                          & "," );
         Ada( E, "       " & Natural'Image(No)                        & "," );
         Ada( E, "       " & Name(variable) & "'Address"              & "," );
         Ada( E, "       " & Name(Size) & "'Size"                     & "," );
         Ada( E, "       " & Ctype                                    & "," );
         Ada( e, "       " & Make_Access(E, Indicator)                      );
         Ada( E, "       " & Cursor_Arg(e)                                  );
         Ada( E, "   );" );
      end Bindparameter;
      ---
   begin
      for i in 1..Nbr_Of_Parameters loop
         Ada( e, "", comment => "Parameter Binding " & Parameters(i) );

         case Type_Of_Host_Variable( Parameters(I) ) is
            ---------------------------------------------------------------
            -- Handle all standard data types which requiere an explicit --
            -- indicator type.                                           --
            ---------------------------------------------------------------
            when ISO92_CHAR_TYPE..ISO92_INDICATOR_TYPE | GNADE_BINARY_TYPE  =>
               declare
                  Ind_Variable : Identifier_String  := Blank_Identifier;
               begin
                  if Parameter_Ind(I) /= Blank_Identifier then
                     Ind_Variable := Parameter_Ind(i);
                  else
                     Ind_Variable := To_Identifier(
                        Indicator_Variable(Parameters(I)) );
                  end if;

                  Bindparameter(
                     No       => I,
                     Variable => Parameters(I),
                     Size     => Parameters(I),
                     Ctype    => Host_Var_Type_String(Parameters(I)),
                     Indicator => Ind_Variable
                  );
               end;
            ---------------------------------------------------------------
            -- Handle the GNADE implementation of strings with variable  --
            -- length.                                                   --
            ---------------------------------------------------------------
            when GNADE_VARCHAR_TYPE | GNADE_VARBINARY_TYPE =>
               Bindparameter(
                  No       => I,
                  Variable => Name(Parameters(I)) & ".Value",
                  Size     => Name(Parameters(I)) & ".Value",
                  Ctype    => Host_Var_Type_String(Parameters(I)),
                  Indicator => Name(Parameters(I)) & ".Length"
               );
            when others =>
               Syntax_Error( E, "binding to unknown data type " );
         end case;

      end loop;
   end ODBC_Bind_Parameters;

   -----------------------
   -- ODBC_Bind_Columns --
   -----------------------
   procedure ODBC_Bind_Columns(
      e              :  in ESQL_Reader ) is
      -- Bind the columns as they have been specified e.q. by
      -- the into clause.
      Current_Cursor : Identifier_String renames e.Current_Cursor;
      Columns        : Identifier_Array  renames e.Columns;
      Nbr_Of_Columns : Natural           renames e.Nbr_Of_Columns;
      Indicators     : Identifier_Array  renames e.Indicators;
      Indicator_Var  : Identifier_String;

      procedure BindColumn(
         No        : in Natural;
         Variable  : in String;
         Size      : in String;
         Ctype     : in String;
         Indicator : in String ) is
      begin
         Ada( E, "   " & Invoke("BINDCOLUMN(")                          );
         Ada( E, "       " & GNADE_Handle(E)                      & "," );
         Ada( E, "       " & Natural'Image(No)                    & "," );
         Ada( E, "       " & Name(Variable) & "'Address"          & "," );
         Ada( E, "       " & Name(Size) & "'Size"                 & "," );
         Ada( E, "       " & Ctype                                & "," );
         Ada( E, "       " & Make_Access(E, Indicator)                  );
         Ada( E, "       " & Cursor_Arg(e) & ");"                       );
      end Bindcolumn;

   begin
      if Nbr_Of_Columns > 0 then

         -- only in cases of a named cursor we check the number of bound
         -- columns.
         if Current_Cursor /= Blank_Identifier then
            Ada( E, "IF " & Invoke("Columns(")
                          & GNADE_Handle(E) & Cursor_Arg(E) &") = 0 THEN");
         end if;

         for I in 1..Nbr_Of_Columns loop

            case Type_Of_Host_Variable( Columns(I) ) is
               --------------------------------------------------------------
               -- Handle the ISO standard types.                           --
               --------------------------------------------------------------
               when ISO92_CHAR_TYPE..ISO92_INDICATOR_TYPE | GNADE_BINARY_TYPE =>
                  if Indicators(I) = Blank_Identifier then
                     Indicator_Var := To_Identifier(Indicator_Variable(Columns(I)));
                  else
                     Indicator_Var := Indicators(i);
                  end if;

                  Bindcolumn(
                     No        => I,
                     Variable  => Columns(I),
                     Size      => Columns(I),
                     Ctype     => Host_Var_Type_String(Columns(I)),
                     Indicator => Indicator_Var
                  );
               --------------------------------------------------------------
               -- GNADE implementation of VAHRCHAR                         --
               --------------------------------------------------------------
               when GNADE_VARCHAR_TYPE | GNADE_VARBINARY_TYPE =>
                  Bindcolumn(
                     No        => I,
                     Variable  => Name(Columns(I)) & ".Value",
                     Size      => Name(Columns(I)) & ".Value",
                     Ctype     => Host_Var_Type_String(Columns(I)),
                     Indicator => Name(Columns(I)) & ".Length"
                  );
               --------------------------------------------------------------
               when others =>
                  null;
            end case;

         end loop;

         if Current_Cursor /= Blank_Identifier then
            Ada( E, "END IF;" );
         end if;
      end if;
   end ODBC_Bind_Columns;

   --------------------------
   -- ODBC_Evaluate_Result --
   --------------------------
   procedure ODBC_Evaluate_Result(
      E               : in ESQL_Reader;
      Comment         : in String  := ""  ) is
      -- Generate the Ada code in order to verify the return codes
      -- of the ODBC interface.
      Not_Found_Action     : ESQL_Reserved_Words renames e.Not_Found_Action;
      Not_Found_Target     : Identifier_String   renames e.Not_Found_target;
      SQL_Error_Action     : ESQL_Reserved_Words renames e.SQL_Error_Action;
      SQL_Error_Target     : Identifier_String   renames e.SQL_Error_Target;
      SQL_Warning_Action   : ESQL_Reserved_Words renames e.SQL_Warning_Action;
      SQL_Warning_Target   : Identifier_String   renames e.SQL_Warning_Target;
      Query                : Unbounded_String    renames e.Query_String;
      Query_Variable       : Unbounded_String    renames e.Query_Variable;

      Q                    : Unbounded_String ;
      --
   begin
      if not E.Eval_Result then
         return;
      end if;

      Ada( e, "-- Evaluate_Result" );

      if Query /= Null_Unbounded_String then
         Q := Query ;
      elsif Query_Variable /= Null_Unbounded_String then
         Q := Query_Variable;
      else
         Q := To_Unbounded_String( E.Cursor_Dynamic_SQL );
      end if;

      Ada( E, "SQLCODE  := " & SQLCA_Var(E, "SQLCODE") & ";");
      Ada( E, "SQLSTATE := " & SQLCA_Var(E, "STATE")  & ";");

      if SQL_Error_Action /= ESQL_Continue then
         Ada( e, "IF ( SQLSTATE(1..2) /= ESQL_SUPPORT.SUCCESS_CLASS  AND" );
         Ada( E, "   SQLSTATE(1..2) /= ESQL_SUPPORT.WARNING_CLASS  AND " );
         Ada( E, "   SQLSTATE(1..2) /= ESQL_SUPPORT.NOTFOUND_CLASS)  OR" );
         Ada( E, "   SQLCODE in SQL_STANDARD.SQL_ERROR THEN" );
         Ada_Action( E, SQL_Error_Action, SQL_Error_Target, Comment );
         Ada( e, "END IF;");
      end if;

      -- Pro*Ada extension
      if SQL_Warning_Action /= ESQL_Continue then
         Ada( e, "IF SQLSTATE(1..2) = ESQL_SUPPORT.WARNING_CLASS  THEN" );
         Ada_Action( E, SQL_Warning_Action, SQL_Warning_Target, Comment );
         Ada( e, "END IF;");
      end if;

      if Not_Found_Action /= ESQL_Continue then
         Ada( e, "IF SQLSTATE(1..2) = ESQL_SUPPORT.NOTFOUND_CLASS OR" );
         Ada( e, "   SQLCODE in SQL_STANDARD.NOT_FOUND  THEN" );
         Ada_Action( E, Not_Found_Action, Not_Found_Target, Comment );
         Ada( e, "END IF;" );
      end if;

   end ODBC_Evaluate_Result;

   -----------------
   -- ODBC_Finish --
   -----------------
   procedure ODBC_Finish(
      E : in ESQL_reader ) is
      -- End of an SQL Query block.
      Not_Found_Action   : ESQL_Reserved_Words renames e.Not_Found_Action;
      SQL_Error_Action   : ESQL_Reserved_Words renames e.SQL_Error_Action;
      SQL_Warning_Action : ESQL_Reserved_Words renames e.SQL_Warning_Action;
   begin
      -- Section 19.2 Chapter, general rules 1c for clearing WHENEVER
      -- conditions.
      if not Option_ISO92_Exception  then
         SQL_Error_Action   := ESQL_Continue;
         SQL_Warning_Action := ESQL_Continue;
         Not_Found_Action   := ESQL_Continue;
      end if;

      Ada( E, "exception");
      Ada( E, "   when The_Error : Others =>");

      ODBC_Debug( E, Query => "",  Force => True,
                     Comments => "*** ", 
		     Reason => "Exception_Information(The_Error)" );
      Ada( E, "      raise;");
      Ada_End(e);
   end ODBC_Finish;

   ----------------------
   -- ODBC_SQL_Context --
   ----------------------
   procedure ODBC_SQL_Context(
      e : in ESQL_Reader ) is
      -- This procedure generates the implict SQL context. The procedure
      -- generates the local and the global environment.
   begin
      if not e.Global_SQL_Env_Set then
         if not Option_No_SQLSTATE then
            Ada( e, "SQLCODE          : SQL_STANDARD.SQLCODE_TYPE ;"                );
            Ada( e, "SQLSTATE         : SQL_STANDARD.SQLSTATE_TYPE;"                );
         end if;

         SQLCA_Ada_Declare(E);

         Ada( e, ""                                                                 );
         Ada( e, "GNADE_DB_DEFAULT : " & Invoke("CONNECTION_HANDLE") & " := NULL;"  );
         Ada( E, "GNADE_G_CONTEXT  : " & Invoke("CONTEXT_TYPE") & ";"               );
         Ada( e, ""                                                                 );
         e.Global_SQL_Env_Set := True;
      else
        null;
      end if;

   end ODBC_SQL_Context;

   ------------------
   -- ODBC_Connect --
   ------------------
   procedure ODBC_Connect(
      E : in ESQL_Reader ) is
      -- Generate the name of the connection handle and call the
      -- connect primitive of ESQL_Support.
      Exception_Inserted : Boolean := False;
      Connection_Errors  : Connection_Error_Array renames E.Connection_Errors;
   begin
      Ada_Begin( E );

      if E.DB_Name /= Blank_Identifier then
         Ada( e, GNADE_Handle(E) & ":=" & Invoke_Connector("Connect(")        &
                     String_Arg( E.DB_Name )    & ", "                        &
                     String_Arg( E.Login_Name ) & ", "                        &
                     String_Arg( E.Password )   & ");"                        );
      else
          Ada(e, "   NULL;"                                         );
      end if;
      Ada( e, "" );

      for I in Connection_Error_Type loop
         if Connection_Errors(I).Action /= ESQL_Continue then

            if not Exception_Inserted then
               Exception_Inserted := True;
               Ada_Exception( E );
            end if;

            case I is
               when Connection_Failure =>
                  Ada( E, "when GNU.DB.SQLCLI.CONNECTION_ERROR =>" );

               when Authorization_Failure =>
                  Ada( E, "when GNU.DB.SQLCLI.INVALID_AUTH_SPEC  =>" );

               when Other_Fault =>
                  Ada( E, "when Others =>" );
            end case;
            Ada_Action( E, Connection_Errors(I).Action, Connection_Errors(I).Target );
         end if;

      end loop;

      Ada_End( E );

      E.Connection_Errors := ( others => Null_Connection_Error );

   end ODBC_Connect;

   ---------------------
   -- ODBC_Disconnect --
   ---------------------
   procedure ODBC_Disconnect(
      E                   : in ESQL_Reader ) is
      -- gnerate a diconnect poricedure call
      Connection_To_Close : Identifier_String renames E.Cursor_To_Close;
      Connection          : Identifier_String renames E.Connection;
      Perform_Commit      : Boolean renames E.Perform_Commit;

   begin
      Connection := Connection_To_Close;

      Ada( E, Invoke_Connector("DISCONNECT(") & GNADE_Handle(E)  & ","  &
              Context_Object(E)                                  & ","  &
              "Commit => " & Boolean'Image(Perform_Commit)       & ");"  );
   end ODBC_Disconnect;

   ---========================================================================---
   ---===             P A R S E R    P R O C E D U R E S                   ===---
   ---========================================================================---

   --------------------------
   -- Verify_Host_Variable --
   --------------------------
   procedure Verify_Host_Variable(
      e        : in ESQL_Reader;
      name     : in Identifier_String ) is
      -- Check wether a name is defined in the table of all host variables.
      Variable : Host_Variable_Record ;
   begin
      Fetch( Host_Vars, Trim(Upper_Case(Name),Right), Variable );

   exception
      when Entry_Not_Found =>
         Syntax_Error( e, "'" & Trim(Name,Right) & "' not defined in DECLARE SECTION" );
   end Verify_Host_Variable;

   -----------------------------
   -- Ada_Variable_Constraint --
   -----------------------------
   procedure Ada_Variable_Constraint(
      e       : in ESQL_Reader ) is
      --
      -- This procedure handles constraints e.g. X(1..10). This feature
      -- is not realy supported but it will be provided in the
      -- future.
      --
      Current : ESQL_Reserved_Words;
   begin
      Current := Next_Symbol(e);

      if Current = ESQL_Open_Bracket then
         Pedantic( e, "constraints in host variables not supported");
         Skip_Till( e, ESQL_Close_Bracket );
      else
         Push_Back_Current_Token(e);
      end if;

   end Ada_Variable_Constraint;

   ----------------------------
   -- Embedded_Host_Variable --
   ----------------------------
   procedure Embedded_Host_Variable(
      e              : in  ESQL_Reader;
      Variable_Name  : out Identifier_String;
      Indicator_Name : out Identifier_String ) is
      -- Handle the host variable syntax:
      --  ':' <variable> [ [ 'INDICATOR' ':'<indicator> ] ]
      Current        : ESQL_Reserved_Words;
   begin
      Expect( e, ESQL_Colon );
      Expect( e, ESQL_Identifier );
      Variable_Name := Identifier(e);

      Verify_Host_Variable( e, Variable_Name );
      Ada_Variable_Constraint(e);

      Current := Next_Symbol(e);
      if Current = ESQL_Indicator then   -- Pro*Ada extension
         Pedantic( E, "using non ISO/92 indicator keyword" );
         Current := Next_Symbol(E);
         if Current /= ESQL_Colon then
            Syntax_Error( E, "Expected host variable after INDICATOR");
         end if;
      end if;

      if Current = ESQL_Colon then
         -- expecting an indicator variable for the current column
         Expect( e, ESQL_Identifier );
         Indicator_Name := Identifier(E);
         Verify_Host_Variable( e, Indicator_Name );
      else
         Indicator_Name := Blank_Identifier;
         Push_Back_Current_Token(E);
      end if;

   end Embedded_Host_Variable;

   --------------------------
   -- Embedded_INTO_Clause --
   --------------------------
   procedure Embedded_INTO_Clause(
      e              : in ESQL_Reader ) is
      --  INTO <hostvariable list>
      Current        : ESQL_Reserved_Words;
      Columns        : Identifier_Array renames e.Columns;
      Indicators     : Identifier_Array renames e.Indicators;
      Nbr_Of_Columns : Natural          renames e.Nbr_Of_Columns;
      --
   begin
       for i in Columns'Range loop
          Embedded_Host_Variable(E, Columns(I), Indicators(I));

          Nbr_Of_Columns := Nbr_Of_Columns + 1;

          Current := Next_Symbol(e);
          exit when Current /= ESQL_Comma;
       end loop;

   end Embedded_INTO_Clause;

   -----------------------------
   -- ISO92_Fetch_From_Cursor --
   -----------------------------
   procedure ISO92_Fetch_From_Cursor(
      E               : ESQL_Reader ) is
      -- FETCH FROM <cursor> [ INTO ..... ; ]
      -- FETCH [ INTO ]
      -- FETCH USING [ STATEMENT ] [ INTO .... ; ]
      Current          : ESQL_Reserved_Words;
      Current_Cursor   : Identifier_String renames e.Current_Cursor;
      Current_Statement: Identifier_String renames e.Current_Cursor;
      Inject_Fetch     : Boolean renames e.Inject_Fetch;
      ---
   begin
      Current := Next_Symbol(E);
      case Current is
         when ESQL_From =>
            Expect( E, ESQL_Identifier );
            Current_Cursor  := Upper_Case(Identifier(E));
            --****--
         when ESQL_Using =>
            Current := Next_Symbol(E);
            if Current /= ESQL_Statement then
               Push_Back_Current_Token(E);
            end if;
            Expect( E, ESQL_Identifier );
            Current_Cursor :=  Upper_Case(Identifier(E));
            --****--
         when ESQL_Into =>
            Current_Cursor := Blank_Identifier;
            Push_Back_Current_Token(E);
         when others =>
            Syntax_Error( E, "excpected token in FETCH clause" );
      end case;

      Inject_Fetch    := True;            -- force the generation of a fetch

      Current := Next_Symbol(E);
      if Current = ESQL_Into then
         Embedded_INTO_Clause(E);
      end if;

      Push_Back_Current_Token(E);
      Skip_Till( E, e.ESQL_Terminator );

   end ISO92_Fetch_From_Cursor;

   ---------------------
   -- Append_To_Query --
   ---------------------
   function Append_To_Query(
      E       : in ESQL_Reader;
      Query   : in Unbounded_String ) return Unbounded_String is
      -- this function appends the current token to the query string
      -- passed.
      -- R.1 - If the current symbol is a terminal then no blank
      --       will be appended
      -- R.2 - If we find a dot, this function consumes the next
      --       string as well, because we assume it to be a part
      --       for a relation name (e.g. employees.empno). This is
      --       is an ugly solution but the best place to put it.
      Result  : Unbounded_String    := Query;
      Current : ESQL_Reserved_Words := E.Current_Keyword;
      S       : Unbounded_String    := To_Unbounded_String( Name(E.Current) );
   begin
      while Current = ESQL_DOT loop                             -- R.2
         Result  := Result & ".";
         Current := Next_Symbol(E);
         Result  := Result & To_Unbounded_String( Name( E.Current ) );

         Current := Next_Symbol(E);
         if Current /= ESQL_DOT then
            Push_Back_Current_Token(E);
            S := Null_Unbounded_String;
         end if;
      end loop;

      if not Is_Terminal(E.Current_Keyword) then                -- R.1
         Result := Result & " ";
      end if;

      Result := Result & S;
      return Result;
   end Append_To_Query;

   -------------------------
   -- ISO92_SQL_Statement --
   -------------------------
   procedure ISO92_SQL_Statement(
      E              : in ESQL_Reader;
      Previous_Token : in ESQL_Reserved_Words := ESQL_Null ) is
      -- Parse the queries by reading all data till the current terminator
      -- symbol and building ODBC query and then bind the parameters.
      --
      -- The previous_token parameter shall be used in case  where it
      -- is not possible to use the Push_Back mechanism, as e.g. i case
      -- of the handling of the BEGIN token.
      --
      -- BEGIN DECLARE  ---> Declare section
      -- BEGIN <else>   ---> ISO92 query.
      Current        : ESQL_Reserved_Words;
      Query          : Unbounded_String renames e.Query_String;
      Columns        : Identifier_Array renames e.Columns;
      Indicators     : Identifier_Array renames e.Indicators;
      Nbr_Of_Columns : Natural          renames e.Nbr_Of_Columns;
      Inject_Fetch   : Boolean          renames e.Inject_Fetch;
      Operation      : ESQL_Reserved_Words     := ESQL_Null;
   begin
      if previous_token /= ESQL_NULL then
         Query := Query & To_Unbounded_String( Trim(Keywords(Previous_Token),Right) );
      end if;

      -- process and INTO's in the SELECT or FETCH
      Current := Next_Symbol(e);
      case Current is
         -- Pro*Ada Syntax: Select <cols> [INTO <hostvars>] FROM
         when ESQL_Select =>

            Operation := Current;
            Query     := Append_To_Query( E, Query );

            Current   := Next_Symbol(e);

            while Current /= ESQL_INTO and
                  Current /= E.ESQL_Terminator and
                  Current /= ESQL_FROM
            loop
               Query   := Append_To_Query( E, Query );
               Current := Next_Symbol(E);
            end loop;

            if Current = ESQL_INTO then
               -- this hack is used to identify a temporary table e,g
               -- SELECT ... INTO [TEMPORARY|TEMP]
               Current := Next_Symbol(E);
               if Current = ESQL_TEMPORARY or Current = ESQL_TEMP then
                  Query   := Query & " INTO ";
               else
                  Push_Back_Current_Token(E);
                  Embedded_INTO_Clause(E);

                  Inject_Fetch := True;
               end if;
            end if;
            Push_Back_Current_Token(E);

         -- Pro*Ada Syntax: Fetch FROM <cursor> [ INTO <hostvars> ]
         when ESQL_Fetch =>
            ISO92_Fetch_From_Cursor(E);
            Query := Null_Unbounded_String;
            Push_Back_Current_Token(E);

         when ESQL_Semicolon =>
            -- this is a WoA for queries as i.e. 'EXEC SQL BEGIN ;'
            Push_Back_Current_Token(e);

         when Others =>
            Query := Append_To_Query( E, Query );

      end case;

      -- handle the remaining part of the query
      Current := Next_Symbol( E );
      while Current /= e.ESQL_Terminator loop

         if Current = ESQL_Colon then
            declare
               Parameter : Identifier_String;
            begin
               Expect(e, ESQL_Identifier);
               Parameter := Identifier(E);

               Verify_Host_Variable( e, Identifier(e) );
               Ada_Variable_Constraint(e);

               Query := Query & "? " ;

               Current := Next_Symbol(E);
               if Current = ESQL_Indicator then
                  Expect(E, ESQL_Colon );
                  Expect(E, ESQL_Identifier);
                  Add_Parameter( e, Parameter, Identifier(E) );
               elsif Current = ESQL_Colon then
                  Expect( E, ESQL_Identifier );
                  Add_Parameter( e, Parameter, Identifier(E) );
               else
                  Push_Back_Current_Token(E);
                  Add_Parameter( e, Parameter );
               end if;
            end;
         else
            Query := Append_To_Query( E, Query );
         end if;

         Current := Next_Symbol( e );
      end loop;
   end ISO92_SQL_Statement;

   ----------------------
   -- Table_Definition --
   ----------------------
   procedure Table_Definition(
      E           : in ESQL_Reader;
      Table_Name  : in Identifier_String ) is
      -- Handle a tabledefintion based upon embedded SQL:
      --
      -- '(' <table_defintition> ')'
      -- table_definition := <column> <type> [ ',' <table_definition> ]
      Current     : ESQL_Reserved_Words;
      Column_Name : Identifier_String := Blank_Identifier;
      Column_Type : Identifier_String := Blank_Identifier;
      Length      : Natural := 0;

      procedure Schema(
         S : in String ) is
      begin
         if Option_Schema then
            Put_Line(Schema_File, S );
         end if;
      end Schema;

      function Is_Allowed_Type (
         S : in Identifier_String ) return Boolean is
      begin
         for I in SQL_Type_Mapping'Range loop
            if S = Sql_Type_Mapping(I) then
               return True;
            end if;
         end loop;

         return False;
      end ;

   begin
      Expect( E, ESQL_Open_Bracket );

      if Option_Verbose then
         Put_Line("     Table : " & Table_Name );
      end if;

      Schema(
         "CREATE TABLE " & Trim(Table_Name, Right) & " ( " );

      Current := Next_Symbol(E);
      while Current /= ESQL_Close_Bracket loop
         declare
            Result  : Unbounded_String := Null_Unbounded_String;
            Nesting : Integer := 0;
         begin
            if Current = ESQL_Identifier then
               Column_Name := Identifier(E);
            else
               Syntax_Error(E, "keyword in column name not allowed");
            end if;
            if Option_Verbose then
               Put_Line( "        " &  Column_Name );
            end if;

            -- read in the data type name
            Expect(E, ESQL_Identifier );
            Column_Type := Identifier(E);

            Result := To_Unbounded_String(
                         Name( Column_Name  ) & " " & Name( Column_Type ) & " "
                      );

            -- scan in the rest of the definition e.g. attributes.
            loop
               Current := Next_Symbol(E);
               Result  := Result & To_Unbounded_String( Name( E.Current) &" " );

               case Current is
                  when ESQL_Open_Bracket =>
                     Nesting := Nesting + 1;
                  when ESQL_Close_Bracket =>
                     if Nesting > 0 then
                        Nesting := Nesting - 1;
                     else
                        exit;
                     end if;
                  when ESQL_Comma =>
                        exit;
                  when others =>
                     null;
               end case;
            end loop;
            Schema( To_String( Result ) );

            exit when Current /= ESQL_Comma;
            Current := Next_Symbol(E);
         end;
      end loop;
      Schema( ")" );

   end Table_Definition;

   ----------------------
   -- Embedded_Declare --
   ----------------------
   procedure Embedded_Declare(
      e                 : ESQL_Reader ) is
      -- Handle the declare clause:
      -- DECLARE <name> CUSOR ....
      -- DECLARE <name> DATABASE
      -- DECLARE <name> TABLE
      -- DECLARE <name> STATEMENT
      Current           : ESQL_Reserved_Words;
      Query             : Unbounded_String  renames e.Query_String;
      AName             : Identifier_String;
      Cursor_To_Declare : Identifier_String renames E.Cursor_To_Declare;
   begin
      Expect( E, ESQL_Identifier );
      aName := Identifier(e);

      Current := Next_Symbol(e);
      case Current is
         when ESQL_DATABASE =>                       -- Pro*Ada extension
            Insert_DB_Table_Name(E, aName);
            Ada( E, GNADE_Handle( AName ) & " : " & Invoke("CONNECTION_HANDLE"));

         when ESQL_TABLE =>
            Table_Definition(E, AName);
            Skip_Till( E, E.ESQL_Terminator );

         when ESQL_STATEMENT =>
            Pedantic(E, "using non ISO/92 dynamic SQL declare construct");
            Insert_STATEMENT_Name(E, AName );
            E.Statement_Name := AName;
            Skip_Till( E, e.ESQL_Terminator );

         when others =>
            -- DECLARE name [REOPENABLE | LOCAL ] CURSOR FOR SQL statement
            Cursor_To_Declare := Upper_Case(AName);

            case Current is
               when ESQL_Reopenable =>
                  Pedantic(E, "using non ISO/92 reopenable cursor");
                  E.Cursor_Is_Reopenable := True;
                  Expect( E, ESQL_CURSOR );

               when ESQL_Local =>
                  Pedantic(E, "using non ISO/92 local cursor");
                  E.Cursor_Is_Local := True;
                  Expect( E, ESQL_CURSOR );

               when ESQL_CURSOR =>
                  null;

               when others =>
                  Syntax_Error( E,
                     "unexpected token " & ESQL_Reserved_Words'Image(Current) &
                     " in DECLARE clause");
            end case;
            Expect( E, ESQL_FOR );

            -- Handle the query part
            ISO92_SQL_Statement(e);

            E.Inject_Fetch := False;

            ODBC_Prelude( e );
            ODBC_Prepare_Query( e );
            ODBC_Evaluate_Result (E, Comment => "Prepare");
            ODBC_Bind_Parameters( e );
            ODBC_Bind_Columns( e );
            ODBC_Execute( e );
            ODBC_Evaluate_Result (E, Comment => "Declare Cursor");
            ODBC_Finish(e);
      end case;
   end Embedded_Declare;

   --------------------------
   -- Embedded_Var_Declare --
   --------------------------
   procedure Embedded_Var_Declare(
      E : in ESQL_Reader ) is
      -- Analyze the VAR decalraton of PRO*Ada, but do nothing
      -- with the information.
      -- VAR host_variable is ..... ;
      Current       : ESQL_Reserved_Words;
      --
   begin
      if Option_Pedantic then
         Syntax_Error(E, "non ISO/92 VAR clause is not supported");
      else
         Warning( E, "non ISO/92 VAR clause ignored" );
      end if;

      Current := Next_Symbol(E);
      if Current = ESQL_Colon then
         Current := Next_Symbol(E);
      end if;

      Expect( E, ADA_IS );
      Skip_Till( E, E.ESQL_Terminator );
   end Embedded_Var_Declare;

   ----------------------------
   -- Ada_Variable_Defintion --
   ----------------------------
   procedure Ada_Variable_Definition(
      e         : in ESQL_Reader ) is
      --
      -- Section 19.3
      -- <identifier> [,<identifierlist>] ':' <type> ...... ';'
      --
      -- The information after the type is not analyzed by the
      -- parser, it is simply copied throught into the result
      -- file. The Ada compiler has to deal with it.
      --
      -- All host viariables will be defined with the attribute
      -- aliased.
      --
      -- R.1 - In order to accept package names in front of types only
      --       one package name is prepended:
      --         type ::= [ <package> '.' ] <Base type>
      --
      Names     : Identifier_Array(1..50);
      Base_Type : Identifier_String := Blank_Identifier;
      Length    : Natural           := 0;
      Current   : ESQL_Reserved_Words;
      ---
   begin
      Identifier_List( e,
         Names,
         Length,
         mode => Accept_Token );

      Expect(e, ESQL_Colon, mode => Accept_Token, Where => "host variable defintion");

      Expect(e, ESQL_Identifier, Mode => Defere_Copy );       -- R.1
      Base_Type := Identifier(E);
      Current := Next_Symbol(E, Mode => Defere_Copy );
      if Current = ESQL_Dot then
         Base_Type := Append( Base_Type, "." );
         Expect(e, ESQL_Identifier, Mode => Defere_Copy );
         Base_Type := Append( Base_Type, Identifier(E));
      else
         Push_Back_Current_Token(E);
      end if;

      if SQL92_Type( Base_Type ) = ISO92_INDICATOR_TYPE then
         Accept_String( E.Reader,  " aliased " );
      end if;

      case SQL92_Type( Base_Type ) is
         when ISO92_CHAR_TYPE =>
            Current := Next_Symbol(E, Mode => Accept_Token);
            if Current = ESQL_Character then
               Syntax_Error( E, "CHARACTER SET CLAUSE not supported");
            elsif Current /= ESQL_Open_Bracket then
               Syntax_Error( E, "( missing in CHAR type specification");
            end if;
            Expect( E, ESQL_Identifier, Mode => Accept_Token );
            Expect( E, ESQL_Range, Mode => Accept_Token );
            Expect( E, ESQL_Identifier, Mode => Accept_Token );
            Expect( E, ESQL_Close_Bracket, Mode => Accept_Token);

         when ISO92_BIT_TYPE =>
            Expect( E, ESQL_Open_Bracket, Mode => Accept_Token );
            Expect( E, ESQL_Identifier, Mode => Accept_Token );
            Expect( E, ESQL_Range, Mode => Accept_Token );
            Expect( E, ESQL_Identifier, Mode => Accept_Token );
            Expect( E, ESQL_Close_Bracket, Mode => Accept_Token );

         when ISO92_Unknown_Type =>
            Syntax_Error( e,
                  "Type " & Name(Base_Type) & " is not a predefined SQL/92 type" );

         when others =>
            null;
      end case;

      Skip_Till(e, ESQL_Semicolon, mode => Accept_Token );

      for i in 1..Length loop
         Add_Host_Variable( Names(i), Base_Type );
      end loop;

   end Ada_Variable_Definition;

   ----------------------------------------
   -- Embedded_Character_Set_Declaration --
   ----------------------------------------
   procedure Embedded_Character_Set_Declaration(
      e  : in ESQL_Reader ) is
      -- Handle:
      --    SQL NAMES ARE <character set specification>
      -- Remark:
      --    The semantics of the commandare not yet evaluated
      --    at all.
      Current       : ESQL_Reserved_Words;
      --
   begin
      Current := Next_Symbol( e, mode => Defere_Copy );
      if Current /= ESQL_SQL  then
         Accept_Symbol(e);
         Push_Back_Current_Token(e);
         return;
      end if;

      Expect( e, ESQL_SQL );
      Expect( e, ESQL_Names );
      Expect( e, ESQL_Are );

      Pedantic( e, "embedded character set declaration not supported");
      Skip_Till( e, ESQL_Semicolon, mode => Suppress_Token );

   end Embedded_Character_Set_Declaration;

   ----------------------------------
   -- Embedded_SQL_Declare_Section --
   ----------------------------------
   procedure Embedded_SQL_Declare_Section(
      e             : in out ESQL_Reader ) is
      -- Parse the declaration section for ESQL for parameters.
      --
      -- EXEC SQL BEGIN DECLARE SECTION [<sql-terminator>]
      --
      --   <identifier> ':'  <type> ....';'
      --      .......
      -- EXEC SQL END DECLARE SECTION [<sql-terminator>]
      --
      -- The type will only be parsed partialy, because the
      -- name is suffcient to verify if there exist a mapping
      -- between underlying ODBC and the Ada type.
      Current       : ESQL_Reserved_Words;
      Host_Variable : Identifier_String := Blank_Identifier;
      Host_Type     : Identifier_String := Blank_Identifier;
      ---
   begin
      Expect(e, ESQL_Declare, where => "declare section" );
      Expect(e, ESQL_Section );
      SQL_Terminator(e);

      Ada( E, "" );
      Ada( E, "   GNADE_L_CONTEXT : " & Invoke("CONTEXT_TYPE") & ";" );

      Current := Next_Symbol( e, mode => Defere_Copy);
      if Current = ESQL_SQL then
         Embedded_Character_Set_Declaration(e);
      else
         while Current /= ESQL_EXEC loop
            -- host variable definition
            Push_Back_Current_Token(e);
            Ada_Variable_Definition(e);

            Current := Next_Symbol( e, mode => Defere_Copy );
         end loop;
      end if;

      ODBC_SQL_Context(e);

      Expect( e, ESQL_SQL );
      Expect( e, ESQL_END );
      Expect( e, ESQL_Declare );
      Expect( e, ESQL_Section );

      SQL_Terminator(e);

   end Embedded_SQL_Declare_Section;

   ----------------------
   -- Condition_Action --
   ----------------------
   function Condition_Action (
      e      : ESQL_Reader ) return ESQL_Reserved_Words is
      --
      -- Handle the action clause of a WHENEVER CLAUSE
      --
      --    actions   :==  { 'GOTO' | 'GO' 'TO' } <target>
      --                  | 'RAISE' <exception>
      --                  | 'DO' <procedure_call>
      --                  | 'CONTINUE'
      --                  ;
      Action : ESQL_Reserved_Words;
      --
   begin
      Action := Next_Symbol(e);
      case Action  is
         when ESQL_GO | ESQL_GOTO =>    -- ISO/92
            -- handle GO TO and GOTO
            if Action = ESQL_GO then
               Expect( e, ESQL_TO );
            end if;
            Action := ESQL_GOTO;

         when ESQL_Continue =>          -- ISO/92
            null;

         when ADA_Raise | Ada_DO =>     -- Pro*Ada extension
            Pedantic(e, "using non ISO/92 action in 'whenever' clause");

         when Others =>
            Syntax_Error( e, "unexpected token " & Name(e.Current) &
                             " in action of when ever clause");
      end case;

      return Action;
   end Condition_Action;

   ------------------------------------
   -- Embedded_Exception_Declaration --
   ------------------------------------
   procedure Embedded_Exception_Declaration(
      e                    : in ESQL_Reader ) is
      --
      -- Declare Exception handling :==
      --    WHENEVER <condition> <action>
      --
      --    condition :== { 'NOT FOUND' | 'SQLERROR' | 'SQLWARNING' }
      --
      Current              : ESQL_Reserved_Words;
      Not_Found_Action     : ESQL_Reserved_Words renames e.Not_Found_Action;
      Not_Found_Target     : Identifier_String   renames e.Not_Found_target;
      SQL_Error_Action     : ESQL_Reserved_Words renames e.SQL_Error_Action;
      SQL_Error_Target     : Identifier_String   renames e.SQL_Error_Target;
      SQL_Warning_Action   : ESQL_Reserved_Words renames e.SQL_Warning_Action;
      SQL_Warning_Target   : Identifier_String   renames e.SQL_Warning_Target;
      ---
   begin
      Current := Next_Symbol(e);
      case Current  is
         when  ESQL_Not  =>          -- NOT FOUND <action> <target>
            Expect( e, ESQL_Found );
            Not_Found_Action := Condition_Action(e);
            if Not_Found_Action /= ESQL_Continue then
               Expect( e, ESQL_Identifier );
               Not_Found_Target := Identifier(e);
            end if;

         when  ESQL_SQL_Error =>
            SQL_Error_Action := Condition_Action(e);
            if SQL_Error_Action /= ESQL_Continue then
               Expect( e, ESQL_Identifier );
               SQL_Error_Target := Identifier(e);
            end if;

         when  ESQL_SQL_Warning =>
            -- Pro*Ada Extension
            Pedantic( e, "non ISO/92 SQLWARNING cause in whenever clause used");
            SQL_Warning_Action := Condition_Action(e);
            if SQL_Warning_Action /= ESQL_Continue then
               Expect( e, ESQL_Identifier );
               SQL_Warning_Target := Identifier(e);
            end if;

         when Others =>
            Syntax_Error(e, "unexpected condition " & Name( e.Current) &
                            " in whenever clause");
      end case;

      Skip_Till( E, E.ESQL_Terminator );
   end Embedded_Exception_Declaration;

   ----------------------
   -- Embedded_Connect --
   ----------------------
   procedure Embedded_Connect(
      e : ESQL_Reader ) is
      --
      --  CONNECT [ user ]
      --          [ BY <Connection> ]
      --          [ TO { <db_name> | DEFAULT } ]
      --          [ AT <host>       ]
      --          [ PORT <port>     ]
      --          [ AS <name> ]
      --          [ IDENTIFIED BY ] <password> ]
      --          [ ON ] <error> [RAISE|GOTO] <target>
      --
      -- This syntax is a mixture of several other dialects. Hope fully
      -- if meets the requirements of every body.
      --
      --
      Current    : ESQL_Reserved_Words;
      DB_Name    : Identifier_String renames e.DB_Name;
      Login_Name : Identifier_String renames e.Login_Name;
      Password   : Identifier_String renames e.Password;
      Connection : Identifier_String renames e.Connection;
      Host       : Identifier_String renames E.Host;
      Port       : Identifier_String renames E.Port;

      Error_Name : constant array( Connection_Error_Type ) of Identifier_String := (
         Connection_Failure    => "COMMUNICATION                                   ",
         Authorization_Failure => "AUTHORIZATION                                   ",
         Other_Fault           => "OTHER                                           "
      );
      --
   begin
      Current := Next_Symbol(E);
      if Current = ESQL_Identifier then
         Login_Name := Identifier(E);
         Current    := Next_Symbol(E);
      end if;

      while Current /= ESQL_Semicolon loop
         case Current is
           when ESQL_BY =>
               Expect( E, ESQL_Identifier );
               Connection := Identifier(E);
               if not Is_Database_Name(Connection) then
                  Syntax_Error( E, Name(Connection) & " not a database " );
               end if;

           when ESQL_TO =>
               Current := Next_Symbol(E);
               if Current = ESQL_DEFAULT then
                  Warning( E, "default in to clause of connect ignored");
               end if;
               DB_Name := Identifier(E);

           when ESQL_AS =>
               Expect( E, ESQL_Identifier);
               if Login_Name /= Blank_Identifier then
                  Warning(E, "username already defined, ignoring: "
                              & Trim(Identifier(E),Right));
               else
                  Login_Name := Identifier(E);
               end if;
           when ESQL_Identified =>
               Expect( E, ESQL_BY );
               Expect( E, ESQL_Identifier);
               Password := Identifier(E);

           when ESQL_AT =>
               Expect( E, ESQL_Identifier );
               Host := Identifier(E);

           when ESQL_ON =>
               declare
                  Error : Connection_Error_Type := Other_Fault;
                  Found : Boolean := False;
               begin
                  -- ON <error> ERROR [RAISE|GOTO|CALL] <target>
                  Expect( E, ESQL_Identifier );
                  for I in Connection_Error_Type loop
                     if Error_Name(I) = Identifier(E) then
                        Error := I;
                        Found := True;
                        exit;
                     end if;
                  end loop;
                  if not Found then
                     Syntax_Error( E, "unknown error type in ON clause");
                  end if;

                  Expect( E, ESQL_Error );

                  Current := Next_Symbol(E);
                  if Current /= ADA_Raise and Current /= ADA_DO and
                     Current /= ESQL_Goto
                  then
                     Syntax_Error( E, "unknown action in ON clause" );
                  else
                     E.Connection_Errors( Error ).Action := Current;
                  end if;
                  Expect( E, ESQL_Identifier );
                  E.Connection_Errors( Error ).Target := Identifier(E);
               end;

           when others =>
               Syntax_Error(E, "unexpected token in CONNECT clause" );
         end case;

         Current := Next_Symbol(E);
      end loop;
   end Embedded_Connect;

   -------------------------
   -- Embedded_Disconnect --
   -------------------------
   procedure Embedded_Disconnect(
      E                   : in out ESQL_Reader ) is
      --
      -- diconnect from the selected data base:
      -- DISCONNECT [ ALL | <identifier> ] [ COMMIT ] ';'
      --
      Current             : ESQL_Reserved_Words;
      Connection_To_Close : Identifier_String renames E.Cursor_To_Close;
      Connection          : Identifier_String renames E.Connection;
      Perform_Commit      : Boolean renames E.Perform_Commit;
   begin
      Connection_To_Close := Connection;

      Current := Next_Symbol(E);
      case Current is
         when ESQL_All =>
            Current := Next_Symbol(E);

         when ESQL_Identifier =>
            if Connection /= Default_Connection then
               Pedantic(E, "overriding connection in AT clause");
            end if;
            Connection_To_Close := Identifier(E);
            Current := Next_Symbol(E);

         when others =>
            null;
      end case;

      if Current = ESQL_Commit then                       -- GNU extension
         Perform_Commit := True;
      else
         Push_Back_Current_Token(E);
      end if;

      Skip_Till( E, E.ESQL_Terminator );

   end Embedded_Disconnect;

   -------------------------
   -- Embedded_Connection --
   -------------------------
   procedure Embedded_Database(
      E             : in out ESQL_Reader ) is
      -- DATABASE IS <identifier>
      --
      -- This construct allows to set the referenced data base
      -- to the identifier specified.
      --
      Instance_Name : Identifier_String := Blank_Identifier;
      Cursor_Name   : Identifier_String := Blank_Identifier;

   begin
      Expect( E, ADA_Is );
      Expect( E, ESQL_Identifier);

      Ada( E, GNADE_Handle(E.Connection) &
              ":=" &
              GNADE_Handle(Identifier(E)) &
              ";"
      );
   end Embedded_Database;


   ----------------------
   -- Embedded_Include --
   ----------------------
   procedure Embedded_INCLUDE(
      e             : in out ESQL_Reader ) is
      -- INCLUDE [SQLCA | DATABASE ] [ NAMED [BY] <identifier> ]
      -- INCLUDE STATEMENT HANDLE [ <cursor> ] OF <database>
      --
      -- This command is used to insert certain strings into the Ada 95
      -- code (e.g. the Connection Handle). In order to allow to embedd
      -- the result into a procedure Argument no final ';' is
      -- expected.
      Current       : ESQL_Reserved_Words;
      Object        : ESQL_Reserved_Words;
      Instance_Name : Identifier_String := Blank_Identifier;
      Cursor_Name   : Identifier_String := Blank_Identifier;
   begin
      Object  := Next_Symbol(E);

      Current := Next_Symbol(E);
      case Current is
         when ESQL_NAMED =>
            Current := Next_Symbol(E);
            if Current = ESQL_BY then
               Current := Next_Symbol(E);
            end if;
            if Current = ESQL_Identifier then
               Instance_Name := Identifier(E);
            else
               Syntax_Error( E,
                  "keyword in NAMED BY clause of include not allowed");
            end if;

         when ESQL_Semicolon =>
            null;

         when ESQL_Handle =>
            Pedantic(E, "non ISO/92 include clause used");
            case Object is
               when ESQL_Statement =>
                  Current := Next_Symbol(E);
                  if Current = ESQL_Identifier then
                     Cursor_Name := Identifier(E);
                     Current := Next_Symbol(E);
                  end if;
                  if Current /= ESQL_OF then
                     Syntax_Error(E, "missing OF in DATABASE HANDLE clause");
                  else
                     Current := Next_Symbol(E);
                  end if;
                  Instance_Name := Identifier(E);

               when ESQL_Connection =>
                  Expect( E, ESQL_OF, Where => "DATABASE CONNECTION clause");
                  Expect( E, ESQL_Identifier );
                  Instance_Name := Identifier(E);
               when others =>
                  Syntax_Error(E, "HANDLE clause used in wrong context" );
            end case;

         when others =>
            if Option_Pedantic then
               Syntax_Error( E, "unknown token in INCLUDE clause" );
            else
               Warning( E, "unknown token in INCLUDE clause ignored" );
            end if;
      end case;
      Push_Back_Current_Token(E);

      -- gnerate the code
      case Object is
         when ESQL_SQLCA =>
            if Instance_Name /= Blank_Identifier then
               E.SQLCA_Name := Instance_Name;
            else
               E.SQLCA_Name := Default_SQLCA_Name;
            end if;
            Ada( E, "   " & Trim(E.SQLCA_Name,Right) & " : aliased SQLCA_TYPE; " );
            Skip_Till( E, E.ESQL_Terminator );

         when ESQL_DATABASE  =>
            Accept_String(E.Reader, GNADE_Handle(Trim(Instance_Name,Right)) );

         when ESQL_STATEMENT =>
            if Option_DBCS = "ODBC" then
               Accept_String( E.Reader,
                  "ODBC_Stmt_Handle( " &
                     GNADE_Handle(Trim(Instance_Name,Right)) & "," &
                     String_Arg( Cursor_Name ) & ")"
               );
            else
               Warning(E,
                  "cannot generate STATEMENT HANDLE clause for selected dbcs");
            end if;

         when ESQL_CONNECTION =>
            if Option_DBCS = "ODBC" then
               Accept_String( E.Reader,
                  "ODBC_Con_Handle( " &
                     GNADE_Handle(Trim(Instance_Name,Right)) & ")"
               );
            else
               Warning(E,
                  "cannot generate STATEMENT HANDLE clause for selected dbcs");
            end if;

         when others =>
            Warning( E,
               "include clause for " & Trim(Identifier(E),Right) & " ignored" );
      end case;

   end Embedded_INCLUDE;

   --------------------
   -- Embedded_Reset --
   --------------------
   procedure Embedded_Reset(
      e             : in out ESQL_Reader ) is
      -- This resets include operation for the SQLCA.
      Current       : ESQL_Reserved_Words;
   begin
      Current := Next_Symbol(E);
      if Current = ESQL_SQLCA then

         Current := Next_Symbol(E);
         if Current /= E.ESQL_Terminator then
            E.SQLCA_Name := Identifier(E);
         else
            E.SQLCA_Name := DEFAULT_SQLCA_Name;
            Push_Back_Current_Token(E);
         end if;
      else
         Warning( E, "RESET clause for " & Trim(Identifier(E),Right) & " ignored" );
      end if;

      Skip_Till( E, E.ESQL_Terminator );
   end Embedded_Reset;

   -------------------
   -- Embedded_Set  --
   -------------------
   procedure Embedded_Set(
      E : in out ESQL_Reader ) is
      --
      -- SET CONNECTION <name>
      --    Connect the default data base to the given connection.
      --
      -- All other set targets are currently ignored.
      --
      Current       : ESQL_Reserved_Words;
   begin
      Current := Next_Symbol(E);
      case Current is
         when ESQL_Connection =>
            Expect( E, ESQL_Identifier );
            Ada(E, GNADE_Handle("DEFAULT") & ":=" &
                   GNADE_Handle(Trim(Identifier(E),Right)) );
         when others =>
            Pedantic(E, "Unkown set clause ignored" );
      end case;
   end Embedded_Set;

   ------------------
   -- Close_Cursor --
   ------------------
   procedure Embedded_Close_Cursor(
      e               : ESQL_Reader ) is
      -- Map the close cursor into a ESQL_SUPPORT (ODBC) operation.
      Current         : ESQL_Reserved_Words;
      Finalize_Cursor : Boolean renames e.Finalize_Cursor;
      --
   begin
      Current := Next_Symbol(E);
      if Current = ESQL_Statement then
         Expect( E, ESQL_Identifier );
         if not Is_Statement_Name(Identifier(E)) then
            Syntax_Error(E, "unknown statemen in close clause");
            return;
         end if;
         Push_Back_Current_Token(E);
      else
         Push_Back_Current_Token(E);
      end if;
      Expect( E, ESQL_Identifier );
      e.Cursor_To_Close := Upper_Case( e.Current );

      Current := Next_Symbol(E);                      -- GNU Ada extension
      if Current = ESQL_Final then
         Finalize_Cursor := True;
         Pedantic(E, "using close cursor with FINAL clause");
      else
         Push_Back_Current_Token(E);
      end if;

      Skip_Till( E, e.ESQL_Terminator );
   end Embedded_Close_Cursor;

   --------------------------
   -- Embedded_Open_Cursor --
   --------------------------
   procedure Embedded_Open_Cursor(
      E              : ESQL_Reader ) is
      -- Handle OPEN <curor>  .... ;
      Cursor_To_Open : Identifier_String renames e.Cursor_To_Open;
      --
   begin
      Expect( E, ESQL_Identifier );
      Cursor_To_Open := Upper_Case(Identifier(E));

      Skip_Till( E, E.ESQL_Terminator );
   end Embedded_Open_Cursor;

   ----------------------
   -- Embedded_Prepare --
   ----------------------
   procedure Embedded_Prepare(
      E                 : in ESQL_Reader ) is
      -- PREPARE <statement>
      --    FROM { <hostvar> | <string> }
      --    [ USING <hostvars> ]
      Current           : ESQL_Reserved_Words;
      Statement         : Identifier_String := Blank_Identifier;
      Cursor_To_Declare : Identifier_String renames e.Cursor_To_Declare;
      Query_Variable    : Unbounded_String  renames e.Query_Variable;
      --
   begin
      Pedantic( E, "using non ISO/92 dynamic SQL prepare statement");

      Expect( E, ESQL_Identifier );
      Statement := Identifier(E);

      if not Is_Statement_Name(Statement) then
         Syntax_Error(E, "unknown statement in prepare clause");
         return;
      end if;
      Cursor_To_Declare := Upper_Case(Statement);

      Expect( E, ESQL_From );
      Expect( E, ESQL_Identifier );
      if OPTION_Debug then
         Put_Line( "Dynamic statement " & Identifier(E) );
      end if;
      Query_Variable := To_Unbounded_String( Trim(Identifier(E),Right) );

      Current := Next_Symbol(E);
      if Current = ESQL_Using then
         declare
            Variable  : Identifier_String := Blank_Identifier;
            Indicator : Identifier_String := Blank_Identifier;
         begin
            E.Nbr_Of_Parameters := 0;
            loop
               Embedded_Host_Variable(E, Variable, Indicator );
               Add_Parameter( E, Variable, Indicator );
               Current := Next_Symbol(E);
               exit when Current /= ESQL_Comma;
            end loop;
            Push_Back_Current_Token(E);
         end ;
      end if;

      Skip_Till( E, E.ESQL_Terminator );
   end ;

   ----------------------
   -- Embedded_Execute --
   ----------------------
   procedure Embedded_Execute(
      E                  : in ESQL_Reader ) is
      -- Handle the syntax of the dynmic embedded sql execute
      -- EXECUTE <statement>
      --    [ USING <hostvar list> ]
      -- EXECUTE [ DECLARE ] BEGIN ... [EXCEPTION] END
      -- EXECUTE IMMEDIATE <hostvar>
      Current            : ESQL_Reserved_Words;
      Statement          : Identifier_String := Blank_Identifier;
      Columns            : Identifier_Array  renames e.Columns;
      Indicators         : Identifier_Array  renames e.Indicators;
      Cursor_Dynamic_SQL : Identifier_String renames E.Cursor_Dynamic_SQL;
      Nbr_Of_Columns     : Natural           renames e.Nbr_Of_Columns;
      Inject_Fetch       : Boolean           renames e.Inject_Fetch;
      --
   begin
      Current := Next_Symbol(E);

      case Current is
         when ESQL_Identifier  =>
            Statement := Upper_Case(Identifier(E));
            if not Is_Statement_Name(Statement) then
               Syntax_Error(E, "unknown statement '" & Trim( Statement, Right) &
                               "' in execute clause");
               return;
            end if;

            Cursor_Dynamic_SQL := Statement;

            Current := Next_Symbol(E);
            if Current =  ESQL_Using  then
               for i in Columns'Range loop
                  Embedded_Host_Variable(E, Columns(I), Indicators(I));

                  Nbr_Of_Columns := Nbr_Of_Columns + 1;

                  Current := Next_Symbol(e);
                  exit when Current /= ESQL_Comma;
               end loop;
            end if;

         when ESQL_Immediate =>
            Expect( E, ESQL_Identifier, Where => " execute clause " );

         when ESQL_Declare | ESQL_Begin =>
            Syntax_Error(E, "plsql blocks not supported in execute clause");

         when others =>
            Syntax_Error(E, "unknown statement '" & Trim( Statement, Right) &
                            "' in execute clause");

      end case;

   end Embedded_Execute;

   ----------------------------
   -- Embedded_SQL_Statement --
   ----------------------------
   procedure Embedded_SQL_Statement(
      e             : in out ESQL_Reader ) is
      --
      -- Parse the esql statements. The parsing begins after
      -- the ESQL prefix. If the prefix is not found, the
      -- procedure returns.
      --
      -- Copying to the result file is defered till the
      -- point that we know how to proceed.
      --
      Current       : ESQL_Reserved_Words;
      Limit         : Natural;
      DB_Name       : Identifier_String renames e.DB_Name;
      Nesting_Level : Natural renames e.Nesting_Level;
      ---
   begin
      -- clear the transient data fo the parser
      E.Nbr_Of_Columns       := 0;
      E.Nbr_Of_Parameters    := 0;
      E.DB_Name              := Blank_Identifier;
      E.Connection           := Default_Connection;
      E.Max_Count            := 0;

      E.Ada_Declare          := False;
      E.Query_String         := Null_Unbounded_String;
      E.Query_Variable       := Null_Unbounded_String;
      E.Inject_Fetch         := False;
      E.Cursor_To_Declare    := Blank_Identifier;
      E.Current_Cursor       := Blank_Identifier;
      E.Cursor_To_Close      := Blank_Identifier;
      E.Cursor_To_Open       := Blank_Identifier;
      E.Cursor_Dynamic_SQL   := Blank_Identifier;
      E.Cursor_Is_Reopenable := False;
      E.Finalize_Cursor      := False;
      E.Cursor_Is_Local      := False;
      E.Statement_Name       := Blank_Identifier;
      E.Connection_Errors    := ( others => Null_Connection_Error );
      E.Eval_Result          := False;

      if E.Nbr_Of_Syntax_Errors > Option_Error_Limit then
         Warning( E, "Number of errors exceed limit ");
         Warning( E, "*** Aborted ***" );
         raise End_Of_File_Exception;
      end if;
      -- process the file contents
      Current := Next_Symbol(e, mode => Defere_Copy );

      case Current  is
         -- *******************************************************************
         -- This section is only needed to control the code intendation in
         -- order to get nice code formating. It will work only in most cases.
         -- *******************************************************************
         when ADA_Package =>
            -- Section 19.3, Syntaxrule 14: Insert a global SQL context --
            Accept_Symbol( e );
            Current := Next_Symbol(E, Mode => Defere_Copy );
            Accept_Symbol(E );
            if Current /= ADA_BODY then
               return;
               --****--
            end if;

            ODBC_Packages( e );

            -- read in the package name including any subpackage names
            E.Package_Name := Null_Unbounded_String;
            loop
               Expect( e, ESQL_Identifier, mode => Accept_Token );
               E.Package_Name := E.Package_Name & Name(Identifier(E));

               Current := Next_Symbol(E, Mode => Accept_Token);

               exit when Current /= ESQL_DOT;
               E.Package_Name := E.Package_Name & ".";
            end loop;

            if Current /= ADA_IS then
               Syntax_Error(E, "Keyword 'IS' missing after package identifier");
            end if;

            if Option_Verbose then
               Put_Line("   Package : " & To_String(E.Package_Name));
            end if;

            Flush(e.Reader);            -- write out: package body x is

            Nesting_Level := Nesting_Level + 1;
            ODBC_SQL_Context( E );
            return;
            --****--

         when ESQL_Begin | ADA_Loop |
              ADA_If     | ADA_Case | Ada_Record =>
            Nesting_Level := Nesting_Level + 1;
            Accept_Symbol( E );
            return;
            --****--

         when ESQL_End  =>
            Accept_Symbol( E );

            Current := Next_Symbol(e, mode => Defere_Copy );
            case Current is
                when ADA_RECORD | ADA_CASE | ADA_IF | ADA_Loop =>
                   Nesting_Level := Nesting_Level - 1;
                   Accept_Symbol( E );
                when others =>
                   Push_Back_Current_Token( E );
            end case;
            return;
            --****--

         -- *********************************************************************

         when Ada_Procedure | Ada_Function =>
            if E.Package_Name = Null_Unbounded_String then
               declare
                  Name : Identifier_String := Blank_Identifier;
               begin
                  Expect( E, ESQL_Identifier, Mode => Accept_Token );
                  Name := Identifier(E);

                  Current := Next_Symbol(E, Mode => Accept_Token );
                  if Current = ESQL_Open_Bracket then
                     Skip_Till( E, ESQL_Close_Bracket, Mode => Accept_Token );
                     Current := Next_Symbol(E, Mode => Accept_Token );
                  end if;
                  if Current /= ADA_Is then
                     return;
                  end if;

                  ODBC_Packages( e );

                  E.Package_Name := To_Unbounded_String(Trim(Name, Right ));
                  if Option_Verbose then
                    Put_Line("   Unit : " & To_String(E.Package_Name));
                  end if;

                  Nesting_Level := Nesting_Level + 1;

                  Flush(e.Reader);
                  Ada( E, "--Is--");
                  ODBC_SQL_Context( E );
               end;
            end if;
            return;
            --****--

         -- *******************************************************************
         -- ISO/92 Ada Style of invokation as EXEC SQL. Processing of the
         -- queries is done below
         -- *******************************************************************
         when ESQL_EXEC =>
            Current := Next_Symbol(e, mode => Defere_Copy );
            if Current /= ESQL_SQL then
               Accept_Symbol( E );
               return;
               --***--
            end if;
            E.ESQL_Terminator := ESQL_Semicolon;

         -- ******************************************************************
         -- in all other cases accept and return.
         -- ******************************************************************
         when Others =>
            pragma Debug( Put_Line( "copies " & E.Current ) );
            Accept_Symbol( E );
            return;
            --****--
      end case;

      E.Query_Line_Number := Current_Line( E.Reader );

      -- **********************************************************************
      -- Process the SQL Queries by checking for the AT/FOR extension and then
      -- handling different key words.
      -- **********************************************************************
      Current := Next_Symbol(e);

      -- Pro*Ada extension : AT and FOR attributes
      while Current = ESQL_FOR or Current = ESQL_AT loop
         case Current is
            when ESQL_FOR =>
               Expect( e, ESQL_Identifier );
               Limit := Natural'Value(e.Current);
               Pedantic( E, "FOR clause ignored");

            when ESQL_AT =>
               Expect( e, ESQL_Identifier );
               if not Is_Database_Name( E.Current ) then
                  if Option_Pedantic then
                     Syntax_Error(E,
                         "database " & Name(E.Current) & " not declared" );
                  else
                      Warning(E,
                         "database " & Name(E.Current) & " not declared" );
                  end if;
               end if;
               e.Connection := e.Current;

            when Others =>
               null;
         end case;
         Current := Next_Symbol(e);
      end loop;

      -- process the different major keywords of embedded SQL
      case Current is
         when ESQL_Begin     =>
            -- The following mess is used to handle the following
            -- two different cases:
            --
            -- EXEC SQL BEGIN DECLARE  ---> Declare section
            -- EXEC SQL BEGIN ;        ---> POSTGRES begin query.
            Current := Next_Symbol(e);
            Push_Back_Current_Token(e);

            if Current = ESQL_DECLARE then
               Embedded_SQL_Declare_Section(e);
               Mark_Source_Line( E.Reader );
            else
               ISO92_SQL_Statement(e, previous_token => ESQL_Begin);

               ODBC_Prelude( e );
               ODBC_Prepare_Query( e );
               ODBC_Evaluate_Result (E, Comment => "Prepare");
               ODBC_Bind_Parameters( e );
               ODBC_Bind_Columns( e );
               ODBC_Execute( e );
               ODBC_Evaluate_Result (E, Comment => "Query");
               ODBC_Finish(e);
               Mark_Source_Line( E.Reader );
            end if;

         when ESQL_Declare =>
            Embedded_Declare(e);

         when ESQL_VAR =>                   -- non ISO/92 PRO*Ada
            Embedded_Var_Declare(E);

         when ESQL_WHENEVER  =>
            Embedded_Exception_Declaration(e);

         when ESQL_Connect    =>
            -- GNADE Extension
            Embedded_Connect(e);
            ODBC_Connect(e);
            Mark_Source_Line( E.Reader );

         when ESQL_Disconnect  =>
            Embedded_Disconnect(E);
            ODBC_Disconnect(E);
            Mark_Source_Line( E.Reader );

         when ESQL_Database =>
            Embedded_Database(E);
            Mark_Source_Line( E.Reader );

         when ESQL_Include =>               -- non ISO/92 Pro*Ada
            Embedded_Include(E);

         when ESQL_Reset =>                 -- GNU extension
            Embedded_Reset(E);

         when ESQL_Set =>
            Embedded_Set(E);

         when ESQL_Close =>
            Embedded_Close_Cursor(E);
            ODBC_Execute( E );
            Mark_Source_Line( E.Reader );

         when ESQL_Open =>
            Embedded_Open_Cursor(E);
            ODBC_Execute( e );
            ODBC_Evaluate_Result (E, Comment => " Open ");
            Mark_Source_Line( E.Reader );

         when ESQL_Prepare =>                -- Dynamic SQL support
            Embedded_Prepare(E);
            ODBC_Prelude( E );
            ODBC_Prepare_Query( E );
            ODBC_Evaluate_Result (E, Comment => " Prepare Query ");
            ODBC_Bind_Parameters( E );
            ODBC_Evaluate_Result (E, Comment => " Bind Parameters");
            ODBC_Finish(e);
            Mark_Source_Line( E.Reader );

         when ESQL_Execute =>
            Embedded_Execute(E);
            ODBC_Prelude( e );
            ODBC_Bind_Columns( e );
            ODBC_Evaluate_Result (E, Comment => "Binding columns");
            ODBC_Execute( e );
            ODBC_Evaluate_Result (E, Comment => "Execute Statement");
            ODBC_Finish(e);
            Mark_Source_Line( E.Reader );

         when Others =>
            -- Handle SQL queries
            Push_Back_Current_Token(e);
            ISO92_SQL_Statement(e);

            ODBC_Prelude( e );
            ODBC_Prepare_Query( e );
            ODBC_Evaluate_Result (E, Comment => "Prepare Query");
            ODBC_Bind_Parameters( e );
            ODBC_Bind_Columns( e );
            ODBC_Execute( e );
            ODBC_Evaluate_Result (E, Comment => " Query ");
            ODBC_Finish(e);
            Mark_Source_Line( E.Reader );
      end case;

      -- **********************************************************************
      -- *               H A N D L E   E X C E P T I O N S                    *
      -- **********************************************************************
   exception
      when Syntax_Exception =>
         Skip_Till( e, ESQL_Semicolon );
      when End_Of_File_Exception =>
         raise;

      when The_Error : Others =>
         Warning( e, "Exception " & Exception_Name( The_Error ) & " occured " );
         Warning( E, "          " & Exception_Message( The_Error ) );
         raise;
   end Embedded_SQL_Statement;

   ---------------------
   -- New_ESQL_Parser --
   ---------------------
   function New_ESQL_Parser(
      f                : in File_Reader ) return ESQL_Reader is
      -- Create a new instance of the ESQL Parser
      Result : ESQL_Reader := new ESQL_Reader_Type;
      --
   begin
      Result.Reader := f;
      Result.Nbr_Of_Syntax_Errors := 0;

      return Result;
   end New_ESQL_Parser;

   ------------------------
   -- Delete_ESQL_Reader --
   ------------------------
   procedure Delete_ESQL_Reader(
      E : in out ESQL_Reader ) is
   begin
      null;
   end Delete_ESQL_Reader;

   ----------------------
   -- Number_Of_Errors --
   ----------------------
   function Number_Of_Errors(
      E : ESQL_Reader ) return Natural is
      -- Return the number of errors found by the parser.
   begin
      return E.Nbr_Of_Syntax_Errors;
   end Number_Of_Errors;

   ----------------------
   -- Number_Of_Errors --
   ----------------------
   function Number_Of_Warnings(
      E : ESQL_Reader ) return Natural is
      -- Return the number of errors found by the parser.
   begin
      return E.Nbr_Of_Warnings;
   end Number_Of_Warnings;

end ESQL_Parser;


