-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/contrib/gsql/table_handler.adb,v $
--  Description     : Handle the tables of a project                         --
--  Author          : Michael Erdmann                                        --
--  Created         : 8.8.2001                                               --
--  Last Modified By: $Author: me $
--  Last Modified On: $Date: 2001/11/26 20:31:13 $
--  Status          : $State: Exp $
--
--  Copyright (C) 2000 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
-------------------------------------------------------------------------------
with Ada.Text_IO;              use Ada.Text_IO;
with Ada.Strings.Unbounded;    use Ada.Strings.Unbounded;

with Text_Block;               use Text_Block;
with Contents_Manager;         use Contents_Manager;
with Contents_Manager.GUI;     use Contents_Manager.GUI;
with Datasource_Handler;       use Datasource_Handler;
with Project_File;             use Project_File;
with Preferences;              use Preferences;

with Gui_Preferences;          use GUI_Preferences;
with GUI_Logging;              use GUI_Logging;
with GUI_Tools;                use GUI_Tools;
with GUI_Worksheet;            use GUI_Worksheet;
with GUI_Table;                use GUI_Table;

package body Table_Handler  is

   Version : constant String := "$Id: table_handler.adb,v 1.11 2001/11/26 20:31:13 me Exp $";
   -- ********************************************************************* --
   -- **                 L O C A L   D A T A                             ** --
   -- ********************************************************************* --
   Class_Name    : constant Unbounded_String := To_Unbounded_String(Table_Prefix);

   Table_Section : Element_ID := Null_Element_ID;

   -- ********************************************************************* --
   -- **               P U B L I C   P R O C E D U R E S                 ** --
   -- ********************************************************************* --

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize(
      This : in out Object ) is
      -- initialize the query (instance)
   begin
      Initialize( Contents_Handler.Object(This) );
      This.Id := GUI.Add_Element( Key(This), Table_Section, Self(This) );

      GUI_Worksheet.Initialize( This.Table_Editor );
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize(
      This : in out Object ) is
      Name : constant Unbounded_String := Key( This );
   begin
      if GUI_Worksheet.Is_Open( Name ) then
         GUI_Worksheet.Remove_Sheet( Name );
      end if;
      Project_File.Remove( Table_Prefix, Key(This) );
      GUI.Delete_Element( This.Id );
      Finalize( Contents_Handler.Object(This) );
   end Finalize;

   -----------------
   -- Instanciate --
   -----------------
   function Instanciate(
      Key    : in String ) return Contents_Handler.Handle is
      Result : Handle := new Object;
   begin
      return Contents_Handler.Handle(Result);
   end Instanciate;

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize is
      Item : Handle := new Object;
   begin
      Item.Constructor := Instanciate'Access;

      Table_Section := Add_Element( Title => "Tables",
                                    Content => Contents_Handler.Handle(Item) );

      Contents_Manager.Register( Table_Prefix, Instanciate'Access );
      Contents_Manager.Initialize( Table_Prefix, Table_Section);
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize is
   begin
      Contents_Manager.Finalize( Table_Prefix );
   end Finalize;

   -----------
   -- Value --
   -----------
   procedure Value(
      This : in Object;
      Text : out Text_Buffer;
      Last : out Positive ) is
   begin
      GUI_Table.Get_Result( This.Table_Editor, Text, Last );
   end Value;

   ----------
   -- Save --
   ----------
   procedure Save(
      Item    : in out Object;
      Value   : in Text_Buffer;
      Name    : in String := "" ) is
      -- store the data of the GUI in the project file.
      -- R.1  If the object is already in the project file, then simply
      --      update the data
      -- R.2  if the object is not in the project file, assume, that the
      --      element with the specified name has to be created newly.assume element
      --      to be inserted newly.
      Current : constant String := Key( Item );
      NewObj  : Handle;
   begin
      Item.Name := Value(1);

      if not Section_Exists( Table_Prefix, To_String( Item.Name) ) then        -- R.2
         NewObj := Handle(Contents_Manager.Create(
                      Table_Prefix,
                      To_String(Item.Name))
                   );
         Gui_Worksheet.Remove_Sheet( Item.Table_Editor );
      end if;
                                                                          -- R.1, R.2
      Project_File.Update( Table_Prefix, To_String(Item.Name), Value);
   end Save;

   ----------
   -- Load --
   ----------
   procedure Load(
      Item        : in out Object;
      Value       : out Text_Buffer;
      Last        : out Natural ) is
      -- R.0 - load the table from the project file.
      -- R.1 - If there is no such entry simply return the template
      --       text.
      Description : constant Text_Buffer := (
                  To_Unbounded_String( "Fill in your description" ),
                  To_Unbounded_String( "Author: " & Preference(Prf_Author))
                  );
   begin
      Project_File.Get(                                             -- R.0
         Table_Prefix,
         Key(Item),
         Value,
         Last
      );
   exception
      when Section_Not_Found =>                                     -- R.1
         Last := Value'First;
         Value(Last) := Item.Name;
         Last := Last + 1;
         Put_Text_Block( Value, Description, Last );
   end Load;

   --------------
   -- Selected --
   --------------
   procedure Selected(
      This         : in out Object ) is
   begin
      if This.Constructor /= null then
         GUI_Tools.Activate( Table_Prefix );
      else
         Open_Editor( This );
      end if;
   end Selected;

   -----------------
   -- Open_Editor --
   -----------------
   procedure Open_Editor(
      This : in out Object ) is
      -- load the referenced instance into the query editor
      Name        : Unbounded_String := Key( This );
   begin
      if not GUI_Worksheet.Is_Open( Name ) then
         GUI_Worksheet.Add_Sheet(
            This.Table_Editor,
            Self(This),
            Name );
      else
         GUI_Worksheet.Bring_In_Front( Name );
      end if;
   end Open_Editor;

   ------------------
   -- Close_Editor --
   ------------------
   procedure Close_Editor(
      This : in out Object ) is
      Name : Unbounded_String := Key( This );
   begin
      GUI_Worksheet.Remove_Sheet( Name );
   end Close_Editor;

   -------------
   -- Execute --
   -------------
   procedure Execute(
      This : in out Object ) is
      -- execute the current editor contents
      Name : constant String := To_String( Key( This ) );
      Text : Text_Buffer( 1..Max_Section_Length );
      Last : Positive := Text'First;
   begin
      if Datasource_Handler.Current_Database = Not_Attached then
         Error("No data base connected");
         return;
      end if;

      Log("Droping Table " & Name );
      Datasource_Handler.Execute( "DROP TABLE " & Name );
      Load( This, Text, Last );
      Log("Inserting Table " & Name );
      Put_Line(DML_Create_Table_Command( Text ) );

      Datasource_Handler.Execute( DML_Create_Table_Command( Text ) );

   end Execute;

end Table_Handler;
