/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/************************************
 * Loading and unloading of images. *
 ************************************/

#include "gliv.h"

#include <stdlib.h>             /* exit() */
#include <signal.h>             /* signal(), SIG_DFL, SIGSEGV */
#include <GL/gl.h>

extern rt_struct *rt;
extern options_struct *options;
extern gliv_image *current_image;
extern GtkWidget *gl_widget;

static gliv_image *previous_image = NULL;
static gliv_image *next_image = NULL;

static gchar *loading_filename = NULL;

/*** Image destroying. ***/

static void destroy_image(gliv_image * im)
{
    if (im != NULL) {
        glDeleteTextures(im->nb_tiles, im->tex_ids);
        g_free(im->tex_ids);
        glDeleteLists(im->list, im->nb_tiles);
        g_free(im->tiles);
        g_free(im->ident);
        g_free(im);
    }
}

/*** Misc. ***/

static void segv_handler(gint sig)
{
    const gchar *sig_str;

    sig_str = g_strsignal(sig);

    if (loading_filename == NULL)
        g_printerr(_("%s not while loading an image\n"), sig_str);
    else
        g_printerr(_("%s while loading %s\n"), sig_str, loading_filename);

#ifdef MAGICK_CONVERT
    image_magick_finish();
#endif

    signal(sig, SIG_DFL);
    raise(sig);
}

void fill_ident(gliv_image * im)
{
    gchar *alpha_str;

    g_free(im->ident);

    alpha_str = im->has_alpha ? _("alpha channel") : _("no alpha channel");

    im->ident = g_strdup_printf("%s (%u/%u) (%s)",
                                get_file_name_by_id(im->number),
                                im->number + 1, array_length(), alpha_str);
}

/*** Dithering ***/

static void copy_alpha(GdkPixbuf * dest, GdkPixbuf * src)
{
    guchar *dest_data, *src_data, *max_dest;

    src_data = gdk_pixbuf_get_pixels(src);
    dest_data = gdk_pixbuf_get_pixels(dest);
    max_dest = dest_data +
        gdk_pixbuf_get_height(dest) * gdk_pixbuf_get_rowstride(dest);

    while (dest_data < max_dest) {
        dest_data[3] = src_data[3];
        dest_data += 4;
        src_data += 4;
    }
}

static void dither(gliv_image * im)
{
    GdkPixmap *pmap;
    GdkPixbuf *new;
    GdkGC *gc;

    gdk_rgb_init();

    pmap = gdk_pixmap_new(gl_widget->window, im->width, im->height, -1);
    gc = gdk_gc_new(gl_widget->window);

    if (im->has_alpha == TRUE)
        gdk_draw_rgb_32_image(pmap, gc, 0, 0, im->width, im->height,
                              GDK_RGB_DITHER_MAX,
                              gdk_pixbuf_get_pixels(im->init->im),
                              gdk_pixbuf_get_rowstride(im->init->im));
    else {
        gdk_pixbuf_render_to_drawable(im->init->im, pmap, gc, 0, 0, 0, 0,
                                      im->width, im->height,
                                      GDK_RGB_DITHER_MAX, 0, 0);

        /* We need the alpha channel in order to copy it. */
        gdk_pixbuf_unref(im->init->im);
    }

    new = gdk_pixbuf_new(GDK_COLORSPACE_RGB, im->has_alpha, 8,
                         im->width, im->height);

    gdk_pixbuf_get_from_drawable(new, pmap, gdk_colormap_get_system(),
                                 0, 0, 0, 0, im->width, im->height);

    gdk_pixmap_unref(pmap);

    if (im->has_alpha == TRUE) {
        copy_alpha(new, im->init->im);
        gdk_pixbuf_unref(im->init->im);
    }

    im->init->im = new;
    gdk_gc_unref(gc);
}

/*** Image loading. ***/

/*
 * Blacken image pixels with alpha == 0.
 * These pixels are invisible but may alter the bilinear filtering.
 */
static void clean_alpha(gliv_image * im)
{
    guchar *pixel, *end;

    pixel = gdk_pixbuf_get_pixels(im->init->im);
    end = pixel + gdk_pixbuf_get_rowstride(im->init->im) * im->height;

    while (pixel != end) {
        if (pixel[3] == 0)
            *((guint32 *) pixel) = 0;

        pixel += 4;
    }
}

static void set_loading(gchar * filename)
{
    /* Used if there is a segfault. */
    loading_filename = filename;

    set_loading_entry(filename);
}

static gliv_image *load_file(gchar * filename)
{
    gliv_image *im;
    GdkPixbuf *loaded_image;

    set_loading(filename);

    loaded_image = file_to_pixbuf(filename);
    if (loaded_image == NULL) {
        /* Loading failed. */
        set_loading(NULL);
        return NULL;
    }

    im = g_new(gliv_image, 1);
    im->ident = NULL;
    im->init = g_new(loading_data, 1);
    im->init->im = loaded_image;

    im->width = gdk_pixbuf_get_width(loaded_image);
    im->height = gdk_pixbuf_get_height(loaded_image);
    im->has_alpha = gdk_pixbuf_get_has_alpha(loaded_image);

    if (im->has_alpha == TRUE)
        clean_alpha(im);

    if (options->dither == TRUE)
        dither(im);

    create_display_list(im);

    gdk_pixbuf_unref(im->init->im);
    g_free(im->init);

    /* If there is a segfault it won't be because of the loading. */
    set_loading(NULL);

    return im;
}

static gliv_image *load_in_direction(gint8 dir)
{
    gliv_image *im = NULL;
    guint id;

    if (current_image == NULL)
        /* First time. */
        id = 0;
    else
        id = current_image->number + dir;

    while (id < array_length()) {
        im = load_file(get_file_name_by_id(id));
        if (im == NULL)
            /* Delete bad images from the list. */
            file_array_remove(id);
        else
            break;
    }

    if (im != NULL)
        im->number = id;

    return im;
}

/*
 * After many images are opened we want to load
 * the next image but not the preloaded one.
 */
void open_next_image(void)
{
    if (loading_filename != NULL)
        return;

    destroy_image(next_image);
    destroy_image(previous_image);

    prioritize_textures(current_image, FALSE);
    previous_image = current_image;

    current_image = load_in_direction(1);
    render();

    while (gtk_events_pending() != 0)
        gtk_main_iteration_do(FALSE);

    next_image = load_in_direction(1);
}

static gboolean events_ok(void)
{
    GdkEvent *event;
    gboolean res;

    if (gtk_events_pending() == 0)
        return TRUE;

    event = gtk_get_current_event();

    res = (event == NULL) || (event->type == GDK_BUTTON_RELEASE);

    gdk_event_free(event);

    return res;
}

void load_direction(gint8 dir)
{
    gliv_image **backup;
    gliv_image **next;

    static gchar next_direction;

    /*
     * Before loading the next image in a prebuffer we let GTK process
     * remaining events and if there is a request for the next
     * image in the events pending then -> crash.
     * With this boolean we process loading request only if we know
     * all loadings are finished.
     */
    static gboolean load_finished = TRUE;

    if (loading_filename != NULL)
        return;

    if (load_finished == FALSE) {
        /*
         * We come from a gtk_main_iteration_do(FALSE) called here,
         * so we stop processing events and we load the next image.
         */
        next_direction = dir;
        return;
    }

    load_finished = FALSE;

    while (ABS(dir) == 1 && events_ok() == TRUE) {

        if ((dir == 1 && current_image->number == array_length() - 1) ||
            (dir == -1 && current_image->number == 0)) {
            /* Out of bounds. */
            load_finished = TRUE;
            return;
        }

        if (dir == -1) {
            backup = &next_image;
            next = &previous_image;
        } else {
            /* dir == 1 */
            backup = &previous_image;
            next = &next_image;
        }

        destroy_image(*backup);
        prioritize_textures(current_image, FALSE);
        *backup = current_image;

        if (*next == NULL)
            *next = load_in_direction(dir);

        current_image = *next;
        render();

        next_direction = 0;
        while (gtk_events_pending() != 0 && next_direction == 0)
            gtk_main_iteration_do(FALSE);

        *next = load_in_direction(dir);

        /* Usually next_direction == 0, and the while loop ends. */
        dir = next_direction;
    }

    load_finished = TRUE;
}

void load_first_image(void)
{
    signal(SIGSEGV, segv_handler);
    init_history();

    current_image = load_in_direction(1);
    if (current_image == NULL) {
        g_printerr(_("No images found.\n"));
        exit(1);
    }

    render();
}

void load_second_image(void)
{
    if (next_image == NULL)
        next_image = load_in_direction(1);
}

void open_file(gchar * filename, gboolean add_dir, gboolean shuffle_flag)
{
    gchar *dir;
    gliv_image *im;
    gchar *tmp;

    if (loading_filename != NULL)
        return;

    /* Try to load the requested image. */
    im = load_file(filename);
    if (im == NULL)
        return;

    if (add_dir == TRUE) {
        /* Add all files in the directory to the end of the array. */
        dir = g_path_get_dirname(filename);

        insert_after_current_position(&dir, 1, shuffle_flag);

        /* Put the requested image at current position in the file list. */
        place_at_position(filename, -1);
    } else {
        tmp = g_strdup(filename);
        insert_after_current_position(&tmp, 1, shuffle_flag);
    }

    im->number = current_image->number + 1;

    destroy_image(previous_image);
    destroy_image(next_image);

    prioritize_textures(current_image, FALSE);
    previous_image = current_image;
    current_image = im;

    render();
    refresh(REFRESH_NOW);

    next_image = load_in_direction(1);
}

static gliv_image *reload_image(gliv_image * im)
{
    guint number;

    if (im == NULL)
        return NULL;

    number = im->number;
    destroy_image(im);

    im = load_file(get_file_name_by_id(number));
    im->number = number;

    return im;
}

#define RELOAD(im) im = reload_image(im)

void reload_all_images(void)
{
    if (loading_filename != NULL)
        return;

    RELOAD(current_image);
    prioritize_textures(current_image, TRUE);
    rt->filtering_enabled = FALSE;

    refresh(REFRESH_NOW);

    RELOAD(previous_image);
    RELOAD(next_image);
}

static gboolean check_direction(gchar * filename, gint dir)
{
    gliv_image **prev, **next;
    guint number, max_number;

    if (dir == 1) {
        prev = &previous_image;
        next = &next_image;
        max_number = array_length() - 1;
    } else {
        /* dir == -1 */
        prev = &next_image;
        next = &previous_image;
        max_number = 0;
    }

    if (*next != NULL) {
        number = (*next)->number;

        /* Check if the requested image is just following the current one. */
        if (g_str_equal(filename, get_file_name_by_id(number)) == TRUE) {

            load_direction(dir);
            return TRUE;
        }

        if (number != max_number) {
            number += dir;

            /*
             * Check if the requested image is just before the previous one,
             * or just after the next one.
             */
            if (g_str_equal(filename, get_file_name_by_id(number)) == TRUE) {

                destroy_image(*prev);
                *prev = *next;
                *next = NULL;
                current_image = load_file(filename);
                current_image->number = number;
                render();
                return TRUE;
            }
        }
    }

    return FALSE;
}

void menu_load(gchar * filename)
{
    guint i, len;
    gliv_image *im;

    if (loading_filename != NULL ||
        g_str_equal(filename, get_file_name_by_id(current_image->number)) ||
        check_direction(filename, -1) || check_direction(filename, 1))
        return;

    len = array_length();
    for (i = 0; i < len; i++)
        if (g_str_equal(filename, get_file_name_by_id(i)) == TRUE)
            break;

    if (i == len)
        /* The id was not found, so the image is not in the list anymore. */
        return;

    im = load_file(filename);
    if (im == NULL)
        return;

    destroy_image(previous_image);
    previous_image = NULL;

    destroy_image(next_image);
    next_image = NULL;

    current_image = im;
    current_image->number = i;

    render();
}
