/* --< GKrellWeather 0.2.7 >--{ 20 Mar 2001 }--
 *
 * Author: Franky Lam (franky@frankylam.com)
 *         http://www.frankylam.com/
 *
 */
#include <gkrellm2/gkrellm.h>

#define DEFAULT_STATION_ID      "YSSY"

#define STYLE_NAME      "weather"

/*    net update status    */
static gboolean    net_update;

static gint style_id;

enum {  TEM_DISPLAY = 0,
        DEW_DISPLAY = 1,
        WIN_DISPLAY = 2,
        SKY_DISPLAY = 3,
        NAM_DISPLAY = 4,
        MAX_DISPLAY = 5,
};

static gint     panel_state = 0;
static gint     x_scroll = 0;
static gint     name_xoff = 0;
static gint     sky_cond_xoff = 0;

static GkrellmChart    *chart;
static GkrellmMonitor  *monitor;
static GkrellmPanel    *panel;
static GkrellmDecal    *decal_temperature,     *decal_unit1,
    *decal_humidity,        *decal_unit2,
    *decal_dew_point,       *decal_wind_chill,
    *decal_wind_direction,  *decal_wind_speed,
    *decal_pressure,
    *decal_name,            *decal_sky_cond;

static GtkWidget    *metric_option,
                    *km_per_hour_option,
                    *beaufort_option,
                    *meter_per_second_option,
                    *mmHg_option,
                    *kPa_option,
                    *hPa_option,
                    *station_option,
                    *url_option,
                    *update_interval_option,
                    *switch_interval_option;
static GtkTooltips  *weather_tips = 0;
static gchar        *weather_tips_text; 

typedef struct
{
    GkrellmTextstyle   ts;
    gint lbearing,
	rbearing,
	width,
	ascent,
	descent;
}
Extents;

static Extents  ext_temperature, ext_unit1,
                ext_humidity, ext_unit2,
                ext_dew_point, ext_wind_chill,
                ext_wind_direction, ext_wind_speed,
                ext_pressure,
                ext_name, ext_sky_cond;

typedef struct
{
    gchar   name[512];          /* station name                         */
    gchar   sky_cond[512];
    gdouble temperature[2];     /* 0 imperial 1 metric                  */
    gdouble humidity;
    gdouble pressure[4];        /* 0 imperial 1 mmHg 2 kPa 3 hPa        */
    gdouble dew_point[2];
    gdouble wind_chill[2];
    gdouble wind_direction;
    gdouble wind_speed[4];      /* 0 imperial 1 km/h 2 m/s 3 beaufort   */
    gint    wind_chill_avail;
    gint    sky_cond_avail;
}
Air;

typedef struct
{
    gint    chart_visible;
    gint    panel_visible;
    gint    metric;
    gint    update_interval;
    gint    switch_interval;
    gint    wind_chill;         /* for later release    */
    gint    windspeed_unit;
    gint    pressure_unit;
    gchar   station[512];
    gchar   command[512];       /* for later release    */
    gchar   filename[512];      /* for later release    */
}
Options;

static Air air;
static Options options;


static void
string_extents (gchar *string, Extents *ext)
{
    gdk_string_extents(ext->ts.font, string, &ext->lbearing, &ext->rbearing,
            &ext->width, &ext->ascent, &ext->descent);
}


/* reference from wmWeather.c */
static gboolean
read_air ()
{
    static gchar *c, line[512], str[1024];
    static gdouble sgn;
    static gint i, cursize, spd, scale[] =
    { 1, 3, 4, 10, 16, 21, 27, 33, 40, 47, 55, 63, 71 };
    FILE *fp;
    char *oldlocale;

    // decimal point stuff
    oldlocale = g_strdup(setlocale(LC_NUMERIC, NULL));
    setlocale(LC_NUMERIC, "C");
    if ((fp = fopen(options.filename, "r")) != NULL) {
        fgets(air.name, 512, fp);
        if (air.name[0] == '\0' || air.name[0] == '\n')
        {   /* GrabWeather could not read data from server */
            fclose(fp);
	    setlocale(LC_NUMERIC, oldlocale);
	    g_free(oldlocale);
            return FALSE;
        }
        for (c = air.name; *c && *c != '('; c++); /* extract name */
        *(c-1) = '\0';
        
        fgets(line, 512, fp);

        fgets(air.sky_cond, 512, fp);
        if (air.sky_cond[0] == '\n') {
            air.sky_cond_avail = 0;
            if (panel_state == SKY_DISPLAY) {
                panel_state = NAM_DISPLAY;
                gkrellm_make_decal_invisible(panel, decal_sky_cond);
                gkrellm_make_decal_visible(panel, decal_name);
            }
        }
        else {
            air.sky_cond_avail = 1;
        }
	if (air.sky_cond[strlen(air.sky_cond)-1] == '\n') 
	    air.sky_cond[strlen(air.sky_cond)-1] = '\0';
	

        fgets(line, 512, fp);

        fscanf(fp, "%lf", &air.temperature[0]);
        air.temperature[1] = (air.temperature[0]-32.0)*5.0/9.0 + 0.5;
        fscanf(fp, "%lf", &air.dew_point[0]);
        air.dew_point[1] = (air.dew_point[0]-32.0)*5.0/9.0 + 0.5;

        fscanf(fp, "%lf", &air.wind_chill[0]);
        air.wind_chill_avail = (air.wind_chill[0] < -900.0) ? 0 : 1;
        air.wind_chill[1] = (air.wind_chill[0]-32.0)*5.0/9.0 + 0.5;

        fscanf(fp, "%lf", &air.pressure[0]);
        air.pressure[0] += 0.005;
        air.pressure[1] = air.pressure[0] * 25.4;
        air.pressure[2] = air.pressure[0] * 3.38639;
        air.pressure[3] = air.pressure[0] * 33.8639;
        fscanf(fp, "%lf", &air.humidity);
        fscanf(fp, "%lf", &air.wind_direction);
        fscanf(fp, "%lf", &air.wind_speed[0]);
        air.wind_speed[1] = air.wind_speed[0] * 1.609;
        air.wind_speed[2] = air.wind_speed[0] * 0.4473;
        sgn = (air.wind_speed[0] < 0.0) ? -1.0 : 1.0;
        spd = (gint)sgn * (gint)air.wind_speed[0];
        for (i = 0; i < 13 && spd > scale[i]; i++);
        air.wind_speed[3] = sgn * (gdouble)i;
        fclose(fp);
    } else {
        air.temperature[0]  = -99.0;
        air.dew_point[0]    = -99.0;
        air.wind_chill[0]   = -99.0;
        air.humidity        = -99.0;
        air.pressure[0]     = -99.0;
        air.wind_direction  = -99.0;
        air.wind_speed[0]   = -99.0;
    }
    setlocale(LC_NUMERIC, oldlocale);
    g_free(oldlocale);

    name_xoff = -gdk_string_width(gkrellm_default_font(1), air.name);
    sky_cond_xoff = -gdk_string_width(gkrellm_default_font(1), air.sky_cond);

    // Change tooltips
    if (weather_tips != NULL)
    {
        sprintf(line, "%s/.wmWeatherReports/%s.TXT", getenv("HOME"),
                options.station);
        if ((fp = fopen(line, "r")) != NULL) {
            g_free(weather_tips_text);
            cursize = 0;
            strcpy(str, "");
            while (fgets(line, 512, fp)) {
                if (cursize + strlen(line) >= 1023) {
                    strncat(str, line, 1024 - cursize);
                    break;
                }
                else {
                    strcat(str, line);
                }
               cursize += strlen(line);
            }
            weather_tips_text = g_strdup(str);
            gtk_tooltips_set_tip(weather_tips, panel->drawing_area,
                                 weather_tips_text, NULL);
            fclose(fp);
        }
    }
    return TRUE;
}


static void
calc_xy (gint i)
{
    gint w;

    switch (i) {
    case NAM_DISPLAY:
        decal_name->x = 2;
        break;
    case SKY_DISPLAY:
        decal_sky_cond->x = 2;
        break;
    case WIN_DISPLAY:
        w = decal_wind_direction->w + decal_wind_speed->w;
        decal_wind_direction->x = (gkrellm_chart_width() - w + 1) / 2 + 1;
        decal_wind_speed->x = decal_wind_direction->x +
            decal_wind_direction->w + 1;
        break;
    case DEW_DISPLAY:
        w = decal_dew_point->w + decal_unit1->w + decal_pressure->w;
        decal_dew_point->x = (gkrellm_chart_width() - w + 1) / 2;
        decal_unit1->x = decal_dew_point->x + decal_dew_point->w - 1;
        decal_wind_chill->x = decal_dew_point->x;
        decal_pressure->x = decal_unit1->x + decal_unit1->w + 1;
        break;
    case TEM_DISPLAY:
    default:
        w = decal_temperature->w + decal_unit1->w + decal_humidity->w +
            decal_unit2->w;
        decal_temperature->x = -1;
        decal_unit1->x = decal_temperature->x + decal_temperature->w - 1;
        decal_unit2->x = gkrellm_chart_width() - decal_unit2->w - 1;
        decal_humidity->x = decal_unit2->x - decal_humidity->w;
        break;
    }
}


static void
draw_panel ()
{
    static gchar *compress_direction[] =
    {
        "N", "NNE", "NE", "ENE", "E", "ESE", "SE", "SSE",
        "S", "SSW", "SW", "WSW", "W", "WNW", "NW", "NNW"
    };
    static gchar l1[8], l2[8], *units;
    static gint v, w, wdx;

    if (options.panel_visible == FALSE)
        return;

    v++;  // no worries about overflow

    calc_xy(panel_state);
    switch (panel_state) {
        case NAM_DISPLAY:
            w = gkrellm_chart_width();
	    if (x_scroll > name_xoff)
		x_scroll -= 2;
	    else
		x_scroll = w;
            decal_name->x_off = x_scroll;
            gkrellm_draw_decal_text(panel, decal_name, air.name, w - x_scroll);
            break;
        case SKY_DISPLAY:
            if (air.sky_cond_avail) {
                w = gkrellm_chart_width();
		if (x_scroll > sky_cond_xoff)
		    x_scroll -= 2;
		else
		    x_scroll = w;
		decal_sky_cond->x_off = x_scroll;
                gkrellm_draw_decal_text(panel, decal_sky_cond, air.sky_cond,
                        w - x_scroll);
            }
            break;
        case WIN_DISPLAY:
            wdx = (int)((air.wind_direction + 360.0 + 11.25) / 22.5) % 16;
            gkrellm_draw_decal_text(panel, decal_wind_direction,
                    compress_direction[wdx], wdx);
            if (options.metric && options.windspeed_unit == 3)
            {
                v = (int)air.wind_speed[3];
                sprintf(l2, "F%d", v);
                v += 1000;   /* to distinguish redraws */
            }
            else
            {
                v = (int)air.wind_speed[
                         options.metric ? options.windspeed_unit : 0];
                sprintf(l2, "%d", v);
            }
            gkrellm_draw_decal_text(panel, decal_wind_speed, l2, v);
            break;
        case DEW_DISPLAY:
            v = (int)air.dew_point[options.metric];
            sprintf(l1, "%+4d", v);
            gkrellm_draw_decal_text(panel, decal_dew_point, l1, v);
            v = (int)air.pressure[options.metric ? options.pressure_unit : 0];
            sprintf(l2, "%4d", v);
            gkrellm_draw_decal_text(panel, decal_pressure, l2, v);
            units = options.metric ? "C" : "F";
            gkrellm_draw_decal_text(panel, decal_unit1, units, options.metric);
            break;
        case TEM_DISPLAY:
        default:
            v = (int)air.temperature[options.metric];
            sprintf(l1, "%+4d", v);
            gkrellm_draw_decal_text(panel, decal_temperature, l1, v);
            v = (int)air.humidity;
            sprintf(l2, "%3d", v);
            gkrellm_draw_decal_text(panel, decal_humidity, l2, v);
            units = options.metric ? "C" : "F";
            gkrellm_draw_decal_text(panel, decal_unit1, units, options.metric);
            gkrellm_draw_decal_text(panel, decal_unit2, "%", 0);
    }
}

static void
panel_switch (gint new_state)
{
    switch (panel_state) {
        case NAM_DISPLAY:
            gkrellm_make_decal_invisible(panel, decal_name);
            break;
        case SKY_DISPLAY:
            gkrellm_make_decal_invisible(panel, decal_sky_cond);
            break;
        case WIN_DISPLAY:
            gkrellm_make_decal_invisible(panel, decal_wind_direction);
            gkrellm_make_decal_invisible(panel, decal_wind_speed);
            break;
        case DEW_DISPLAY:
            gkrellm_make_decal_invisible(panel, decal_wind_chill);
            gkrellm_make_decal_invisible(panel, decal_dew_point);
            gkrellm_make_decal_invisible(panel, decal_pressure);
            gkrellm_make_decal_invisible(panel, decal_unit1);
            gkrellm_make_decal_invisible(panel, decal_unit2);
            break;
        case TEM_DISPLAY:
        default:
            gkrellm_make_decal_invisible(panel, decal_temperature);
            gkrellm_make_decal_invisible(panel, decal_unit1);
            gkrellm_make_decal_invisible(panel, decal_humidity);
            gkrellm_make_decal_invisible(panel, decal_unit2);
    }
    panel_state = new_state;
    draw_panel();
    switch (panel_state) {
        case NAM_DISPLAY:
            gkrellm_make_decal_visible(panel, decal_name);
            break;
        case SKY_DISPLAY:
            gkrellm_make_decal_visible(panel, decal_sky_cond);
            break;
        case WIN_DISPLAY:
            gkrellm_make_decal_visible(panel, decal_wind_direction);
            gkrellm_make_decal_visible(panel, decal_wind_speed);
            break;
        case DEW_DISPLAY:
            if (options.wind_chill && air.wind_chill_avail)
                gkrellm_make_decal_visible(panel, decal_wind_chill);
            else
                gkrellm_make_decal_visible(panel, decal_dew_point);
            gkrellm_make_decal_visible(panel, decal_unit1);
            gkrellm_make_decal_visible(panel, decal_pressure);
            break;
        case TEM_DISPLAY:
        default:
            gkrellm_make_decal_visible(panel, decal_temperature);
            gkrellm_make_decal_visible(panel, decal_unit1);
            gkrellm_make_decal_visible(panel, decal_humidity);
            gkrellm_make_decal_visible(panel, decal_unit2);
    }
}

static FILE *command_pipe;

static void
run_command ()
{
    if (command_pipe)
        return;
    command_pipe = popen(options.command, "r");
    if (command_pipe)
        fcntl(fileno(command_pipe), F_SETFL, O_NONBLOCK);
    net_update = TRUE;
}

static gboolean
command_done ()
{
    gchar buf[64];

    while (fread(buf, 1, sizeof(buf) - 1, command_pipe) > 0)
        ;
    if (feof(command_pipe))
    {
        pclose(command_pipe);
        command_pipe = NULL;
        return TRUE;
    }
    return FALSE;
}

static void
update_air ()
{
    static gint switch_timer = 0, minute_timer = 0;
    
    if (command_pipe)
    {
        net_update =  command_done() && read_air();
    }
    if(!net_update && GK.timer_ticks % 600 == 0)
    {
        run_command();
    }
    if (GK.second_tick && options.switch_interval > 0 &&
            switch_timer++ >= options.switch_interval)
    {
        switch_timer = 0;
        if (panel_state == WIN_DISPLAY && !air.sky_cond_avail)
             panel_switch (NAM_DISPLAY);
        else panel_switch ((panel_state+1) % MAX_DISPLAY);
    }
    if (GK.minute_tick && ++minute_timer >= options.update_interval)
    {
        minute_timer = 0;
        run_command();
    }
    draw_panel();
    gkrellm_draw_panel_layers(panel);
}


static gint
expose_event (GtkWidget *widget, GdkEventExpose *ev)
{   
    if (widget == panel->drawing_area)
    {
        gdk_draw_pixmap(widget->window,
            widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
            panel->pixmap, ev->area.x, ev->area.y, ev->area.x, ev->area.y,
            ev->area.width, ev->area.height);
    }
    else if (widget == chart->drawing_area)
    {
        gdk_draw_pixmap(widget->window,
            widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
            chart->pixmap,
            ev->area.x, ev->area.y, ev->area.x, ev->area.y,
        ev->area.width, ev->area.height);
    }
    return FALSE;
}

static gint
panel_press (GtkWidget *widget, GdkEventButton *ev)
{
    if (ev->button == 2 || ev->button == 3)
    {
        if (panel_state == WIN_DISPLAY && !air.sky_cond_avail)
             panel_switch (NAM_DISPLAY);
        else panel_switch ((panel_state+1) % MAX_DISPLAY);
    }
    return TRUE;
}


static void
create_air (GtkWidget *vbox, gint first_create)
{
    GkrellmStyle *style;
    gint i;

    if (first_create)
    {
        chart = gkrellm_chart_new0();
        panel = gkrellm_panel_new0();
    }
    else
    {
        gkrellm_destroy_krell_list(panel);
        gkrellm_destroy_decal_list(panel);
    }

    style = gkrellm_meter_style(style_id);

    panel->textstyle = gkrellm_meter_alt_textstyle(style_id);

    ext_temperature.ts = *gkrellm_meter_alt_textstyle(style_id);
    string_extents("+888", &ext_temperature);
    ext_unit1.ts = *gkrellm_meter_textstyle(style_id);
    string_extents("C", &ext_unit1);
    ext_humidity.ts = *gkrellm_meter_alt_textstyle(style_id);
    string_extents("88", &ext_humidity);
    ext_unit2.ts = *gkrellm_meter_textstyle(style_id);
    string_extents("%", &ext_unit2);
    ext_dew_point.ts = *gkrellm_meter_alt_textstyle(style_id);
    string_extents("+888", &ext_dew_point);
    ext_wind_chill.ts = *gkrellm_meter_alt_textstyle(style_id);
    string_extents("+888", &ext_wind_chill);
    ext_pressure.ts = *gkrellm_meter_alt_textstyle(style_id);
    string_extents("8888", &ext_pressure);
    ext_wind_direction.ts = *gkrellm_meter_textstyle(style_id);
    string_extents("WWW", &ext_wind_direction);
    ext_wind_speed.ts = *gkrellm_meter_alt_textstyle(style_id);
    string_extents("888", &ext_wind_speed);
    ext_name.ts = *gkrellm_meter_alt_textstyle(style_id);
    string_extents("Ay", &ext_name);
    ext_sky_cond.ts = *gkrellm_meter_alt_textstyle(style_id);
    string_extents("Ay", &ext_sky_cond);

    decal_temperature = gkrellm_create_decal_text(panel, "+888",
            &ext_temperature.ts, style, 0, 3, ext_temperature.width + 2);
    decal_unit1 = gkrellm_create_decal_text(panel, "C", &ext_unit1.ts,
            style, 0, 3, ext_unit1.width + 2);
    decal_humidity = gkrellm_create_decal_text(panel, "88", &ext_humidity.ts,
            style, 0, 3, ext_humidity.width + 4);
    decal_unit2 = gkrellm_create_decal_text(panel, "%", &ext_unit2.ts,
            style, 0, 3, ext_unit2.width + 2);
    decal_dew_point = gkrellm_create_decal_text(panel, "+888",
            &ext_dew_point.ts, style, 0, 3, ext_dew_point.width + 2);
    decal_wind_chill = gkrellm_create_decal_text(panel, "+888",
            &ext_wind_chill.ts, style, 0, 3, ext_wind_chill.width + 2);
    decal_pressure = gkrellm_create_decal_text(panel, "8888", &ext_pressure.ts,
            style, 0, 3, ext_pressure.width + 2);
    decal_wind_direction = gkrellm_create_decal_text(panel, "WWW",
            &ext_wind_direction.ts, style, 0, 3, ext_wind_direction.width + 2);
    decal_wind_speed = gkrellm_create_decal_text(panel, "888",
            &ext_wind_speed.ts, style, 0, 3, ext_wind_speed.width + 2);
    decal_name = gkrellm_create_decal_text(panel, "Ay", &ext_name.ts,
            style, 0, 3, gkrellm_chart_width() - 4);
    decal_sky_cond = gkrellm_create_decal_text(panel, "Ay", &ext_sky_cond.ts,
            style, 0, 3, gkrellm_chart_width() - 4);

    if (decal_unit1->h < decal_temperature->h) {
        decal_unit1->y = decal_temperature->y + decal_temperature->h -
            decal_unit1->h;
        if (decal_pressure->h < decal_temperature->h)
            decal_pressure->y = decal_temperature->y + decal_temperature->h -
                decal_pressure->h;
        if (decal_name->h < decal_temperature->h) {
            decal_name->y = decal_temperature->y + decal_temperature->h -
                decal_name->h;
            decal_sky_cond->y = decal_name->y;
        }
    }
    else if (decal_unit1->h > decal_temperature->h) {
        decal_temperature->y = decal_unit1->y + decal_unit1->h -
            decal_temperature->h;
        if (decal_pressure->h < decal_unit1->h)
            decal_pressure->y = decal_unit1->y + decal_unit1->h -
                decal_pressure->h;
        if (decal_name->h < decal_unit1->h) {
            decal_name->y = decal_unit1->y + decal_unit1->h - decal_name->h;
            decal_sky_cond->y = decal_name->y;
        }
    }
    decal_unit2->y = decal_unit1->y;
    decal_dew_point->y = decal_wind_chill->y = decal_temperature->y;
    decal_humidity->y = decal_wind_speed->y = decal_temperature->y;
    decal_wind_direction->y = decal_wind_speed->y + decal_wind_speed->h -
        decal_wind_direction->h;
    gkrellm_panel_configure(panel, NULL, gkrellm_meter_style(style_id));
    gkrellm_panel_create(vbox, monitor, panel);
//    chart->h = 36;
    /* gkrellm_create_chart(vbox, chart, DEFAULT_STYLE); */
    /* gkrellm_monitor_height_adjust(chart->h + panel->h); */
//    gkrellm_monitor_height_adjust(panel->h);

    if (weather_tips == NULL)
    {
        weather_tips = gtk_tooltips_new();
        weather_tips_text = g_strdup("GKrellWeather");
        gtk_tooltips_set_tip(weather_tips, panel->drawing_area,
                             weather_tips_text, NULL);
        gtk_tooltips_set_delay(weather_tips, 1000);
    }

    x_scroll = 0;
    for (i = MAX_DISPLAY - 1; i >= 0; i--)
    {
        panel_state = i; draw_panel();
    }

    gkrellm_make_decal_invisible(panel, decal_sky_cond);
    gkrellm_make_decal_invisible(panel, decal_name);
    gkrellm_make_decal_invisible(panel, decal_dew_point);
    gkrellm_make_decal_invisible(panel, decal_wind_chill);
    gkrellm_make_decal_invisible(panel, decal_pressure);
    gkrellm_make_decal_invisible(panel, decal_wind_direction);
    gkrellm_make_decal_invisible(panel, decal_wind_speed);
    gkrellm_make_decal_visible(panel, decal_temperature);
    gkrellm_make_decal_visible(panel, decal_unit1);
    gkrellm_make_decal_visible(panel, decal_humidity);
    gkrellm_make_decal_visible(panel, decal_unit2);
    if (first_create)
    {
        /*
        gtk_signal_connect(GTK_OBJECT(chart->drawing_area),
            "expose_event", (GtkSignalFunc) expose_event, NULL);
        */
        gtk_signal_connect(GTK_OBJECT(panel->drawing_area),
            "expose_event", (GtkSignalFunc) expose_event, NULL);
        gtk_signal_connect(GTK_OBJECT(panel->drawing_area),
            "button_press_event", (GtkSignalFunc) panel_press, NULL);

    }
}

#define PLUGIN_CONFIG_KEYWORD    "gkrellweather"

static void
save_air_config (FILE *f)
{
    fprintf(f, "%s chart_visible %d\n", PLUGIN_CONFIG_KEYWORD,
            options.chart_visible);
    fprintf(f, "%s panel_visible %d\n", PLUGIN_CONFIG_KEYWORD,
            options.panel_visible);
    fprintf(f, "%s metric %d\n", PLUGIN_CONFIG_KEYWORD, options.metric);
    fprintf(f, "%s update_interval %d\n", PLUGIN_CONFIG_KEYWORD,
            options.update_interval);
    fprintf(f, "%s switch_interval %d\n", PLUGIN_CONFIG_KEYWORD,
            options.switch_interval);
    fprintf(f, "%s wind_chill %d\n", PLUGIN_CONFIG_KEYWORD,
            options.wind_chill);
    fprintf(f, "%s windspeed_unit %d\n", PLUGIN_CONFIG_KEYWORD,
            options.windspeed_unit);
    fprintf(f, "%s pressure_unit %d\n", PLUGIN_CONFIG_KEYWORD,
            options.pressure_unit);
    fprintf(f, "%s station %s\n", PLUGIN_CONFIG_KEYWORD, options.station);
    fprintf(f, "%s command %s\n", PLUGIN_CONFIG_KEYWORD, options.command);
    fprintf(f, "%s filename %s\n", PLUGIN_CONFIG_KEYWORD, options.filename);
}


static void
load_air_config (gchar *arg)
{
    gchar config[64], item[256];
    gint n;

    n = sscanf(arg, "%s %[^\n]", config, item);
    if (n == 2)
    {
        if (strcmp(config, "chart_visible") == 0)
            sscanf(item, "%d\n", &(options.chart_visible));
        if (strcmp(config, "panel_visible") == 0)
            sscanf(item, "%d\n", &(options.panel_visible));
        if (strcmp(config, "metric") == 0)
            sscanf(item, "%d\n", &(options.metric));
        if (strcmp(config, "update_interval") == 0)
            sscanf(item, "%d\n", &(options.update_interval));
        if (strcmp(config, "switch_interval") == 0)
            sscanf(item, "%d\n", &(options.switch_interval));
        if (strcmp(config, "wind_chill") == 0)
            sscanf(item, "%d\n", &(options.wind_chill));
        if (strcmp(config, "windspeed_unit") == 0)
            sscanf(item, "%d\n", &(options.windspeed_unit));
        if (strcmp(config, "pressure_unit") == 0)
            sscanf(item, "%d\n", &(options.pressure_unit));
        if (strcmp(config, "station") == 0)
            sscanf(item, "%s\n", options.station);
        if (strcmp(config, "command") == 0)
            strcpy(options.command, item);
        if (strcmp(config, "filename") == 0)
            sscanf(item, "%s\n", options.filename);
    }
}


static void
apply_air_config (void)
{
    gint old_metric = options.metric;
    gchar *c;

    options.metric = GTK_TOGGLE_BUTTON(metric_option)->active;
    if (old_metric != options.metric)
    {
        draw_panel();
        gkrellm_draw_panel_layers(panel);
    }
    c = g_strdup(gtk_entry_get_text(GTK_ENTRY(station_option)));
    if (strcmp(options.station, c)) {
        options.station[0] = c[0];
        options.station[1] = c[1];
        options.station[2] = c[2];
        options.station[3] = c[3];
        sprintf(options.command, PREFIX "/bin/GrabWeather %s", options.station);
        sprintf(options.filename, "%s/.wmWeatherReports/%s.dat",
                getenv("HOME"), options.station);
	net_update = FALSE;
	run_command();
    }
    g_free(c);
    options.update_interval = gtk_spin_button_get_value_as_int(
            GTK_SPIN_BUTTON(update_interval_option));
    options.switch_interval = gtk_spin_button_get_value_as_int(
            GTK_SPIN_BUTTON(switch_interval_option));
}

static void windspeed_unit_set(GtkWidget *w, gpointer data)
{
    options.windspeed_unit = GPOINTER_TO_INT(data);
    draw_panel();
    gkrellm_draw_panel_layers(panel);
}


static void pressure_unit_set(GtkWidget *w, gpointer data)
{
    options.pressure_unit = GPOINTER_TO_INT(data);
    draw_panel();
    gkrellm_draw_panel_layers(panel);
}

static void
create_air_tab (GtkWidget *tab)
{
    GtkWidget *laptop, *frame, *ybox, *hbox, *zbox, *vbox,
              *label, *text, *info_window, *about_label;
    GtkAdjustment *switch_adjust, *update_adjust;
    GSList  *wind_speed_group = NULL,
            *pressure_group = NULL;
    gchar *about_text = NULL;
    gint i;
    static gchar *help_text[] =
    {
        N_("GKrellWeather shows the current weather under the clock.\n" \
        "\n" \
        "Right click the panel to toggle between different display:\n" \
        "\n" \
        " - Temperature, Relative Humidity\n" \
        " - DewPoint, Pressure (altimeter)\n" \
        " - Wind Direction, Wind Speed\n" \
        " - Sky Condition (if available)\n" \
        " - Station name\n" \
        "\n" \
        "Options:\n" \
        " switch interval - number of seconds (0 = never)" \
        " between switching display\n")
    };
    static gchar url[] = "http://www.nws.noaa.gov/tg/siteloc.shtml";
    laptop = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(laptop), GTK_POS_TOP);
    gtk_box_pack_start(GTK_BOX(tab), laptop, TRUE, TRUE, 0);

    /* options */
    frame = gtk_frame_new(NULL);
    gtk_container_border_width(GTK_CONTAINER(frame), 3);

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_border_width(GTK_CONTAINER(vbox), 3);

    hbox = gtk_hbox_new(FALSE, 0);
    label = gtk_label_new(_("4 letter Station ID:"));
    station_option = gtk_entry_new_with_max_length(8);
    gtk_entry_set_text(GTK_ENTRY(station_option), options.station);
    gtk_entry_set_editable(GTK_ENTRY(station_option), TRUE);
    gtk_container_add(GTK_CONTAINER(hbox), label);
    gtk_container_add(GTK_CONTAINER(hbox), station_option);
    gtk_container_add(GTK_CONTAINER(vbox), hbox);

    hbox = gtk_hbox_new(FALSE, 0);
    label = gtk_label_new(_("Get your station ID at:"));
    url_option = gtk_entry_new_with_max_length(64);
    gtk_entry_set_text(GTK_ENTRY(url_option), url);
    gtk_entry_set_editable(GTK_ENTRY(url_option), FALSE);
    gtk_container_add(GTK_CONTAINER(hbox), label);
    gtk_container_add(GTK_CONTAINER(hbox), url_option);
    gtk_container_add(GTK_CONTAINER(vbox), hbox);

    hbox = gtk_hbox_new(FALSE, 0);
    /* check boxes */
    zbox = gtk_vbox_new(FALSE, 0);
    metric_option = gtk_check_button_new_with_label(_("Display in metric unit"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(metric_option),
            options.metric);
    gtk_container_add(GTK_CONTAINER(zbox), metric_option);

    ybox = gtk_hbox_new(FALSE, 0);
    switch_adjust = (GtkAdjustment *) gtk_adjustment_new((gfloat)
            options.switch_interval, 0.0, 100.0, 1.0, 5.0, 0.0);
    switch_interval_option = gtk_spin_button_new(switch_adjust, 1.0, 1);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(switch_interval_option),
            (guint) 0);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(switch_interval_option),
            options.switch_interval);
    gtk_box_pack_start(GTK_BOX(ybox), switch_interval_option, FALSE, FALSE, 0);
    label = gtk_label_new(_("switch interval (secs)"));
    gtk_box_pack_start(GTK_BOX(ybox), label, FALSE, FALSE, 0);
    gtk_container_add(GTK_CONTAINER(zbox), ybox);

    ybox = gtk_hbox_new(FALSE, 0);
    update_adjust = (GtkAdjustment *) gtk_adjustment_new((gfloat)
            options.update_interval, 0.0, 100.0, 1.0, 5.0, 0.0);
    update_interval_option = gtk_spin_button_new(update_adjust, 1.0, 1);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(update_interval_option),
            (guint) 0);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(update_interval_option),
            options.update_interval);
    gtk_box_pack_start(GTK_BOX(ybox), update_interval_option, FALSE, FALSE, 0);
    label = gtk_label_new(_("update interval (mins)"));
    gtk_box_pack_start(GTK_BOX(ybox), label, FALSE, FALSE, 0);
    gtk_container_add(GTK_CONTAINER(zbox), ybox);

    gtk_container_add(GTK_CONTAINER(hbox), zbox);

    /* windspeed radio buttons */
    zbox = gtk_vbox_new(FALSE, 0);
    label = gtk_label_new(_("metric windspeed unit:"));
    gtk_container_add(GTK_CONTAINER(zbox), label);

    km_per_hour_option = gtk_radio_button_new_with_label(NULL, _("km/h"));
    wind_speed_group = gtk_radio_button_group(
            GTK_RADIO_BUTTON(km_per_hour_option));
    gtk_container_add(GTK_CONTAINER(zbox), km_per_hour_option);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(km_per_hour_option),
        options.windspeed_unit == 1);
    gtk_signal_connect(GTK_OBJECT(km_per_hour_option), "pressed",
        (GtkSignalFunc) windspeed_unit_set, GINT_TO_POINTER(1));

    meter_per_second_option = gtk_radio_button_new_with_label(wind_speed_group,
            _("m/s"));
    wind_speed_group = gtk_radio_button_group(
            GTK_RADIO_BUTTON(meter_per_second_option));
    gtk_container_add(GTK_CONTAINER(zbox), meter_per_second_option);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(meter_per_second_option),
        options.windspeed_unit == 2);
    gtk_signal_connect(GTK_OBJECT(meter_per_second_option), "pressed",
        (GtkSignalFunc) windspeed_unit_set, GINT_TO_POINTER(2));

    beaufort_option = gtk_radio_button_new_with_label(wind_speed_group,
            _("beaufort"));
    gtk_container_add(GTK_CONTAINER(zbox), beaufort_option);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(beaufort_option),
        options.windspeed_unit == 3);
    gtk_signal_connect(GTK_OBJECT(beaufort_option), "pressed",
        (GtkSignalFunc) windspeed_unit_set, GINT_TO_POINTER(3));
    gtk_container_add(GTK_CONTAINER(hbox), zbox);

    /* pressure radio buttons */
    zbox = gtk_vbox_new(FALSE, 0);
    label = gtk_label_new(_("metric pressure unit:"));
    gtk_container_add(GTK_CONTAINER(zbox), label);
    mmHg_option = gtk_radio_button_new_with_label(NULL, _("mmHg"));
    pressure_group = gtk_radio_button_group(GTK_RADIO_BUTTON(mmHg_option));
    gtk_container_add(GTK_CONTAINER(zbox), mmHg_option);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mmHg_option),
        options.pressure_unit == 1);
    gtk_signal_connect(GTK_OBJECT(mmHg_option), "pressed",
        (GtkSignalFunc) pressure_unit_set, GINT_TO_POINTER(1));

    kPa_option = gtk_radio_button_new_with_label(pressure_group, _("kPa"));
    pressure_group = gtk_radio_button_group(GTK_RADIO_BUTTON(kPa_option));
    gtk_container_add(GTK_CONTAINER(zbox), kPa_option);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(kPa_option),
        options.pressure_unit == 2);
    gtk_signal_connect(GTK_OBJECT(kPa_option), "pressed",
        (GtkSignalFunc) pressure_unit_set, GINT_TO_POINTER(2));

    hPa_option = gtk_radio_button_new_with_label(pressure_group, _("hPa"));
    gtk_container_add(GTK_CONTAINER(zbox), hPa_option);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(hPa_option),
        options.pressure_unit == 3);
    gtk_signal_connect(GTK_OBJECT(hPa_option), "pressed",
        (GtkSignalFunc) pressure_unit_set, GINT_TO_POINTER(3));
    gtk_container_add(GTK_CONTAINER(hbox), zbox);

    gtk_container_add(GTK_CONTAINER(vbox), hbox);

    label = gtk_label_new(_("Options"));
    gtk_container_add(GTK_CONTAINER(frame), vbox);
    gtk_notebook_append_page(GTK_NOTEBOOK(laptop), frame, label);

    /* help */
    frame = gtk_frame_new(NULL);
    gtk_container_border_width(GTK_CONTAINER(frame), 3);
    info_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(info_window),
    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(frame), info_window);

    text = gtk_text_view_new();

    for (i=0; i < sizeof(help_text)/sizeof(gchar *); ++i)
      gkrellm_gtk_text_view_append(text, _(help_text[i]));
    gtk_container_add(GTK_CONTAINER(info_window), text);

    label = gtk_label_new(_("Help"));
    gtk_notebook_append_page(GTK_NOTEBOOK(laptop), frame, label);

    /* about */
    about_text = g_strdup_printf(
        _("GKrellWeather %s\n" \
        "GKrellM weather Plugin\n" \
        "\n" \
        "Copyright (C) 2001 Franky Lam\n" \
        "franky@frankylam.com\n" \
        "http://www.frankylam.com/\n" \
        "\n" \
        "Released under the GNU Public License\n") \
        , VERSION
    );
    about_label = gtk_label_new(about_text);
    g_free(about_text);
    label = gtk_label_new(_("About"));
    gtk_notebook_append_page(GTK_NOTEBOOK(laptop), about_label, label);
}


static GkrellmMonitor air_mon =
{
    "GKrellWeather",    /* Name, for config tab.    */
    0,                  /* Id,  0 if a plugin       */
    create_air,         /* The create function      */
    update_air,         /* The update function      */
    create_air_tab,     /* The config tab create function   */
    apply_air_config,   /* Apply the config function        */

    save_air_config,        /* Save user config         */
    load_air_config,        /* Load user config         */
    PLUGIN_CONFIG_KEYWORD,  /* config keyword           */

    NULL,                   /* Undefined 2    */
    NULL,                   /* Undefined 1    */
    NULL,                   /* Undefined 0    */

    MON_CPU,                /* Insert plugin before this monitor.  Choose   */
                            /*   MON_CLOCK, MON_CPU, MON_PROC, MON_DISK,    */
                            /*   MON_INET, MON_NET, MON_FS, MON_MAIL,       */
                            /*   MON_APM, or MON_UPTIME                     */

    NULL,                   /* Handle if a plugin, filled in by GKrellM     */
    NULL                    /* path if a plugin, filled in by GKrellM       */
};


static void
read_default(void)
{
    panel_state = 0;
    options.chart_visible = FALSE;
    options.panel_visible = TRUE;
    options.metric = 1;
    options.update_interval = 15;
    options.switch_interval = 0;
    options.wind_chill = 0;
    options.windspeed_unit = 1;
    options.pressure_unit = 1;
    strcpy(options.station, DEFAULT_STATION_ID);
    sprintf(options.command, "/usr/share/gkrellm/GrabWeather %s", options.station);
    sprintf(options.filename, "%s/.wmWeatherReports/%s.dat",
            getenv("HOME"), options.station);
}


GkrellmMonitor *
gkrellm_init_plugin(void)
{
    #ifdef ENABLE_NLS
        bind_textdomain_codeset(PACKAGE, "UTF-8");
    #endif /* ENABLE_NLS */
    style_id = gkrellm_add_meter_style(&air_mon, STYLE_NAME);

    read_default();

    net_update = FALSE;

    monitor = &air_mon;
    return &air_mon;
}
