/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <time.h>
#include <string.h>

#include "gimageview.h"
#include "dnd.h"
#include "fileutil.h"
#include "gimv_image.h"
#include "thumbnail.h"
#include "thumbnail_view.h"
#include "thumbview_detail.h"
#include "thumbnail_window.h"


typedef struct ThumbViewData_Tag
{
   GtkWidget   *clist;
   gint         page_pos_x[3];
   gint         page_pos_y[3];
   gint         dest_mode;
} ThumbViewData;


/* for converting thumbnail data to string */
typedef gchar *(*ColDataStr) (Thumbnail *thumb);

typedef struct _ListColumn {
   gchar        *title;  /* column title */
   gint          width;  /* Column width. Currentry, this value not used */
   gboolean      free;   /* Free after setting text data to columns or not */
   ColDataStr    func;   /* Pointer to function convert data to string */
} ListColumn;


/* callback functions */
static void cb_clist_button_press (GtkWidget      *widget,
				   GdkEventButton *event,
				   gpointer        data);
static void cb_click_column       (GtkWidget      *widget,
				   gint            col,
				   ThumbView      *tv);

/* for set clist text data */
static gchar *column_data_filename (Thumbnail *thumb);
static gchar *column_data_size     (Thumbnail *thumb);
static gchar *column_data_type     (Thumbnail *thumb);
static gchar *column_data_time     (Thumbnail *thumb);
static gchar *column_data_uid      (Thumbnail *thumb);
static gchar *column_data_gid      (Thumbnail *thumb);
static gchar *column_data_mode     (Thumbnail *thumb);
static gchar *column_data_cache    (Thumbnail *thumb);

/* other private function */
static ThumbViewData *detailview_new (ThumbView *tv);
static void detailview_set_text_data (ThumbView *tv,
				      gint       dest_mode,
				      GList     *start);
static void detailview_set_pixmaps   (ThumbView *tv,
				      gint       dest_mode);
static void store_page_pos           (ThumbView *tv);
static gint idle_reset_page_pos      (gpointer data);


static ListColumn list_columns [] =
{
   {NULL,                0,   FALSE, NULL},
   {N_("Name"),          100, TRUE,  column_data_filename},
   {N_("Size (byte)"),   0,   TRUE,  column_data_size},
   {N_("Type"),          50,  TRUE,  column_data_type},
   {N_("Cache type"),    100, FALSE, column_data_cache},
   {N_("Date Modified"), 150, TRUE,  column_data_time},
   {N_("User"),          40,  TRUE,  column_data_uid},
   {N_("Group"),         40,  TRUE,  column_data_gid},
   {N_("Mode"),          100, TRUE,  column_data_mode},
};


/******************************************************************************
 *
 *   Callback functions.
 *
 ******************************************************************************/
static void
cb_clist_button_press (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
   ThumbView *tv = data;
   gint row;
   Thumbnail *thumb;

   g_return_if_fail (tv);

   gtk_clist_get_selection_info (GTK_CLIST (widget),
				 event->x, event->y, &row, NULL);
   thumb = gtk_clist_get_row_data (GTK_CLIST (widget), row);
   if (!thumb) return;

   thumbview_thumb_button_press_cb (widget, event, thumb);
}


static void
cb_clist_button_release (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
   ThumbView *tv = data;
   gint row;
   Thumbnail *thumb;

   g_return_if_fail (tv);

   gtk_clist_get_selection_info (GTK_CLIST (widget),
				 event->x, event->y, &row, NULL);
   thumb = gtk_clist_get_row_data (GTK_CLIST (widget), row);
   if (!thumb) return;

   thumbview_thumb_button_release_cb (widget, event, thumb);
}


static void
cb_clist_motion_notify (GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
   ThumbView *tv = data;
   gint row;
   Thumbnail *thumb;

   g_return_if_fail (tv);

   gtk_clist_get_selection_info (GTK_CLIST (widget),
				 event->x, event->y, &row, NULL);
   thumb = gtk_clist_get_row_data (GTK_CLIST (widget), row);
   if (!thumb) return;

   thumbview_motion_notify_cb (widget, event, thumb);
}


static void
cb_click_column (GtkWidget *widget, gint col, ThumbView *tv)
{
   ThumbWindow *tw;
   GtkWidget *sort_item;

   if (!tv || tv->progress) return;

   tw = tv->thumb_window;
   if (!tw) return;

   switch (col) {
   case 1:
      sort_item = tw->menuitem.sort_name;
      break;
   case 2:
      sort_item = tw->menuitem.sort_size;
      break;
   case 3:
      sort_item = tw->menuitem.sort_type;
      break;
   case 5:
      sort_item = tw->menuitem.sort_time;
      break;
   default:
      return;
   }

   if (!sort_item) return;

   if (GTK_CHECK_MENU_ITEM(sort_item)->active) {
      gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(tw->menuitem.sort_rev),
				      !GTK_CHECK_MENU_ITEM(tw->menuitem.sort_rev)->active);
   } else {
      gtk_check_menu_item_set_active
	 (GTK_CHECK_MENU_ITEM(sort_item), TRUE);
   }
}


static void
cb_select_row (GtkCList *clist, gint row, gint column,
	       GdkEventButton *event, gpointer user_data)
{
   ThumbView *tv = user_data;
   Thumbnail *thumb;
   GList *node;

   g_return_if_fail (tv);
   if (!tv->thumblist) return;

   node = g_list_nth (tv->thumblist, row);
   thumb = node->data;

   if (thumb) {
      thumb->selected = TRUE;
   }
}


static void
cb_unselect_row (GtkCList *clist, gint row, gint column,
		 GdkEventButton *event, gpointer user_data)
{
   ThumbView *tv = user_data;
   Thumbnail *thumb;
   GList *node;

   g_return_if_fail (tv);
   if (!tv->thumblist) return;

   node = g_list_nth (tv->thumblist, row);
   thumb = node->data;

   if (thumb) {
      thumb->selected = FALSE;
   }
}



/******************************************************************************
 *
 *   private functions.
 *
 ******************************************************************************/
static ThumbViewData *
detailview_new (ThumbView *tv)
{
   ThumbViewData *tv_data;
   gint i, num;

   g_return_val_if_fail (tv, NULL);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, DETAIL_VIEW_LABEL);
   if (!tv_data) {
      tv_data = g_new0 (ThumbViewData, 1);
      tv_data->clist = NULL;
      num = sizeof (tv_data->page_pos_x) / sizeof (tv_data->page_pos_x[0]);
      for (i = 0; i < num; i++) {
         tv_data->page_pos_x[i] = 0;
         tv_data->page_pos_y[i] = 0;
      }
      g_hash_table_insert (tv->disp_mode_data, DETAIL_VIEW_LABEL, tv_data);
   }

   return tv_data;
}


static gchar *
column_data_filename (Thumbnail *thumb)
{
   ThumbView *tv;

   if (!thumb) return NULL;
   tv = thumb->thumb_view;

   if (tv->mode == THUMB_MODE_DIR)
      return g_strdup (g_basename (thumb->info->filename));
   else
      return g_strdup (thumb->info->filename);
}


static gchar *
column_data_size (Thumbnail *thumb)
{
   if (!thumb) return NULL;

   return fileutil_size2str (thumb->info->st.st_size, FALSE);
}


static gchar *
column_data_type (Thumbnail *thumb)
{
   if (!thumb) return NULL;

   return g_strdup (gimv_image_detect_type_by_ext (thumb->info->filename));
}


static gchar *
column_data_time (Thumbnail *thumb)
{
   if (!thumb) return NULL;

   return fileutil_time2str (thumb->info->st.st_mtime);
}


static gchar *
column_data_uid (Thumbnail *thumb)
{
   if (!thumb) return NULL;

   return fileutil_uid2str (thumb->info->st.st_uid);
}


static gchar *
column_data_gid (Thumbnail *thumb)
{
   if (!thumb) return NULL;

   return fileutil_gid2str (thumb->info->st.st_gid);
}


static gchar *
column_data_mode (Thumbnail *thumb)
{
   if (!thumb) return NULL;

   return fileutil_mode2str (thumb->info->st.st_mode);
}


static gchar *
column_data_cache (Thumbnail *thumb)
{
   if (!thumb) return NULL;

   return thumb_prefs[thumb->cache_type].label;
}


/*
 *  detailview_set_text_data:
 *     @ 
 *
 *  tv : Pointer to the ThumbView struct.
 *  dest_mode : New thumbnail view display mode.
 *  start  : Pointer to Start of the Thumbnail list.
 */
static void
detailview_set_text_data (ThumbView *tv, gint dest_mode, GList *start)
{
   Thumbnail *thumb;
   ThumbViewData *tv_data;
   GList *node;
   gint i, j, start_pos, num, colnum;
   gint font_height = 16;
   gchar *image_data[] = {
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
   };
   const gint image_data_num = sizeof (image_data) / sizeof (image_data[0]);

   g_return_if_fail (tv);
   if (!start) return;

   node = start;
   start_pos = g_list_position (tv->thumblist, start);
   num = g_list_length (node);
   colnum = sizeof (list_columns) / sizeof (list_columns[0]);
   g_return_if_fail (image_data_num == colnum);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, DETAIL_VIEW_LABEL);
   g_return_if_fail (tv_data && tv_data->clist);

   /* set text data */
   for (i = start_pos ; i < num ; i++) {
      thumb = node->data;

      for (j = 0; j < colnum && j < image_data_num; j++)
	 if (list_columns[j].func)
	    image_data[j] = list_columns[j].func (thumb);

      gtk_clist_append (GTK_CLIST (tv_data->clist), image_data);
      gtk_clist_set_row_data (GTK_CLIST (tv_data->clist), i, thumb);

      if (thumb->selected)
	 gtk_clist_select_row (GTK_CLIST (tv_data->clist), i, -1);
      else
	 gtk_clist_unselect_row (GTK_CLIST (tv_data->clist), i, -1);

      for (j = 0; j < colnum && j < image_data_num; j++)
	 if (list_columns[j].free && image_data[j])
	    g_free (image_data[j]);

      if (!strcmp ("Detail + Thumbnail", thumbview_num_to_label (dest_mode)))
	 for (j = 1; j < colnum; j++) {
	    font_height = gdk_string_height (tv_data->clist->style->font, "aho");
	    if (tv->ThumbnailSize > font_height)
	       gtk_clist_set_shift (GTK_CLIST (tv_data->clist), i, j,
				    (tv->ThumbnailSize - font_height) / 2, 0);
	 }
      node = g_list_next (node);
   }
}


/*
 *  detailview_set_pixmaps:
 *     @ set thumbnails to columns
 *
 *  tv : Pointer to the ThumbView struct.
 *  dest_mode : New thumbnail view display mode.
 */
static void
detailview_set_pixmaps (ThumbView *tv, gint dest_mode)
{
   Thumbnail *thumb;
   GList *node;
   gint i, num;

   if (!tv) return;

   node = g_list_first (tv->thumblist);
   num = g_list_length (node);

   /* set images */
   for ( i = 0 ; i < num ; i++ ) {
      thumb = node->data;
      detailview_add_thumbnail (thumb, dest_mode, LOAD_CACHE);
      node = g_list_next (node);
   }
}


static void
store_page_pos (ThumbView *tv)
{
   ThumbViewData *tv_data;
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *hadj, *vadj;
   gchar *label;

   g_return_if_fail (tv);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, DETAIL_VIEW_LABEL);
   g_return_if_fail (tv_data);

   label = thumbview_num_to_label (tv->disp_mode);
   scrollwin = GTK_SCROLLED_WINDOW (tv->container);
   hadj = gtk_scrolled_window_get_hadjustment (scrollwin);
   vadj = gtk_scrolled_window_get_vadjustment (scrollwin);

   if (!strcmp (label, DETAIL_VIEW_LABEL)) {
      tv_data->page_pos_x[0] = hadj->value;
      tv_data->page_pos_y[0] = vadj->value;
   } else if (!strcmp (label, DETAIL_ICON_LABEL)) {
      tv_data->page_pos_x[1] = hadj->value;
      tv_data->page_pos_y[1] = vadj->value;
   } else if (!strcmp (label, DETAIL_THUMB_LABEL)) {
      tv_data->page_pos_x[2] = hadj->value;
      tv_data->page_pos_y[2] = vadj->value;
   }
   /* g_print ("store: %s %f\n", label, vadj->value); */
}


static gint
idle_reset_page_pos (gpointer data)
{
   ThumbView *tv = data;
   ThumbViewData *tv_data;
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *hadj, *vadj;
   gchar *label;
   gfloat page_pos_x, page_pos_y;

   g_return_val_if_fail (tv, FALSE);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, DETAIL_VIEW_LABEL);
   g_return_val_if_fail (tv_data, FALSE);

   label = thumbview_num_to_label (tv_data->dest_mode);
   scrollwin = GTK_SCROLLED_WINDOW (tv->container);
   hadj = gtk_scrolled_window_get_hadjustment (scrollwin);
   vadj = gtk_scrolled_window_get_vadjustment (scrollwin);

   if (!strcmp (label, DETAIL_VIEW_LABEL)) {
      page_pos_x = tv_data->page_pos_x[0];
      page_pos_y = tv_data->page_pos_y[0];
   } else if (!strcmp (label, DETAIL_ICON_LABEL)) {
      page_pos_x = tv_data->page_pos_x[1];
      page_pos_y = tv_data->page_pos_y[1];
   } else if (!strcmp (label, DETAIL_THUMB_LABEL)) {
      page_pos_x = tv_data->page_pos_x[2];
      page_pos_y = tv_data->page_pos_y[2];
   } else {
      return FALSE;
   }

   /* g_print ("reset: %s %f\n", label, page_pos); */
   gtk_adjustment_set_value (hadj, page_pos_x);
   gtk_adjustment_set_value (vadj, page_pos_y);

   return FALSE;
}



/******************************************************************************
 *
 *   public functions.
 *
 ******************************************************************************/
/*
 *  detailview_remove_thumbnail_data:
 *
 *  thumb : Pointer to the Thumbnail struct.
 */
void
detailview_remove_thumbview_data (ThumbView *tv)
{
   ThumbViewData *tv_data;

   if (!tv) return;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, DETAIL_VIEW_LABEL);
   if (!tv_data) return;

   g_hash_table_remove (tv->disp_mode_data, DETAIL_VIEW_LABEL); 
   g_free (tv_data);
}


/*
 *  detailview_append_thumb_frames:
 *     @ add thumbnail flame (only redraw clist widget).
 *
 *  tv     : Pointer to the ThumbView struct.
 *  start  : Pointer to Start of the Thumbnail list.
 *  Return : Return FALSE if canceled by user.
 */
gboolean
detailview_append_thumb_frames (ThumbView *tv, GList *start, gint dest_mode)
{
   g_return_val_if_fail (tv && start, FALSE);

   if (detailview_redraw (tv, dest_mode, tv->container)) {
      return TRUE;
   } else {
      return FALSE;
   }
}


/*
 *  detailview_add_thumbnail:
 *     @ append thumbnail to list.
 *
 *  thumb     : Pointer to the Thumbnail struct.
 *  dest_mode : New thumbnail view display mode.
 *  Return    : Added thumbnail pixmap widget.
 */
GtkWidget *
detailview_add_thumbnail (Thumbnail *thumb, gint dest_mode,
			  ThumbLoadType type)
{
   ThumbView *tv;
   ThumbViewData *tv_data;
   GdkPixmap *gdk_pixmap;
   GdkBitmap *mask;
   GList *node;
   gint pos, row, col;

   g_return_val_if_fail (thumb, NULL);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, NULL);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, DETAIL_VIEW_LABEL);
   g_return_val_if_fail (tv_data && tv_data->clist, NULL);

   node = g_list_find (tv->thumblist, thumb);
   pos = g_list_position(tv->thumblist, node);

   row = pos;
   col = 0;

   /*
   if (!thumb->pixmap)
      thumb->pixmap = create_thumbnail_auto (thumb, tv->ThumbnailSize, type);
   */

   /* set icon */
   if (!strcmp ("Detail + Icon", thumbview_num_to_label (dest_mode))) {
      if (thumb->icon) {
	 gtk_pixmap_get (GTK_PIXMAP (thumb->icon), &gdk_pixmap, &mask);
	 gtk_clist_set_pixmap (GTK_CLIST (tv_data->clist),
			       row, col, gdk_pixmap, mask);
	 gtk_clist_set_text (GTK_CLIST (tv_data->clist),
			     row, 4, column_data_cache (thumb));
	 return thumb->icon;
      }
   }

   /* set thumbnail */
   if (!strcmp ("Detail + Thumbnail", thumbview_num_to_label (dest_mode))) {
      if (thumb->pixmap) {
	 gtk_pixmap_get (GTK_PIXMAP (thumb->pixmap), &gdk_pixmap, &mask);
	 gtk_clist_set_pixmap (GTK_CLIST (tv_data->clist),
			       row, col, gdk_pixmap, mask);
	 gtk_clist_set_text (GTK_CLIST (tv_data->clist),
			     row, 4, column_data_cache (thumb));
	 return thumb->pixmap;
      }
   }

   return NULL;
}


/*
 *  detailview_redraw:
 *     @ remove old detailview widget and redraw.
 *
 *  tv        : Pointer to ThumbView struct for redraw.
 *  dest_mode : New thumbnail view display mode.
 *  Return    : Pointer to the new GtkTable widget.
 */
GtkWidget *
detailview_redraw (ThumbView *tv, gint dest_mode, GtkWidget *scroll_win)
{
   ThumbViewData *tv_data;
   GList *node;

   g_return_val_if_fail (tv, NULL);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return NULL;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, DETAIL_VIEW_LABEL);
   if (!tv_data)
      tv_data = detailview_new (tv);

   tv_data->dest_mode = dest_mode;

   if (tv_data->clist) {
      store_page_pos (tv);
      gtk_widget_destroy (tv_data->clist);
      tv_data->clist = NULL;
   }

   if (scroll_win) {
      if (GTK_BIN (tv->container)->child)
	 gtk_widget_destroy (GTK_BIN (tv->container)->child);   
      tv_data->clist = detailview_create (tv, dest_mode);
      gtk_container_add (GTK_CONTAINER (scroll_win), tv_data->clist);
      gtk_idle_add (idle_reset_page_pos, tv);
   }

   return tv_data->clist;
}


/*
 *  detailview_refresh_thumbnail:
 *     @ Create thumbnail from original image and draw it.
 *
 *  thumb  : Pointer to the Thumbnail struct.
 *  Return : TRUE if success.
 */
gboolean
detailview_refresh_thumbnail (Thumbnail *thumb)
{
   GtkWidget *widget;
   ThumbView *tv;

   if (!thumb) return FALSE;

   tv = thumb->thumb_view;
   if (!tv) return FALSE;

   widget = detailview_add_thumbnail (thumb, tv->disp_mode, LOAD_CACHE);

   if (widget)
      return TRUE;
   else
      return FALSE;
}


/*
 *  detailview_resize:
 *     @ do nothing.
 *
 *  tv     : Pointer to the ThumbView struct.
 *  Return : Pointer to the New thumbnail table widget.
 */
GtkWidget *
detailview_resize (ThumbView *tv)
{
   ThumbViewData *tv_data;
   GList *node;

   g_return_val_if_fail (tv, NULL);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return NULL;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, DETAIL_VIEW_LABEL);
   g_return_val_if_fail (tv_data, NULL);

   return tv_data->clist;
}


/*
 *  detailview_adjust:
 *     @ .
 *
 *  tv     : Pointer to the ThumbView struct.
 *  tv     : Pointer to the Thumbnail struct.
 */
void
detailview_adjust (ThumbView *tv, Thumbnail *thumb)
{
   ThumbViewData *tv_data;
   GList *node;
   gint pos;

   g_return_if_fail (tv);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, DETAIL_VIEW_LABEL);
   g_return_if_fail (tv_data);

   node = g_list_find (tv->thumblist, thumb);
   if (!node) return;
   pos = g_list_position (tv->thumblist, node);

   gtk_clist_moveto (GTK_CLIST (tv_data->clist), pos, 0, 0, 0);

   return;
}


gboolean
detailview_set_selection (Thumbnail *thumb, gboolean select)
{
   ThumbView *tv;
   ThumbViewData *tv_data;
   GList *node;
   gint pos;

   g_return_val_if_fail (thumb, FALSE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, FALSE);
   if (g_list_length (tv->thumblist) < 1) return FALSE;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, DETAIL_VIEW_LABEL);
   g_return_val_if_fail (tv_data && tv_data->clist, FALSE);

   node = g_list_find (tv->thumblist, thumb);
   pos = g_list_position (tv->thumblist, node);

   if (pos >= 0) {
      thumb->selected = select;
      if (thumb->selected)
	 gtk_clist_select_row (GTK_CLIST (tv_data->clist), pos, -1);
      else
	 gtk_clist_unselect_row (GTK_CLIST (tv_data->clist), pos, -1);
   }

   return TRUE;
}


/*
 *  detailview_create:
 *     @ Create thumbnail detail view widget.
 *
 *  tv        : Pointer to the ThumbView struct.
 *  dest_mode : New display mode.
 *  Return    : GtkTable widget.
 */
GtkWidget *
detailview_create (ThumbView *tv, gint dest_mode)
{
   gint i, num;
   ThumbViewData *tv_data;

   g_return_val_if_fail (tv, NULL);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, DETAIL_VIEW_LABEL);
   if (!tv_data) {
      tv_data = detailview_new (tv);
      g_return_val_if_fail (tv_data, NULL);
   }

   /* create clist widget */
   num = sizeof (list_columns) / sizeof (list_columns[0]);
   tv_data->clist = gtk_clist_new (num);
   gtk_clist_set_selection_mode(GTK_CLIST (tv_data->clist),
				GTK_SELECTION_EXTENDED);
   gtk_clist_set_shadow_type (GTK_CLIST (tv_data->clist), GTK_SHADOW_OUT);
   /* gtk_clist_set_button_actions (GTK_CLIST (clist), 0, GTK_BUTTON_SELECTS); */
   gtk_widget_show (tv_data->clist);

   gtk_signal_connect_after (GTK_OBJECT (tv_data->clist),"button_press_event",
			     GTK_SIGNAL_FUNC (cb_clist_button_press), tv);
   gtk_signal_connect (GTK_OBJECT (tv_data->clist),"button_release_event",
		       GTK_SIGNAL_FUNC (cb_clist_button_release), tv);
   gtk_signal_connect (GTK_OBJECT (tv_data->clist),"motion_notify_event",
		       GTK_SIGNAL_FUNC (cb_clist_motion_notify), tv);
   gtk_signal_connect (GTK_OBJECT (tv_data->clist), "click_column",
		       GTK_SIGNAL_FUNC (cb_click_column), tv);
   gtk_signal_connect (GTK_OBJECT (tv_data->clist), "select-row",
		       GTK_SIGNAL_FUNC (cb_select_row), tv);
   gtk_signal_connect (GTK_OBJECT (tv_data->clist), "unselect-row",
		       GTK_SIGNAL_FUNC (cb_unselect_row), tv);

   gtk_clist_set_use_drag_icons (GTK_CLIST (tv_data->clist), FALSE);

   /* for drop file list */
   dnd_src_set  (tv_data->clist, dnd_types, dnd_types_num);
   dnd_dest_set (tv_data->clist, dnd_types, dnd_types_num);

   gtk_signal_connect(GTK_OBJECT (tv_data->clist), "drag_begin",
		      GTK_SIGNAL_FUNC (thumbview_drag_begin_cb), tv);
   gtk_signal_connect(GTK_OBJECT (tv_data->clist), "drag_data_get",
		      GTK_SIGNAL_FUNC (thumbview_drag_data_get_cb), tv);
   gtk_signal_connect(GTK_OBJECT (tv_data->clist), "drag_data_received",
		      GTK_SIGNAL_FUNC (thumbview_drag_data_received_cb), tv);
   gtk_signal_connect(GTK_OBJECT (tv_data->clist), "drag-data-delete",
		      GTK_SIGNAL_FUNC (thumbview_drag_data_delete_cb), tv);
   gtk_signal_connect(GTK_OBJECT (tv_data->clist), "drag_end",
		      GTK_SIGNAL_FUNC (thumbview_drag_end_cb), tv);

   /* set column titles */
   for (i = 0; i < num; i++)
      gtk_clist_set_column_title (GTK_CLIST (tv_data->clist), i, _(list_columns[i].title));
   gtk_clist_column_titles_show (GTK_CLIST (tv_data->clist));
   /* set column width and height */
   for (i = 0; i < num; i++) {
      gtk_clist_set_column_auto_resize (GTK_CLIST (tv_data->clist), i, TRUE);
      /*
      gtk_clist_set_column_resizeable (GTK_CLIST (clist), i, TRUE);
      if (list_columns[i].width > 0)
	 gtk_clist_set_column_width (GTK_CLIST(clist), i, list_columns[i].width);
      */
   }
   gtk_clist_set_column_justification(GTK_CLIST (tv_data->clist), 2, GTK_JUSTIFY_RIGHT);
   gtk_clist_set_column_justification(GTK_CLIST (tv_data->clist), 3, GTK_JUSTIFY_CENTER);

   if (!strcmp ("Detail + Icon", thumbview_num_to_label (dest_mode))) {
      gtk_clist_set_row_height (GTK_CLIST(tv_data->clist), ICON_SIZE);
   }
   if (!strcmp ("Detail + Thumbnail", thumbview_num_to_label (dest_mode))) {
      gtk_clist_set_column_width  (GTK_CLIST (tv_data->clist), 0, tv->ThumbnailSize);
      gtk_clist_set_row_height (GTK_CLIST (tv_data->clist), tv->ThumbnailSize);
   }

   /* set data */
   detailview_set_text_data (tv, dest_mode, tv->thumblist);
   detailview_set_pixmaps (tv, dest_mode);

   return tv_data->clist;
}
