/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#include "gimageview.h"

#include "icon.h"
#include "image_window.h"
#include "prefs.h"
#include "thumbnail_window.h"

Config conf;
KeyConf akey;


extern GtkItemFactoryEntry imagewin_menu_items[];
extern GtkItemFactoryEntry imagewin_view_items[];
extern GtkItemFactoryEntry imageview_zoom_items[];
extern GtkItemFactoryEntry imageview_rotate_items[];
extern GtkItemFactoryEntry thumbwin_menu_items[];
extern GtkItemFactoryEntry thumbwin_view_items[];
extern GtkItemFactoryEntry thumbwin_sort_items[];


typedef enum {
   D_NULL,
   D_STRING,
   D_INT,
   D_FLOAT,
   D_BOOL,
   D_ENUM
} DataType;

typedef struct _ConfParm
{
   gchar    *keyname;
   DataType  data_type;
   gchar    *default_val;
   gpointer  data;
} ConfParam;


typedef struct _KeyBindIfactory
{
   gpointer keyconf;
   GtkItemFactoryEntry *ifactory;
   gchar   *path;
} KeyBindIfactory;


typedef struct _PrefsSection
{
   gchar     *section_name;
   ConfParam *param;
} PrefsSection;


typedef struct _PrefsFile
{
   gchar        *filename;
   PrefsSection *sections;
} PrefsFile;


/* default config value */
static ConfParam param_common [] = {
   {"default_file_open_window", D_ENUM, "0",     &conf.default_file_open_window},
   {"default_dir_open_window",  D_ENUM, "1",     &conf.default_dir_open_window},
   {"scan_dir_recursive",       D_BOOL, "FALSE", &conf.scan_dir_recursive},
   {"recursive_follow_link",    D_BOOL, "FALSE", &conf.recursive_follow_link},
   {"recursive_follow_parent",  D_BOOL, "FALSE", &conf.recursive_follow_parent},
   {"read_dotfile",             D_BOOL, "FALSE", &conf.read_dotfile},

   {"detect_filetype_by_ext",   D_BOOL, "TRUE",  &conf.detect_filetype_by_ext},
   {"disp_filename_stdout",     D_BOOL, "TRUE",  &conf.disp_filename_stdout},
#if HAVE_GDK_PIXBUF
   {"interpolation",            D_ENUM, "0",     &conf.interpolation},
#else
#if HAVE_GDK_IMLIB
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */
   {"conv_rel_path_to_abs",     D_BOOL,   "TRUE",    &conf.conv_rel_path_to_abs},
   {"iconset",                  D_STRING, DEFAULT_ICONSET, &conf.iconset},
   {NULL, D_NULL, NULL, NULL}
};

static ConfParam param_startup [] = {
   {"startup_read_dir",         D_BOOL, "FALSE", &conf.startup_read_dir},
   {"startup_open_thumbwin",    D_BOOL, "FALSE", &conf.startup_open_thumbwin},
   {"startup_no_warning",       D_BOOL, "FALSE", &conf.startup_no_warning},
   {NULL, D_NULL, NULL, NULL}
};

static ConfParam param_imageview [] = {
   {"imgwin_width",             D_INT,  "600",   &conf.imgwin_width},
   {"imgwin_height",            D_INT,  "450",   &conf.imgwin_height},
   {"imgwin_fit_to_image",      D_BOOL, "FALSE", &conf.imgwin_fit_to_image},
   {"imgwin_open_new_win",      D_BOOL, "FALSE", &conf.imgwin_open_new_win},
   {"imgwin_raise_window",      D_BOOL, "TRUE",  &conf.imgwin_raise_window},

   {"imgwin_show_menubar",      D_BOOL, "TRUE",  &conf.imgwin_show_menubar},
   {"imgwin_show_toolbar",      D_BOOL, "TRUE",  &conf.imgwin_show_toolbar},
   {"imgwin_show_statusbar",    D_BOOL, "TRUE",  &conf.imgwin_show_statusbar},
   {"imgwin_toolbar_style",     D_ENUM, "0",     &conf.imgwin_toolbar_style},

   {"imgview_fit_image_to_win", D_BOOL, "TRUE",  &conf.imgview_fit_image_to_win},
   {"imgview_keep_aspect",      D_BOOL, "TRUE",  &conf.imgview_keep_aspect},
   {"imgview_scale",            D_INT,  "100",   &conf.imgview_scale},
   {"imgview_buffer",           D_BOOL, "FALSE", &conf.imgview_buffer},
   {"imgview_scroll_nolimit",   D_BOOL, "FALSE", &conf.imgview_scroll_nolimit},
   {NULL, D_NULL, NULL, NULL}
};


static ConfParam param_thumbview [] = {
   {"thumbwin_save_win_state",  D_BOOL,   "TRUE",  &conf.thumbwin_save_win_state},
   {"thumbwin_width",           D_INT,    "550",   &conf.thumbwin_width},
   {"thumbwin_height",          D_INT,    "500",   &conf.thumbwin_height},
   {"thumbwin_layout_type",     D_INT,    "0",     &conf.thumbwin_layout_type},
   {"thumbwin_pane1_horizontal",D_BOOL,   "TRUE",  &conf.thumbwin_pane1_horizontal},
   {"thumbwin_pane2_horizontal",D_BOOL,   "TRUE",  &conf.thumbwin_pane2_horizontal},
   {"thumbwin_pane2_attach_1",  D_BOOL,   "FALSE", &conf.thumbwin_pane2_attach_1},
   {"thumbwin_widget[0]",       D_INT,    "1",     &conf.thumbwin_widget[0]},
   {"thumbwin_widget[1]",       D_INT,    "2",     &conf.thumbwin_widget[1]},
   {"thumbwin_widget[2]",       D_INT,    "3",     &conf.thumbwin_widget[2]},
   {"thumbwin_pane_size1",      D_INT,    "150",   &conf.thumbwin_pane_size1},
   {"thumbwin_pane_size2",      D_INT,    "200",   &conf.thumbwin_pane_size2},

   {"thumbwin_show_dir_view",   D_BOOL,   "TRUE",  &conf.thumbwin_show_dir_view},
   {"thumbwin_show_preview",    D_BOOL,   "FALSE", &conf.thumbwin_show_preview},
   {"thumbwin_show_menubar",    D_BOOL,   "TRUE",  &conf.thumbwin_show_menubar},
   {"thumbwin_show_toolbar",    D_BOOL,   "TRUE",  &conf.thumbwin_show_toolbar},
   {"thumbwin_show_statusbar",  D_BOOL,   "TRUE",  &conf.thumbwin_show_statusbar},
   {"thumbwin_show_tab",        D_BOOL,   "TRUE",  &conf.thumbwin_show_tab},

   {"thumbwin_raise_window",    D_BOOL,   "TRUE",  &conf.thumbwin_raise_window},
   {"thumbwin_toolbar_style",   D_ENUM,   "0",     &conf.thumbwin_toolbar_style},

   {"thumbwin_disp_mode",       D_STRING, DEFAULT_DISP_MODE, &conf.thumbwin_disp_mode},
   {"thumbwin_tab_pos",         D_ENUM,   "2",     &conf.thumbwin_tab_pos},
   {"thumbwin_move_to_newtab",  D_BOOL,   "TRUE",  &conf.thumbwin_move_to_newtab},
   {"thumbwin_show_tab_close",  D_BOOL,   "TRUE",  &conf.thumbwin_show_tab_close},
   {"thumbwin_tab_fullpath",    D_BOOL,   "TRUE",  &conf.thumbwin_tab_fullpath},
   {"thumbwin_force_open_tab",  D_BOOL,   "FALSE", &conf.thumbwin_force_open_tab},

   {"thumbwin_thumb_size",      D_INT,    "64",    &conf.thumbwin_thumb_size},
   {"thumbwin_redraw_interval", D_INT,    "1",     &conf.thumbwin_redraw_interval},

   {"thumbtable_colnum_min",    D_INT,    "1",     &conf.thumbtable_colnum_min},
   {"thumbtable_colnum_max",    D_INT,    "10",    &conf.thumbtable_colnum_max},
   {"thumbtable_row_space",     D_INT,    "8",     &conf.thumbtable_row_space},
   {"thumbtable_col_space",     D_INT,    "8",     &conf.thumbtable_col_space},
   {"thumbbutton_border_width", D_INT,    "2",     &conf.thumbbutton_border_width},
   {NULL, D_NULL, NULL, NULL}
};

static ConfParam param_dirview [] = {
   {"dirview_show_toolbar",     D_BOOL,   "TRUE",   &conf.dirview_show_toolbar},
   {"dirview_show_dotfile",     D_BOOL,   "FALSE",  &conf.dirview_show_dotfile},
   {"dirview_show_current_dir", D_BOOL,   "TRUE",   &conf.dirview_show_current_dir},
   {"dirview_show_parent_dir",  D_BOOL,   "TRUE",   &conf.dirview_show_parent_dir},
   {"dirview_line_style",       D_ENUM,   "2",      &conf.dirview_line_style},
   {"dirview_expander_style",   D_ENUM,   "1",      &conf.dirview_expander_style},
   {"dirview_auto_scroll",      D_BOOL,   "TRUE",   &conf.dirview_auto_scroll},
   {"dirview_auto_scroll_time", D_INT,    "50",     &conf.dirview_auto_scroll_time},
   {"dirview_auto_expand",      D_BOOL,   "FALSE",  &conf.dirview_auto_expand},
   {"dirview_auto_expand_time", D_INT,    "500",    &conf.dirview_auto_expand_time},
   {NULL, D_NULL, NULL, NULL}
};


static ConfParam param_cache [] = {
   {"cache_gimv",               D_BOOL, "TRUE",  &conf.cache_gimv},
   {"cache_nautilus",           D_BOOL, "TRUE",  &conf.cache_nautilus},
   {"cache_konqueror_l",        D_BOOL, "TRUE",  &conf.cache_konqueror_l},
   {"cache_konqueror_m",        D_BOOL, "TRUE",  &conf.cache_konqueror_m},
   {"cache_konqueror_s",        D_BOOL, "TRUE",  &conf.cache_konqueror_s},
   {"cache_kde_2_2",            D_BOOL, "TRUE",  &conf.cache_kde_2_2},
   {"cache_gqview",             D_BOOL, "TRUE",  &conf.cache_gqview},
   {"cache_ee_l",               D_BOOL, "TRUE",  &conf.cache_ee_l},
   {"cache_ee_m",               D_BOOL, "TRUE",  &conf.cache_ee_m},
   {"cache_ee_s",               D_BOOL, "TRUE",  &conf.cache_ee_s},
   {"cache_picview",            D_BOOL, "FALSE", &conf.cache_picview},
   {"cache_xvpic",              D_BOOL, "TRUE",  &conf.cache_xvpics},
   {"cache_write_type",         D_ENUM, "0",     &conf.cache_write_type},
   {"cache_gimv_write_size",    D_INT,  "96",    &conf.cache_gimv_write_size},
   {"cache_gqview_write_size",  D_ENUM, "2",     &conf.cache_gqview_write_size},
   {NULL, D_NULL, NULL, NULL}
};

static ConfParam param_slideshow [] = {
   {"slideshow_fit_to_win",     D_BOOL, "FALSE", &conf.slideshow_fit_to_win},
   {"slideshow_img_scale",      D_INT,  "100",   &conf.slideshow_img_scale},
   {"slideshow_fullscreen",     D_BOOL, "TRUE",  &conf.slideshow_fullscreen},
   {"slideshow_menubar",        D_BOOL, "FALSE", &conf.slideshow_menubar},
   {"slideshow_toolbar",        D_BOOL, "FALSE", &conf.slideshow_toolbar},
   {"slideshow_statusbar",      D_BOOL, "FALSE", &conf.slideshow_statusbar},
   {"slideshow_scrollbar",      D_BOOL, "FALSE", &conf.slideshow_scrollbar},
   {"slideshow_keep_aspcet",    D_BOOL, "TRUE",  &conf.slideshow_keep_aspect},
   {"slideshow_interval",       D_FLOAT,"5.00",  &conf.slideshow_interval},
   {"slideshow_repeat",         D_BOOL, "FALSE", &conf.slideshow_repeat},
   {NULL, D_NULL, NULL, NULL}
};

static ConfParam param_dnd [] = {
   {"dnd_enable_to_external",   D_BOOL, "FALSE",  &conf.dnd_enable_to_external},
   {"dnd_enable_from_external", D_BOOL, "FALSE",  &conf.dnd_enable_from_external},
   {"dnd_refresh_list_always",  D_BOOL, "TRUE",   &conf.dnd_refresh_list_always},
   {NULL, D_NULL, NULL, NULL}
};


static ConfParam param_progs [] = {
   {"progs[0]",    D_STRING, "Gimp,gimp-remote -n", &conf.progs[0]},
   {"progs[1]",    D_STRING, "XPaint,xpaint",       &conf.progs[1]},
   {"progs[2]",    D_STRING, "ElectricEyes,ee",     &conf.progs[2]},
   {"progs[3]",    D_STRING, "animate,animate",     &conf.progs[3]},
   {"progs[4]",    D_STRING, "enfle,enfle",         &conf.progs[4]},
   {"progs[5]",    D_STRING, NULL,                  &conf.progs[5]},
   {"progs[6]",    D_STRING, NULL,                  &conf.progs[6]},
   {"progs[7]",    D_STRING, NULL,                  &conf.progs[7]},
   {"progs[8]",    D_STRING, NULL,                  &conf.progs[8]},
   {"progs[9]",    D_STRING, NULL,                  &conf.progs[9]},
   {"progs[10]",   D_STRING, NULL,                  &conf.progs[10]},
   {"progs[11]",   D_STRING, NULL,                  &conf.progs[11]},
   {"progs[12]",   D_STRING, NULL,                  &conf.progs[12]},
   {"progs[13]",   D_STRING, NULL,                  &conf.progs[13]},
   {"progs[14]",   D_STRING, NULL,                  &conf.progs[14]},
   {"progs[15]",   D_STRING, NULL,                  &conf.progs[15]},
   {"text_viewer", D_STRING, "emacs",               &conf.text_viewer},
   {"use_internal_text_viewer",  D_BOOL, "TRUE", &conf.text_viewer_use_internal},
   {NULL, D_NULL, NULL, NULL}
};

ConfParam keyprefs_common[] = {
   {"common_open_dialog",       D_STRING, "<control>F", &akey.common_open},
   {"common_open_thumbwindow",  D_STRING, "<control>W", &akey.common_thumbwin},
   {"common_quit",              D_STRING, "<control>C", &akey.common_quit},
   {"common_open_prefs",        D_STRING, "<control>O", &akey.common_prefs},
   {"common_toggle_menubar",    D_STRING, "M", &akey.common_toggle_menubar},
   {"common_toggle_toolbar",    D_STRING, "N", &akey.common_toggle_toolbar},
   {"common_toggle_tab",        D_STRING, "B", &akey.common_toggle_tab},
   {"common_toggle_statusbar",  D_STRING, "V", &akey.common_toggle_statusbar},
   {"common_toggle_fullscreen", D_STRING, "F", &akey.common_fullscreen},
   {NULL, D_NULL, NULL, NULL}
};

ConfParam keyprefs_imgwin[] = {
   {"imgwin_buffer",              D_STRING, "<control>B", &akey.imgwin_buffer},
   {"imgwin_close",               D_STRING, "Q",          &akey.imgwin_close},
   {"imgwin_zoomin",              D_STRING, "S",          &akey.imgwin_zoomin},
   {"imgwin_zoomout",             D_STRING, "A",          &akey.imgwin_zoomout},
   {"imgwin_fit_img",             D_STRING, "W",          &akey.imgwin_fit_img},
   {"imgwin_keep_aspect",         D_STRING, "<shift>A",   &akey.imgwin_keep_aspect},
   {"imgwin_zoom10",              D_STRING, "1",          &akey.imgwin_zoom10},
   {"imgwin_zoom25",              D_STRING, "2",          &akey.imgwin_zoom25},
   {"imgwin_zoom50",              D_STRING, "3",          &akey.imgwin_zoom50},
   {"imgwin_zoom75",              D_STRING, "4",          &akey.imgwin_zoom75},
   {"imgwin_zoom100",             D_STRING, "5",          &akey.imgwin_zoom100},
   {"imgwin_zoom125",             D_STRING, "6",          &akey.imgwin_zoom125},
   {"imgwin_zoom150",             D_STRING, "7",          &akey.imgwin_zoom150},
   {"imgwin_zoom175",             D_STRING, "8",          &akey.imgwin_zoom175},
   {"imgwin_zoom200",             D_STRING, "9",          &akey.imgwin_zoom200},

   {"imgwin_rotate_ccw",          D_STRING, "E",          &akey.imgwin_rotate_ccw},
   {"imgwin_rotate_cw",           D_STRING, "R",          &akey.imgwin_rotate_cw},
   {"imgwin_rotate_180",          D_STRING, "D",          &akey.imgwin_rotate_180},

   {"imgwin_fit_window_to_image", D_STRING, "I", &akey.imgwin_fit_win},
   {NULL, D_NULL, NULL, NULL}
};

ConfParam keyprefs_thumbwin[] = {
   {"thumbwin_new_tab",           D_STRING, "<control>T", &akey.thumbwin_new_tab},
   {"thumbwin_new_collection",    D_STRING, NULL,         &akey.thumbwin_new_collection},
   {"thumbwin_close_tab",         D_STRING, "Q",          &akey.thumbwin_close_tab},
   {"thumbwin_close_win",         D_STRING, "<control>Q", &akey.thumbwin_close_win},

   {"thumbwin_select_all",        D_STRING, "<control>A", &akey.thumbwin_select_all},
   {"thumbwin_unselect_all",      D_STRING, "<control>N", &akey.thumbwin_unselect_all},
   {"thumbwin_refresh_list",      D_STRING, "R",          &akey.thumbwin_refresh_list},
   {"thumbwin_reload_cache",      D_STRING, "<shift>R",   &akey.thumbwin_reload_cache},
   {"thumbwin_recreate_thumb",    D_STRING, "<control>R", &akey.thumbwin_recreate_thumb},
   {"thumbwin_move_tab_forward",  D_STRING, "<shift>F",   &akey.thumbwin_move_tab_forward},
   {"thumbwin_move_tab_backward", D_STRING, "<shift>B",   &akey.thumbwin_move_tab_backward},
   {"thumbwin_detach_tab",        D_STRING, "<control>D", &akey.thumbwin_detach_tab},

   {"thumbwin_layout0",           D_STRING, "<control>0", &akey.thumbwin_layout0},
   {"thumbwin_layout1",           D_STRING, "<control>1", &akey.thumbwin_layout1},
   {"thumbwin_layout2",           D_STRING, "<control>2", &akey.thumbwin_layout2},
   {"thumbwin_layout3",           D_STRING, "<control>3", &akey.thumbwin_layout3},
   {"thumbwin_layout4",           D_STRING, "<control>4", &akey.thumbwin_layout4},
   {"thumbwin_custom",            D_STRING, NULL,         &akey.thumbwin_custom},
   {"thumbwin_slideshow",         D_STRING, "<control>S", &akey.thumbwin_slideshow},
   {"thumbwin_first_page",        D_STRING, "K",          &akey.thumbwin_first_page},
   {"thumbwin_last_page",         D_STRING, "J",          &akey.thumbwin_last_page},
   {"thumbwin_next_page",         D_STRING, "L",          &akey.thumbwin_next_page},
   {"thumbwin_prev_page",         D_STRING, "H",          &akey.thumbwin_prev_page},
   {"thumbwin_toggle_dir_toolbar",D_STRING, "<shift>N",   &akey.thumbwin_toggle_dir_toolbar},
   {"thumbwin_toggle_dirview",    D_STRING, "<shift>D",   &akey.thumbwin_toggle_dirview},
   {"thumbwin_toggle_preview",    D_STRING, "<shift>P",   &akey.thumbwin_toggle_preview},

   {"thumbwin_sort_name",         D_STRING, NULL,         &akey.thumbwin_sort_name},
   {"thumbwin_sort_atime",        D_STRING, NULL,         &akey.thumbwin_sort_atime},
   {"thumbwin_sort_mtime",        D_STRING, NULL,         &akey.thumbwin_sort_mtime},
   {"thumbwin_sort_ctime",        D_STRING, NULL,         &akey.thumbwin_sort_ctime},
   {"thumbwin_sort_size",         D_STRING, NULL,         &akey.thumbwin_sort_size},
   {"thumbwin_sort_type",         D_STRING, NULL,         &akey.thumbwin_sort_type},
   {"thumbwin_sort_rev",          D_STRING, NULL,         &akey.thumbwin_sort_rev},
   {NULL, D_NULL, NULL, NULL}
};

KeyBindIfactory keybind_ifactory [] = {
   /*
    * image window 
    */
   /* Main Menu */
   {&akey.common_open,      imagewin_menu_items, "/_File/_Open"},
   {&akey.common_thumbwin,  imagewin_menu_items, "/_File/_Open Thumbnail Window"},
   {&akey.imgwin_buffer,    imagewin_menu_items, "/_File/Memory _Buffer"},
   {&akey.imgwin_close,     imagewin_menu_items, "/_File/Close"},
   {&akey.common_quit,      imagewin_menu_items, "/_File/Quit"},
   {&akey.common_prefs,     imagewin_menu_items, "/_Edit/Options"},

   /* View submenu */
   {&akey.common_toggle_menubar,   imagewin_view_items, "/Menu Bar"},
   {&akey.common_toggle_toolbar,   imagewin_view_items, "/Tool Bar"},
   {&akey.common_toggle_statusbar, imagewin_view_items, "/Status Bar"},
   {&akey.common_fullscreen,       imagewin_view_items, "/Full Screen"},
   {&akey.imgwin_fit_win,          imagewin_view_items, "/Fit to Image"},

   /* Zoom submenu */
   {&akey.imgwin_zoomin,           imageview_zoom_items, "/Zoom In"},
   {&akey.imgwin_zoomout,          imageview_zoom_items, "/Zoom Out"},
   {&akey.imgwin_fit_img,          imageview_zoom_items, "/Fit to Widnow"},
   {&akey.imgwin_keep_aspect,      imageview_zoom_items, "/Keep aspect ratio"},
   {&akey.imgwin_zoom10,           imageview_zoom_items, "/10%"},
   {&akey.imgwin_zoom25,           imageview_zoom_items, "/25%"},
   {&akey.imgwin_zoom50,           imageview_zoom_items, "/50%"},
   {&akey.imgwin_zoom75,           imageview_zoom_items, "/75%"},
   {&akey.imgwin_zoom100,          imageview_zoom_items, "/100%"},
   {&akey.imgwin_zoom125,          imageview_zoom_items, "/125%"},
   {&akey.imgwin_zoom150,          imageview_zoom_items, "/150%"},
   {&akey.imgwin_zoom175,          imageview_zoom_items, "/175%"},
   {&akey.imgwin_zoom200,          imageview_zoom_items, "/200%"},

   {&akey.imgwin_rotate_ccw,       imageview_rotate_items, "/Rotate 90 deg"},
   {&akey.imgwin_rotate_cw,        imageview_rotate_items, "/Rotate -90 deg"},
   {&akey.imgwin_rotate_180,       imageview_rotate_items, "/Rotate 180 deg"},

   /*
    * thumbnail window 
    */
   /* Main Menu */
   {&akey.common_open,                thumbwin_menu_items, "/_File/_Open"},
   {&akey.common_thumbwin,            thumbwin_menu_items, "/_File/New Window"},
   {&akey.thumbwin_new_tab,           thumbwin_menu_items, "/_File/New Tab"},
   {&akey.thumbwin_new_collection,    thumbwin_menu_items, "/_File/New Collection"},
   {&akey.thumbwin_close_tab,         thumbwin_menu_items, "/_File/Close Tab"},
   {&akey.thumbwin_close_win,         thumbwin_menu_items, "/_File/Close Window"},
   {&akey.common_quit,                thumbwin_menu_items, "/_File/Quit"},

   {&akey.thumbwin_select_all,        thumbwin_menu_items, "/_Edit/Select All"},
   {&akey.thumbwin_unselect_all,      thumbwin_menu_items, "/_Edit/Unselect All"},
   {&akey.thumbwin_refresh_list,      thumbwin_menu_items, "/_Edit/Refresh List"},
   {&akey.thumbwin_reload_cache,      thumbwin_menu_items, "/_Edit/Reload Cache"},
   {&akey.thumbwin_recreate_thumb,    thumbwin_menu_items, "/_Edit/Update All Thumbnail"},
   {&akey.common_prefs,               thumbwin_menu_items, "/_Edit/Options"},
   {&akey.thumbwin_first_page,        thumbwin_menu_items, "/_Tab/First Page"},
   {&akey.thumbwin_last_page,         thumbwin_menu_items, "/_Tab/Last Page"},
   {&akey.thumbwin_next_page,         thumbwin_menu_items, "/_Tab/Next Page"},
   {&akey.thumbwin_prev_page,         thumbwin_menu_items, "/_Tab/Previous Page"},
   {&akey.thumbwin_move_tab_forward,  thumbwin_menu_items, "/_Tab/Move tab forward"},
   {&akey.thumbwin_move_tab_backward, thumbwin_menu_items, "/_Tab/Move tab backward"},
   {&akey.thumbwin_detach_tab,        thumbwin_menu_items, "/_Tab/Detach tab"},

   /* View sub menu */
   {&akey.thumbwin_layout0,           thumbwin_view_items,  "/Layout/Layout0"},
   {&akey.thumbwin_layout1,           thumbwin_view_items,  "/Layout/Layout1"},
   {&akey.thumbwin_layout2,           thumbwin_view_items,  "/Layout/Layout2"},
   {&akey.thumbwin_layout3,           thumbwin_view_items,  "/Layout/Layout3"},
   {&akey.thumbwin_layout4,           thumbwin_view_items,  "/Layout/Layout4"},
   {&akey.thumbwin_custom,            thumbwin_view_items,  "/Layout/Custom"},
   {&akey.thumbwin_slideshow,         thumbwin_view_items,  "/Slideshow"},
   {&akey.thumbwin_toggle_dir_toolbar,thumbwin_view_items,  "/DirView Tool Bar"},
   {&akey.thumbwin_toggle_dirview,    thumbwin_view_items,  "/Directory View"},
   {&akey.thumbwin_toggle_preview,    thumbwin_view_items,  "/Preview"},
   {&akey.common_toggle_menubar,      thumbwin_view_items,  "/Menu Bar"},
   {&akey.common_toggle_toolbar,      thumbwin_view_items,  "/Tool Bar"},
   {&akey.common_toggle_tab,          thumbwin_view_items,  "/Tab"},
   {&akey.common_toggle_statusbar,    thumbwin_view_items,  "/Status Bar"},
   {&akey.common_fullscreen,          thumbwin_view_items,  "/Full Screen"},

   {&akey.thumbwin_sort_name,      thumbwin_sort_items,  "/by Name"},
   {&akey.thumbwin_sort_mtime,     thumbwin_sort_items,  "/by Modification Time"},
   {&akey.thumbwin_sort_atime,     thumbwin_sort_items,  "/by Access Time"},
   {&akey.thumbwin_sort_ctime,     thumbwin_sort_items,  "/by Change Time"},
   {&akey.thumbwin_sort_size,      thumbwin_sort_items,  "/by Size"},
   {&akey.thumbwin_sort_type,      thumbwin_sort_items,  "/by Type"},
   {&akey.thumbwin_sort_rev,       thumbwin_sort_items,  "/Reverse Order"},
   {NULL, NULL, NULL},
};


static PrefsSection conf_sections [] = {
   {"Common",           param_common},
   {"Startup",          param_startup},
   {"Image View",       param_imageview},
   {"Thumbnail View",   param_thumbview},
   {"Directory View",   param_dirview},
   {"Cache",            param_cache},
   {"Slide Show",       param_slideshow},
   {"Drag and Drop",    param_dnd},
   {"External Program", param_progs},
   {NULL, NULL},
};


static PrefsSection keyconf_sections [] = {
   {"Common",           keyprefs_common},
   {"Image Window",     keyprefs_imgwin},
   {"Thumbnail Window", keyprefs_thumbwin},
   {NULL, NULL},
};


static PrefsFile conf_files [] = {
   {GIMAGEVIEW_RC, conf_sections},
   {KEYCONF_RC,    keyconf_sections},
   {NULL, NULL},
};


/*
 *  store_config:
 *     @ Convert config data string to eache data type and store to memory.
 *
 *  data   : Pointer to memory chunk to store data.
 *  val    : Data value string.
 *  type   : Data type to convert.
 */
static void
store_config (gpointer data, gchar *val, DataType type)
{
   switch (type) {
   case D_STRING:
      g_free (*((gchar **)data));
      if (!val)
	 *((gchar **) data) = NULL;
      else
	 *((gchar **) data) = g_strdup(val);
      break;
   case D_INT:
      if (!val) break;
      *((gint *) data) =  atoi(val);
      break;
   case D_FLOAT:
      if (!val) break;
      *((gfloat *) data) =  atof(val);
      break;
   case D_BOOL:
      if (!val) break;
      if (!g_strcasecmp (val, "TRUE"))
	 *((gboolean *) data) = TRUE;
      else
	 *((gboolean *) data) = FALSE;
      break;
   case D_ENUM:
      if (!val) break;
      *((gint *) data) =  atoi(val);
      break;
   default:
      break;
   }
}


/*
 *  prefs_load_config_default:
 *     @ Load default config.
 */
void
prefs_load_config_default (PrefsSection *sections)
{
   gint i, j;

   for (j = 0; sections[j].section_name; j++) {
      ConfParam *param = sections[j].param;
      for (i = 0; param[i].keyname; i++)
         store_config (param[i].data, param[i].default_val, param[i].data_type);
   }
}


/*
 *  prefs_load_rc:
 *     @ Load config from disk.
 */
void
prefs_load_rc (gchar *filename, PrefsSection *sections)
{
   gchar rcfile[MAX_PATH_LEN];
   FILE *gimvrc;
   gchar buf[BUF_SIZE];
   gchar **pair;
   gint i, j;

   prefs_load_config_default (sections);

   g_snprintf (rcfile, MAX_PATH_LEN, "%s/%s/%s",
	       getenv("HOME"), RC_DIR, filename);
   gimvrc = fopen(rcfile, "r");
   if (!gimvrc) {
      g_warning (_("Can't open rc file: %s\n"
		   "Use default setting ..."), rcfile);
      return;
   }

   while (fgets (buf, sizeof(buf), gimvrc)) {
      if (!strncmp(buf, "[", 1) || !strncmp(buf, "\n", 1))
	 continue;

      g_strstrip (buf);

      pair = g_strsplit (buf, "=", 2);
      if (pair[0]) g_strstrip(pair[0]);
      if (pair[1]) g_strstrip(pair[1]);

      for (j = 0; sections[j].section_name; j++) {
	 ConfParam *param = sections[j].param;
	 for (i = 0; param[i].keyname; i++) {
	    if (!g_strcasecmp (param[i].keyname, pair[0])) {
	       store_config (param[i].data, pair[1], param[i].data_type);
	       break;
	    }
	 }
      }
      g_strfreev (pair);
   }

   fclose (gimvrc);
}


void
prefs_load_config ()
{
   gint i;

   for (i = 0; conf_files[i].filename; i++)
      prefs_load_rc (conf_files[i].filename, conf_files[i].sections);
}


/*
 *  prefs_save_rc:
 *     @ Save config to disk.
 */
void
prefs_save_rc (gchar *filename, PrefsSection *sections)
{
   gchar dir[MAX_PATH_LEN], rcfile[MAX_PATH_LEN];
   FILE *gimvrc;
   gint i, j;
   gchar *bool;
   struct stat st;

   g_snprintf (dir, MAX_PATH_LEN, "%s/%s", getenv("HOME"), RC_DIR);

   if (stat(dir, &st)) {
      mkdir(dir, S_IRWXU);
      g_warning (_("Directory \"%s\" not found. Created it ..."), dir);
   } else {
      if (!S_ISDIR(st.st_mode)) {
	 g_warning (_("\"%s\" found, but it's not directory. Abort creating ..."), dir);
	 return;
      }
   }

   g_snprintf (rcfile, MAX_PATH_LEN, "%s/%s/%s", getenv("HOME"),
	       RC_DIR, filename);
   gimvrc = fopen (rcfile, "w");
   if (!gimvrc) {
      g_warning (_("Can't open rc file for write."));
      return;
   }

   for (j = 0; sections[j].section_name; j++) {
      ConfParam *param = sections[j].param;
      fprintf (gimvrc, "[%s]\n", sections[j].section_name);
      for (i = 0; param[i].keyname; i++) {
	 switch (param[i].data_type) {
	 case D_STRING:
	    if (!*(gchar **) param[i].data)
	       fprintf (gimvrc, "%s=\n", param[i].keyname);
	    else
	       fprintf (gimvrc, "%s=%s\n", param[i].keyname,
			*(gchar **) param[i].data);
	    break;
	 case D_INT:
	    fprintf (gimvrc, "%s=%d\n", param[i].keyname,
		     *((gint *) param[i].data));
	    break;
	 case D_FLOAT:
	    fprintf (gimvrc, "%s=%f\n", param[i].keyname,
		     *((gfloat *) param[i].data));
	    break;
	 case D_BOOL:
	    if (*((gboolean *)param[i].data))
	       bool = "TRUE";
	    else
	       bool = "FALSE";
	    fprintf (gimvrc, "%s=%s\n", param[i].keyname, bool);
	    break;
	 case D_ENUM:
	    fprintf (gimvrc, "%s=%d\n", param[i].keyname,
		     *((gint *) param[i].data));
	    break;
	 default:
	    break;
         }
      }
      fprintf (gimvrc, "\n");
   }

   fclose (gimvrc);
}


void
prefs_save_config ()
{
   gint i;

   for (i = 0; conf_files[i].filename; i++)
      prefs_save_rc (conf_files[i].filename, conf_files[i].sections);
}


/*
 *  prefs_keybind_to_ifactory
 */
void prefs_bind_key_to_ifactory (GtkItemFactoryEntry *ifactory)
{
   gint i, j;

   for (i = 0; keybind_ifactory[i].keyconf; i++) {
      if (ifactory == keybind_ifactory[i].ifactory) {
	 for (j = 0; ifactory[j].path; j++) {
	    if (!strcmp (ifactory[j].path, keybind_ifactory[i].path)) {
	       //g_free (ifactory[j].accelerator);
	       ifactory[j].accelerator = g_strdup(*(gchar **) keybind_ifactory[i].keyconf);
	    }
	 }
      }
   }
}


void prefs_bind_key ()
{
   /* image window item factory entry*/
   prefs_bind_key_to_ifactory (imagewin_menu_items);
   prefs_bind_key_to_ifactory (imagewin_view_items);

   /* image view module item factory entry*/
   prefs_bind_key_to_ifactory (imageview_zoom_items);
   prefs_bind_key_to_ifactory (imageview_rotate_items);

   /* thumbnail window item factory entry*/
   prefs_bind_key_to_ifactory (thumbwin_menu_items);
   prefs_bind_key_to_ifactory (thumbwin_view_items);
   prefs_bind_key_to_ifactory (thumbwin_sort_items);
}
