/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>

#include "gimageview.h"

#include "fileutil.h"
#include "gfileutil.h"
#include "help.h"
#include "menu.h"
#include "prefs.h"
#include "text_viewer.h"

#define DOC_TEXT_DIR "text"

/* callback functions */
static void cb_open_text               (GtkWidget *menuitem,
					gchar     *filename);
static void cb_gimv_info_win_ok_button (GtkWidget *widget,
					GtkWidget *window);
static void cb_open_info               (gpointer   data,
					guint      action,
					GtkWidget *widget);

/* other private functions */
static gchar     *get_lang            ();
static gchar     *get_text_doc_dir    (gchar *lang);
static GtkWidget *get_plain_text_menu ();


GtkItemFactoryEntry gimvhelp_menu_items[] =
{
   {N_("/_Manual"),               NULL, NULL,    0, NULL},
   {N_("/_Document"),             NULL, NULL,    0, "<Branch>"},
   {N_("/_Document/_HTML"),       NULL, NULL,    0, "<Branch>"},
   {N_("/_Document/Plain _Text"), NULL, NULL,    0, "<Branch>"},
   {N_("/_About"),                NULL, cb_open_info, 0, NULL},
   {NULL, NULL, NULL, 0, NULL},
};


GList *text_filelist = NULL;


/******************************************************************************
 *
 *   Callback functions for menubar.
 *
 ******************************************************************************/
static void
cb_open_text (GtkWidget *menuitem, gchar *filename)
{
   gchar *cmd = NULL;

   g_return_if_fail (filename);

   if (!conf.text_viewer_use_internal && conf.text_viewer) {
      cmd = g_strconcat (conf.text_viewer, " ", filename, " &", NULL);
      system (cmd);
   } else {
      text_viewer_create (filename);
   }

   g_free (cmd);
}


static void
cb_gimv_info_win_ok_button (GtkWidget *widget, GtkWidget *window) {
   gtk_widget_destroy (window);
}


static void
cb_open_info (gpointer data, guint action, GtkWidget *widget)
{
   gimvhelp_info ();
}


/******************************************************************************
 *
 *   Private functions.
 *
 ******************************************************************************/
static gchar *
get_lang ()
{
   gchar *lang = NULL;

   lang = g_getenv ("LANGUAGE");

   if (!lang)
      lang = g_getenv ("LC_ALL");

   if (!lang)
      lang = g_getenv ("LC_MESSAGES");

   if (!lang)
      lang = g_getenv ("LANG");

   if (!lang)
      lang = "C";

   return lang;
}


static gchar *
get_text_doc_dir (gchar *lang)
{
   gchar *dir, *lang_fallback, *tmp;

   if (!lang)
      lang = get_lang ();

   dir = g_strconcat (DOCDIR, "/", DOC_TEXT_DIR, "/", lang, NULL);

   if (isdir (dir)) return dir;

   lang_fallback = g_strdup (lang);
   tmp = strchr (lang, '.');
   if (tmp) {
      *tmp = '\0';
      g_free (dir);
      dir = g_strconcat (DOCDIR, "/", DOC_TEXT_DIR, "/", lang_fallback, NULL);
      g_free (lang_fallback);
      if (isdir (dir)) {
	 return dir;
      }
   } else {
      g_free (lang_fallback);
   }

   if (strlen (lang) > 2 && lang[2] == '_') {
      lang_fallback = g_strdup (lang);
      lang_fallback[2] = '\0';
      g_free (dir);
      dir = g_strconcat (DOCDIR, "/", DOC_TEXT_DIR, "/", lang_fallback, NULL);
      g_free (lang_fallback);
      if (isdir (dir)) {
	 return dir;
      }
   }

   return NULL;
}


static GtkWidget *
get_plain_text_menu ()
{
   GtkWidget *menu = NULL, *menuitem;
   GList *node;
   gchar *dir;

   menu = gtk_menu_new();

   dir = get_text_doc_dir (NULL);
   if (!dir) goto FUNC_END;

   if (!text_filelist)
      get_dir (dir, GETDIR_FOLLOW_SYMLINK, &text_filelist, NULL);

   node = text_filelist;
   while (node) {
      gchar *filename = node->data;

      node = g_list_next (node);

      if (!filename) continue;

      menuitem = gtk_menu_item_new_with_label (g_basename(filename));
      gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			  GTK_SIGNAL_FUNC (cb_open_text), filename);
      gtk_menu_append (GTK_MENU (menu), menuitem);
      gtk_widget_show (menuitem);

   }

 FUNC_END:
   g_free (dir);
   return menu;
}


/******************************************************************************
 *
 *   Public functions.
 *
 ******************************************************************************/
GtkWidget *
gimvhelp_create_menu (GtkWidget *window)
{
   GtkWidget *menu = NULL, *menuitem = NULL, *submenu;
   GtkItemFactory *ifactory;
   guint n_menu_items;

   n_menu_items = sizeof (gimvhelp_menu_items)
                     / sizeof (gimvhelp_menu_items[0]) - 1;
   menu = menu_create_items(window, gimvhelp_menu_items,
			    n_menu_items, "<HelpSubMenu>", NULL);

   submenu = get_plain_text_menu ();
   menu_set_submenu (menu, "/Document/Plain Text", submenu);

   /* not implemented yet */
   ifactory = gtk_item_factory_from_widget (menu);
   menuitem  = gtk_item_factory_get_item (ifactory, "/Manual");
   gtk_widget_set_sensitive (menuitem, FALSE);
   menuitem  = gtk_item_factory_get_item (ifactory, "/Document/HTML");
   gtk_widget_set_sensitive (menuitem, FALSE);

   return menu;
}


void
gimvhelp_info () {
   GtkWidget *window;
   GtkWidget *label;
   GtkWidget *button;
   GtkWidget *scrolledwin, *text;
   gchar buf[BUF_SIZE];

   window = gtk_dialog_new ();
   gtk_container_set_border_width (GTK_CONTAINER (GTK_DIALOG (window)->vbox), 5);
   gtk_window_set_default_size (GTK_WINDOW (window), 450, 280);
   gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);

   g_snprintf (buf, BUF_SIZE, _("About %s"), PROG_NAME);
   gtk_window_set_title (GTK_WINDOW (window), buf); 

   g_snprintf (buf, BUF_SIZE,
	       _("%s\n\n"
		 "Copyright (C) 2001 %s <%s>\n\n"
		 "Web: %s"),
	       PROG_VERSION, PROG_AUTHOR, PROG_ADDRESS, PROG_URI);
   label = gtk_label_new (buf);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), 
		       label, TRUE, TRUE, 0);

   scrolledwin = gtk_scrolled_window_new (NULL, NULL);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrolledwin),
				   GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), scrolledwin, TRUE, TRUE, 0);

   text = gtk_text_new (gtk_scrolled_window_get_hadjustment
			(GTK_SCROLLED_WINDOW (scrolledwin)),
			gtk_scrolled_window_get_vadjustment
			(GTK_SCROLLED_WINDOW (scrolledwin)));
   gtk_container_add (GTK_CONTAINER (scrolledwin), text);

   gtk_text_freeze (GTK_TEXT (text));

   gtk_text_insert (GTK_TEXT (text), NULL, NULL, NULL,
		    _("This program is free software; you can redistribute it and/or modify\n"
		      "it under the terms of the GNU General Public License as published by\n"
		      "the Free Software Foundation; either version 2, or (at your option)\n"
		      "any later version.\n\n"

		      "This program is distributed in the hope that it will be useful,\n"
		      "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
		      "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"
		      "See the GNU General Public License for more details.\n\n"

		      "You should have received a copy of the GNU General Public License\n"
		      "along with this program; if not, write to the Free Software\n"
		      "Foundation, Inc., 59 Temple Place - Suite 330, Boston,\n"
		      "MA 02111-1307, USA."), -1);

   gtk_text_thaw (GTK_TEXT (text));

   button = gtk_button_new_with_label (_("OK"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
		       button, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (cb_gimv_info_win_ok_button),
		       window);
   gtk_widget_show_all (window);

   gtk_widget_grab_focus (button);

   gtk_grab_add (window);
}
