%
% (c) AQUA Project, Glasgow University, 1998
%

Cheap and cheerful dynamic types.

The Dynamic interface is part of the Hugs/GHC standard
libraries, providing basic support for dynamic types.

Operations for injecting values of arbitrary type into
a dynamically typed value, Dynamic, are provided, together
with operations for converting dynamic values into a concrete
(monomorphic) type.

The Dynamic implementation provided is closely based on code
contained in Hugs library of the same name.

NOTE: test code at the end, but commented out.

\begin{code}
module Dynamic
	(
	-- dynamic type
	  Dynamic	-- abstract, instance of: Show, Typeable
	, toDyn		-- :: Typeable a => a -> Dynamic
	, fromDyn	-- :: Typeable a => Dynamic -> a -> a
	, fromDynamic	-- :: Typeable a => Dynamic -> Maybe a
	
	-- type representation

	, Typeable(
	     typeOf)	-- :: a -> TypeRep

	  -- Dynamic defines Typeable instances for the following
	-- Prelude types: [a], (), (a,b), (a,b,c), (a,b,c,d),
	-- (a,b,c,d,e), (a->b), (Array a b), Bool, Char,
	-- (Complex a), Double, (Either a b), Float, Handle,
	-- Int, Integer, (IO a), (Maybe a), Ordering

	, TypeRep	-- abstract, instance of: Eq, Show, Typeable
	, TyCon		-- abstract, instance of: Eq, Show, Typeable

	-- type representation constructors/operators:
	, mkTyCon	-- :: String  -> TyCon
	, mkAppTy	-- :: TyCon   -> [TypeRep] -> TypeRep
	, mkFunTy	-- :: TypeRep -> TypeRep   -> TypeRep
	, applyTy	-- :: TypeRep -> TypeRep   -> Maybe TypeRep

	-- 
	-- let fTy = mkTyCon "Foo" in show (mkAppTy (mkTyCon ",,")
	--                                 [fTy,fTy,fTy])
	-- 
	-- returns "(Foo,Foo,Foo)"
	--
	-- The TypeRep Show instance promises to print tuple types
	-- correctly. Tuple type constructors are specified by a 
	-- sequence of commas, e.g., (mkTyCon ",,,,") returns
	-- the 5-tuple tycon.
	) where

import IOExts		( unsafePerformIO,
			  IORef, newIORef, readIORef, writeIORef
			)

import Array		( Array )
import Complex		( Complex )
import IO		( Handle )

import Addr		( Addr )
import Ptr              ( Ptr )
import ForeignObj	( ForeignObj ) {- DEPRECATED -}
import ForeignPtr	( ForeignPtr )
import Int		( Int8, Int16, Int32, Int64 )
import StablePtr	( StablePtr )
import ST		( ST )
import Word		( Word8, Word16, Word32, Word64 )

import ByteArray	( ByteArray )
import IArray		( UArray )
import IOExts		( IOArray )
import MArray		( IOUArray, STUArray )
import MutableArray	( MutableByteArray )
import PackedString	( PackedString )
import ST		( STArray )
import StableName	( StableName )
import Weak		( Weak )
import CTypes
import CTypesISO

#ifdef __STGHUGS__
import PrelPrim 	( primUnsafeCoerce
			, Dynamic(..) , TyCon(..) , TypeRep(..)
			)
			
unsafeCoerce :: a -> b
unsafeCoerce = primUnsafeCoerce
#else
import GlaExts
import PrelDynamic

unsafeCoerce :: a -> b
unsafeCoerce = unsafeCoerce#
#endif
\end{code}

The dynamic type is represented by Dynamic, carrying
the dynamic value along with its type representation:

\begin{code}
-- the instance just prints the type representation.
instance Show Dynamic where
   showsPrec _ (Dynamic t _) = 
          showString "<<" . 
	  showsPrec 0 t   . 
	  showString ">>"
\end{code}

Operations for going to and from Dynamic:

\begin{code}
toDyn :: Typeable a => a -> Dynamic
toDyn v = Dynamic (typeOf v) (unsafeCoerce v)

fromDyn :: Typeable a => Dynamic -> a -> a
fromDyn (Dynamic t v) def
  | typeOf def == t = unsafeCoerce v
  | otherwise       = def

fromDynamic :: Typeable a => Dynamic -> Maybe a
fromDynamic (Dynamic t v) =
  case unsafeCoerce v of 
    r | t == typeOf r -> Just r
      | otherwise     -> Nothing
\end{code}

(Abstract) universal datatype:

\begin{code}
instance Show TypeRep where
  showsPrec p (App tycon tys) =
    case tys of
      [] -> showsPrec p tycon
      [x] | tycon == listTc    -> showChar '[' . shows x . showChar ']'
      xs  
        | isTupleTyCon tycon -> showTuple tycon xs
	| otherwise	     ->
	    showParen (p > 9) $
   	    showsPrec p tycon . 
	    showChar ' '      . 
	    showArgs tys

  showsPrec p (Fun f a) =
     showParen (p > 8) $
     showsPrec 9 f . showString " -> " . showsPrec 8 a
\end{code}

To make it possible to convert values with user-defined types
into type Dynamic, we need a systematic way of getting
the type representation of an arbitrary type. A type
class provides just the ticket,

\begin{code}
class Typeable a where
  typeOf :: a -> TypeRep
\end{code}

NOTE: The argument to the overloaded `typeOf' is only
used to carry type information, and Typeable instances
should *never* *ever* look at its value.

\begin{code}
isTupleTyCon :: TyCon -> Bool
isTupleTyCon (TyCon _ (',':_)) = True
isTupleTyCon _		       = False

instance Show TyCon where
  showsPrec _ (TyCon _ s) = showString s

\end{code}
 
If we enforce the restriction that there is only one
@TyCon@ for a type & it is shared among all its uses,
we can map them onto Ints very simply. The benefit is,
of course, that @TyCon@s can then be compared efficiently.

Provided the implementor of other @Typeable@ instances
takes care of making all the @TyCon@s CAFs (toplevel constants),
this will work. 

If this constraint does turn out to be a sore thumb, changing
the Eq instance for TyCons is trivial.

\begin{code}
mkTyCon :: String -> TyCon
mkTyCon str = unsafePerformIO $ do
   v <- readIORef uni
   writeIORef uni (v+1)
   return (TyCon v str)

{-# NOINLINE uni #-}
uni :: IORef Int
uni = unsafePerformIO ( newIORef 0 )
\end{code}

Some (Show.TypeRep) helpers:

\begin{code}
showArgs :: Show a => [a] -> ShowS
showArgs [] = id
showArgs [a] = showsPrec 10 a
showArgs (a:as) = showsPrec 10 a . showString " " . showArgs as 

showTuple :: TyCon -> [TypeRep] -> ShowS
showTuple (TyCon _ str) args = showChar '(' . go str args
 where
  go [] [a] = showsPrec 10 a . showChar ')'
  go _  []  = showChar ')' -- a failure condition, really.
  go (',':xs) (a:as) = showsPrec 10 a . showChar ',' . go xs as
  go _ _   = showChar ')'
\end{code}

\begin{code}
mkAppTy  :: TyCon   -> [TypeRep] -> TypeRep
mkAppTy tyc args = App tyc args

mkFunTy  :: TypeRep -> TypeRep   -> TypeRep
mkFunTy f a = Fun f a
\end{code}

Auxillary functions

\begin{code}
-- (f::(a->b)) `dynApply` (x::a) = (f a)::b
dynApply :: Dynamic -> Dynamic -> Maybe Dynamic
dynApply (Dynamic t1 f) (Dynamic t2 x) =
  case applyTy t1 t2 of
    Just t3 -> Just (Dynamic t3 ((unsafeCoerce f) x))
    Nothing -> Nothing

dynApp :: Dynamic -> Dynamic -> Dynamic
dynApp f x = case dynApply f x of 
             Just r -> r
             Nothing -> error ("Type error in dynamic application.\n" ++
                               "Can't apply function " ++ show f ++
                               " to argument " ++ show x)

applyTy :: TypeRep -> TypeRep -> Maybe TypeRep
applyTy (Fun t1 t2) t3
  | t1 == t3    = Just t2
applyTy _ _     = Nothing

\end{code}

Prelude types

\begin{code}
listTc :: TyCon
listTc = mkTyCon "[]"

instance Typeable a => Typeable [a] where
  typeOf ls = mkAppTy listTc [typeOf ((undefined:: [a] -> a) ls)]

unitTc :: TyCon
unitTc = mkTyCon "()"

instance Typeable () where
  typeOf _ = mkAppTy unitTc []

tup2Tc :: TyCon
tup2Tc = mkTyCon ","

instance (Typeable a, Typeable b) => Typeable (a,b) where
  typeOf tu = mkAppTy tup2Tc [typeOf ((undefined :: (a,b) -> a) tu),
			      typeOf ((undefined :: (a,b) -> b) tu)]

tup3Tc :: TyCon
tup3Tc = mkTyCon ",,"

instance ( Typeable a , Typeable b , Typeable c) => Typeable (a,b,c) where
  typeOf tu = mkAppTy tup3Tc [typeOf ((undefined :: (a,b,c) -> a) tu),
			      typeOf ((undefined :: (a,b,c) -> b) tu),
			      typeOf ((undefined :: (a,b,c) -> c) tu)]

tup4Tc :: TyCon
tup4Tc = mkTyCon ",,,"

instance ( Typeable a
	 , Typeable b
	 , Typeable c
	 , Typeable d) => Typeable (a,b,c,d) where
  typeOf tu = mkAppTy tup4Tc [typeOf ((undefined :: (a,b,c,d) -> a) tu),
			      typeOf ((undefined :: (a,b,c,d) -> b) tu),
			      typeOf ((undefined :: (a,b,c,d) -> c) tu),
			      typeOf ((undefined :: (a,b,c,d) -> d) tu)]

tup5Tc :: TyCon
tup5Tc = mkTyCon ",,,,"

instance ( Typeable a
	 , Typeable b
	 , Typeable c
	 , Typeable d
	 , Typeable e) => Typeable (a,b,c,d,e) where
  typeOf tu = mkAppTy tup5Tc [typeOf ((undefined :: (a,b,c,d,e) -> a) tu),
			      typeOf ((undefined :: (a,b,c,d,e) -> b) tu),
			      typeOf ((undefined :: (a,b,c,d,e) -> c) tu),
			      typeOf ((undefined :: (a,b,c,d,e) -> d) tu),
			      typeOf ((undefined :: (a,b,c,d,e) -> e) tu)]

instance (Typeable a, Typeable b) => Typeable (a -> b) where
  typeOf f = mkFunTy (typeOf ((undefined :: (a -> b) -> a) f))
		     (typeOf ((undefined :: (a -> b) -> b) f))

arrayTc :: TyCon
arrayTc = mkTyCon "Array"

instance (Typeable a, Typeable b) => Typeable (Array a b) where
  typeOf a = mkAppTy arrayTc [typeOf ((undefined :: Array a b -> a) a),
			      typeOf ((undefined :: Array a b -> b) a)]


#define SIMPLE_INSTANCE_TYPEABLE(tycon,tcname,str) \
tcname = mkTyCon str; \
instance Typeable tycon where { typeOf _ = mkAppTy tcname [] } \

SIMPLE_INSTANCE_TYPEABLE(Bool,boolTc,"Bool")
SIMPLE_INSTANCE_TYPEABLE(Char,charTc,"Char")

complexTc :: TyCon
complexTc = mkTyCon "Complex"

instance Typeable a => Typeable (Complex a) where
  typeOf a = mkAppTy complexTc [typeOf ((undefined :: Complex a -> a) a)]

SIMPLE_INSTANCE_TYPEABLE(Float,floatTc,"Float")
SIMPLE_INSTANCE_TYPEABLE(Double,doubleTc,"Double")
SIMPLE_INSTANCE_TYPEABLE(Int,intTc,"Int")
SIMPLE_INSTANCE_TYPEABLE(Integer,integerTc,"Integer")

eitherTc :: TyCon
eitherTc = mkTyCon "Either"

instance (Typeable a, Typeable b) => Typeable (Either a b) where
  typeOf ei = mkAppTy eitherTc [typeOf ((undefined :: Either a b -> a) ei),
			        typeOf ((undefined :: Either a b -> b) ei)]

SIMPLE_INSTANCE_TYPEABLE(Handle,handleTc,"Handle")

iOTc :: TyCon
iOTc = mkTyCon "IO"

instance Typeable a => Typeable (IO a) where
  typeOf action = mkAppTy iOTc [typeOf ((undefined :: IO a -> a) action)]

maybeTc :: TyCon
maybeTc = mkTyCon "Maybe"

instance Typeable a => Typeable (Maybe a) where
  typeOf mb = mkAppTy maybeTc [typeOf ((undefined :: Maybe a -> a) mb)]

SIMPLE_INSTANCE_TYPEABLE(Ordering,orderingTc,"Ordering")
\end{code}

FFI typeable instances

\begin{code}
ptrTc :: TyCon
ptrTc = mkTyCon "Ptr"

instance Typeable a => Typeable (Ptr a) where
  typeOf _ = mkAppTy ptrTc [typeOf (undefined :: a)]

foreignPtrTc :: TyCon
foreignPtrTc = mkTyCon "ForeignPtr"

instance Typeable a => Typeable (ForeignPtr a) where
  typeOf _ = mkAppTy foreignPtrTc [typeOf (undefined :: a)]

SIMPLE_INSTANCE_TYPEABLE(Int8,int8Tc, "Int8" )
SIMPLE_INSTANCE_TYPEABLE(Int16,int16Tc,"Int16")
SIMPLE_INSTANCE_TYPEABLE(Int32,int32Tc,"Int32")
SIMPLE_INSTANCE_TYPEABLE(Int64,int64Tc,"Int64")

SIMPLE_INSTANCE_TYPEABLE(Word8,word8Tc, "Word8" )
SIMPLE_INSTANCE_TYPEABLE(Word16,word16Tc,"Word16")
SIMPLE_INSTANCE_TYPEABLE(Word32,word32Tc,"Word32")
SIMPLE_INSTANCE_TYPEABLE(Word64,word64Tc,"Word64")

SIMPLE_INSTANCE_TYPEABLE(TyCon,tyconTc,"TyCon")
SIMPLE_INSTANCE_TYPEABLE(TypeRep,typeRepTc,"TypeRep")
SIMPLE_INSTANCE_TYPEABLE(Dynamic,dynamicTc,"Dynamic")

SIMPLE_INSTANCE_TYPEABLE(ForeignObj,foreignObjTc,"ForeignObj") -- DEPRECATED
SIMPLE_INSTANCE_TYPEABLE(Addr,addrTc,"Addr")                   -- DEPRECATED

-- CTypes
SIMPLE_INSTANCE_TYPEABLE(CChar,cCharTc,"CChar")
SIMPLE_INSTANCE_TYPEABLE(CSChar,cSCharTc,"CSChar")
SIMPLE_INSTANCE_TYPEABLE(CUChar,cUCharTc,"CUChar")

SIMPLE_INSTANCE_TYPEABLE(CShort,cShortTc,"CShort")
SIMPLE_INSTANCE_TYPEABLE(CUShort,cUShortTc,"CUShort")

SIMPLE_INSTANCE_TYPEABLE(CInt,cIntTc,"CInt")
SIMPLE_INSTANCE_TYPEABLE(CUInt,cUIntTc,"CUInt")

SIMPLE_INSTANCE_TYPEABLE(CLong,cLongTc,"CLong")
SIMPLE_INSTANCE_TYPEABLE(CULong,cULongTc,"CULong")

SIMPLE_INSTANCE_TYPEABLE(CLLong,cLLongTc,"CLLong")
SIMPLE_INSTANCE_TYPEABLE(CULLong,cULLongTc,"CULLong")

SIMPLE_INSTANCE_TYPEABLE(CFloat,cFloatTc,"CFloat")
SIMPLE_INSTANCE_TYPEABLE(CDouble,cDoubleTc,"CDouble")
SIMPLE_INSTANCE_TYPEABLE(CLDouble,cLDoubleTc,"CLDouble")

-- CTypesISO
SIMPLE_INSTANCE_TYPEABLE(CPtrdiff,cPtrdiffTc,"CPtrdiff")
SIMPLE_INSTANCE_TYPEABLE(CSize,cSizeTc,"CSize")
SIMPLE_INSTANCE_TYPEABLE(CWchar,cWcharTc,"CWchar")
SIMPLE_INSTANCE_TYPEABLE(CSigAtomic,cSigAtomicTc,"CSigAtomic")
SIMPLE_INSTANCE_TYPEABLE(CClock,cClockTc,"CClock")
SIMPLE_INSTANCE_TYPEABLE(CTime,cTimeTc,"CTime")
\end{code}

GHC extension lib types

\begin{code}
iORefTc :: TyCon
iORefTc = mkTyCon "IORef"

instance Typeable a => Typeable (IORef a) where
  typeOf ref = mkAppTy iORefTc [typeOf ((undefined :: IORef a -> a) ref)]

sTTc :: TyCon
sTTc = mkTyCon "ST"

instance (Typeable a, Typeable b) => Typeable (ST a b) where
  typeOf st = mkAppTy sTTc [typeOf ((undefined :: ST a b -> a) st),
			    typeOf ((undefined :: ST a b -> b) st)]

stablePtrTc :: TyCon
stablePtrTc = mkTyCon "StablePtr"

instance Typeable a => Typeable (StablePtr a) where
  typeOf sp = mkAppTy stablePtrTc [typeOf ((undefined :: StablePtr a -> a) sp)]

byteArrayTc :: TyCon
byteArrayTc = mkTyCon "ByteArray"

instance Typeable a => Typeable (ByteArray a) where
  typeOf sp = mkAppTy byteArrayTc [typeOf ((undefined :: ByteArray a -> a) sp)]

iOArrayTc :: TyCon
iOArrayTc = mkTyCon "IOArray"

instance (Typeable a, Typeable b) => Typeable (IOArray a b) where
  typeOf a = mkAppTy iOArrayTc [typeOf ((undefined :: IOArray a b -> a) a),
				typeOf ((undefined :: IOArray a b -> b) a)]

iOUArrayTc :: TyCon
iOUArrayTc = mkTyCon "IOUArray"

instance (Typeable a, Typeable b) => Typeable (IOUArray a b) where
  typeOf a = mkAppTy iOUArrayTc [typeOf ((undefined :: IOUArray a b -> a) a),
				 typeOf ((undefined :: IOUArray a b -> b) a)]

mutableByteArrayTc :: TyCon
mutableByteArrayTc = mkTyCon "MutableByteArray"

instance (Typeable a, Typeable b) => Typeable (MutableByteArray a b) where
  typeOf a = mkAppTy mutableByteArrayTc [typeOf ((undefined :: MutableByteArray a b -> a) a),
					 typeOf ((undefined :: MutableByteArray a b -> b) a)]

packedStringTc :: TyCon
packedStringTc = mkTyCon "PackedString"

instance Typeable PackedString where
  typeOf _ = mkAppTy packedStringTc []

sTArrayTc :: TyCon
sTArrayTc = mkTyCon "STArray"

instance (Typeable a, Typeable b, Typeable c) => Typeable (STArray a b c) where
  typeOf a = mkAppTy sTArrayTc [typeOf ((undefined :: STArray a b c -> a) a),
				typeOf ((undefined :: STArray a b c -> b) a),
				typeOf ((undefined :: STArray a b c -> c) a)]

sTUArrayTc :: TyCon
sTUArrayTc = mkTyCon "STUArray"

instance (Typeable a, Typeable b, Typeable c) => Typeable (STUArray a b c) where
  typeOf a = mkAppTy sTUArrayTc [typeOf ((undefined :: STUArray a b c -> a) a),
				 typeOf ((undefined :: STUArray a b c -> b) a),
				 typeOf ((undefined :: STUArray a b c -> c) a)]

stableNameTc :: TyCon
stableNameTc = mkTyCon "StableName"

instance Typeable a => Typeable (StableName a) where
  typeOf sp = mkAppTy stableNameTc [typeOf ((undefined :: StableName a -> a) sp)]

uArrayTc :: TyCon
uArrayTc = mkTyCon "UArray"

instance (Typeable a, Typeable b) => Typeable (UArray a b) where
  typeOf a = mkAppTy uArrayTc [typeOf ((undefined :: UArray a b -> a) a),
			       typeOf ((undefined :: UArray a b -> b) a)]

weakTc :: TyCon
weakTc = mkTyCon "Weak"

instance Typeable a => Typeable (Weak a) where
  typeOf sp = mkAppTy weakTc [typeOf ((undefined :: Weak a -> a) sp)]
\end{code}
