{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 2.7 (Vertex Specification) of the
OpenGL 1.2.1 specs.
-}

module GL_VertexSpec (
   Vertex1(..), Vertex2(..), Vertex3(..), Vertex4(..),
   Vertex(..), VertexV(..),

   TexCoord1(..), TexCoord2(..), TexCoord3(..), TexCoord4(..),
   TexCoord(..), TexCoordV(..),

   Normal3(..),
   Normal(..), NormalV(..),

   Color3(..), Color4(..),
   Color(..), ColorV(..),

   ColorIndex(..),
   Index(..), IndexV(..)
) where

import Foreign          (Ptr, Storable(..))

import GL_BasicTypes    (GLbyte, GLubyte, GLshort, GLushort, GLint, GLuint,
                         GLfloat, GLdouble,
                         peek1, peek2, peek3, peek4,
                         poke1, poke2, poke3, poke4)

---------------------------------------------------------------------------

newtype Vertex1 a = Vertex1 a   deriving (Eq,Ord)

instance Storable a => Storable (Vertex1 a) where
   sizeOf    ~(Vertex1 x) = sizeOf x
   alignment ~(Vertex1 x) = alignment x
   peek                   = peek1 Vertex1
   poke ptr   (Vertex1 x) = poke1 ptr x

data Vertex2 a = Vertex2 a a   deriving (Eq,Ord)

instance Storable a => Storable (Vertex2 a) where
   sizeOf    ~(Vertex2 x _) = 2 * sizeOf x
   alignment ~(Vertex2 x _) = alignment x
   peek                     = peek2 Vertex2
   poke ptr   (Vertex2 x y) = poke2 ptr x y

data Vertex3 a = Vertex3 a a a   deriving (Eq,Ord)

instance Storable a => Storable (Vertex3 a) where
   sizeOf    ~(Vertex3 x _ _) = 3 * sizeOf x
   alignment ~(Vertex3 x _ _) = alignment x
   peek                       = peek3 Vertex3
   poke ptr   (Vertex3 x y z) = poke3 ptr x y z

data Vertex4 a = Vertex4 a a a a   deriving (Eq,Ord)

instance Storable a => Storable (Vertex4 a) where
   sizeOf    ~(Vertex4 x _ _ _) = 4 * sizeOf x
   alignment ~(Vertex4 x _ _ _) = alignment x
   peek                         = peek4 Vertex4
   poke ptr   (Vertex4 x y z w) = poke4 ptr x y z w

---------------------------------------------------------------------------

class Vertex a where
   vertex :: a -> IO ()

instance Vertex (Vertex2 GLshort ) where
   vertex (Vertex2 x y) = glVertex2s x y

instance Vertex (Vertex2 GLint   ) where
   vertex (Vertex2 x y) = glVertex2i x y

instance Vertex (Vertex2 GLfloat ) where
   vertex (Vertex2 x y) = glVertex2f x y

instance Vertex (Vertex2 GLdouble) where
   vertex (Vertex2 x y) = glVertex2d x y

instance Vertex (Vertex3 GLshort ) where
   vertex (Vertex3 x y z) = glVertex3s x y z

instance Vertex (Vertex3 GLint   ) where
   vertex (Vertex3 x y z) = glVertex3i x y z

instance Vertex (Vertex3 GLfloat ) where
   vertex (Vertex3 x y z) = glVertex3f x y z

instance Vertex (Vertex3 GLdouble) where
   vertex (Vertex3 x y z) = glVertex3d x y z

instance Vertex (Vertex4 GLshort ) where
   vertex (Vertex4 x y z w) = glVertex4s x y z w

instance Vertex (Vertex4 GLint   ) where
   vertex (Vertex4 x y z w) = glVertex4i x y z w

instance Vertex (Vertex4 GLfloat ) where
   vertex (Vertex4 x y z w) = glVertex4f x y z w

instance Vertex (Vertex4 GLdouble) where
   vertex (Vertex4 x y z w) = glVertex4d x y z w

foreign import "glVertex2s" unsafe glVertex2s :: GLshort  -> GLshort                          -> IO ()
foreign import "glVertex2i" unsafe glVertex2i :: GLint    -> GLint                            -> IO ()
foreign import "glVertex2f" unsafe glVertex2f :: GLfloat  -> GLfloat                          -> IO ()
foreign import "glVertex2d" unsafe glVertex2d :: GLdouble -> GLdouble                         -> IO ()
foreign import "glVertex3s" unsafe glVertex3s :: GLshort  -> GLshort  -> GLshort              -> IO ()
foreign import "glVertex3i" unsafe glVertex3i :: GLint    -> GLint    -> GLint                -> IO ()
foreign import "glVertex3f" unsafe glVertex3f :: GLfloat  -> GLfloat  -> GLfloat              -> IO ()
foreign import "glVertex3d" unsafe glVertex3d :: GLdouble -> GLdouble -> GLdouble             -> IO ()
foreign import "glVertex4s" unsafe glVertex4s :: GLshort  -> GLshort  -> GLshort  -> GLshort  -> IO ()
foreign import "glVertex4i" unsafe glVertex4i :: GLint    -> GLint    -> GLint    -> GLint    -> IO ()
foreign import "glVertex4f" unsafe glVertex4f :: GLfloat  -> GLfloat  -> GLfloat  -> GLfloat  -> IO ()
foreign import "glVertex4d" unsafe glVertex4d :: GLdouble -> GLdouble -> GLdouble -> GLdouble -> IO ()

---------------------------------------------------------------------------

class VertexV a where
   vertexV :: Ptr a -> IO ()

instance VertexV (Vertex2 GLshort ) where
   vertexV = glVertex2sv

instance VertexV (Vertex2 GLint   ) where
   vertexV = glVertex2iv

instance VertexV (Vertex2 GLfloat ) where
   vertexV = glVertex2fv

instance VertexV (Vertex2 GLdouble) where
   vertexV = glVertex2dv

instance VertexV (Vertex3 GLshort ) where
   vertexV = glVertex3sv

instance VertexV (Vertex3 GLint   ) where
   vertexV = glVertex3iv

instance VertexV (Vertex3 GLfloat ) where
   vertexV = glVertex3fv

instance VertexV (Vertex3 GLdouble) where
   vertexV = glVertex3dv

instance VertexV (Vertex4 GLshort ) where
   vertexV = glVertex4sv

instance VertexV (Vertex4 GLint   ) where
   vertexV = glVertex4iv

instance VertexV (Vertex4 GLfloat ) where
   vertexV = glVertex4fv

instance VertexV (Vertex4 GLdouble) where
   vertexV = glVertex4dv

foreign import "glVertex2sv" unsafe glVertex2sv :: Ptr (Vertex2 GLshort ) -> IO ()
foreign import "glVertex2iv" unsafe glVertex2iv :: Ptr (Vertex2 GLint   ) -> IO ()
foreign import "glVertex2fv" unsafe glVertex2fv :: Ptr (Vertex2 GLfloat ) -> IO ()
foreign import "glVertex2dv" unsafe glVertex2dv :: Ptr (Vertex2 GLdouble) -> IO ()

foreign import "glVertex3sv" unsafe glVertex3sv :: Ptr (Vertex3 GLshort ) -> IO ()
foreign import "glVertex3iv" unsafe glVertex3iv :: Ptr (Vertex3 GLint   ) -> IO ()
foreign import "glVertex3fv" unsafe glVertex3fv :: Ptr (Vertex3 GLfloat ) -> IO ()
foreign import "glVertex3dv" unsafe glVertex3dv :: Ptr (Vertex3 GLdouble) -> IO ()

foreign import "glVertex4sv" unsafe glVertex4sv :: Ptr (Vertex4 GLshort ) -> IO ()
foreign import "glVertex4iv" unsafe glVertex4iv :: Ptr (Vertex4 GLint   ) -> IO ()
foreign import "glVertex4fv" unsafe glVertex4fv :: Ptr (Vertex4 GLfloat ) -> IO ()
foreign import "glVertex4dv" unsafe glVertex4dv :: Ptr (Vertex4 GLdouble) -> IO ()

---------------------------------------------------------------------------

newtype TexCoord1 a = TexCoord1 a   deriving (Eq,Ord)

instance Storable a => Storable (TexCoord1 a) where
   sizeOf    ~(TexCoord1 x) = sizeOf x
   alignment ~(TexCoord1 x) = alignment x
   peek                     = peek1 TexCoord1
   poke ptr   (TexCoord1 x) = poke1 ptr x

data TexCoord2 a = TexCoord2 a a   deriving (Eq,Ord)

instance Storable a => Storable (TexCoord2 a) where
   sizeOf    ~(TexCoord2 x _) = 2 * sizeOf x
   alignment ~(TexCoord2 x _) = alignment x
   peek                       = peek2 TexCoord2
   poke ptr   (TexCoord2 x y) = poke2 ptr x y

data TexCoord3 a = TexCoord3 a a a   deriving (Eq,Ord)

instance Storable a => Storable (TexCoord3 a) where
   sizeOf    ~(TexCoord3 x _ _) = 3 * sizeOf x
   alignment ~(TexCoord3 x _ _) = alignment x
   peek                         = peek3 TexCoord3
   poke ptr   (TexCoord3 x y z) = poke3 ptr x y z

data TexCoord4 a = TexCoord4 a a a a   deriving (Eq,Ord)

instance Storable a => Storable (TexCoord4 a) where
   sizeOf    ~(TexCoord4 x _ _ _) = 4 * sizeOf x
   alignment ~(TexCoord4 x _ _ _) = alignment x
   peek                           = peek4 TexCoord4
   poke ptr   (TexCoord4 x y z w) = poke4 ptr x y z w

---------------------------------------------------------------------------

class TexCoord a where
   texCoord :: a -> IO ()

instance TexCoord (TexCoord1 GLshort ) where
   texCoord (TexCoord1 s) = glTexCoord1s s

instance TexCoord (TexCoord1 GLint   ) where
   texCoord (TexCoord1 s) = glTexCoord1i s

instance TexCoord (TexCoord1 GLfloat ) where
   texCoord (TexCoord1 s) = glTexCoord1f s

instance TexCoord (TexCoord1 GLdouble) where
   texCoord (TexCoord1 s) = glTexCoord1d s

instance TexCoord (TexCoord2 GLshort ) where
   texCoord (TexCoord2 s t) = glTexCoord2s s t

instance TexCoord (TexCoord2 GLint   ) where
   texCoord (TexCoord2 s t) = glTexCoord2i s t

instance TexCoord (TexCoord2 GLfloat ) where
   texCoord (TexCoord2 s t) = glTexCoord2f s t

instance TexCoord (TexCoord2 GLdouble) where
   texCoord (TexCoord2 s t) = glTexCoord2d s t

instance TexCoord (TexCoord3 GLshort ) where
   texCoord (TexCoord3 s t r) = glTexCoord3s s t r

instance TexCoord (TexCoord3 GLint   ) where
   texCoord (TexCoord3 s t r) = glTexCoord3i s t r

instance TexCoord (TexCoord3 GLfloat ) where
   texCoord (TexCoord3 s t r) = glTexCoord3f s t r

instance TexCoord (TexCoord3 GLdouble) where
   texCoord (TexCoord3 s t r) = glTexCoord3d s t r

instance TexCoord (TexCoord4 GLshort ) where
   texCoord (TexCoord4 s t r q) = glTexCoord4s s t r q

instance TexCoord (TexCoord4 GLint   ) where
   texCoord (TexCoord4 s t r q) = glTexCoord4i s t r q

instance TexCoord (TexCoord4 GLfloat ) where
   texCoord (TexCoord4 s t r q) = glTexCoord4f s t r q

instance TexCoord (TexCoord4 GLdouble) where
   texCoord (TexCoord4 s t r q) = glTexCoord4d s t r q

foreign import "glTexCoord1s" unsafe glTexCoord1s :: GLshort                                      -> IO ()
foreign import "glTexCoord1i" unsafe glTexCoord1i :: GLint                                        -> IO ()
foreign import "glTexCoord1f" unsafe glTexCoord1f :: GLfloat                                      -> IO ()
foreign import "glTexCoord1d" unsafe glTexCoord1d :: GLdouble                                     -> IO ()
foreign import "glTexCoord2s" unsafe glTexCoord2s :: GLshort  -> GLshort                          -> IO ()
foreign import "glTexCoord2i" unsafe glTexCoord2i :: GLint    -> GLint                            -> IO ()
foreign import "glTexCoord2f" unsafe glTexCoord2f :: GLfloat  -> GLfloat                          -> IO ()
foreign import "glTexCoord2d" unsafe glTexCoord2d :: GLdouble -> GLdouble                         -> IO ()
foreign import "glTexCoord3s" unsafe glTexCoord3s :: GLshort  -> GLshort  -> GLshort              -> IO ()
foreign import "glTexCoord3i" unsafe glTexCoord3i :: GLint    -> GLint    -> GLint                -> IO ()
foreign import "glTexCoord3f" unsafe glTexCoord3f :: GLfloat  -> GLfloat  -> GLfloat              -> IO ()
foreign import "glTexCoord3d" unsafe glTexCoord3d :: GLdouble -> GLdouble -> GLdouble             -> IO ()
foreign import "glTexCoord4s" unsafe glTexCoord4s :: GLshort  -> GLshort  -> GLshort  -> GLshort  -> IO ()
foreign import "glTexCoord4i" unsafe glTexCoord4i :: GLint    -> GLint    -> GLint    -> GLint    -> IO ()
foreign import "glTexCoord4f" unsafe glTexCoord4f :: GLfloat  -> GLfloat  -> GLfloat  -> GLfloat  -> IO ()
foreign import "glTexCoord4d" unsafe glTexCoord4d :: GLdouble -> GLdouble -> GLdouble -> GLdouble -> IO ()

---------------------------------------------------------------------------

class TexCoordV a where
   texCoordV :: Ptr a -> IO ()

instance TexCoordV (TexCoord1 GLshort ) where
   texCoordV = glTexCoord1sv

instance TexCoordV (TexCoord1 GLint   ) where
   texCoordV = glTexCoord1iv

instance TexCoordV (TexCoord1 GLfloat ) where
   texCoordV = glTexCoord1fv

instance TexCoordV (TexCoord1 GLdouble) where
   texCoordV = glTexCoord1dv

instance TexCoordV (TexCoord2 GLshort ) where
   texCoordV = glTexCoord2sv

instance TexCoordV (TexCoord2 GLint   ) where
   texCoordV = glTexCoord2iv

instance TexCoordV (TexCoord2 GLfloat ) where
   texCoordV = glTexCoord2fv

instance TexCoordV (TexCoord2 GLdouble) where
   texCoordV = glTexCoord2dv

instance TexCoordV (TexCoord3 GLshort ) where
   texCoordV = glTexCoord3sv

instance TexCoordV (TexCoord3 GLint   ) where
   texCoordV = glTexCoord3iv

instance TexCoordV (TexCoord3 GLfloat ) where
   texCoordV = glTexCoord3fv

instance TexCoordV (TexCoord3 GLdouble) where
   texCoordV = glTexCoord3dv

instance TexCoordV (TexCoord4 GLshort ) where
   texCoordV = glTexCoord4sv

instance TexCoordV (TexCoord4 GLint   ) where
   texCoordV = glTexCoord4iv

instance TexCoordV (TexCoord4 GLfloat ) where
   texCoordV = glTexCoord4fv

instance TexCoordV (TexCoord4 GLdouble) where
   texCoordV = glTexCoord4dv

foreign import "glTexCoord1sv" unsafe glTexCoord1sv :: Ptr (TexCoord1 GLshort ) -> IO ()
foreign import "glTexCoord1iv" unsafe glTexCoord1iv :: Ptr (TexCoord1 GLint   ) -> IO ()
foreign import "glTexCoord1fv" unsafe glTexCoord1fv :: Ptr (TexCoord1 GLfloat ) -> IO ()
foreign import "glTexCoord1dv" unsafe glTexCoord1dv :: Ptr (TexCoord1 GLdouble) -> IO ()
foreign import "glTexCoord2sv" unsafe glTexCoord2sv :: Ptr (TexCoord2 GLshort ) -> IO ()
foreign import "glTexCoord2iv" unsafe glTexCoord2iv :: Ptr (TexCoord2 GLint   ) -> IO ()
foreign import "glTexCoord2fv" unsafe glTexCoord2fv :: Ptr (TexCoord2 GLfloat ) -> IO ()
foreign import "glTexCoord2dv" unsafe glTexCoord2dv :: Ptr (TexCoord2 GLdouble) -> IO ()
foreign import "glTexCoord3sv" unsafe glTexCoord3sv :: Ptr (TexCoord3 GLshort ) -> IO ()
foreign import "glTexCoord3iv" unsafe glTexCoord3iv :: Ptr (TexCoord3 GLint   ) -> IO ()
foreign import "glTexCoord3fv" unsafe glTexCoord3fv :: Ptr (TexCoord3 GLfloat ) -> IO ()
foreign import "glTexCoord3dv" unsafe glTexCoord3dv :: Ptr (TexCoord3 GLdouble) -> IO ()
foreign import "glTexCoord4sv" unsafe glTexCoord4sv :: Ptr (TexCoord4 GLshort ) -> IO ()
foreign import "glTexCoord4iv" unsafe glTexCoord4iv :: Ptr (TexCoord4 GLint   ) -> IO ()
foreign import "glTexCoord4fv" unsafe glTexCoord4fv :: Ptr (TexCoord4 GLfloat ) -> IO ()
foreign import "glTexCoord4dv" unsafe glTexCoord4dv :: Ptr (TexCoord4 GLdouble) -> IO ()

---------------------------------------------------------------------------

data Normal3 a = Normal3 a a a   deriving (Eq,Ord)

instance Storable a => Storable (Normal3 a) where
   sizeOf    ~(Normal3 x _ _) = 3 * sizeOf x
   alignment ~(Normal3 x _ _) = alignment x
   peek                       = peek3 Normal3
   poke ptr   (Normal3 x y z) = poke3 ptr x y z

---------------------------------------------------------------------------

class Normal a where
   normal :: a -> IO ()

instance Normal (Normal3 GLbyte  ) where
   normal (Normal3 x y z) = glNormal3b x y z

instance Normal (Normal3 GLshort ) where
   normal (Normal3 x y z) = glNormal3s x y z

instance Normal (Normal3 GLint   ) where
   normal (Normal3 x y z) = glNormal3i x y z

instance Normal (Normal3 GLfloat ) where
   normal (Normal3 x y z) = glNormal3f x y z

instance Normal (Normal3 GLdouble) where
   normal (Normal3 x y z) = glNormal3d x y z

foreign import "glNormal3b" unsafe glNormal3b :: GLbyte   -> GLbyte   -> GLbyte   -> IO ()
foreign import "glNormal3s" unsafe glNormal3s :: GLshort  -> GLshort  -> GLshort  -> IO ()
foreign import "glNormal3i" unsafe glNormal3i :: GLint    -> GLint    -> GLint    -> IO ()
foreign import "glNormal3f" unsafe glNormal3f :: GLfloat  -> GLfloat  -> GLfloat  -> IO ()
foreign import "glNormal3d" unsafe glNormal3d :: GLdouble -> GLdouble -> GLdouble -> IO ()

---------------------------------------------------------------------------

class NormalV a where
   normalV :: Ptr a -> IO ()

instance NormalV (Normal3 GLbyte  ) where
   normalV = glNormal3bv

instance NormalV (Normal3 GLshort ) where
   normalV = glNormal3sv

instance NormalV (Normal3 GLint   ) where
   normalV = glNormal3iv

instance NormalV (Normal3 GLfloat ) where
   normalV = glNormal3fv

instance NormalV (Normal3 GLdouble) where
   normalV = glNormal3dv

foreign import "glNormal3bv" unsafe glNormal3bv :: Ptr (Normal3 GLbyte  ) -> IO ()
foreign import "glNormal3sv" unsafe glNormal3sv :: Ptr (Normal3 GLshort ) -> IO ()
foreign import "glNormal3iv" unsafe glNormal3iv :: Ptr (Normal3 GLint   ) -> IO ()
foreign import "glNormal3fv" unsafe glNormal3fv :: Ptr (Normal3 GLfloat ) -> IO ()
foreign import "glNormal3dv" unsafe glNormal3dv :: Ptr (Normal3 GLdouble) -> IO ()

---------------------------------------------------------------------------

data Color3 a = Color3 a a a   deriving (Eq,Ord)

instance Storable a => Storable (Color3 a) where
   sizeOf    ~(Color3 r _ _) = 3 * sizeOf r
   alignment ~(Color3 r _ _) = alignment r
   peek                      = peek3 Color3
   poke ptr   (Color3 r g b) = poke3 ptr r g b

data Color4 a = Color4 a a a a   deriving (Eq,Ord)

instance Storable a => Storable (Color4 a) where
   sizeOf    ~(Color4 r _ _ _) = 4 * sizeOf r
   alignment ~(Color4 r _ _ _) = alignment r
   peek                        = peek4 Color4
   poke ptr   (Color4 r g b a) = poke4 ptr r g b a

---------------------------------------------------------------------------

class Color a where
   color :: a -> IO ()

instance Color (Color3 GLbyte  ) where
   color (Color3 r g b) = glColor3b r g b

instance Color (Color3 GLshort ) where
   color (Color3 r g b) = glColor3s r g b

instance Color (Color3 GLint   ) where
   color (Color3 r g b) = glColor3i r g b

instance Color (Color3 GLfloat ) where
   color (Color3 r g b) = glColor3f r g b

instance Color (Color3 GLdouble) where
   color (Color3 r g b) = glColor3d r g b

instance Color (Color3 GLubyte ) where
   color (Color3 r g b) = glColor3ub r g b

instance Color (Color3 GLushort) where
   color (Color3 r g b) = glColor3us r g b

instance Color (Color3 GLuint  ) where
   color (Color3 r g b) = glColor3ui r g b

instance Color (Color4 GLbyte  ) where
   color (Color4 r g b a) = glColor4b r g b a

instance Color (Color4 GLshort ) where
   color (Color4 r g b a) = glColor4s r g b a

instance Color (Color4 GLint   ) where
   color (Color4 r g b a) = glColor4i r g b a

instance Color (Color4 GLfloat ) where
   color (Color4 r g b a) = glColor4f r g b a

instance Color (Color4 GLdouble) where
   color (Color4 r g b a) = glColor4d r g b a

instance Color (Color4 GLubyte ) where
   color (Color4 r g b a) = glColor4ub r g b a

instance Color (Color4 GLushort) where
   color (Color4 r g b a) = glColor4us r g b a

instance Color (Color4 GLuint  ) where
   color (Color4 r g b a) = glColor4ui r g b a

foreign import "glColor3b"  unsafe glColor3b  :: GLbyte   -> GLbyte   -> GLbyte               -> IO ()
foreign import "glColor3s"  unsafe glColor3s  :: GLshort  -> GLshort  -> GLshort              -> IO ()
foreign import "glColor3i"  unsafe glColor3i  :: GLint    -> GLint    -> GLint                -> IO ()
foreign import "glColor3f"  unsafe glColor3f  :: GLfloat  -> GLfloat  -> GLfloat              -> IO ()
foreign import "glColor3d"  unsafe glColor3d  :: GLdouble -> GLdouble -> GLdouble             -> IO ()
foreign import "glColor3ub" unsafe glColor3ub :: GLubyte  -> GLubyte  -> GLubyte              -> IO ()
foreign import "glColor3us" unsafe glColor3us :: GLushort -> GLushort -> GLushort             -> IO ()
foreign import "glColor3ui" unsafe glColor3ui :: GLuint   -> GLuint   -> GLuint               -> IO ()
foreign import "glColor4b"  unsafe glColor4b  :: GLbyte   -> GLbyte   -> GLbyte   -> GLbyte   -> IO ()
foreign import "glColor4s"  unsafe glColor4s  :: GLshort  -> GLshort  -> GLshort  -> GLshort  -> IO ()
foreign import "glColor4i"  unsafe glColor4i  :: GLint    -> GLint    -> GLint    -> GLint    -> IO ()
foreign import "glColor4f"  unsafe glColor4f  :: GLfloat  -> GLfloat  -> GLfloat  -> GLfloat  -> IO ()
foreign import "glColor4d"  unsafe glColor4d  :: GLdouble -> GLdouble -> GLdouble -> GLdouble -> IO ()
foreign import "glColor4ub" unsafe glColor4ub :: GLubyte  -> GLubyte  -> GLubyte  -> GLubyte  -> IO ()
foreign import "glColor4us" unsafe glColor4us :: GLushort -> GLushort -> GLushort -> GLushort -> IO ()
foreign import "glColor4ui" unsafe glColor4ui :: GLuint   -> GLuint   -> GLuint   -> GLuint   -> IO ()

---------------------------------------------------------------------------

class ColorV a where
   colorV :: Ptr a -> IO ()

instance ColorV (Color3 GLbyte  ) where
   colorV = glColor3bv

instance ColorV (Color3 GLshort ) where
   colorV = glColor3sv

instance ColorV (Color3 GLint   ) where
   colorV = glColor3iv

instance ColorV (Color3 GLfloat ) where
   colorV = glColor3fv

instance ColorV (Color3 GLdouble) where
   colorV = glColor3dv

instance ColorV (Color3 GLubyte ) where
   colorV = glColor3ubv

instance ColorV (Color3 GLushort) where
   colorV = glColor3usv

instance ColorV (Color3 GLuint  ) where
   colorV = glColor3uiv

instance ColorV (Color4 GLbyte  ) where
   colorV = glColor4bv

instance ColorV (Color4 GLshort ) where
   colorV = glColor4sv

instance ColorV (Color4 GLint   ) where
   colorV = glColor4iv

instance ColorV (Color4 GLfloat ) where
   colorV = glColor4fv

instance ColorV (Color4 GLdouble) where
   colorV = glColor4dv

instance ColorV (Color4 GLubyte ) where
   colorV = glColor4ubv

instance ColorV (Color4 GLushort) where
   colorV = glColor4usv

instance ColorV (Color4 GLuint  ) where
   colorV = glColor4uiv

foreign import "glColor3bv"  unsafe glColor3bv  :: Ptr (Color3 GLbyte  ) -> IO ()
foreign import "glColor3sv"  unsafe glColor3sv  :: Ptr (Color3 GLshort ) -> IO ()
foreign import "glColor3iv"  unsafe glColor3iv  :: Ptr (Color3 GLint   ) -> IO ()
foreign import "glColor3fv"  unsafe glColor3fv  :: Ptr (Color3 GLfloat ) -> IO ()
foreign import "glColor3dv"  unsafe glColor3dv  :: Ptr (Color3 GLdouble) -> IO ()
foreign import "glColor3ubv" unsafe glColor3ubv :: Ptr (Color3 GLubyte ) -> IO ()
foreign import "glColor3usv" unsafe glColor3usv :: Ptr (Color3 GLushort) -> IO ()
foreign import "glColor3uiv" unsafe glColor3uiv :: Ptr (Color3 GLuint  ) -> IO ()
foreign import "glColor4bv"  unsafe glColor4bv  :: Ptr (Color4 GLbyte  ) -> IO ()
foreign import "glColor4sv"  unsafe glColor4sv  :: Ptr (Color4 GLshort ) -> IO ()
foreign import "glColor4iv"  unsafe glColor4iv  :: Ptr (Color4 GLint   ) -> IO ()
foreign import "glColor4fv"  unsafe glColor4fv  :: Ptr (Color4 GLfloat ) -> IO ()
foreign import "glColor4dv"  unsafe glColor4dv  :: Ptr (Color4 GLdouble) -> IO ()
foreign import "glColor4ubv" unsafe glColor4ubv :: Ptr (Color4 GLubyte ) -> IO ()
foreign import "glColor4usv" unsafe glColor4usv :: Ptr (Color4 GLushort) -> IO ()
foreign import "glColor4uiv" unsafe glColor4uiv :: Ptr (Color4 GLuint  ) -> IO ()

---------------------------------------------------------------------------

newtype ColorIndex a = ColorIndex a   deriving (Eq,Ord)

instance Storable a => Storable (ColorIndex a) where
   sizeOf    ~(ColorIndex i) = sizeOf i
   alignment ~(ColorIndex i) = alignment i
   peek                      = peek1 ColorIndex
   poke ptr   (ColorIndex i) = poke1 ptr i

---------------------------------------------------------------------------

-- Collision with Prelude.index
class Index a where
   index :: a -> IO ()

instance Index (ColorIndex GLshort ) where
   index = glIndexs

instance Index (ColorIndex GLint   ) where
   index = glIndexi

instance Index (ColorIndex GLfloat ) where
   index = glIndexf

instance Index (ColorIndex GLdouble) where
   index = glIndexd

instance Index (ColorIndex GLubyte ) where
   index = glIndexub

foreign import "glIndexs"  unsafe glIndexs  :: ColorIndex GLshort  -> IO ()
foreign import "glIndexi"  unsafe glIndexi  :: ColorIndex GLint    -> IO ()
foreign import "glIndexf"  unsafe glIndexf  :: ColorIndex GLfloat  -> IO ()
foreign import "glIndexd"  unsafe glIndexd  :: ColorIndex GLdouble -> IO ()
foreign import "glIndexub" unsafe glIndexub :: ColorIndex GLubyte  -> IO ()

---------------------------------------------------------------------------

class IndexV a where
   indexV :: Ptr a -> IO ()

instance IndexV (ColorIndex GLshort ) where
   indexV = glIndexsv

instance IndexV (ColorIndex GLint   ) where
   indexV = glIndexiv

instance IndexV (ColorIndex GLfloat ) where
   indexV = glIndexfv

instance IndexV (ColorIndex GLdouble) where
   indexV = glIndexdv

instance IndexV (ColorIndex GLubyte ) where
   indexV = glIndexubv

foreign import "glIndexsv"  unsafe glIndexsv  :: Ptr (ColorIndex GLshort ) -> IO ()
foreign import "glIndexiv"  unsafe glIndexiv  :: Ptr (ColorIndex GLint   ) -> IO ()
foreign import "glIndexfv"  unsafe glIndexfv  :: Ptr (ColorIndex GLfloat ) -> IO ()
foreign import "glIndexdv"  unsafe glIndexdv  :: Ptr (ColorIndex GLdouble) -> IO ()
foreign import "glIndexubv" unsafe glIndexubv :: Ptr (ColorIndex GLubyte ) -> IO ()
