{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 2.13 (Colors and Coloring) of the
OpenGL 1.2.1 specs.
-}

module GL_Colors (
   Lighting(..), FrontOrientation(..),
   unmarshalFrontOrientation,              -- internal use only
   frontFace,
   Face(..),
   marshalFace, unmarshalFace,             -- internal use only
   LightType(..),
   marshalLightType, unmarshalLightType,   -- internal use only
   MaterialParam(..), material,
   Light(..), Attenuation,
   marshalAttenuation,                     -- internal use only
   LightParam(..), light,
   ColorControl(..),                       -- @GL_1_2@
   unmarshalColorControl,                  -- @GL_1_2@, internal use only
   LightModel(..), lightModel,
   ColorMaterial(..), colorMaterial,
   ShadeModel(..),
   unmarshalShadeModel,                    -- internal use only
   shadeModel
) where

import Foreign          ( Ptr, withObject, withArray )
import Maybe            ( fromMaybe )

import GL_Constants     ( gl_LIGHTING, gl_CCW, gl_CW, gl_FRONT, gl_BACK,
                          gl_FRONT_AND_BACK, gl_AMBIENT, gl_DIFFUSE,
                          gl_AMBIENT_AND_DIFFUSE, gl_SPECULAR, gl_EMISSION,
                          gl_SHININESS, gl_COLOR_INDEXES, gl_LIGHT0, gl_POSITION,
                          gl_SPOT_DIRECTION, gl_SPOT_EXPONENT, gl_SPOT_CUTOFF,
                          gl_CONSTANT_ATTENUATION, gl_LINEAR_ATTENUATION,
                          gl_QUADRATIC_ATTENUATION, gl_SINGLE_COLOR,
                          gl_SEPARATE_SPECULAR_COLOR, gl_LIGHT_MODEL_AMBIENT,
                          gl_LIGHT_MODEL_LOCAL_VIEWER, gl_LIGHT_MODEL_TWO_SIDE,
                          gl_LIGHT_MODEL_COLOR_CONTROL, gl_COLOR_MATERIAL,
                          gl_FLAT, gl_SMOOTH )
import GL_BasicTypes    ( GLenum, GLint, boolToGLint, GLfloat, Capability(..) )
import GL_VertexSpec    ( Vertex4, Normal3, Color4 )

---------------------------------------------------------------------------
-- Section 2.13.1 (Lighting)

data Lighting =
     Lighting
     deriving (Eq,Ord)

instance Capability Lighting where
   marshalCapability Lighting = gl_LIGHTING

data FrontOrientation =
     Ccw
   | Cw
   deriving (Eq,Ord)

marshalFrontOrientation :: FrontOrientation -> GLenum
marshalFrontOrientation Ccw = gl_CCW
marshalFrontOrientation Cw  = gl_CW

unmarshalFrontOrientation :: GLenum -> FrontOrientation
unmarshalFrontOrientation frontOrientation
   | frontOrientation == gl_CCW = Ccw
   | frontOrientation == gl_CW  = Cw
   | otherwise                  = error "unmarshalFrontOrientation"

frontFace :: FrontOrientation -> IO ()
frontFace = glFrontFace . marshalFrontOrientation

foreign import "glFrontFace" unsafe glFrontFace :: GLenum -> IO ()

---------------------------------------------------------------------------
-- Section 2.13.2 (Lighting Parameter Specification)

-- GL_FRONT:          Collision with GL_Framebuffer.DrawBuffer (resolved there)
-- GL_BACK:           Collision with GL_Framebuffer.DrawBuffer (resolved there)
-- GL_FRONT_AND_BACK: Collision with GL_Framebuffer.DrawBuffer (resolved there)
data Face =
     Front
   | Back
   | FrontAndBack
   deriving (Eq,Ord)

marshalFace :: Face -> GLenum
marshalFace Front        = gl_FRONT
marshalFace Back         = gl_BACK
marshalFace FrontAndBack = gl_FRONT_AND_BACK

unmarshalFace :: GLenum -> Face
unmarshalFace face
   | face == gl_FRONT          = Front
   | face == gl_BACK           = Back
   | face == gl_FRONT_AND_BACK = FrontAndBack
   | otherwise                 = error "unmarshalFace"

data LightType =
     Ambient
   | Diffuse
   | AmbientAndDiffuse
   | Specular
   | Emission
   deriving (Eq,Ord)

marshalLightType :: LightType -> GLenum
marshalLightType Ambient           = gl_AMBIENT
marshalLightType Diffuse           = gl_DIFFUSE
marshalLightType AmbientAndDiffuse = gl_AMBIENT_AND_DIFFUSE
marshalLightType Specular          = gl_SPECULAR
marshalLightType Emission          = gl_EMISSION

unmarshalLightType :: GLenum -> LightType
unmarshalLightType lightType
   | lightType == gl_AMBIENT             = Ambient
   | lightType == gl_DIFFUSE             = Diffuse
   | lightType == gl_AMBIENT_AND_DIFFUSE = AmbientAndDiffuse
   | lightType == gl_SPECULAR            = Specular
   | lightType == gl_EMISSION            = Emission
   | otherwise                           = error "unmarshalLightType"

data MaterialParam =
     MaterialColor LightType (Color4 GLfloat)
   | Shininess GLfloat
   | ColorIndexes GLint GLint GLint
   deriving (Eq,Ord)

material :: Face -> MaterialParam -> IO ()
material face (MaterialColor lightType col) =
   withObject col $ glMaterialfv (marshalFace face) (marshalLightType lightType)
material face (Shininess s) =
   glMaterialf (marshalFace face) gl_SHININESS s
material face (ColorIndexes a d s) =
   withArray [a, d, s] $ glMaterialiv (marshalFace face) gl_COLOR_INDEXES

foreign import "glMaterialf"  unsafe glMaterialf  :: GLenum -> GLenum -> GLfloat              -> IO ()
foreign import "glMaterialiv" unsafe glMaterialiv :: GLenum -> GLenum -> Ptr GLint            -> IO ()
foreign import "glMaterialfv" unsafe glMaterialfv :: GLenum -> GLenum -> Ptr (Color4 GLfloat) -> IO ()

newtype Light = Light Int deriving (Eq,Ord)

instance Capability Light where
   marshalCapability (Light n) = gl_LIGHT0 + fromIntegral n   -- TODO: add check

data Attenuation =
     Constant
   | Linear
   | Quadratic
   deriving (Eq, Ord)

marshalAttenuation :: Attenuation -> GLenum
marshalAttenuation Constant  = gl_CONSTANT_ATTENUATION
marshalAttenuation Linear    = gl_LINEAR_ATTENUATION
marshalAttenuation Quadratic = gl_QUADRATIC_ATTENUATION

data LightParam =
     LightColor LightType (Color4 GLfloat)
   | Position (Vertex4 GLfloat)
   | SpotDirection (Normal3 GLfloat)
   | SpotExponent GLfloat
   | SpotCutoff (Maybe GLfloat)
   | Attenuation Attenuation GLfloat
   deriving (Eq,Ord)

light :: Light -> LightParam -> IO ()
-- NOTE: AmbientAndDiffuse is emulated below, Emission is still invalid
light l (LightColor AmbientAndDiffuse col) = do
   withObject col $ glLightC4 (marshalCapability l) (marshalLightType Ambient)
   withObject col $ glLightC4 (marshalCapability l) (marshalLightType Diffuse)
light l (LightColor lightType col) =
   withObject col $ glLightC4 (marshalCapability l) (marshalLightType lightType)
light l (Position pos) =
   withObject pos $ glLightV4 (marshalCapability l) gl_POSITION
light l (SpotDirection dir) =
   withObject dir $ glLightN3 (marshalCapability l) gl_SPOT_DIRECTION
light l (SpotExponent e) =
   glLightf (marshalCapability l) gl_SPOT_EXPONENT e
light l (SpotCutoff c) =
   glLightf (marshalCapability l) gl_SPOT_CUTOFF (fromMaybe 180 c)
light l (Attenuation a x) =
   glLightf (marshalCapability l) (marshalAttenuation a) x

foreign import "glLightf"  unsafe glLightf  :: GLenum -> GLenum -> GLfloat               -> IO ()
foreign import "glLightfv" unsafe glLightC4 :: GLenum -> GLenum -> Ptr (Color4  GLfloat) -> IO ()
foreign import "glLightfv" unsafe glLightV4 :: GLenum -> GLenum -> Ptr (Vertex4 GLfloat) -> IO ()
foreign import "glLightfv" unsafe glLightN3 :: GLenum -> GLenum -> Ptr (Normal3 GLfloat) -> IO ()

-- @GL_1_2@
data ColorControl =
     SingleColor
   | SeparateSpecularColor
   deriving (Eq,Ord)

marshalColorControl :: ColorControl -> GLint
marshalColorControl SingleColor           = gl_SINGLE_COLOR
marshalColorControl SeparateSpecularColor = gl_SEPARATE_SPECULAR_COLOR

unmarshalColorControl :: GLint -> ColorControl
unmarshalColorControl colorControl
   | colorControl == gl_SINGLE_COLOR            = SingleColor
   | colorControl == gl_SEPARATE_SPECULAR_COLOR = SeparateSpecularColor
   | otherwise                                  = error "unmarshalColorControl"

data LightModel =
     LightModelAmbient (Color4 GLfloat)
   | LightModelLocalViewer Bool
   | LightModelTwoSide Bool
   | LightModelColorControl ColorControl   -- @GL_1_2@

lightModel :: LightModel -> IO ()
lightModel (LightModelAmbient c) =
   withObject c $ glLightModelfv gl_LIGHT_MODEL_AMBIENT
lightModel (LightModelLocalViewer l) =
   glLightModeli gl_LIGHT_MODEL_LOCAL_VIEWER (boolToGLint l)
lightModel (LightModelTwoSide t) =
   glLightModeli gl_LIGHT_MODEL_TWO_SIDE (boolToGLint t)
lightModel (LightModelColorControl c) =
   glLightModeli gl_LIGHT_MODEL_COLOR_CONTROL (marshalColorControl c)

foreign import "glLightModeli"  unsafe glLightModeli  :: GLenum -> GLint                -> IO ()
foreign import "glLightModelfv" unsafe glLightModelfv :: GLenum -> Ptr (Color4 GLfloat) -> IO ()

---------------------------------------------------------------------------
-- Section 2.13.3 (Color Material)

data ColorMaterial = ColorMaterial deriving (Eq,Ord)

instance Capability ColorMaterial where
   marshalCapability ColorMaterial = gl_COLOR_MATERIAL

colorMaterial :: Face -> LightType -> IO ()
colorMaterial face lightType = glColorMaterial (marshalFace face) (marshalLightType lightType)

foreign import "glColorMaterial" unsafe glColorMaterial :: GLenum -> GLenum -> IO ()

---------------------------------------------------------------------------
-- Section 2.13.7 (Flatshading)

data ShadeModel =
     Flat
   | Smooth
   deriving (Eq,Ord)

marshalShadeModel :: ShadeModel -> GLenum
marshalShadeModel Flat   = gl_FLAT
marshalShadeModel Smooth = gl_SMOOTH

unmarshalShadeModel :: GLenum -> ShadeModel
unmarshalShadeModel shadeModel
   | shadeModel == gl_FLAT   = Flat
   | shadeModel == gl_SMOOTH = Smooth
   | otherwise               = error "unmarshalShadeModel"

shadeModel :: ShadeModel -> IO ()
shadeModel = glShadeModel . marshalShadeModel

foreign import "glShadeModel" unsafe glShadeModel :: GLenum -> IO ()
