/*
** 1998-05-29 -	A command to change the access flags of a file or directory.
**		Made significantly simpler by the new cmd_generic module.
** 1999-03-06 -	Adapted for new selection/generic handling.
*/

#include "gentoo.h"
#include "errors.h"
#include "dirpane.h"
#include "fileutil.h"
#include "strutil.h"
#include "window.h"

#include "cmd_generic.h"
#include "cmd_chmod.h"

#define	CMD_ID	"chmod"

/* ----------------------------------------------------------------------------------------- */

typedef struct {
	GtkWidget	*frame;
	GtkWidget	*vbox;
	GtkWidget	*check[3];
	gulong		signal[3];
} PFrame;

typedef struct {
	GtkWidget	*vbox;
	GtkWidget	*label;
	GtkWidget	*fbox;
	PFrame		frame[4];
	GtkWidget	*entry_text;
	GtkWidget	*entry_octal;
	GtkWidget	*bbox;
	GtkWidget	*all, *none, *toggle, *revert;
	mode_t		last_mode;
	GtkWidget	*recurse;
	gboolean	last_recurse;
	GtkWidget	*nodirs;
	gboolean	last_nodirs;
} ChmInfo;

static const mode_t mask[] = {  S_ISUID, S_ISGID, S_ISVTX,
				S_IRUSR, S_IWUSR, S_IXUSR, S_IRGRP, S_IWGRP, S_IXGRP, S_IROTH, S_IWOTH, S_IXOTH };

/* ----------------------------------------------------------------------------------------- */

static mode_t get_checks(const ChmInfo *chm)
{
	mode_t	mode = 0U;
	guint	i, j, k;

	for(i = k = 0; i < sizeof chm->frame / sizeof *chm->frame; i++)
	{
		for(j = 0; j < sizeof chm->frame[i].check / sizeof *chm->frame[i].check; j++, k++)
		{
			if(GTK_TOGGLE_BUTTON(chm->frame[i].check[j])->active)
				mode |= mask[k];
		}
	}
	return mode;
}

/* 2009-03-25 -	Update the textual representations. These are currently read-only. */
static void set_texts(ChmInfo *chm, mode_t mode)
{
	gchar	buf[32];

	stu_mode_to_text(buf, sizeof buf, mode);
	gtk_entry_set_text(GTK_ENTRY(chm->entry_text), buf + 1);	/* Skip the directory indicator. */
	g_snprintf(buf, sizeof buf, "%o", mode);
	gtk_entry_set_text(GTK_ENTRY(chm->entry_octal), buf);
}

static void set_checks(ChmInfo *chm, mode_t mode)
{
	guint	i, j, k;

	for(i = k = 0; i < sizeof chm->frame / sizeof *chm->frame; i++)
	{
		for(j = 0; j < sizeof chm->frame[i].check / sizeof *chm->frame[i].check; j++, k++)
		{
			g_signal_handler_block(G_OBJECT(chm->frame[i].check[j]), chm->frame[i].signal[j]);
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chm->frame[i].check[j]), mode & mask[k]);
			g_signal_handler_unblock(G_OBJECT(chm->frame[i].check[j]), chm->frame[i].signal[j]);
		}
	}
	set_texts(chm, mode);
}

static void chm_body(MainInfo *min, DirPane *src, DirRow *row, gpointer user)
{
	ChmInfo		*chm = user;
	gchar		temp[2 * FILENAME_MAX + 128];
	mode_t		mode = DP_ROW_LSTAT(row).st_mode;

	g_snprintf(temp, sizeof temp, _("Set protection bits for \"%s\":"), DP_ROW_NAMED(row));
	gtk_label_set_text(GTK_LABEL(chm->label), temp);
	set_checks(chm, chm->last_mode = (mode & 07777));
}

/* ----------------------------------------------------------------------------------------- */

static gboolean chmod_file(MainInfo *min, const gchar *name, mode_t mode)
{
	errno = 0;
	if(chmod(name, mode) != 0)
	{
		err_set(min, errno, CMD_ID, name);
		return 0;
	}
	return 1;
}

static gboolean chmod_dir(MainInfo *min, const gchar *name, mode_t mode, gboolean recurse, gboolean nodirs)
{
	errno = 0;
	if(nodirs || chmod(name, mode) == 0)
	{
		if(recurse)
		{
			gchar	old_dir[PATH_MAX];
			DIR	*dir;

			if(!fut_cd(name, old_dir, sizeof old_dir))
				return FALSE;

			if((dir = opendir(".")) != NULL)
			{
				struct dirent	*de;
				struct stat	stat;
				gboolean	ok = TRUE;

				while(errno == 0 && ((de = readdir(dir)) != NULL) && ok)
				{
					if(!min->cfg.dir_filter(de->d_name))
						continue;
					if(lstat(de->d_name, &stat) == 0)
					{
						gchar	buf[PATH_MAX];

						g_snprintf(buf, sizeof buf, "%s%c%s", name, G_DIR_SEPARATOR, de->d_name);
						if(S_ISDIR(stat.st_mode))
							ok = chmod_dir(min, de->d_name, mode, recurse, nodirs);
						else
							ok = chmod_file(min, de->d_name, mode);
					}
				}
				closedir(dir);
			}
			fut_cd(old_dir, NULL, 0U);
		}
	}
	else
		err_set(min, errno, CMD_ID, name);
	return errno == 0;
}

static gint chm_action(MainInfo *min, DirPane *src, DirPane *dst, DirRow *row, gpointer user)
{
	ChmInfo	*chm = user;
	mode_t	mode;
	gint	ok;

	if(!fut_cd(src->dir.path, NULL, 0))
		return 0;

	mode = get_checks(chm);

	chm->last_recurse = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chm->recurse));
	chm->last_nodirs  = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chm->nodirs));

	if(S_ISDIR(DP_ROW_LSTAT(row).st_mode))
		ok = chmod_dir(min, DP_ROW_NAME(row), mode, chm->last_recurse, chm->last_nodirs);
	else
		ok = chmod_file(min, DP_ROW_NAME(row), mode);
	if(ok)
		dp_unselect(src, DP_ROW_INDEX(src, row));

	return ok;
}

/* ----------------------------------------------------------------------------------------- */

static void evt_clicked(GtkWidget *wid, gpointer user)
{
	ChmInfo	*chm = user;

	if(wid == chm->all)
		set_checks(chm, S_ISUID | S_ISGID | S_ISVTX | S_IRUSR | S_IWUSR | S_IXUSR |
				S_IRGRP | S_IWGRP | S_IXGRP | S_IROTH | S_IWOTH | S_IXOTH);
	else if(wid == chm->none)
		set_checks(chm, 0);
	else if(wid == chm->toggle)
		set_checks(chm, (~get_checks(chm)) & 07777);
	else if(wid == chm->revert)
		set_checks(chm, chm->last_mode);
	else
		set_texts(chm, get_checks(chm));
}

/* 1998-05-29 -	Build a protection frame, with three checkboxes. If type is 0, we build a
**		special one (with setuid/setgid/sticky), otherwise a standard (read/write/exec).
*/
static void build_frame(ChmInfo *ci, gint pos, gint type)
{
	gchar	*label[] = { N_("Special"), N_("Owner"),   N_("Group"),  N_("Others") };
	gchar	*check[] = { N_("Set UID"), N_("Set GID"), N_("Sticky"), N_("Read"), N_("Write"), N_("Execute") };
	PFrame	*fr = &ci->frame[pos];
	gint	i;

	fr->frame = gtk_frame_new(_(label[pos]));
	fr->vbox = gtk_vbox_new(FALSE, 0);
	for(i = 0; i < 3; i++)
	{
		fr->check[i] = gtk_check_button_new_with_label(_(check[type * 3 + i]));
		fr->signal[i] = g_signal_connect(G_OBJECT(fr->check[i]), "toggled", G_CALLBACK(evt_clicked), ci);
		gtk_box_pack_start(GTK_BOX(fr->vbox), fr->check[i], TRUE, TRUE, 0);
	}
	gtk_container_add(GTK_CONTAINER(fr->frame), fr->vbox);
}

gint cmd_chmod(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	static ChmInfo	ci;
	guint		i;
	GtkWidget	*hbox, *w;

	ci.vbox  = gtk_vbox_new(FALSE, 5);
	ci.label = gtk_label_new(_("Protection Bits"));
	ci.fbox  = gtk_hbox_new(FALSE, 0);
	for(i = 0; i < sizeof ci.frame / sizeof ci.frame[0]; i++)
	{
		build_frame(&ci, i, (i == 0) ? 0 : 1);
		gtk_box_pack_start(GTK_BOX(ci.fbox), ci.frame[i].frame, TRUE, TRUE, 5);
	}
	gtk_box_pack_start(GTK_BOX(ci.vbox), ci.label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(ci.vbox), ci.fbox,  TRUE,  TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 0);
	w = gtk_label_new(_("Textual"));
	gtk_box_pack_start(GTK_BOX(hbox), w, FALSE, FALSE, 0);
	ci.entry_text = gtk_entry_new_with_max_length(10);
	gtk_entry_set_width_chars(GTK_ENTRY(ci.entry_text), 10);
	gtk_editable_set_editable(GTK_EDITABLE(ci.entry_text), FALSE);
	gtk_box_pack_start(GTK_BOX(hbox), ci.entry_text, FALSE, FALSE, 0);

	ci.entry_octal = gtk_entry_new_with_max_length(4);
	gtk_entry_set_width_chars(GTK_ENTRY(ci.entry_octal), 4);
	gtk_editable_set_editable(GTK_EDITABLE(ci.entry_octal), FALSE);
	gtk_box_pack_end(GTK_BOX(hbox), ci.entry_octal, FALSE, FALSE, 0);
	w = gtk_label_new(_("Octal"));
	gtk_box_pack_end(GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(ci.vbox), hbox, FALSE, FALSE, 0);

	ci.bbox = gtk_hbox_new(FALSE, 0);
	ci.all	  = gtk_button_new_with_label(_("All"));
	ci.none	  = gtk_button_new_with_label(_("None"));
	ci.toggle = gtk_button_new_with_label(_("Toggle"));
	ci.revert = gtk_button_new_with_label(_("Revert"));

	g_signal_connect(G_OBJECT(ci.all),    "clicked", G_CALLBACK(evt_clicked), &ci);
	g_signal_connect(G_OBJECT(ci.none),   "clicked", G_CALLBACK(evt_clicked), &ci);
	g_signal_connect(G_OBJECT(ci.toggle), "clicked", G_CALLBACK(evt_clicked), &ci);
	g_signal_connect(G_OBJECT(ci.revert), "clicked", G_CALLBACK(evt_clicked), &ci);
	gtk_box_pack_start(GTK_BOX(ci.bbox), ci.all,    TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(ci.bbox), ci.none,   TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(ci.bbox), ci.toggle, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(ci.bbox), ci.revert, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(ci.vbox), ci.bbox, TRUE, TRUE, 0);
	hbox = gtk_hbox_new(FALSE, 0);
	ci.recurse = gtk_check_button_new_with_label(_("Recurse Directories?"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ci.recurse), ci.last_recurse);
	gtk_box_pack_start(GTK_BOX(hbox), ci.recurse, TRUE, TRUE, 0);
	ci.nodirs = gtk_check_button_new_with_label(_("Don't Touch Directories?"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ci.nodirs), ci.last_nodirs);
	gtk_box_pack_start(GTK_BOX(hbox), ci.nodirs, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(ci.vbox), hbox, FALSE, FALSE, 0);

	return cmd_generic(min, _("Change Mode"), CGF_SRC, chm_body, chm_action, NULL, &ci);
}
