/*
** 1999-03-16 -	Configure controls. Initially, this will only have global keyboard shortcuts
**		to deal with, but it may get more in the future. This module relies heavily
**		on the services provided by the 'controls' module, of course.
*/

#include "gentoo.h"

#include "cmdseq_dialog.h"

#include "cmdseq.h"
#include "controls.h"
#include "dialog.h"
#include "guiutil.h"
#include "strutil.h"

#include "cfg_gui.h"
#include "cfg_module.h"
#include "cfg_cmdseq.h"				/* For ccs_current(). */
#include "cfg_controls.h"

#define	NODE	"Controls"

/* ----------------------------------------------------------------------------------------- */

#include "graphics/icon_mouse1.xpm"
#include "graphics/icon_mouse2.xpm"
#include "graphics/icon_mouse3.xpm"
#include "graphics/icon_mouse4.xpm"
#include "graphics/icon_mouse5.xpm"

/* ----------------------------------------------------------------------------------------- */

typedef struct {
	GtkWidget	*vbox;			/* Root page container. Very standard. */

	GtkWidget	*kframe;		/* Keyboard frame. */

	GtkWidget	*kscwin;		/* Scrolling window for keyboard shortcuts. */
	GtkWidget	*kclist;		/* Actual list holding the keyboard shortcuts. */
	GtkWidget	*kdtable;		/* Key definition table. */
	GtkWidget	*kdkey;			/* Definition key name (entry). */
	GtkWidget	*kdcmd;			/* Definition cmdsequence (entry). */
	GtkWidget	*kdcmdpick;		/* Definition command pick button. */
	GtkWidget	*khbox;			/* Horizontal box for keyboard add & del commands. */
	GtkWidget	*kadd;			/* Keyboard add button. */
	GtkWidget	*kdel;			/* Keyboard delete button. */

	GtkWidget	*mframe;		/* Dirpane mouse button config frame. */
	GtkWidget	*mscwin;
	GtkWidget	*mclist;		/* List of defined mouse buttons. */
	GtkWidget	*mdtable;		/* Definition table. */
	GtkWidget	*mdmenu;		/* Option menu showing mouse button for mapping. */
	GtkWidget	*mdcmd;
	GtkWidget	*mdcmdpick;
	GtkWidget	*mhbox;
	GtkWidget	*madd, *mdel;

	GdkPixmap	*mpix[5];		/* Mouse icon pixmaps and masks. */
	GdkBitmap	*mmsk[5];

	GtkWidget	*nonumlock;		/* Ignore numlock check button. */

	MainInfo	*min;			/* This is handy sometimes. */
	gboolean	modified;		/* Indicates that a change has been made. */
	CtrlInfo	*ctrlinfo;		/* The control info we're editing. */
	CtrlKey		*curr_key;		/* Current key (if selected). */
	gint		curr_key_row;		/* Row of current key in list. */
	CtrlMouse	*curr_mouse;		/* Current mouse button binding (if any). */
	gint		curr_mouse_row;		/* List row number for currently selected mouse row. */
} P_Controls;

static P_Controls	the_page;

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-16 -	Set key definition widgets to non-idle state. */
static void set_key_widgets(P_Controls *page, CtrlKey *key)
{
	if(page != NULL)
	{

		if(key != NULL)
		{
			const gchar	*cmdseq;

			gtk_entry_set_text(GTK_ENTRY(page->kdkey), ctrl_key_get_keyname(key));
			if((cmdseq = ctrl_key_get_cmdseq(key)) != NULL)
				gtk_entry_set_text(GTK_ENTRY(page->kdcmd), cmdseq);
			gtk_widget_set_sensitive(page->kdtable, TRUE);
			gtk_widget_set_sensitive(page->kdel, TRUE);
		}
	}
}

/* 1999-06-13 -	Set state of mouse binding def. widgets to match selected binding. */
static void set_mouse_widgets(P_Controls *page, CtrlMouse *mouse)
{
	if((page != NULL) && (mouse != NULL))
	{
		gtk_option_menu_set_history(GTK_OPTION_MENU(page->mdmenu), ctrl_mouse_get_button(mouse) - 1);
		gtk_entry_set_text(GTK_ENTRY(page->mdcmd), ctrl_mouse_get_cmdseq(mouse));
		gtk_widget_set_sensitive(page->mdtable, TRUE);
		gtk_widget_set_sensitive(page->mdel, TRUE);
	}
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-16 -	Reset the key definition widgets to their idle state. */
static void reset_key_widgets(P_Controls *page)
{
	gtk_entry_set_text(GTK_ENTRY(page->kdkey), "");
	gtk_entry_set_text(GTK_ENTRY(page->kdcmd), "");
	gtk_widget_set_sensitive(page->kdtable, FALSE);
	gtk_widget_set_sensitive(page->kadd, TRUE);
	gtk_widget_set_sensitive(page->kdel, FALSE);
}

/* 1999-06-13 -	Reset mouse editing widgets. */
static void reset_mouse_widgets(P_Controls *page)
{
	gtk_option_menu_set_history(GTK_OPTION_MENU(page->mdmenu), 0);
	gtk_entry_set_text(GTK_ENTRY(page->mdcmd), "");
	gtk_widget_set_sensitive(page->mdtable, FALSE);
	gtk_widget_set_sensitive(page->madd, TRUE);
	gtk_widget_set_sensitive(page->mdel, FALSE);
}

/* 1999-03-16 -	Reset all widgets on page to their most idle state. */
static void reset_widgets(P_Controls *page)
{
	page->curr_key   = NULL;
	page->curr_mouse = NULL;
	reset_key_widgets(page);
	reset_mouse_widgets(page);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(page->nonumlock), ctrl_numlock_ignore_get(page->ctrlinfo));
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-16 -	Populate keyboard clist. */
static void populate_key_clist(P_Controls *page)
{
	gchar	*crow[] = { NULL, NULL };
	GSList	*list, *iter;

	gtk_clist_clear(GTK_CLIST(page->kclist));
	gtk_clist_freeze(GTK_CLIST(page->kclist));
	if((list = ctrl_keys_get_list(page->ctrlinfo)) != NULL)
	{
		gint	row;

		for(iter = list; iter != NULL; iter = g_slist_next(iter))
		{
			crow[0] = (gchar *) ctrl_key_get_keyname((CtrlKey *) iter->data);
			crow[1] = (gchar *) ctrl_key_get_cmdseq((CtrlKey *) iter->data);
			row = gtk_clist_append(GTK_CLIST(page->kclist), crow);
			gtk_clist_set_row_data(GTK_CLIST(page->kclist), row, iter->data);
		}
	}
	gtk_clist_thaw(GTK_CLIST(page->kclist));
}

/* 1999-06-13 -	Populate the mouse command mapping list. */
static void populate_mouse_clist(P_Controls *page)
{
	gchar	*bname[] = { N_("Left"), N_("Middle"), N_("Right"), N_("Wheel Up"), N_("Wheel Down") },
		*crow[] = { NULL, NULL, NULL, NULL };
	gint	row;
	guint	btn;
	GSList	*list, *iter;

	gtk_clist_freeze(GTK_CLIST(page->mclist));
	gtk_clist_clear(GTK_CLIST(page->mclist));
	for(list = ctrl_mouse_get_list(page->ctrlinfo), iter = list; iter != NULL; iter = g_slist_next(iter))
	{
		btn = ctrl_mouse_get_button(iter->data) - 1;
		crow[1] = _(bname[btn]);
		crow[2] = gtk_accelerator_name(0U, ctrl_mouse_get_state(iter->data));
		crow[3] = (gchar *) ctrl_mouse_get_cmdseq(iter->data);
		row = gtk_clist_append(GTK_CLIST(page->mclist), crow);
		gtk_clist_set_row_data(GTK_CLIST(page->mclist), row, iter->data);
		gtk_clist_set_pixmap(GTK_CLIST(page->mclist), row, 0, page->mpix[btn], page->mmsk[btn]);
	}
	gtk_clist_thaw(GTK_CLIST(page->mclist));
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-16 -	Keyboard clist selection signal handler. */
static void evt_kclist_select(GtkWidget *wid, gint row, gint column, GdkEventButton *evt, gpointer user)
{
	P_Controls	*page = user;

	set_key_widgets(page, gtk_clist_get_row_data(GTK_CLIST(page->kclist), row));
	page->curr_key = gtk_clist_get_row_data(GTK_CLIST(page->kclist), row);
	page->curr_key_row = row;
}

/* 1999-03-16 -	Keyboard clist unselection handler. */
static void evt_kclist_unselect(GtkWidget *wid, gint row, gint column, GdkEventButton *evt, gpointer user)
{
	P_Controls	*page = user;

	page->curr_key = NULL;
	page->curr_key_row = -1;
	reset_key_widgets(page);
}

/* 1999-03-16 -	User pressed something in the key definition entry. Convert to name and set it. */
static gint evt_kkey_press(GtkWidget *wid, GdkEventKey *evt, gpointer user)
{
	P_Controls	*page = user;

	if((page != NULL) && (page->curr_key != NULL))
	{
		gchar	*keyname = gtk_accelerator_name(evt->keyval, evt->state);

		page->modified = TRUE;
		ctrl_key_set_keyname(page->ctrlinfo, page->curr_key, keyname);
		gtk_clist_set_text(GTK_CLIST(page->kclist), page->curr_key_row, 0, keyname);
		set_key_widgets(page, page->curr_key);
		gtk_signal_emit_stop_by_name(GTK_OBJECT(wid), "key_press_event");
	}
	return TRUE;
}

/* 1999-03-16 -	User edited the command sequence, so store the new one. */
static gint evt_kcmd_changed(GtkWidget *wid, gpointer user)
{
	P_Controls	*page = user;

	if((page != NULL) && (page->curr_key != NULL))
	{
		gchar	*cmdseq = gtk_entry_get_text(GTK_ENTRY(wid));

		if(!ctrl_key_has_cmdseq(page->ctrlinfo, page->curr_key, cmdseq))
		{
			page->modified = TRUE;
			ctrl_key_set_cmdseq(page->ctrlinfo, page->curr_key, cmdseq);
			gtk_clist_set_text(GTK_CLIST(page->kclist), page->curr_key_row, 1, cmdseq);
		}
	}

	return TRUE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-16 -	User clicked the "pick" button for key command sequence.
** 1999-03-29 -	Rewritten for new, simpler, csq_dialog() semantics.
*/
static gint evt_kcmdpick_clicked(GtkWidget *wid, gpointer user)
{
	P_Controls	*page = user;
	const gchar	*cmd;

	if((cmd = csq_dialog_sync_new_wait(page->min, ccs_get_current())) != NULL)
		gtk_entry_set_text(GTK_ENTRY(page->kdcmd), cmd);
	return TRUE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-16 -	User just hit the "Add" key button so, er, add a key. */
static gint evt_kadd_clicked(GtkWidget *wid, gpointer user)
{
	P_Controls	*page = user;

	if(page != NULL)
	{
		gchar	*crow[] = { NULL, NULL };
		gint	row;
		CtrlKey	*key;

		key = ctrl_key_add_unique(page->ctrlinfo);
		crow[0] = (gchar *) ctrl_key_get_keyname(key);
		crow[1] = (gchar *) ctrl_key_get_cmdseq(key);
		row = gtk_clist_append(GTK_CLIST(page->kclist), crow);
		gtk_clist_set_row_data(GTK_CLIST(page->kclist), row, key);
		gtk_clist_select_row(GTK_CLIST(page->kclist), row, -1);
		gtk_widget_grab_focus(page->kdkey);
		page->modified = TRUE;
	}

	return TRUE;
}

/* 1999-03-16 -	User hit the "Delete" button for keys. */
static gint evt_kdel_clicked(GtkWidget *wid, gpointer user)
{
	P_Controls	*page = user;

	if((page != NULL) && (page->curr_key != NULL))
	{
		gint	row = page->curr_key_row;

		ctrl_key_remove(page->ctrlinfo, page->curr_key);
		gtk_clist_remove(GTK_CLIST(page->kclist), row);
		while(row >= GTK_CLIST(page->kclist)->rows)
			row--;
		gtk_clist_select_row(GTK_CLIST(page->kclist), row, -1);
		page->modified = TRUE;
	}
	return TRUE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-13 -	A mouse binding row has been selected. */
static void evt_mclist_select_row(GtkWidget *wid, gint row, gint col, GdkEventButton *evt, gpointer user)
{
	P_Controls	*page = user;

	set_mouse_widgets(page, gtk_clist_get_row_data(GTK_CLIST(wid), row));
	page->curr_mouse     = gtk_clist_get_row_data(GTK_CLIST(wid), row);
	page->curr_mouse_row = row;
}

/* 1999-06-13 -	We lost the selected row in the mouse CList. Clear other widgets. */
static void evt_mclist_unselect_row(GtkWidget *wid, gint row, gint col, GdkEventButton *evt, gpointer user)
{
	P_Controls	*page = user;

	page->curr_mouse = NULL;
	page->curr_mouse_row = -1;
	reset_mouse_widgets(page);
}

static void evt_mbutton_activated(GtkWidget *wid, gpointer user)
{
	P_Controls	*page = user;
	gchar		*blabel[] = { N_("Left"), N_("Middle"), N_("Right"), N_("Wheel Up"), N_("Wheel Down") };
	guint		btn;

	btn = GPOINTER_TO_UINT(gtk_object_get_user_data(GTK_OBJECT(wid)));
	ctrl_mouse_set_button(page->curr_mouse, btn + 1);
	gtk_clist_set_pixmap(GTK_CLIST(page->mclist), page->curr_mouse_row, 0, page->mpix[btn], page->mmsk[btn]);
	gtk_clist_set_text(GTK_CLIST(page->mclist), page->curr_mouse_row, 1, _(blabel[btn]));
}

/* 1999-06-13 -	User just hit the "Edit Modifiers..." button. Bring up a little dialog. */
static void evt_mstate_clicked(GtkWidget *wid, gpointer user)
{
	P_Controls	*page = user;
	Dialog		*dlg;
	guint		i, mask[] = { GDK_SHIFT_MASK, GDK_CONTROL_MASK, GDK_MOD1_MASK };
	GtkWidget	*vbox, *label, *check[sizeof mask / sizeof mask[0]];

	if((page == NULL) || (page->curr_mouse == NULL))
		return;

	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new(_("The following modifier key(s) must\nbe held down when the mouse button"
				"\n is clicked to trigger the command:"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);
	for(i = 0; i < sizeof mask / sizeof mask[0]; i++)
	{
		check[i] = gtk_check_button_new_with_label(gtk_accelerator_name(0, mask[i]));
		if(ctrl_mouse_get_state(page->curr_mouse) & mask[i])
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check[i]), TRUE);
		gtk_box_pack_start(GTK_BOX(vbox), check[i], FALSE, FALSE, 0);
		gtk_widget_show(check[i]);
	}
	dlg = dlg_dialog_sync_new(vbox, _("Edit Modifiers"), _("OK|Cancel"));
	if(dlg_dialog_sync_wait(dlg) == DLG_POSITIVE)
	{
		guint	ns = 0U;

		for(i = 0; i < sizeof mask / sizeof mask[0]; i++)
		{
			if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check[i])))
				ns |= mask[i];
		}
		ctrl_mouse_set_state(page->curr_mouse, ns);
		gtk_clist_set_text(GTK_CLIST(page->mclist), page->curr_mouse_row, 2, gtk_accelerator_name(0, ns));
		page->modified = TRUE;
	}
	dlg_dialog_sync_destroy(dlg);
}

/* 1999-06-13 -	User is editing the command name. Update mapping. */
static void evt_mcmd_changed(GtkWidget *wid, gpointer user)
{
	P_Controls	*page = user;

	if((page != NULL) && (page->curr_mouse != NULL))
	{
		gchar	*cmdseq;

		if((cmdseq = gtk_entry_get_text(GTK_ENTRY(page->mdcmd))) != NULL)
		{
			ctrl_mouse_set_cmdseq(page->curr_mouse, cmdseq);
			gtk_clist_set_text(GTK_CLIST(page->mclist), page->curr_mouse_row, 3, cmdseq);
			page->modified = TRUE;
		}
	}
}

/* 1999-06-13 -	User clicked details button for mouse mapping command. Pop up dialog. */
static void evt_mcmdpick_clicked(GtkWidget *wid, gpointer user)
{
	P_Controls	*page = user;

	if((page != NULL) && (page->curr_mouse != NULL))
	{
		const gchar	*cmd;

		if((cmd = csq_dialog_sync_new_wait(page->min, ccs_get_current())) != NULL)
			gtk_entry_set_text(GTK_ENTRY(page->mdcmd), cmd);
	}
}

/* 1999-06-20 -	User clicked the "Add" button for mouse bindings, so add one. */
static void evt_madd_clicked(GtkWidget *wid, gpointer user)
{
	P_Controls	*page = user;

	if(page != NULL)
	{
		ctrl_mouse_add(page->ctrlinfo, 1U, 0U, "");
		populate_mouse_clist(page);
		gtk_clist_select_row(GTK_CLIST(page->mclist), 0, -1);
		gtk_clist_moveto(GTK_CLIST(page->mclist), 0, -1, 0.0f, -1.0f);
		page->modified = TRUE;
	}
}

/* 1999-06-21 -	The Delete-button has been clicked, remove current mouse command mapping. */
static void evt_mdel_clicked(GtkWidget *wid, gpointer user)
{
	P_Controls	*page = user;

	if((page != NULL) && (page->curr_mouse != NULL))
	{
		gint	row = page->curr_mouse_row, nr;

		ctrl_mouse_remove(page->ctrlinfo, page->curr_mouse);
		gtk_clist_unselect_row(GTK_CLIST(page->mclist), row, 0);
		page->curr_mouse = NULL;
		page->curr_mouse_row = -1;
		populate_mouse_clist(page);
		nr = (row >= GTK_CLIST(page->mclist)->rows) ? GTK_CLIST(page->mclist)->rows - 1 : row;
		gtk_clist_select_row(GTK_CLIST(page->mclist), nr, 0);
		gtk_clist_moveto(GTK_CLIST(page->mclist), nr, -1, 1.0f, -1.0f);
		page->modified = TRUE;
	}
}

/* ----------------------------------------------------------------------------------------- */

/* 2000-02-18 -	User hit the "Ignore Num Lock?" toggle. Update editing state. */
static void evt_nonumlock_clicked(GtkWidget *wid, gpointer user)
{
	P_Controls	*page = user;

	if(page != NULL)
	{
		ctrl_numlock_ignore_set(page->ctrlinfo, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(wid)));
		page->modified = TRUE;
	}
}

/* ----------------------------------------------------------------------------------------- */

static void build_mouse_menu(P_Controls *page)
{
	gchar		**gfx[] = { icon_mouse1_xpm, icon_mouse2_xpm, icon_mouse3_xpm, icon_mouse4_xpm, icon_mouse5_xpm },
			*label[] = { N_("Left"), N_("Middle"), N_("Right"), N_("Wheel Up"), N_("Wheel Down") };
	GtkWidget	*menu, *item, *ihbox, *pmap, *lab;
	guint		i;

	menu = gtk_menu_new();
	gtk_menu_set_title(GTK_MENU(menu), _("Menu"));
	for(i = 0; i < 5; i++)
	{
		item = gtk_menu_item_new();
		ihbox = gtk_hbox_new(FALSE, 0);
		page->mpix[i] = gdk_pixmap_colormap_create_from_xpm_d(NULL, gtk_widget_get_default_colormap(), &page->mmsk[i], NULL, gfx[i]);
		pmap = gtk_pixmap_new(page->mpix[i], page->mmsk[i]);
		gtk_box_pack_start(GTK_BOX(ihbox), pmap, FALSE, FALSE, 0);
		gtk_widget_show(pmap);
		lab = gtk_label_new(_(label[i]));
		gtk_box_pack_start(GTK_BOX(ihbox), lab, FALSE, FALSE, 0);
		gtk_widget_show(lab);
		gtk_container_add(GTK_CONTAINER(item), ihbox);
		gtk_widget_show(ihbox);
		gtk_menu_append(GTK_MENU(menu), item);
		gtk_widget_show(item);
		gtk_object_set_user_data(GTK_OBJECT(item), GUINT_TO_POINTER(i));
		gtk_signal_connect(GTK_OBJECT(item), "activate", GTK_SIGNAL_FUNC(evt_mbutton_activated), page);
	}
	page->mdmenu = gtk_option_menu_new();
	gtk_option_menu_set_menu(GTK_OPTION_MENU(page->mdmenu), menu);
	gtk_widget_show(menu);
	gtk_table_attach(GTK_TABLE(page->mdtable), page->mdmenu, 1, 2, 0, 1,  GTK_FILL|GTK_EXPAND,0,0,0);
	gtk_widget_show(page->mdmenu);
}

static GtkWidget * cct_init(MainInfo *min, gchar **name)
{
	P_Controls	*page = &the_page;
	GtkWidget	*vbox, *label, *wid;

	if(name == NULL)
		return NULL;

	*name = _("Controls");

	page->min = min;
	page->modified = FALSE;

	page->vbox = gtk_vbox_new(FALSE, 0);


	page->kframe = gtk_frame_new(_("Global Keyboard Shortcuts"));
	vbox = gtk_vbox_new(FALSE, 0);
	page->kscwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(page->kscwin), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	page->kclist = gtk_clist_new(2);
	gtk_clist_set_column_auto_resize(GTK_CLIST(page->kclist), 0, TRUE);
	gtk_clist_set_column_min_width(GTK_CLIST(page->kclist), 0, 80);
	gtk_signal_connect(GTK_OBJECT(page->kclist), "select_row", GTK_SIGNAL_FUNC(evt_kclist_select), page);
	gtk_signal_connect(GTK_OBJECT(page->kclist), "unselect_row", GTK_SIGNAL_FUNC(evt_kclist_unselect), page);
	gtk_container_add(GTK_CONTAINER(page->kscwin), page->kclist);
	gtk_box_pack_start(GTK_BOX(vbox), page->kscwin, TRUE, TRUE, 0);
	page->kdtable = gtk_table_new(2, 3, FALSE);
	label = gtk_label_new(_("Key"));
	gtk_table_attach(GTK_TABLE(page->kdtable), label, 0, 1, 0, 1,  0,0,0,0);
	page->kdkey = gtk_entry_new();
	gtk_signal_connect(GTK_OBJECT(page->kdkey), "key_press_event", GTK_SIGNAL_FUNC(evt_kkey_press), page);
	gtk_entry_set_editable(GTK_ENTRY(page->kdkey), FALSE);
	gtk_table_attach(GTK_TABLE(page->kdtable), page->kdkey, 1, 3, 0, 1,  GTK_EXPAND|GTK_FILL,0,0,0);
	label = gtk_label_new(_("Command"));
	gtk_table_attach(GTK_TABLE(page->kdtable), label, 0, 1, 1, 2,  0,0,0,0);
	page->kdcmd = gtk_entry_new();
	gtk_signal_connect(GTK_OBJECT(page->kdcmd), "changed", GTK_SIGNAL_FUNC(evt_kcmd_changed), page);
	gtk_table_attach(GTK_TABLE(page->kdtable), page->kdcmd, 1, 2, 1, 2,  GTK_EXPAND|GTK_FILL,0,0,0);
	page->kdcmdpick = gui_details_button_new(min->gui->window->window);
	gtk_signal_connect(GTK_OBJECT(page->kdcmdpick), "clicked", GTK_SIGNAL_FUNC(evt_kcmdpick_clicked), page);
	gtk_table_attach(GTK_TABLE(page->kdtable), page->kdcmdpick, 2, 3, 1, 2,  0,0,0,0);
	gtk_box_pack_start(GTK_BOX(vbox), page->kdtable, FALSE, FALSE, 0);
	page->khbox = gtk_hbox_new(FALSE, 0);
	page->kadd = gtk_button_new_with_label(_("Add"));
	gtk_signal_connect(GTK_OBJECT(page->kadd), "clicked", GTK_SIGNAL_FUNC(evt_kadd_clicked), page);
	gtk_box_pack_start(GTK_BOX(page->khbox), page->kadd, TRUE, TRUE, 5);
	page->kdel = gtk_button_new_with_label(_("Delete"));
	gtk_signal_connect(GTK_OBJECT(page->kdel), "clicked", GTK_SIGNAL_FUNC(evt_kdel_clicked), page);
	gtk_box_pack_start(GTK_BOX(page->khbox), page->kdel, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), page->khbox, FALSE, FALSE, 5);
	gtk_container_add(GTK_CONTAINER(page->kframe), vbox);
	gtk_box_pack_start(GTK_BOX(page->vbox), page->kframe, TRUE, TRUE, 0);

	page->mframe = gtk_frame_new(_("Dirpane Mouse Buttons"));
	vbox = gtk_vbox_new(FALSE, 0);
	page->mscwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(page->mscwin), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	page->mclist = gtk_clist_new(4);
	gtk_clist_set_column_width(GTK_CLIST(page->mclist), 0, 16);
	gtk_clist_set_column_justification(GTK_CLIST(page->mclist), 0, GTK_JUSTIFY_CENTER);
	gtk_clist_set_column_width(GTK_CLIST(page->mclist), 1, 80);
	gtk_clist_set_column_justification(GTK_CLIST(page->mclist), 1, GTK_JUSTIFY_LEFT);
	gtk_clist_set_column_width(GTK_CLIST(page->mclist), 2, 160);
	gtk_clist_set_column_justification(GTK_CLIST(page->mclist), 2, GTK_JUSTIFY_LEFT);
	gtk_signal_connect(GTK_OBJECT(page->mclist), "select_row", GTK_SIGNAL_FUNC(evt_mclist_select_row), page);
	gtk_signal_connect(GTK_OBJECT(page->mclist), "unselect_row", GTK_SIGNAL_FUNC(evt_mclist_unselect_row), page);
	gtk_container_add(GTK_CONTAINER(page->mscwin), page->mclist);
	gtk_box_pack_start(GTK_BOX(vbox), page->mscwin, TRUE, TRUE, 0);
	page->mdtable = gtk_table_new(2, 4, FALSE);
	label = gtk_label_new(_("Button"));
	gtk_table_attach(GTK_TABLE(page->mdtable), label, 0, 1, 0, 1,  0,0,0,0);
	build_mouse_menu(page);
	wid = gtk_button_new_with_label(_("Edit Modifiers..."));
	gtk_signal_connect(GTK_OBJECT(wid), "clicked", GTK_SIGNAL_FUNC(evt_mstate_clicked), page);
	gtk_table_attach(GTK_TABLE(page->mdtable), wid, 2, 3, 0, 1, GTK_FILL|GTK_EXPAND,0,5,0);
	label = gtk_label_new(_("Command"));
	gtk_table_attach(GTK_TABLE(page->mdtable), label, 0, 1, 1, 2,  0,0,0,0);
	page->mdcmd = gtk_entry_new();
	gtk_signal_connect(GTK_OBJECT(page->mdcmd), "changed", GTK_SIGNAL_FUNC(evt_mcmd_changed), page);
	gtk_table_attach(GTK_TABLE(page->mdtable), page->mdcmd, 1, 3, 1, 2,  GTK_EXPAND|GTK_FILL,0,0,0);
	page->mdcmdpick = gui_details_button_new(min->gui->window->window);
	gtk_signal_connect(GTK_OBJECT(page->mdcmdpick), "clicked", GTK_SIGNAL_FUNC(evt_mcmdpick_clicked), page);
	gtk_table_attach(GTK_TABLE(page->mdtable), page->mdcmdpick, 3, 4, 1, 2,  0,0,0,0);
	gtk_box_pack_start(GTK_BOX(vbox), page->mdtable, FALSE, FALSE, 0);
	page->mhbox = gtk_hbox_new(FALSE, 0);
	page->madd = gtk_button_new_with_label(_("Add"));
	gtk_signal_connect(GTK_OBJECT(page->madd), "clicked", GTK_SIGNAL_FUNC(evt_madd_clicked), page);
	gtk_box_pack_start(GTK_BOX(page->mhbox), page->madd, TRUE, TRUE, 5);
	page->mdel = gtk_button_new_with_label(_("Delete"));
	gtk_signal_connect(GTK_OBJECT(page->mdel), "clicked", GTK_SIGNAL_FUNC(evt_mdel_clicked), page);
	gtk_box_pack_start(GTK_BOX(page->mhbox), page->mdel, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), page->mhbox, FALSE, FALSE, 5);
	gtk_container_add(GTK_CONTAINER(page->mframe), vbox);
	gtk_box_pack_start(GTK_BOX(page->vbox), page->mframe, TRUE, TRUE, 0);

	page->nonumlock = gtk_check_button_new_with_label(_("Ignore Num Lock For All Bindings?"));
	gtk_signal_connect(GTK_OBJECT(page->nonumlock), "clicked", GTK_SIGNAL_FUNC(evt_nonumlock_clicked), page);
	gtk_box_pack_start(GTK_BOX(page->vbox), page->nonumlock, FALSE, FALSE, 0);
	
	gtk_widget_show_all(page->vbox);
	return page->vbox;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-16 -	Update the controls config page by grabbing current settings. */
static void cct_update(MainInfo *min)
{
	the_page.ctrlinfo = ctrl_copy(min->cfg.ctrlinfo);
	populate_key_clist(&the_page);
	populate_mouse_clist(&the_page);
	reset_widgets(&the_page);
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-17 -	The user just accepted the settings, so make them current. */
static void cct_accept(MainInfo *min)
{
	P_Controls	*page = &the_page;

	if(page->modified)
	{
		if(ctrl_mouse_ambiguity_exists(page->ctrlinfo))
			dlg_dialog_async_new_simple(_("Note: The mouse button control settings\n"
						    "are ambiguous: the same button+modifier\n"
						    "is used for more than one function. This\n"
						    "might make their behaviour pretty weird..."),
							_("Warning"), _("OK"), NULL, NULL);
		ctrl_destroy(min->cfg.ctrlinfo);
		min->cfg.ctrlinfo = page->ctrlinfo;
		page->ctrlinfo = NULL;
		cfg_set_flags(CFLG_RESET_KEYBOARD);
		page->modified = FALSE;
	}
}

/* ----------------------------------------------------------------------------------------- */

static void keys_save(CtrlInfo *ctrl, FILE *out)
{
	GSList	*keys, *iter;

	xml_put_node_open(out, "Keys");
	if((keys = ctrl_keys_get_list(ctrl)) != NULL)
	{
		for(iter = keys; iter != NULL; iter = g_slist_next(iter))
		{
			xml_put_node_open(out, "Key");
			xml_put_text(out, "keyname", ctrl_key_get_keyname((CtrlKey *) iter->data));
			xml_put_text(out, "cmdseq", ctrl_key_get_cmdseq((CtrlKey *) iter->data));
			xml_put_node_close(out, "Key");
		}
		g_slist_free(keys);
	}
	xml_put_node_close(out, "Keys");
}

/* 1999-06-20 -	Save out all mouse command bindings. */
static void mouse_save(CtrlInfo *ctrl, FILE *out)
{
	GSList	*mouse, *iter;

	xml_put_node_open(out, "MouseButtons");
	if((mouse = ctrl_mouse_get_list(ctrl)) != NULL)
	{
		for(iter = mouse; iter != NULL; iter = g_slist_next(iter))
		{
			xml_put_node_open(out, "MouseButton");
			xml_put_uinteger(out, "button", ctrl_mouse_get_button(iter->data));
			xml_put_uinteger(out, "state", ctrl_mouse_get_state(iter->data));
			xml_put_text(out, "cmdseq", ctrl_mouse_get_cmdseq(iter->data));
			xml_put_node_close(out, "MouseButton");
		}
		g_slist_free(mouse);
	}
	xml_put_node_close(out, "MouseButtons");
}

/* 1999-03-17 -	Save the current control settings right out in given <file>. */
static gint cct_save(MainInfo *min, FILE *out)
{
	xml_put_node_open(out, "Controls");
	keys_save(min->cfg.ctrlinfo, out);
	mouse_save(min->cfg.ctrlinfo, out);
	xml_put_boolean(out, "ignore_numlock", ctrl_numlock_ignore_get(min->cfg.ctrlinfo));
	xml_put_node_close(out, "Controls");

	return TRUE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-17 -	Load in and install a single key mapping. */
static void key_load(const XmlNode *node, gpointer user)
{
	const gchar	*name, *cmd;

	if(xml_get_text(node, "keyname", &name) && xml_get_text(node, "cmdseq", &cmd))
		ctrl_key_add(((MainInfo *) user)->cfg.ctrlinfo, name, csq_cmdseq_map_name(cmd, "keyboard shortcut"));
}

/* 1999-03-17 -	Load in all key mappings. */
static void keys_load(MainInfo *min, const XmlNode *node)
{
	xml_node_visit_children(node, key_load, min);
}

/* 1999-06-20 -	Load and install a single mouse command mapping. */
static void mousebutton_load(const XmlNode *node, gpointer user)
{
	const gchar	*cmd;
	guint		button, state;

	if(xml_get_uinteger(node, "button", &button) && xml_get_uinteger(node, "state", &state) && xml_get_text(node, "cmdseq", &cmd))
		ctrl_mouse_add(((MainInfo *) user)->cfg.ctrlinfo, button, state, csq_cmdseq_map_name(cmd, "dirpane mouse button"));
}

/* 1999-06-20 -	Load in the mouse command bindings. */
static void mousebuttons_load(MainInfo *min, const XmlNode *node)
{
	xml_node_visit_children(node, mousebutton_load, min);
}

/* 1999-03-17 -	Load in the control config information. Replaces current. */
static void cct_load(MainInfo *min, const XmlNode *node)
{
	const XmlNode	*data;

	if((node = xml_tree_search(node, "Controls")) != NULL)
	{
		gboolean	tmp;

		if((data = xml_tree_search(node, "Keys")) != NULL)
		{
			ctrl_keys_uninstall_all(min->cfg.ctrlinfo);
			ctrl_key_remove_all(min->cfg.ctrlinfo);
			keys_load(min, data);
		}
		if((data = xml_tree_search(node, "MouseButtons")) != NULL)
		{
			ctrl_mouse_remove_all(min->cfg.ctrlinfo);
			mousebuttons_load(min, data);
		}
		if(xml_get_boolean(node, "ignore_numlock", &tmp))
			ctrl_numlock_ignore_set(min->cfg.ctrlinfo, tmp);
	}
}

/* ----------------------------------------------------------------------------------------- */	

CfgPage * cct_describe(MainInfo *min)
{
	static CfgPage	desc = { NODE, cct_init, cct_update, cct_accept, cct_save, cct_load, NULL };

	return &desc;
}
