/*  $Id$
    PixmapArray.h - Object containing an array of Pixmaps and masks.

    gengameng - Generic 2D Game Engine library
    Copyright (C) 1999 Pierre Sarrazin <sarrazip@iname.com>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef H_PixmapArray
#define H_PixmapArray

#include "Couple.h"

#include "SDL.h"
#include "SDL_image.h"


class PixmapArray
/*  Object containing an array of Pixmaps and masks.
*/
{
public:

    typedef SDL_Surface *Pixmap;
    /*  The Pixmap type is local to this class.
	This helps avoid global namespace conflicts.
	It was defined in the global name space in version 2.0 by
	the X11 #includes.
    */


    PixmapArray(size_t numImages);
    /*  Creates a pixmap array capable of contains 'numImages' images
	and 'numImages' masks.
	'numImages' must be positive.
	This objects is the owner of the created array and pixmaps,
	and the destructor will take care of freeing these resources.
    */

    PixmapArray(Pixmap imagesArray[], Pixmap masksArray[], size_t size);
    /*  Keeps a copy of the three arguments and becomes the owner of
	the two arrays.
	'imagesArray' and 'masksArray' must not be null and must have
	been allocated with new Pixmap[].
	'size' must be positive.

	Deprecated. Use PixmapArray(size_t).
    */

    static void freePixmap(Pixmap pm);
    /*  Frees the designated pixmap.
	Does nothing if 'pm' is null.
	Portability wrapper for XFreePixmap().
    */

    void freePixmaps();
    /*  Frees the pixmaps that are stored in this object.
	'display' must not be null.
	Frees the arrays that hold the pixmaps.
	Assumes that the arrays passed to the constructor of this object
	were allocated with new Pixmap[], or that the PixmapArray(size_t)
	constructor was used.
    */

    ~PixmapArray();
    /*  Calls freePixmaps().
    */

    Pixmap getImage(size_t i) const;
    Pixmap getMask(size_t i) const;
    size_t getNumImages() const;
    /*  Returns the Pixmap of the image or mask at index 'i' of the
	arrays given to the constructor of this object.
	'i' must be lower than the value returned by getNumImages().
	The number of masks is the same as the number of images.
	This method must not be called if freePixmaps() has been called
	on this object.
    */

    void setArrayElement(size_t i, Pixmap image, Pixmap mask);
    /*  'i' must be a valid index in the array.
	'image' must not be null.
	'mask' is allowed to be null.
    */

    void setImageSize(Couple size);
    Couple getImageSize() const;
    /*  Sets or gets the size in pixels of the images in the pixmap array.
	All images in the array are assumed to be of the same size.
	Neither size.x nor size.y are allowed to be zero.
    */

private:
    #if 0
    static Display *theDisplay;  // must be init. before objects are created
    #endif

    Pixmap *images;
    Pixmap *masks;
    size_t arraySize;  // size of the array 'images' and 'masks'
    Couple imageSize;  // size in pixels of the images; all assumed same size

    /*	Forbidden operations:
    */
    PixmapArray(const PixmapArray &x);
    PixmapArray &operator = (const PixmapArray &x);
};


/*  INLINE METHODS
*/

inline PixmapArray::Pixmap
PixmapArray::getImage(size_t i) const { return images[i]; }
inline PixmapArray::Pixmap
PixmapArray::getMask(size_t i) const { return masks[i]; }
inline size_t
PixmapArray::getNumImages() const { return arraySize; }
inline Couple
PixmapArray::getImageSize() const { return imageSize; }


#endif  /* H_PixmapArray */
