/*
Copyright (C) 1993 by David H. Gay and Andrew L. Rohl 
 
dgay@ricx.royal-institution.ac.uk
andrew@ricx.royal-institution.ac.uk

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <glib.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/times.h>

#include "timer.h"

/* local prototypes */
unsigned timer_hash(char *s);
timer_type *lookup(char *s);
timer_type *find_fn(char *name);
double this_time(void);

/* local variables */
static timer_type *timer_table[TIMER_HASH_SIZE];

unsigned timer_hash(char *s)
{
unsigned hashval;
  
for (hashval=0; *s!='\0'; s++)
  hashval = *s + 31 * hashval;
return(hashval % TIMER_HASH_SIZE);
}

timer_type *lookup(char *s)
{
timer_type *np;
  
for (np = timer_table[timer_hash(s)]; np != NULL; np = np->next)
  if (strcmp(s, np->fn_name) == 0)
    return np; /* found */
return NULL; /* not found */
}

timer_type *find_fn(char *name)
{
timer_type *np;
unsigned    hashval;

/* found? */
if ((np = lookup(name)) == NULL) 
  {
  np = (timer_type *) malloc(sizeof(*np));
  if (np == NULL)
    printf("Error 1\n");
  if ((np->fn_name = strdup(name)) == NULL)
    printf("Error 2\n");
  np->n_call = 0;
  np->time = 0.0;
  hashval = timer_hash(name);
  np->next = timer_table[hashval];
  timer_table[hashval] = np;
  }
return np;
}

double this_time(void)
{
double wall_clock;

struct tms ticks;
  
wall_clock = (double)times(&ticks)/(double)CLK_TCK;
return (double) (ticks.tms_utime + ticks.tms_stime + 
                 ticks.tms_cutime + ticks.tms_cstime);
}

void start_timer(char *fn)
{
timer_type *np;
  
np = find_fn(fn);
np->n_call++;
np->start_call = this_time();
}

void stop_timer(char *fn)
{
timer_type *np;
  
np = find_fn(fn);
np->time += this_time() - np->start_call;
}

void print_times(void)
{
int i, t;
timer_type *np;

printf("\nTIMING BREAKDOWN (Ticks per second = %d)\n",(int) CLK_TCK);
printf("-----------------------------------------------------\n");
printf(" Routine                     No. calls     No. Ticks\n");
printf("-----------------------------------------------------\n");

for (i=0; i<TIMER_HASH_SIZE; i++) 
  {
  if ((np = timer_table[i]) != NULL) 
    {
    while (np != NULL) 
      {
      t = (int) np->time;
      printf(" %-23s  %9d           %d\n", np->fn_name, np->n_call, t);
      np = np->next;
      }
    }
  }
printf("\n");
}

/**************************/
/* a simple sleep routine */
/**************************/
void delay(gint micro_seconds)
{
unsigned long usec;

usec = (unsigned long) (micro_seconds);
usleep(micro_seconds);
}

