/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include "gdis.h"

/* top level data structure */
extern struct sysenv_pak sysenv;

/* task manager globals */
GtkWidget *task_list=NULL;
GtkWidget *task_label1, *task_label2, *task_label3;
gint selected_task = -1;

/***********************************/
/* process replacement system call */
/***********************************/
#define DEBUG_EXEC 0
gint exec(gchar *command)
{
gchar *argv[4];

#if DEBUG_EXEC
printf("executing [%s]\n", command);
#endif

argv[0] = "sh";
argv[1] = "-c";
argv[2] = command;
argv[3] = NULL;
execvp("sh", argv);
printf("exec() error: shoudn't get here.\n");
exit(-1);
}

/************************************/
/* interruptable forked system call */
/************************************/
gint my_system(gchar *command) 
{
gint pid, status;

if (command == 0)
  return(1);

pid = fork();
switch(pid)
  {
  case -1:
    printf("Fork failed.\n");
    return(-1);
  case 0:
    {
    gchar *argv[4];
    argv[0] = "sh";
    argv[1] = "-c";
    argv[2] = command;
    argv[3] = NULL;
    execvp("sh", argv);
    }
  default:
    do 
      {
      if (waitpid(pid, &status, 0) == -1) 
        return(-1);
      else
        return(status);
      } 
    while(1);
  }
}

/*********************************/
/* begin a queued/suspended task */
/*********************************/
gint start_queued_task(gint t)
{
pid_t pid;

/* NB: the task must have already been commited to the task list */
if (t < 0 || t >= sysenv.num_tasks)
  {
  printf("start_queued_task(): bad task number.\n");
  return(1);
  }

pid = fork();
switch(pid)
  {
/* start the main task */
  case 0:
    (sysenv.task+t)->primary ((sysenv.task+t)->ptr1);
    _exit(-1);
/* failure */
  case -1:
    printf("start_queued_task() error: fork failed.\n");
    return(1);
/* parent */
  default:
     (sysenv.task+t)->pid = pid;
     (sysenv.task+t)->status = RUNNING;
/* signal handler for child exit */
     signal(SIGCHLD, (void *) task_exit_handler);
  }

/* only the parent should get this far (if fork was successful) */
sysenv.running_tasks++;
return(0);
}

/*************************************************/
/* check if there are any queued tasks to be run */
/*************************************************/
void queued_tasks(void)
{
gint t=0;

/* try to run as many processes as allowed */
while (sysenv.running_tasks < sysenv.max_running_tasks && t < sysenv.num_tasks)
  {
/* TODO - status column could have a % if it's a gdis routine */
/* ie use a periodic signal (interrupt) to update */
  if ((sysenv.task+t)->status == QUEUED)
    start_queued_task(t);
/* next task */
  t++;
  }
}

/*******************************/
/* begin a new background task */
/*******************************/
gint start_task(struct task_pak *task)
{
gint t;

/* task list alloc */
if (!sysenv.num_tasks)
  sysenv.task = g_malloc(sizeof(struct task_pak));
else
  sysenv.task = g_renew(struct task_pak, sysenv.task, sysenv.num_tasks+1);

/* save input task */
t = sysenv.num_tasks;
memcpy((sysenv.task+t), task, sizeof(struct task_pak));
(sysenv.task+t)->pid = 0;
(sysenv.task+t)->status = QUEUED;
sysenv.num_tasks++;

/* run if possible (otherwise it'll default to queued) */
if (sysenv.running_tasks < sysenv.max_running_tasks)
  start_queued_task(t);

/* update dialog info */
update_tasks();
return(0);
}

/***********************/
/* update process info */
/***********************/
gint update_tasks(void)
{
gchar buff[5];

/* redraw task number in main display */
sprintf(buff, " %-2d", sysenv.num_tasks);
gtk_label_set_text(GTK_LABEL(sysenv.task_label), buff);

/* redraw task manager clist */
update_task_info();
return(0);
}

/********************************/
/* remove and reorder task list */
/********************************/
gint remove_task(gint i)
{
gint j;

/* free task data */
g_free((sysenv.task+i)->label);
/* reorder list - overwrite if necessary */
j = sysenv.num_tasks - i - 1;
if (j > 0)
  memcpy((sysenv.task+i), (sysenv.task+i+1), j*sizeof(struct task_pak));
/* shrink the task list & memory allocation */
sysenv.num_tasks--;
/*
sysenv.running_tasks--;
*/
if (sysenv.num_tasks)
  sysenv.task = g_renew(struct task_pak, sysenv.task, sysenv.num_tasks);
else
  g_free(sysenv.task);

return(0);
}

/*************************************************/
/* signal handler to catch the exit of any child */
/*************************************************/
#define DEBUG_TASK_EXIT_HANDLER 0
void task_exit_handler(gint signum)
{
gint i, j, pid, status;

/* don't block while getting exited children */
while((pid = waitpid(-1, &status, WNOHANG)) > 0)
  {
#if DEBUG_TASK_EXIT_HANDLER
printf("Caught exit of child: %d, status: %d\n", pid, status);
#endif
/* check the task list to see if it was one of ours */
  for (i=0 ; i<sysenv.num_tasks ; i++)
    {
    if ((sysenv.task+i)->pid == pid)
      {
#if DEBUG_TASK_EXIT_HANDLER
printf("Completed task: %d, pid: %d\n", i, pid);
#endif
      if (status)
        show_text("job terminated abnormally.");
      else
        {
        show_text("job successfully completed.");
        (sysenv.task+i)->cleanup ((sysenv.task+i)->ptr2);
        }
/* free task data */
      g_free((sysenv.task+i)->label);
/* reorder list - overwrite if necessary */
      j = sysenv.num_tasks - i - 1;
      if (j > 0)
        memcpy((sysenv.task+i), (sysenv.task+i+1), j*sizeof(struct task_pak));
/* shrink the task list & memory allocation */
      sysenv.num_tasks--;
      sysenv.running_tasks--;
      if (sysenv.num_tasks)
        sysenv.task = g_renew(struct task_pak, sysenv.task, sysenv.num_tasks);
      else
        g_free(sysenv.task);
      }
    }

  queued_tasks();

  update_tasks();
  }
/* done */
}

/*********************************/
/* rewrite the current task list */
/*********************************/
void update_task_info()
{
gint i, flag;
gchar *txt, *info[3];

/* check for dialog */
/* NB: if not done this causes a wierd bug - */
/* open task manager : do a run & wait until completed : close task manager */
/* the next run will core dump when the task ends & this routine is run */
/* presumably because task_list is a completely valid widget, but no */
/* longer is displayed in the task manger dialog */
flag=0;
for (i=MAX_DIALOGS ; i-- ; )
  if (sysenv.dialog[i].active)
    if (sysenv.dialog[i].type == TASKMAN)
      {
      flag++;
      break;
      }
if (!flag)
  return;

/* checks */
if (!GTK_IS_CLIST(task_list))
  return;

/* update task summary info */
txt = g_strdup_printf("%d", sysenv.num_tasks);
gtk_label_set_text(GTK_LABEL(task_label1), txt);
g_free(txt);

txt = g_strdup_printf("%d", sysenv.running_tasks);
gtk_label_set_text(GTK_LABEL(task_label2), txt);
g_free(txt);

txt = g_strdup_printf("%d", sysenv.max_running_tasks);
gtk_label_set_text(GTK_LABEL(task_label3), txt);
g_free(txt);


/* delete all existing entries */
gtk_clist_freeze(GTK_CLIST(task_list));
gtk_clist_clear(GTK_CLIST(task_list));

for (i=0 ; i<sysenv.num_tasks ; i++)
  {
/* TODO - status column could have a % if it's a gdis routine */
/* ie use a periodic signal (interrupt) to update */
  if ((sysenv.task+i)->pid)
    info[0] = g_strdup_printf("%d", (sysenv.task+i)->pid);
  else
    info[0] = g_strdup(" ");

  if ((sysenv.task+i)->label)
    info[1] = g_strdup((sysenv.task+i)->label);
  else
    info[1] = g_strdup(" Unknown ");

  switch((sysenv.task+i)->status)
    {
    case QUEUED:
      info[2] = g_strdup("Queued");
      break;
    case RUNNING:
      info[2] = g_strdup("Running");
      break;
    case SUSPENDED:
      info[2] = g_strdup("Suspended");
      break;
    default:
      info[2] = g_strdup("Unknown");
    }
/* add to the list */
  gtk_clist_append(GTK_CLIST(task_list), &info[0]);
/* free */
  g_free(info[0]);
  g_free(info[1]);
  g_free(info[2]);
  } 

/* update */
gtk_clist_thaw(GTK_CLIST(task_list));
}

/******************************/
/* task dialog event handlers */
/******************************/

/* select */
gint select_task(GtkWidget *clist, gint row, gint column,
                 GdkEventButton *event, gpointer data)
{
selected_task = row;
return(FALSE);
}
/* unselect */
gint unselect_task(GtkWidget *clist, gint row, gint column,
                   GdkEventButton *event, gpointer data)
{
selected_task = row;
return(FALSE);
}

/* kill */
gint kill_task(gint mode)
{
gint i, flag=0;

/* NB: signals will trigger the task update stuff */
switch(mode)
  {
  case SINGLE:
    if (selected_task >= 0 && selected_task < sysenv.num_tasks)
      {
/* if running kill, else delete from list */
      if ((sysenv.task+selected_task)->status == RUNNING)
        {
        kill((sysenv.task+selected_task)->pid, SIGTERM);
        flag++;
        }
      else
        remove_task(selected_task);
      }
    break;
  case ALL:
    for (i=0 ; i<sysenv.num_tasks ; i++)
      {
      if ((sysenv.task+i)->status == RUNNING)
        {
        kill((sysenv.task+i)->pid, SIGTERM);
        flag++;
        }
      else
        remove_task(i);
      }
    break;
  default:
    printf("kill_task(): bad argument.\n");
  }

/* if a kill was issued - update_task_info will automatically be invoked */
if (!flag)
  update_task_info();

/* the select highligh will vanish -> nothing selected */
selected_task = -1;
return(FALSE);
}

/****************************************************/
/* a task dialog for viewing/killing existing tasks */
/****************************************************/
void task_dialog()
{
gint id;
GtkWidget *frame, *vbox, *hbox, *button, *label;
GtkWidget *scr_win;
gchar *titles[3] = {"  PID  ", "       Job       ", "   Status   "};
gchar *txt;
struct dialog_pak *task_dialog;

/* request a new dialog */
if ((id = request_dialog(sysenv.active, TASKMAN)) < 0)
  return;
task_dialog = &sysenv.dialog[id];

/* create new dialog */
task_dialog->win = gtk_dialog_new();
gtk_window_set_title(GTK_WINDOW (task_dialog->win), "Task Manager");
gtk_window_set_default_size(GTK_WINDOW(task_dialog->win), 280, 350);
gtk_signal_connect(GTK_OBJECT(task_dialog->win), "destroy",
                   GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* Frame */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(task_dialog->win)->vbox),frame,FALSE,FALSE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);

/* create a hbox in the frame */
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), hbox);
gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);

/* create vbox */
vbox = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
/* labels - running tasks, total tasks, allowed running tasks */
label = gtk_label_new("Number of tasks ");
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, FALSE, 0);
label = gtk_label_new("Number of running tasks ");
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, FALSE, 0);
label = gtk_label_new("Allowed number of running tasks ");
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, FALSE, 0);

/* create vbox */
vbox = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);
/* values - running tasks, total tasks, allowed running tasks */
txt = g_strdup_printf("%d", sysenv.num_tasks);
task_label1 = gtk_label_new(txt);
gtk_misc_set_alignment(GTK_MISC(task_label1), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), task_label1, TRUE, FALSE, 0);
g_free(txt);

txt = g_strdup_printf("%d", sysenv.running_tasks);
task_label2 = gtk_label_new(txt);
gtk_misc_set_alignment(GTK_MISC(task_label2), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), task_label2, TRUE, FALSE, 0);
g_free(txt);

txt = g_strdup_printf("%d", sysenv.max_running_tasks);
task_label3 = gtk_label_new(txt);
gtk_misc_set_alignment(GTK_MISC(task_label3), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), task_label3, TRUE, FALSE, 0);
g_free(txt);

/* Frame */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(task_dialog->win)->vbox),frame,TRUE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
/* create a vbox in the frame */
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width (GTK_CONTAINER(GTK_BOX(vbox)), 3);

/* hbox for the modes */
/*
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
label = gtk_label_new("Task list");
gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, FALSE, 0);
*/

/* scrolled model pane */
scr_win = gtk_scrolled_window_new (NULL, NULL);
gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr_win),
                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(vbox), scr_win, TRUE, TRUE, 0);
/* Create the CList */
task_list = gtk_clist_new_with_titles(3, titles);
/* events */
gtk_signal_connect(GTK_OBJECT(task_list), "select_row",
                   GTK_SIGNAL_FUNC(select_task), NULL);
gtk_signal_connect(GTK_OBJECT(task_list), "unselect_row",
                   GTK_SIGNAL_FUNC(unselect_task), NULL);

/* set some properties of the CList */
gtk_clist_set_shadow_type (GTK_CLIST(task_list), GTK_SHADOW_OUT);
gtk_container_add(GTK_CONTAINER(scr_win), task_list);

/* control buttons */
hbox = gtk_hbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
/* col 1 - only one */
button = gtk_button_new_with_label ("  Kill  ");
gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (kill_task),
                           (gpointer) SINGLE);
/* col 2 - all labels */
button = gtk_button_new_with_label (" Kill all ");
gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (kill_task),
                           (gpointer) ALL);

/* terminating button */
button = gtk_button_new_with_label ("  Close  ");
gtk_box_pack_start (GTK_BOX (GTK_DIALOG (task_dialog->win)->action_area),
                                         button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC(close_dialog),
                           (gpointer) id);

/* done */
gtk_widget_show_all(task_dialog->win);
/* refresh labels */
update_task_info();
}
