/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "gdis.h"
#include "elem.h"

/* main data structure */
struct sysenv_pak sysenv;

/**********************************/
/* read the init file if possible */
/**********************************/
gint read_gdisrc(void)
{
gint i, element_flag;
gchar line[LINELEN], **buff;
gfloat version;
FILE *fp;

/* attempt to open */
fp = fopen(sysenv.gdisrc, "r");

/* check for an old/bad gdisrc */
if (fp)
  {
/* scan the 1st line */
  fgetline(fp, line);
/* test for right format */
  buff = get_tokens(line, 2);
  if (g_strncasecmp(*buff,"gdis",4) == 0)
    {
/* test for right version */
    sscanf(*(buff+1),"%f",&version);
    if (version < (gfloat) atof(VERSION))
      return(1);
    }
  else
    return(1);
  g_strfreev(buff);
  }
else
  return(1);

printf("Reading %s\n", sysenv.gdisrc);

/* read until EOF, or (failsafe) reached elements[] array allocation */
element_flag=0;
i=0;
while(!fgetline(fp,line))
  {
  buff = get_tokens(line, 11);
/* decide what to read */
  if (g_strncasecmp("size",*buff,4) == 0)
    {
    sscanf(*(buff+1),"%3d",&sysenv.width);
    sscanf(*(buff+2),"%3d",&sysenv.height);
    if (sysenv.width < MIN_WIDTH)
      sysenv.width = MIN_WIDTH;
    if (sysenv.height < MIN_HEIGHT)
      sysenv.height = MIN_HEIGHT;
    } 
/* povray executable */
  if (g_strncasecmp("povray_exe",*buff,10) == 0)
    {
    if (strlen(*(buff+1)))
      {
      g_free(sysenv.povray_exe);
      sysenv.povray_exe = g_strdup(*(buff+1));
      }
    }
/* animation creation executable */
  if (g_strncasecmp("convert_exe",*buff,11) == 0)
    {
    if (strlen(*(buff+1)))
      {
      g_free(sysenv.convert_exe);
      sysenv.convert_exe = g_strdup(*(buff+1));
      }
    }
/* image viewing executable */
  if (g_strncasecmp("viewer_exe",*buff,10) == 0)
    {
    if (strlen(*(buff+1)))
      {
      g_free(sysenv.viewer_exe);
      sysenv.viewer_exe = g_strdup(*(buff+1));
      }
    }
/* gulp executable */
  if (g_strncasecmp("gulp",*buff,4) == 0)
    {
    if (strlen(*(buff+1)))
      {
      g_free(sysenv.gulp_exe);
      sysenv.gulp_exe = g_strdup(*(buff+1));
      }
    }
/* file conversion executable */
  if (g_strncasecmp("babel_exe",*buff,9) == 0)
    {
    if (strlen(*(buff+1)))
      {
      g_free(sysenv.babel_exe);
      sysenv.babel_exe = g_strdup(*(buff+1));
      }
    }

/* graphics label font */
  if (g_strncasecmp("font",*buff,4) == 0)
    if (strlen(*(buff+1)))
      strcpy(sysenv.font, *(buff+1));

/* element data */
  if (g_strncasecmp("elem",*buff,4) == 0)
    element_flag=1;

  if (element_flag)
    {
/* skip anything that has < 6 tokens */
    if (!strlen(*(buff+5)))
      continue;
/* done? */
    if (g_strncasecmp("end",*buff,3) == 0 || g_strncasecmp("eof",*buff,3) == 0)
      {
      element_flag=0;
      continue;
      }
/* read in the element data */
    sscanf(*(buff+0),"%s",&(elements[i].symbol[0]));
    sscanf(*(buff+1),"%s",&(elements[i].name[0]));
    sscanf(*(buff+2),"%f",&(elements[i].weight));
    sscanf(*(buff+3),"%f",&(elements[i].cova));
    sscanf(*(buff+4),"%f",&(elements[i].vdw));
/* ??? why is #6 read in as one number token??? */
    sscanf(*(buff+5),"%d",&(elements[i].colour[0]));
    sscanf(*(buff+6),"%d",&(elements[i].colour[1]));
    sscanf(*(buff+7),"%d",&(elements[i].colour[2]));
    i++;
    }
  g_strfreev(buff);
  }

return(0);
}

/*********************************************/
/* write setup & elements to the gdisrc file */
/*********************************************/
gint write_gdisrc(void)
{
gint i;
FILE *fp;

fp = fopen(sysenv.gdisrc,"w");
if (!fp)
  {
  printf("Error: unable to create %s\n", sysenv.gdisrc);
  return(1);
  }

fprintf(fp,"gdis %s\n",VERSION);
fprintf(fp,"size %d %d\n",sysenv.width,sysenv.height);
fprintf(fp,"font %s\n",sysenv.font);
fprintf(fp,"povray_exe %s\n",sysenv.povray_exe);
fprintf(fp,"convert_exe %s\n",sysenv.convert_exe);
fprintf(fp,"viewer_exe %s\n",sysenv.viewer_exe);
fprintf(fp,"gulp_exe %s\n",sysenv.gulp_exe);
fprintf(fp,"babel_exe %s\n",sysenv.babel_exe);
fprintf(fp,"elements\n");

/* TODO - eliminate the need for size to be passed */
for(i=0 ; i<sysenv.num_elements ; i++)
  {
  fprintf(fp, "%-3s  %-15s   %-8.4f  %-6.4f   %-6.4f   %-5d %-5d %-5d\n",
               elements[i].symbol,
               elements[i].name,
               elements[i].weight,
               elements[i].cova,
               elements[i].vdw,
               elements[i].colour[0],
               elements[i].colour[1],
               elements[i].colour[2]);
  }
fprintf(fp,"EOF\n");
fclose(fp);
return(0);
}

/****************************/
/* check for file existance */
/****************************/
#define DEBUG_CHECK_FILE 0
gint check_file(const gchar *name, gint status)
{
gint i, ret;
gchar **prefix, *file, *path;

/* get all PATH entries from environ */
path = g_strdup(getenv("PATH"));
g_return_val_if_fail(path != NULL,0);
prefix = g_strsplit(path, ":", MAX_PATH_ITEMS); 

/* check file for required status */
ret=0;
switch(status)
  {
  case EXEC_PATH:
#if DEBUG_CHECK_FILE
printf("Checking for executable: %s\n",name);
#endif
    i=0;
    while(*(prefix+i) != NULL)
      {
      file = g_strconcat(*(prefix+i), "/", name, NULL);
      if (!access(file, X_OK))
        {
#if DEBUG_CHECK_FILE
printf("found: %s\n",file);
#endif
        g_free(file);
        ret = 1;
        break;
        }
      g_free(file);
      i++;
      }
    break;
  default:
    printf("check_file() error: unknown check type.\n");
    break;
  }

/* notify the user */
if (!ret)
  printf("WARNING: %s was not found in your path.\n",name);

/* clean up & exit */
g_free(path);
g_strfreev(prefix);
return(ret);
}

/**************/
/* main setup */
/**************/
void sys_init()
{
gint i;
gchar name[LINELEN], *tmp;

/* top level structure initialization */
sysenv.num_tasks = 0;
sysenv.running_tasks = 0;
sysenv.max_running_tasks = 1;
sysenv.num_models = 0;
sysenv.width = START_WIDTH;
sysenv.height = START_HEIGHT;
/* default to single model display */
sysenv.displayed[0] = -1;
sysenv.active = 0;
sysenv.select_mode = ATOM;
mod_screen(1,1);

/* default masks */
sysenv.file_type = DATA;
sysenv.babel_type = NONE;

/* ascertain how many elements defined in elem.h */
sysenv.num_elements = sizeof(elements) / sizeof(struct elem_pak);

/* NEW - unified rendering setup */
sysenv.render.type = BALL_STICK;
sysenv.render.width = 600;
sysenv.render.height = 600;
sysenv.render.vp_dist = 5.0;
sysenv.render.background = 0;
sysenv.render.num_lights = 0;
sysenv.render.light = NULL;
sysenv.render.antialias = FALSE;
sysenv.render.shadowless = FALSE;
sysenv.render.animate = FALSE;
sysenv.render.atype = FALSE;
sysenv.render.axes = FALSE;
sysenv.render.wire_csurf = FALSE;
sysenv.render.wire_morph = FALSE;
sysenv.render.delay = 20;
sysenv.render.ambience = 0.2;
sysenv.render.diffuse = 0.8;
sysenv.render.specular = 0.8;
sysenv.render.ball_rad = 0.35;
sysenv.render.stick_rad = 0.1;
sysenv.render.line_thickness = GTKGL_LINE_WIDTH;
sysenv.render.frame_thickness = GTKGL_LINE_WIDTH;
sysenv.render.cpk_scale = 1.0;
VEC3SET(sysenv.render.morph_colour, 0.0, 1.0, 1.0);

/* generate name of users RC file */
tmp = (gchar *) getenv("HOME");
if (tmp)
  {
  strcpy(name, tmp);
  strcat(name, "/");
  strcat(name, RCFILE);
/* save for later updating */
  strcpy(sysenv.gdisrc,name);
  }
else
  {
  printf("sys_init() error: your HOME environment variable is not set.\n");
  exit(1);
  }

/* set cwd for gdis */
tmp = (gchar *) getenv("PWD");
/* gdis core dumps if PWD is not defined, (eg bash-1.14) */
if (!tmp)
  {
  printf("sys_init() warning: PWD not defined.\n");
  tmp = (gchar *) getenv("HOME");
  }
strcpy(sysenv.cwd, tmp);
strcat(sysenv.cwd, "/");

/* defaults */
sysenv.povray_exe = g_strdup("povray");
sysenv.convert_exe = g_strdup("convert");
sysenv.viewer_exe = g_strdup("display");
sysenv.gulp_exe = g_strdup("gulp");
sysenv.babel_exe = g_strdup("babel");
strcpy(sysenv.font,FONT);

/* if read failed - write a new one */
/* TODO - if overwrite old version, save/rename first? */
if (read_gdisrc())
  {
  printf("Writing %s\n",name);
  write_gdisrc();
  }

/* NEW - set flags according to whether executables were found or not */
sysenv.have_povray = sysenv.have_convert = sysenv.have_viewer = 0;
sysenv.have_gulp = sysenv.have_babel = 0;
if (check_file(sysenv.povray_exe, EXEC_PATH))
  sysenv.have_povray = 1;
if (check_file(sysenv.convert_exe, EXEC_PATH))
  sysenv.have_convert = 1;
if (check_file(sysenv.viewer_exe, EXEC_PATH))
  sysenv.have_viewer = 1;
if (check_file(sysenv.gulp_exe, EXEC_PATH))
  sysenv.have_gulp = 1;
if (check_file(sysenv.babel_exe, EXEC_PATH))
  sysenv.have_babel = 1;

/* dialog init */
for (i=0 ; i<MAX_DIALOGS ; i++)
  sysenv.dialog[i].active = FALSE;
/* tree init */
sysenv.num_trees = 0;
/* copied selection */
sysenv.select_source = NULL;

/* NEW - trig lookup table */
init_trig();
}

/*****************/
/* PROMPT thread */
/*****************/
/*
void prompt()
{
gchar input[10];
 
for (;;)
  {
  printf("Command: ");
  fgets(input,10,stdin);
  if (!g_strncasecmp(input,"quit",4))
    {
    gtk_exit(0);
    }
  }
}
*/

/********/
/* MAIN */
/********/
gint main (int argc, char *argv[])
{

/* arguments */
if (argc > 1)
  {
  if (g_strncasecmp(argv[1],"-v",2) == 0)
    {
    printf("GDIS version %s.%-d\n",VERSION,PATCH);
    printf("(c) 2000 Sean Fleming\n\n");
    return(0);
    }
  }
/* GPL blurb */
printf("GDIS version %s.%d, Copyright (C) 2000 by Sean Fleming\n\n",VERSION,PATCH);
printf("GDIS comes with ABSOLUTELY NO WARRANTY.\n\n");
printf("This is free software. You are welcome to redistribute copies\n");
printf("provided the conditions of the GNU Public License (GPL) are met.\n");
printf("You should have received a copy of the file GPL.txt along with\n");
printf("this program; if not, write to the Free Software Foundation Inc.,\n");
printf("59 Temple Place, Suite 330, Boston, MA  02111-1307  USA\n\n");

/* initialize */
sys_init();
gtk_init(&argc, &argv);

/* set up main window and event handlers */
connect_events();
/* default, don't write */
sysenv.write_gdisrc = FALSE;

/* NEW - create gdis command prompt thread */
/*
pthread_create(&thread1, NULL, (void *) &prompt, (void*) NULL);
*/

/* wait for events */
gtk_main();
return(0);
}

