/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/param.h>
#ifdef __sun
#include <sys/dirent.h>
#else
#include <sys/dir.h>
#endif

#include "gdis.h"

#define DEBUG_MORE 0
#define MAX_KEYS 15

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/************************/
/* save morphology data */
/************************/
gint write_gmf(gchar *name, struct model_pak *data)
{
gint h, k, l;
FILE *fp;
struct plane_pak *plane;
GSList *plist;

g_return_val_if_fail(data != NULL, 1);
/* TODO - flag visible / option to save or not? */

/* open for saving */
fp = fopen(name,"wt");
if (!fp)
  return(1);

fprintf(fp, "TITLE morphology\n");
fprintf(fp, "CELL %f %f %f  %f %f %f\n", data->pbc[0], data->pbc[1], data->pbc[2],
             180.0*data->pbc[3]/PI, 180.0*data->pbc[4]/PI, 180.0*data->pbc[5]/PI);

/* TODO - spacename/number */

/* TODO - + general format string */
fprintf(fp,"HKL len dhkl eatt esurf\n");

/* print out the planes */
plist = data->planes;
while (plist != NULL)
  {
  plane = (struct plane_pak *) plist->data;
  if (plane->num_points)
    {
    h = (gint) plane->m[0];
    k = (gint) plane->m[1];
    l = (gint) plane->m[2];
    fprintf(fp,"%3d  %3d  %3d    ", h, k, l);
    fprintf(fp,"%10.4f  %10.4f  %10.4f  %10.4f\n",
                plane->len, plane->dhkl, plane->Eatt, plane->Esurf);
    }
  plist = g_slist_next(plist);
  }

fclose(fp);
return(0);
}

/******************************************/
/* load miller & normal lengths from file */
/******************************************/
#define DEBUG_LOAD_PLANES 0
gint load_planes(FILE *fp, struct model_pak *data)
{
gint i, j, flag, num_cols;
gint item[5];
gchar line[LINELEN], **key, **buff;
struct plane_pak *plane;

g_return_val_if_fail(fp != NULL, 1);

flag=0;
i=0;
while(!fgetline(fp,line))
  {
  key = get_tokens(line, 8);
/* scan for PBC */
  if (g_strncasecmp(*key,"CELL",4) == 0)
    {
    sscanf(*(key+1),"%f",&(data->pbc[0]));
    sscanf(*(key+2),"%f",&(data->pbc[1]));
    sscanf(*(key+3),"%f",&(data->pbc[2]));
    sscanf(*(key+4),"%f",&(data->pbc[3]));
    sscanf(*(key+5),"%f",&(data->pbc[4]));
    sscanf(*(key+6),"%f",&(data->pbc[5]));
/* degrees -> radians */
    data->pbc[3] *= PI/180.0;
    data->pbc[4] *= PI/180.0;
    data->pbc[5] *= PI/180.0;
    } 

/* TODO - space group */

/* scan for data */
  if (g_strncasecmp(*key,"DATA",4) == 0 || g_strncasecmp(*key,"HKL",3) == 0)
    {
    buff = get_tokens(line, 6);
/* parse data column format (if any) */
    i=1;
    num_cols=0;
    while(strlen(*(buff+i)))
      {
      if (g_strncasecmp(*(buff+i), "len", 3) == 0)
        {
        item[num_cols] = LENGTH;
        num_cols++;
        }
      if (g_strncasecmp(*(buff+i), "dhkl", 4) == 0)
        {
        item[num_cols] = DHKL;
        num_cols++;
        }
      if (g_strncasecmp(*(buff+i), "Ea", 2) == 0)
        {
        item[num_cols] = EATT;
        num_cols++;
        }
      if (g_strncasecmp(*(buff+i), "Es", 2) == 0)
        {
        item[num_cols] = ESURF;
        num_cols++;
        }
      i++;
      }
    g_strfreev(buff);

#if DEBUG_LOAD_PLANES
printf("Expecting %d column data items, type(s)", num_cols);
for (j=0 ; j<num_cols ; j++)
  printf(" : %d", item[j]);
printf("\n");
#endif

/* if no data format - assume length values */
   if (!num_cols)
     {
     num_cols++;
     item[0] = LENGTH;
     }

/* get subequent data lines */
    i=0;
    while(!fgetline(fp,line))
      {
      buff = get_tokens(line, 10);
      if (strlen(*(buff+3)))
        {
/* make a new plane */
        plane = g_malloc(sizeof(struct plane_pak));
        plane->m[0] = str_to_float(*(buff+0));
        plane->m[1] = str_to_float(*(buff+1));
        plane->m[2] = str_to_float(*(buff+2));
/* default vals */
        plane->len = plane->dhkl = plane->Eatt = plane->Esurf = 0.0;
/* check data column format for which (len/dhjl,Eatt,Esurf) to assign */
        for (j=0 ; j<num_cols ; j++)
          {
          switch(item[j]) 
            {
            case LENGTH:
              plane->len = str_to_float(*(buff+j+3));
              if (plane->len > FRACTION_TOLERANCE)
                data->type = LENGTH;
              break;
            case DHKL:
              plane->dhkl = str_to_float(*(buff+j+3));
              if (plane->dhkl > FRACTION_TOLERANCE)
                data->type = DHKL;
              break;
            case EATT:
              plane->Eatt = str_to_float(*(buff+j+3));
              if (plane->Eatt > FRACTION_TOLERANCE)
                data->type = EATT;
              break;
            case ESURF:
              plane->Esurf = str_to_float(*(buff+j+3));
              if (plane->Esurf > FRACTION_TOLERANCE)
                data->type = ESURF;
              break;
            default:
              printf("Shouldn't get here.\n");
            }
          }

/* add to the list */
        data->planes = g_slist_prepend(data->planes, plane);
        g_strfreev(buff);
        i++;
        }
      }
/* we've been prepending - so reverse the list order */
    data->planes = g_slist_reverse(data->planes);
    data->num_planes = i;
    }
  g_strfreev(key);
  }
#if DEBUG_LOAD_PLANES
printf("Read in %d planes.\n", data->num_planes);
#endif
return(0);
}

/***********************************************************/
/* determine if two facets in a given model are equivalent */
/***********************************************************/
/* f1 & f2 are vecs with the h k l values */
gint facet_equiv(struct model_pak *data, gint *f1, gint *f2)
{
gint i;
gfloat vec[3];

if (data->periodic == 3)
  {
  if (data->sginfo.spacenum)
    {
    for (i=0 ; i<data->sginfo.order ; i++)
      {
      ARR3SET(vec, f1);
      vecmat(*(data->sginfo.matrix+i), vec);
      if (*vec == *f2 && *(vec+1) == *(f2+1) && *(vec+2) == *(f2+2))
        return(TRUE);  
      }
    }
  else
    printf("No space group information.\n");
  }
else
  {
  printf("Sorry can't do point group only symmetry yet.\n");
  }
return(FALSE);
}

/***********************************/
/* determine if a plane is visible */
/***********************************/
#define DEBUG_FACET 0
gint facet_visible(struct model_pak *data, struct plane_pak *plane)
{
gfloat a, norm[3], vec[3];

/* get (unrotated!) normal */
ARR3SET(norm, plane->norm);

/* rotate it */
vecmat(data->rotmat, norm);

#if DEBUG_FACET
printf("facet (%2d%2d%2d), norm: %5.2f,%5.2f,%5.2f    " 
,plane->index[0] ,plane->index[1] ,plane->index[2]
,norm[0],norm[1],norm[2]);
#endif
/* absolute viewing vector, determining if visible or not */
VEC3SET(vec, 0.0, 0.0, -1.0);

/* got correct facet normal - is it visible? */
a = via(norm,vec,3);
if (a < PI/2.0)
  {
#if DEBUG_FACET
printf("view: %5.2f,%5.2f,%5.2f (%5.1f) YES\n",vec[0],vec[1],vec[2],180.0*a/PI);
#endif
  return(1);
  }
/* else... */
#if DEBUG_FACET
printf("view: %5.2f,%5.2f,%5.2f (%5.1f) NO\n",vec[0],vec[1],vec[2],180.0*a/PI);
#endif
return(0);
}

/***********************************/
/* determine if a ridge is visible */
/***********************************/
gint ridge_visible(struct model_pak *data, gint i, gint j)
{
gint p, id, flag;
GSList *plist;
struct plane_pak *plane;

/* loop through all planes */
plist = data->planes;
while (plist != NULL)
  {
  plane = (struct plane_pak *) plist->data;
  if (plane->present && plane->visible)
    {
    flag=0;
/* if facet is visible, check if bounded by i or j  */
    for (p=0 ; p<plane->num_points ; p++)
      {
      id = *(plane->points+p);
      if (id == i || id == j)
        flag++;
      }
/* found a visible face containing BOTH pts -> ridge is visible */
    if (flag == 2)
      return(1);
    }
  plist = g_slist_next(plist);
  }
return(0);
}

/***********************/
/* Morphology creation */
/***********************/
#define DEBUG_MORPH_CREATE 0
gint load_gmf(FILE *fp, gint model, gchar *filename)
{
gfloat len;
gfloat norm[3];
GSList *plist;
struct model_pak *data=NULL;
struct plane_pak *plane;

/* make a new model */
data = model_ptr(model, ASSIGN);
if (data == NULL)
  return(1);
sysenv.active = model;
data->mode = FREE;
/* setup model parameters */
data->id = MORPH;
data->axes_type = OTHER;
data->hpr = TRUE;
data->morph_label = TRUE;
strcpy(data->filename, filename);
g_free(data->basename);
data->basename = strdup_no_extension(g_basename(data->filename));

/* read from a gmf file */
if (load_planes(fp, data))
  {
  printf("File load failed, check %s.\n",data->filename);
  delete_model(model);
  return(1);
  }

/* CURRENT - input length is cartesian, miller indices are lattice space */
make_xlat(data);
make_rlat(data);

/* convert miller indices to cartesian inequalities */
plist = data->planes;
while (plist != NULL)
  {
  plane = (struct plane_pak *) plist->data;
/* use reciprocal vectors to get the direct lattice plane normal */
  ARR3SET(norm, plane->m); 
  vecmat(data->rlatmat, norm);

/* normalize */
  len = VEC3MAG(norm);
  if (len > FRACTION_TOLERANCE)
    {
    VEC3MUL(norm, 1.0/len);
/* NB: cdd ineq. format needs the *negative* of the miller index */
    VEC3MUL(norm, -1.0);
    ARR3SET(plane->x, norm);
    }
  else
    {
/* make it so gdis safely ignores such a thing */
    printf("Warning: 0 length face found.\n");
    return(1);
    }
  plist = g_slist_next(plist);
  }

/* construct the hull (convert to H representation) */
data->num_vertices=0;
data->num_facets=0;
exec_cdd(data);

#if DEBUG_MORPH_CREATE
printf("-------------------------\n");
printf("       Input planes: %d\n",data->num_planes);
printf("Calculated vertices: %d\n",data->num_vertices);
printf("  Calculated facets: %d\n",data->num_facets);
printf("-------------------------\n");
#endif

if (!data->num_facets)
  {
/* cleanup!!! */
  delete_model(model);
  show_text("Morphology calculation failed.");
  return(1);
  }

/* vertices are cartesian - so a fudge is needed */
latmat_fudge(data);

/* initialize */
data->axes_on = TRUE;
init_objs(INIT_COORDS, data);   /* setup non90 stuff */

/* proper rmax scaling & visibility calc */
init_objs(REDO_COORDS, data);

new_tree_item(model, APPEND);

return(0);
}

#if GTK_EXTRA
/***********************************************/
/* Write current morphology data to a GtkSheet */
/***********************************************/
void update_morph_table(struct model_pak *data)
{
gint i, h, k, l;
GString *line;
GSList *plist;
struct plane_pak *plane;

/* checks */
g_return_if_fail(data != NULL);
g_return_if_fail(GTK_IS_SHEET(data->morph_sheet) != FALSE);

/* init */
line = g_string_new(NULL);

/* loop over rows (hkl) */
i=0;
plist = data->planes;
while (plist != NULL)
  {
  plane = (struct plane_pak *) plist->data;
/* col 1 */
  h = (gint) plane->m[0];
  k = (gint) plane->m[1];
  l = (gint) plane->m[2];
  g_string_sprintf(line, "%d %d %d", h, k, l);
  gtk_sheet_set_cell_text(GTK_SHEET(data->morph_sheet), i, 0, line->str);

/* col 2 */
  g_string_sprintf(line, "%f", plane->len);
  gtk_sheet_set_cell_text(GTK_SHEET(data->morph_sheet), i, 1, line->str);
/* col 3 */
  g_string_sprintf(line, "%f", plane->dhkl);
  gtk_sheet_set_cell_text(GTK_SHEET(data->morph_sheet), i, 2, line->str);
/* col 4 */
  g_string_sprintf(line, "%f", plane->Eatt);
  gtk_sheet_set_cell_text(GTK_SHEET(data->morph_sheet), i, 3, line->str);
/* col 5 */
  g_string_sprintf(line, "%f", plane->Esurf);
  gtk_sheet_set_cell_text(GTK_SHEET(data->morph_sheet), i, 4, line->str);

  plist = g_slist_next(plist);
/* TODO - spreadsheet MAX ROWS */
  if (i++ > 90)
    break; 
  }

g_string_free(line, TRUE);
}

/*****************************/
/* callback for cell changes */
/*****************************/
void cell_change(GtkWidget *w, gint row, gint col, gpointer ptr)
{
gchar *txt;
struct plane_pak *plane;
struct model_pak *data;

txt = gtk_sheet_cell_get_text(GTK_SHEET(w), row, col);
/*
printf("[%d %d] : %s\n",row,col,txt);
*/

data = (struct model_pak *) ptr;

/* boundary checks */
if (row < 0 || row >= data->num_planes)
  return;
if (col < 1 || col > 4)
  return;

plane = (struct plane_pak *) g_slist_nth_data(data->planes, row);

/* NB: correct order must be kept for this to work!!! */
col--;
switch(col)
  {
  case LENGTH:
    plane->len = str_to_float(txt);
    break;
  case DHKL:
    plane->dhkl = str_to_float(txt);
    break;
  case EATT:
    plane->Eatt = str_to_float(txt);
    break;
  case ESURF:
    plane->Esurf = str_to_float(txt);
    break;
  }
}

/******************************/
/* init and redraw morphology */
/******************************/
gint redo_morph(struct model_pak *data)
{
/* TODO - rescan spreadsheet here??? (instead of continues updates) */

/* recalc */
if (!exec_cdd(data))
  {
/* reinit */
  latmat_fudge(data);
  init_objs(INIT_COORDS, data);   /* setup non90 stuff */
  init_objs(REDO_COORDS, data);
  redraw_canvas(SINGLE);
  show_text("Morphology drawn.");
  }
else
  show_text("Bad input plane list.");

return(FALSE);
}
/******************************************/
/* callback for morphology type selection */
/******************************************/
gint switch_type(gpointer *button)
{
gint type;
struct model_pak *data;

/* get data */
data = (struct model_pak *) gtk_object_get_data(GTK_OBJECT(button), "ptr");
type = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(button), "id"));
g_return_val_if_fail(data != NULL, FALSE);

/* recalc new morph */
data->type = type;

redo_morph(data);
return(FALSE);
}
#endif

/******************************************/
/* Display & edit a morphology facet list */
/******************************************/
void morph_widget(GtkWidget *w, struct model_pak *data)
{
#if GTK_EXTRA
gint id;
GString *line;
GSList *radio;
GtkWidget *scr_win, *frame, *vbox, *label, *button;
struct dialog_pak *dialog;

/* checks */
g_return_if_fail(data != NULL);

if ((id = request_dialog(data->number, MORPH)) < 0)
  return;
dialog = &sysenv.dialog[id];

/* init */
line = g_string_new(NULL);

/* create dialog */
dialog->win = gtk_dialog_new();
gtk_window_set_title(GTK_WINDOW(dialog->win), "Morphology table");
/* FIXME - there's gotta be a better way to (automatically) set the width */
gtk_window_set_default_size(GTK_WINDOW(dialog->win), 550, 500);
gtk_signal_connect(GTK_OBJECT (dialog->win), "destroy",
                   GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* 1st frame */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog->win)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
vbox = gtk_hbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
/* name */
g_string_sprintf(line, "Name: %s", data->basename);
label = gtk_label_new(line->str);
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

/* 2nd frame */
/* type of display - BFDH, Growth, Equilibrium (ie which column */
/* to use in the hull construction) */
frame = gtk_frame_new("Hull construction method");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog->win)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* radio group for morphology types */
button = gtk_radio_button_new_with_label (NULL, "Length");
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (switch_type),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) LENGTH);
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_widget_show(button);
if (data->type == LENGTH)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

radio = gtk_radio_button_group (GTK_RADIO_BUTTON (button));

/* do the next button */
button = gtk_radio_button_new_with_label (radio, "Dhkl");
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_type),
                           (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) DHKL);
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_widget_show(button);
if (data->type == DHKL)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(
         gtk_radio_button_group (GTK_RADIO_BUTTON (button)), "Eatt");
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_type),
                           (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) EATT);
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_widget_show(button);
if (data->type == EATT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(
         gtk_radio_button_group (GTK_RADIO_BUTTON (button)), "Esurf");
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_type),
                           (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) ESURF);
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_widget_show(button);
if (data->type == ESURF)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);


/* 3rd frame - listing */
frame = gtk_frame_new (NULL);
gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dialog->win)->vbox),frame,TRUE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
vbox = gtk_hbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
/* scrolled window  */
scr_win = gtk_scrolled_window_new (NULL, NULL);
gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr_win),
                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(vbox), scr_win, TRUE, TRUE, 0);
/* gtk-extra, spreadsheet */
data->morph_sheet = gtk_sheet_new(100, 5, "my sheet");
gtk_container_add(GTK_CONTAINER(scr_win), data->morph_sheet);
gtk_signal_connect(GTK_OBJECT(data->morph_sheet), "changed", 
                   (GtkSignalFunc) cell_change, (gpointer) data);

/* set column titles */
gtk_sheet_column_button_add_label(GTK_SHEET(data->morph_sheet), 0, "miller");
gtk_sheet_column_button_add_label(GTK_SHEET(data->morph_sheet), 1, "length");
gtk_sheet_column_button_add_label(GTK_SHEET(data->morph_sheet), 2, "dhkl");
gtk_sheet_column_button_add_label(GTK_SHEET(data->morph_sheet), 3, "Eatt");
gtk_sheet_column_button_add_label(GTK_SHEET(data->morph_sheet), 4, "Esurf");


update_morph_table(data);


/* terminating button(s) */
button = gtk_button_new_with_label ("Close");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog->win)->action_area),
                                    button, FALSE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                          GTK_SIGNAL_FUNC(close_dialog),
                         (gpointer) id);
/* done */
gtk_widget_show_all(dialog->win);

g_string_free(line, TRUE);
#else
show_text("Sorry, you need the gtk-extra library installed for this.");
#endif
}

