/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "gdis.h"

#define DEBUG 0

gint create_active;
GtkWidget *elem_entry, *grid_dim, *grid_sep;

/****************/
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];
/****************/

/************/
/* shutdown */
/************/
void c_close(GtkWidget *w)
{
gint i;
struct model_pak *data;

/* remove the creation widget */
gtk_widget_destroy(w);
create_active = 0;

/* update all models (just in case) */
for (i=0 ; i<sysenv.num_models ; i++)
  {
/* new - make permanent any deletions */
  data = model_ptr(i, RECALL);
  mem_shrink(data);
/* remove the mesh */
  init_objs(MESH_OFF, data);
/* return to normal viewing */
  data->mode = FREE;
/* update model data */
  init_objs(CENT_COORDS, data);
  calc_bonds(data);
  calc_mols(data);
  }

redraw_canvas(ALL);
}

/****************************/
/* connectivity only update */
/****************************/
void c_update(GtkWidget *w)
{
struct model_pak *data;

data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;

init_objs(REDO_COORDS, data);
calc_bonds(data);
calc_mols(data);
redraw_canvas(SINGLE);
}

/********************/
/* mesh only update */
/********************/
void grid_changed(gint mode)
{
gfloat r;
struct model_pak *data;

data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;

switch(mode)
  {
  case MESH_POINTS:
    data->mesh.px = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(grid_dim));
    data->mesh.py = data->mesh.px;
/* fall through to redo (auto) coordinate scaling */ 
  case MESH_SPACING:
    r = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(grid_sep));
    data->rmax = 0.5 * r * (gfloat) data->mesh.px;
    break;
  default:
    printf("Error: no such mode.\n");
    return;
  }
/* redo coords to match scaling */
init_objs(REDO_COORDS, data);
redraw_canvas(SINGLE);
}

void mesh_toggle()
{
gint i;
struct model_pak *data;

data = model_ptr(sysenv.active, RECALL);
g_return_if_fail(data != NULL);

data->mesh_on ^= 1;
/* clean up after */
if (data->mesh_on == FALSE)
  for (i=data->num_atoms ; i-- ; )
    (data->atoms+i)->status &= ~SQUARE_HL;

redraw_canvas(SINGLE);
}

/*******************************/
/* model switching update hook */
/*******************************/
void update_creator()
{
}

/*********/
/* setup */
/*********/
/* TODO - split the left & right panes into separate dialogues */
/* - it'll be easier on smaller screens */
void creation_widget()
{
gfloat cur_space, min_space, max_space;
GtkWidget *c_win, *vbox, *vbox1, *vbox2, *table, *frame, *button, *label;
GtkAdjustment *adj;
struct model_pak *data;

/* checks */
if (create_active)
  return;
/* model data contains grid parameters */
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;

/* setup the creator dialog */
/* TODO - use dialog request function */
c_win = gtk_dialog_new();
gtk_signal_connect (GTK_OBJECT (c_win), "destroy",
                    GTK_SIGNAL_FUNC(c_close),
                    GTK_OBJECT (c_win));
gtk_window_set_title (GTK_WINDOW (c_win), "The Creator");

/* NEW - dual pane creator layour */
table = gtk_table_new(2, 1, FALSE);
gtk_container_add(GTK_CONTAINER(GTK_BOX(GTK_DIALOG(c_win)->vbox)),table);
/* left & right vboxes */
vbox1 = gtk_vbox_new (FALSE, 0);
gtk_table_attach_defaults(GTK_TABLE(table),vbox1,0,1,0,1);
vbox2 = gtk_vbox_new (FALSE, 0);
gtk_table_attach_defaults(GTK_TABLE(table),vbox2,1,2,0,1);

/* FRAME - ATOM TYPE */
/* FIXME - how to shorten the length of the text entry box??? */
frame = gtk_frame_new ("Atom type");
gtk_box_pack_start (GTK_BOX (vbox1), frame, FALSE, TRUE, 0);
gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
vbox = gtk_vbox_new (FALSE, 0);
gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
gtk_container_add (GTK_CONTAINER(frame),vbox);
elem_entry = gtk_entry_new();
gtk_entry_set_max_length(GTK_ENTRY(elem_entry), 2);
gtk_box_pack_start (GTK_BOX (vbox), elem_entry, FALSE, TRUE, 0);
gtk_entry_set_text(GTK_ENTRY(elem_entry),"C");

/* FRAME - ATOM OPERATIONS */
/* editing modes */
frame = gtk_frame_new ("Atom modification");
gtk_box_pack_start (GTK_BOX (vbox1), frame, FALSE, TRUE, 0);
gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
vbox = gtk_vbox_new (FALSE, 0);
gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
gtk_container_add (GTK_CONTAINER (frame), vbox);
/* use table for nice spacing */
table = gtk_table_new(2, 4, FALSE);
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox)),table);

label = gtk_label_new ("Add atoms");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,0,1);
button = gtk_button_new_with_label (" X ");
gtk_table_attach_defaults(GTK_TABLE(table),button,1,2,0,1);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_mode),
                           (gpointer) ATOM_ADD);

label = gtk_label_new ("Move atoms");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);
button = gtk_button_new_with_label (" X ");
gtk_table_attach_defaults(GTK_TABLE(table),button,1,2,1,2);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_mode),
                           (gpointer) ATOM_MOVE);
 
label = gtk_label_new ("Delete atoms");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);
button = gtk_button_new_with_label (" X ");
gtk_table_attach_defaults(GTK_TABLE(table),button,1,2,2,3);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_mode),
                           (gpointer) ATOM_DELETE);

label = gtk_label_new ("Change type");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,3,4);
button = gtk_button_new_with_label (" X ");
gtk_table_attach_defaults(GTK_TABLE(table),button,1,2,3,4);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_mode),
                           (gpointer) ATOM_CHANGE_TYPE);

/* do this last => makes all the above conform? */
gtk_table_set_row_spacings (GTK_TABLE(table), 5);
gtk_table_set_col_spacings (GTK_TABLE(table), 5);

/* FRAME - BOND OPERATIONS */
/* editing modes */
frame = gtk_frame_new ("Bond modification");
gtk_box_pack_start (GTK_BOX (vbox1), frame, FALSE, TRUE, 0);
gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
vbox = gtk_vbox_new (FALSE, 0);
gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
gtk_container_add (GTK_CONTAINER (frame), vbox);
/* use table for nice spacing */
table = gtk_table_new(2, 3, FALSE);
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox)),table);

label = gtk_label_new ("Make single");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,0,1);
button = gtk_button_new_with_label (" X ");
gtk_table_attach_defaults(GTK_TABLE(table),button,1,2,0,1);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_mode),
                           (gpointer) BOND_SINGLE);

label = gtk_label_new ("Make double");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);
button = gtk_button_new_with_label (" X ");
gtk_table_attach_defaults(GTK_TABLE(table),button,1,2,1,2);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_mode),
                           (gpointer) BOND_DOUBLE);
 
label = gtk_label_new ("Make triple");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);
button = gtk_button_new_with_label (" X ");
gtk_table_attach_defaults(GTK_TABLE(table),button,1,2,2,3);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_mode),
                           (gpointer) BOND_TRIPLE);

/* do this last => makes all the above conform? */
gtk_table_set_row_spacings (GTK_TABLE(table), 5);
gtk_table_set_col_spacings (GTK_TABLE(table), 5);


/* FRAME - MOLECULE OPERATIONS */
/* editing modes */
frame = gtk_frame_new ("Molecule modification");
gtk_box_pack_start (GTK_BOX (vbox1), frame, FALSE, TRUE, 0);

gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
vbox = gtk_vbox_new (FALSE, 0);
gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
gtk_container_add (GTK_CONTAINER (frame), vbox);
/* use table for nice spacing */
table = gtk_table_new(2, 1, FALSE);
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox)),table);

label = gtk_label_new ("Delete mols");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);
button = gtk_button_new_with_label (" X ");
gtk_table_attach_defaults(GTK_TABLE(table),button,1,2,1,2);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_mode),
                           (gpointer) MOL_DELETE);


/* FRAME - GRID SETUP */
/* Grid dimensions */
frame = gtk_frame_new ("Grid setup");
gtk_box_pack_start (GTK_BOX (vbox2), frame, FALSE, TRUE, 0);
gtk_container_set_border_width (GTK_CONTAINER (frame), 5);

/* table for nice spacing */
table = gtk_table_new(2, 3, FALSE);
gtk_container_add(GTK_CONTAINER(frame), table);

/* dimension checks */
if (data->mesh.px < MESH_POINTS_MIN)
  {
  data->mesh.px = MESH_POINTS_MIN;
  data->mesh.py = MESH_POINTS_MIN;
  }
if (data->mesh.px > MESH_POINTS_MAX)
  {
  data->mesh.px = MESH_POINTS_MAX;
  data->mesh.py = MESH_POINTS_MAX;
  }
/* make the dimension adjustment */
label = gtk_label_new ("Grid dimension");
gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
adj = (GtkAdjustment *) gtk_adjustment_new 
      (data->mesh.px, MESH_POINTS_MIN, MESH_POINTS_MAX, 1, 2, 0);
grid_dim = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (grid_dim), FALSE);
gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (grid_dim),
                                   GTK_SHADOW_OUT);
gtk_table_attach_defaults(GTK_TABLE(table), grid_dim, 1, 2, 0, 1);
gtk_signal_connect_object (GTK_OBJECT (adj), "value_changed",
                           GTK_SIGNAL_FUNC (grid_changed),
                           (gpointer) MESH_POINTS);

/* auto recalibrated spacing */
cur_space = data->rmax/RMAX_FUDGE * 2.0/(gfloat) data->mesh.px;
min_space = (cur_space < MESH_SPACE_MIN) ? cur_space : MESH_SPACE_MIN;
max_space = (cur_space > MESH_SPACE_MAX) ? cur_space : MESH_SPACE_MAX;

/* make the spacing adjustment */
label = gtk_label_new ("Grid spacing (A)");
gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
adj = (GtkAdjustment *) gtk_adjustment_new 
      (cur_space, min_space, max_space, 0.1, 0.5, 0);
grid_sep = gtk_spin_button_new (adj, 0.1, 2);
gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (grid_sep), FALSE);
gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (grid_sep),
                                   GTK_SHADOW_OUT);
gtk_table_attach_defaults(GTK_TABLE(table), grid_sep, 1, 2, 1, 2);
gtk_signal_connect_object (GTK_OBJECT (adj), "value_changed",
                           GTK_SIGNAL_FUNC (grid_changed),
                           (gpointer) MESH_SPACING);

/* terminating buttons */
/* TODO - reset? this button looks ugly - fix!!! */
button = gtk_button_new_with_label("Grid toggle");
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (mesh_toggle), NULL);
gtk_table_attach_defaults(GTK_TABLE(table), button, 0, 1, 2, 3);

/* do this last => makes all the above conform? */
gtk_table_set_row_spacings (GTK_TABLE(table), 5);
gtk_table_set_col_spacings (GTK_TABLE(table), 5);

/* MAYBE - minimizer links in here too (if potential set loaded) */
/* TODO - update connectivity etc. button */
/* the redo button */
button = gtk_button_new_with_label ("Update");
gtk_box_pack_start (GTK_BOX (GTK_DIALOG (c_win)->action_area), button,
                                         FALSE, TRUE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (c_update),
                           GTK_OBJECT (c_win));
/* Do the exit button */
button = gtk_button_new_with_label ("Close");
gtk_box_pack_start (GTK_BOX (GTK_DIALOG (c_win)->action_area), button,
                                         FALSE, TRUE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (c_close),
                           GTK_OBJECT (c_win));
gtk_widget_show_all(c_win);
create_active=1;
 
redraw_canvas(ALL);
}

/****************************/
/* modify an existing model */
/****************************/
void modify_model()
{
/* check */
if (!sysenv.num_models)
  return;

creation_widget();

/* CURRENT - update grid */

}

/*******************************/
/* creates a new (blank) model */
/*******************************/
void create_new_model()
{
struct model_pak *data;

/* make a slot for the new model */
data = model_ptr(sysenv.num_models, ASSIGN);
g_return_if_fail(data != NULL);

sysenv.active = sysenv.num_models-1;
data->mode = ATOM_ADD;

/* setup model parameters */
data->id = CREATOR;
data->num_frames = 1;
data->cur_frame = 1;
strcpy(data->filename, "new_model");
g_free(data->basename);
data->basename = g_strdup("new model");

/* initialize */
data->axes_on = TRUE;
init_objs(INIT_COORDS, data);
calc_bonds(data);
calc_mols(data);
init_objs(MESH_ON, data);

/* needed? */
/*
data->scale = 1.0;
*/
/* rmax has to be after init_objs() as INIT_COORDS will set it to 1.0 */

/* number of mesh points (default = 10) / 2*rmax = angstrom spacing */
data->rmax = 5.0*RMAX_FUDGE;  /* default = 1A per grid pt */

/* update/redraw */
creation_widget();
new_tree_item(sysenv.active, APPEND);
redraw_canvas(ALL);
}

/*********************/
/* copy an atom core */
/*********************/
/* NEW - return the number of items copied (ie 0-2) */
#define DEBUG_COPY_CORE 0
gint copy_core(struct model_pak *dest, struct model_pak *src, gint n)
{
gint i, items=0;
struct atom_pak *copy, *orig;
struct shell_pak *copys, *origs;

#if DEBUG_COPY_CORE
printf("copy_core() #%d [%s]\n",n,(src->atoms+n)->label);
#endif

/* checks */
g_return_val_if_fail(dest != NULL, 0);
g_return_val_if_fail(src != NULL, 0);
g_return_val_if_fail(n >= 0 && n < src->num_atoms, 0);

/* enough memory? */
if (mem_check(dest, ATOM))
  mem_grow(dest, ATOM, 10);

/* pointers for ease of reference */
orig = src->atoms + n;
copy = dest->atoms + dest->num_atoms;

#if DEBUG_COPY_CORE
printf("adding core #%d ", dest->num_atoms);
#endif

/* main info */
copy->atom_code = orig->atom_code;
copy->status = orig->status;
copy->sitecount = 0;
copy->sof = orig->sof;
/* so that gulp doesn't ignore it when saving? */
copy->orig = TRUE; 
copy->primary = TRUE;
/* init_objs should calculate this correctly for selections */
copy->atom_bonds = 0;
/* calculated via init_objs */
copy->has_shell = FALSE;
copy->region = orig->region;
/* coords n stuff */
/* coords n stuff */
copy->cx = orig->cx;
copy->cy = orig->cy;
copy->cz = orig->cz;
copy->x = orig->rx;
copy->y = orig->ry;
copy->z = orig->rz;
copy->offx = 0;
copy->offy = 0;
/* element/label stuff */
strcpy(copy->label,orig->label); 
strcpy(copy->element,orig->element); 
copy->colour[0] = orig->colour[0];
copy->colour[1] = orig->colour[1];
copy->colour[2] = orig->colour[2];
/* FIXME */
if (strlen(orig->tail))
  strcpy(copy->tail, orig->tail); 
else
  strncpy(copy->tail, "", 1); 
items++;

/* attached shell? */
if (orig->has_shell)
  {
  i = orig->idx_shell;
/* check if enough memory was allocated */
  if (mem_check(dest, SHELL))
    mem_grow(dest, SHELL, 10);
/* FIXME - bit of redundancy here, can we use a generic gpointer */
/*  to transfer most of the core and shell data with the same code? */
/* update core */
  copy->has_shell = TRUE;
  copy->idx_shell = dest->num_shells;
/* pointers for ease of reference */
  origs = src->shells + i;
  copys = dest->shells + dest->num_shells;
#if DEBUG_COPY_CORE
printf("and shell #%d", dest->num_shells);
#endif
/* main info */
  copys->atom_code = origs->atom_code;
  copys->status = origs->status;
  copys->primary=FALSE;
  copys->sitecount=0;
  copys->orig=FALSE; 
/* also calculated via init_objs */
  copys->has_core = TRUE;
  copys->idx_core = dest->num_atoms;
  copys->region = origs->region;
/* coords n stuff */
  copys->cx = origs->cx;
  copys->cy = origs->cy;
  copys->cz = origs->cz;
/* FIXME - is this a sensible way to merge? */
  copys->x = origs->rx;
  copys->y = origs->ry;
  copys->z = origs->rz;
  copys->offx = 0;
  copys->offy = 0;
/* element/label stuff */
  strcpy(copys->label,origs->label); 
  strcpy(copys->element,origs->element); 
  copys->colour[0] = origs->colour[0];
  copys->colour[1] = origs->colour[1];
  copys->colour[2] = origs->colour[2];
  strcpy(copys->tail,orig->tail); 
  dest->num_shells++;
  items++;
  }

/* we have a new atom to be processed */
dest->num_atoms++;
#if DEBUG_COPY_CORE
printf("\ncore [%d/%d] : shell [%d/%d]\n",dest->num_atoms,dest->atom_limit,
                                          dest->num_shells, dest->shell_limit);
#endif
return(items);
}

/***********************/
/* add atom event hook */
/***********************/
void add_atom(gint px, gint py)
{
gint i;
gchar *orig,*elem;
struct model_pak *data;

/* get the atom from the gtk entry */
orig = (gchar *) gtk_entry_get_text(GTK_ENTRY(elem_entry));
elem = g_strdup(orig);

/* check if enough memory was allocated */
data = model_ptr(sysenv.active,RECALL);
if (data->num_atoms == data->atom_limit)
  mem_grow(data,ATOM,10);

#if DEBUG
printf("Adding atom number %d\n",data->num_atoms+1);
#endif

/* add the atom to the end */
i = data->num_atoms;
strcpy((data->atoms+i)->element,elem); 
strcpy((data->atoms+i)->label,elem);
strcpy((data->atoms+i)->tail,"\0");
/* get element code & default colour */
elem_seek(i, ATOM, data);
/* creator plane goes through the origin */
(data->atoms+i)->rz = 0.0;         /* MUST occur before pixel_coord_map */
pixel_coord_map(px, py, data, i);  /* NEW - this should init the coords */
/* init the atom */
(data->atoms+i)->status = NORMAL;
(data->atoms+i)->atom_bonds = 0;
(data->atoms+i)->has_shell = FALSE;
(data->atoms+i)->offx = 0;
(data->atoms+i)->offy = 0;
/* done */
data->num_atoms++;
g_free(elem);
}

/*********************/
/* move atoms around */
/*********************/
void move_atom(gint px, gint py, gint mode)
{
static int atom=-1;
struct model_pak *data;

/* START is called by button_press_event */
/* UPDATE is called by mouse_motion_event */
/* STOP is called when a switch_mode event occurs - ie cleanup */

data = model_ptr(sysenv.active, RECALL);

switch(mode)
  {
  case START:
/* try to find an atom at the mouse pointer */
    atom = seek_atom(px,py,data);     
    break;
  case UPDATE:
/* move held atom to the mouse pointer */
    if (atom >= 0)
      pixel_coord_map(px,py,data,atom);
    break;
  case STOP:
    atom = -1;
  default:
    return;
  }
}

/************************/
/* Delete atom by index */
/************************/
void delete_atom(gint n, struct model_pak *data)
{
gint s;

(data->atoms+n)->status |= DELETED;
if ((data->atoms+n)->has_shell)
  {
  s = (data->atoms+n)->idx_shell;
  (data->shells+s)->status |= DELETED;
  }
}

/*********************************/
/* Delete atom by pixel position */
/*********************************/
void delete_atom_at(gint x, gint y, struct model_pak *data)
{
gint i;
gchar txt[15];

/* attempt to match point with an atom */
i = seek_atom(x, y, data);

/* no match - leave */
if (i < 0)
  return;

delete_atom(i, data);

g_snprintf(txt,12,"Deleted: %-2s", (data->atoms+i)->element);
show_text(txt);
}

/*********************************/
/* Delete atom by pixel position */
/*********************************/
void change_atom_at(gint x, gint y, struct model_pak *data)
{
gint i;
gchar *elem;

/* attempt to match point with an atom */
i = seek_atom(x, y, data);

/* no match - leave */
if (i < 0)
  return;

elem = (gchar *) gtk_entry_get_text(GTK_ENTRY(elem_entry));
if (!elem)
  return;

/* change type */
strcpy((data->atoms+i)->element, elem);
strcpy((data->atoms+i)->label, elem);
/* setup */
elem_seek(i, ATOM, data);
}

/************************************/
/* pick molecule - create selection */
/************************************/
void pick_mol(gint x, gint y, struct model_pak *data)
{
gint i;
gint atom, mol;

/* try to find an atom at the mouse pointer */
atom = seek_atom(x,y,data);     
if (atom < 0)
  return;

/* get the mol to which the atom belongs */
mol = (data->atoms+atom)->molecule;

/* NEW - append all atoms in the molecule to the list */
for (i=0 ; i<(data->mols+mol)->num_atoms ; i++)
  {
  atom = *((data->mols+mol)->atom+i);
  add_select(data, atom);
  (data->atoms+atom)->status |= SELECT_HL;
  }
}

/*************************/
/* move molecules around */
/*************************/
void move_mol(gint px, gint py, gint mode)
{
gint i, j, x, y, offx, offy;
static int atom=-1, mol;
struct model_pak *data;

/* START is called by button_press_event */
/* UPDATE is called by mouse_motion_event */
/* STOP is called when a switch_mode event occurs - ie cleanup */

data = model_ptr(sysenv.active, RECALL);

switch(mode)
  {
  case START:
/* try to find an atom at the mouse pointer */
    atom = seek_atom(px,py,data);     
/* find which molecule it belongs to */
/* find the mol to which the atom belongs */
    mol = -1;
    for (i=0 ; i<data->num_mols ; i++)
      for (j=0 ; j<(data->mols+i)->num_atoms ; j++)
        if (atom == *((data->mols+i)->atom+j))
          {
          mol = i; 
          return;
          }
    break;
  case UPDATE:
/* move held atom to the mouse pointer */
    if (atom >= 0 && mol >= 0)
      {
      offx = px - (data->atoms+atom)->px;
      offy = py - (data->atoms+atom)->py;
      for (i=0 ; i<(data->mols+mol)->num_atoms ; i++)
        {
        j = *((data->mols+mol)->atom+i);
        x = (data->atoms+j)->px + offx;
        y = (data->atoms+j)->py + offy;
        pixel_coord_map(x,y,data,j);
        }
      }
    break;
  case STOP:
    atom = -1;
    mol = -1;
  default:
    return;
  }
}

/*******************/
/* Delete molecule */
/*******************/
void delete_mol(gint x, gint y, struct model_pak *data)
{
gint i, j, k, t;

/* attempt to match point with an atom */
i = seek_atom(x, y, data);

/* no match - leave */
if (i < 0)
  return;

/* find the mol to which the atom belongs */
t=-1;
for (j=0 ; j<data->num_mols ; j++)
  for (k=0 ; k<(data->mols+j)->num_atoms ; k++)
    if (i == *((data->mols+j)->atom+k))
      {
      t = j; 
      break;
      }

/* no match - leave */
if (t < 0)
  return;

/* delete the mol */
for (j=0 ; j<(data->mols+t)->num_atoms ; j++)
  {
  k = *((data->mols+t)->atom+j);
  (data->atoms+k)->status |= DELETED;
  }
}

