{
GPC demo program for the CRT unit.

Copyright (C) 1999 Free Software Foundation, Inc.

Author: Frank Heckenbach <frank@pascal.gnu.de>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING. If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.

As a special exception, if you incorporate even large parts of the
code of this demo program into another program with substantially
different functionality, this does not cause the other program to
be covered by the GNU General Public License. This exception does
not however invalidate any other reasons why it might be covered
by the GNU General Public License.
}

program CRTDemo;

uses GPC, CRT;

type
  TFrameChars = array [1 .. 8] of Char;

const
  SingleFrame : TFrameChars = (chCornerTLS, chLineHS, chCornerTRS, chLineVS, chLineVS, chCornerBLS, chLineHS, chCornerBRS);
  DoubleFrame : TFrameChars = (chCornerTLD, chLineHD, chCornerTRD, chLineVD, chLineVD, chCornerBLD, chLineHD, chCornerBRD);

var
  ScrollState : Boolean = True;
  CursorShape : TCursorShape = CursorNormal;
  MainPanel : TPanel;

procedure FrameWin (const Title : String; const Frame : TFrameChars);
var w, h, y : Integer;
begin
  HideCursor;
  SetPCCharSet (True);
  ClrScr;
  w := GetXMax;
  h := GetYMax;
  WriteCharAt (1, 1, 1,     Frame [1], TextAttr);
  WriteCharAt (2, 1, w - 2, Frame [2], TextAttr);
  WriteCharAt (w, 1, 1,     Frame [3], TextAttr);
  for y := 2 to h - 1 do
    begin
      WriteCharAt (1, y, 1, Frame [4], TextAttr);
      WriteCharAt (w, y, 1, Frame [5], TextAttr)
    end;
  WriteCharAt (1, h, 1,     Frame [6], TextAttr);
  WriteCharAt (2, h, w - 2, Frame [7], TextAttr);
  WriteCharAt (w, h, 1,     Frame [8], TextAttr);
  SetPCCharSet (False);
  WriteStrAt ((w - Length (Title)) div 2 + 1, 1, Title, TextAttr)
end;

procedure MainDraw;
begin
  Writeln ('3, F3  : Open a window');
  Writeln ('4, F4  : Close a window');
  Writeln ('5, F5  : Previous window');
  Writeln ('6, F6  : Next window');
  Writeln ('7, F7  : Move window');
  Writeln ('8, F8  : Resize window');
  Write   ('q, Esc : Quit')
end;

procedure StatusDraw;
const
  YesNo : array [Boolean] of String [3] = ('No', 'Yes');
  CursorShapeNames : array [TCursorShape] of String [7] = ('Ignored', 'Hidden ', 'Normal ', 'Fat    ', 'Block  ');
begin
  Writeln ('Current settings. You can toggle some of them by pressing');
  Writeln ('the key in parentheses. Changing the screen size, cursor');
  Writeln ('shape and colour naturally does not work on all terminals.');
  Writeln;
  Writeln ('XCurses version:  ', YesNo [XCRT]);
  Writeln ('(M)onochrome:     ', YesNo [IsMonoMode]);
  with GetScreenSize do Writeln ('Screen si(z)e:    ', X, 'x', Y);
  Writeln ('(B)reak checking: ', YesNo [CheckBreak]);
  Writeln ('(S)crolling:      ', YesNo [ScrollState]);
  Write   ('(C)ursor shape:   ', CursorShapeNames [CursorShape])
end;

procedure SetTextMode (Mode : Integer); forward;

procedure StatusKey (Key : TKey);
var Ch : Char;
begin
  Ch := Key2Char (Key);
  case LoCase (Ch) of
    'm' : SetTextMode (BW80 + (CO80 - BW80) * Ord (IsMonoMode) + LastMode and Font8x8);
    'z' : SetTextMode (LastMode xor Font8x8);
    'b' : CheckBreak := not CheckBreak;
    's' : ScrollState := not ScrollState;
    'c' : begin
            case CursorShape of
              CursorNormal : CursorShape := CursorBlock;
              CursorFat,
              CursorBlock  : CursorShape := CursorHidden;
              else           CursorShape := CursorNormal
            end
          end;
  end;
  ClrScr;
  StatusDraw
end;

procedure TextAttrDemo;
const HexDigits : array [0 .. $f] of Char = '0123456789ABCDEF';
var f, b, x1, y1, x2, y2 : Integer;
begin
  GetWindow (x1, y1, x2, y2);
  Window (x1 - 1, y1, x2, y2);
  SetScroll (False);
  for b := 0 to 15 do
    begin
      GotoXY (1, b + 1);
      for f := 0 to 15 do
        begin
          TextAttr := f + 16 * b;
          Write (' ', HexDigits [b], HexDigits [f], ' ')
        end
    end
end;

procedure CharSetDemo (UsePCCharSet : Boolean);
var h, l : Integer;
begin
  SetScroll (False);
  SetPCCharSet (UsePCCharSet);
  SetControlChars (False);
  Write ('    ');
  for l := 0 to 15 do Write (l : 2, ' ');
  for h := 0 to 15 do
    begin
      GotoXY (1, h + 2);
      Write (16 * h : 3, ' ');
      for l := 0 to 15 do Write (' ', Chr (16 * h + l), ' ')
    end
end;

procedure NormalCharSetDemo;
begin
  CharSetDemo (False)
end;

procedure PCCharSetDemo;
begin
  CharSetDemo (True)
end;

procedure FKeyDemoDraw;
var x1, y1, x2, y2 : Integer;
begin
  GetWindow (x1, y1, x2, y2);
  Window (x1, y1, x2 - 1, y2);
  ClrScr;
  SetScroll (False);
  Writeln ('You can type the following keys (function keys if present');
  Writeln ('on the terminal, and letters as alternatives):');
  Writeln;
  Writeln ('S, Left           : move left (with wrap-around)');
  Writeln ('D, Right          : move right (with wrap-around)');
  Writeln ('E, Up             : move up (with wrap-around)');
  Writeln ('X, Down           : move down (with wrap-around)');
  Writeln ('A, Home           : go to first column');
  Writeln ('F, End            : go to last column');
  Writeln ('R, Page Up        : go to first line');
  Writeln ('C, Page Down      : go to last line');
  Writeln ('Y, Ctrl-Page Up   : go to first column, first line');
  Writeln ('B, Ctrl-Page Down : go to last column, last line');
  Writeln ('Z, Ctrl-Home      : clear screen');
  Writeln ('N, Ctrl-End       : clear to end of line');
  Writeln ('V, Insert         : insert a line');
  Writeln ('T, Delete         : delete a line');
  Writeln ('#                 : beep');
  Writeln ('*                 : flash');
  Writeln ('Tab, Enter, Backspace, other normal characters : write text');
  GotoXY (1, 21)
end;

procedure FKeyDemoKey (Key : TKey);
const TabSize = 8;
var
  Ch : Char;
  NewX : Integer;
begin
  case LoCaseKey (Key) of
    Ord ('s'), kbLeft     : if WhereX = 1 then GotoXY (GetXMax, WhereY) else GotoXY (WhereX - 1, WhereY);
    Ord ('d'), kbRight    : if WhereX = GetXMax then GotoXY (1, WhereY) else GotoXY (WhereX + 1, WhereY);
    Ord ('e'), kbUp       : if WhereY = 1 then GotoXY (WhereX, GetYMax) else GotoXY (WhereX, WhereY - 1);
    Ord ('x'), kbDown     : if WhereY = GetYMax then GotoXY (WhereX, 1) else GotoXY (WhereX, WhereY + 1);
    Ord ('a'), kbHome     : Write (chCR);
    Ord ('f'), kbEnd      : GotoXY (GetXMax, WhereY);
    Ord ('r'), kbPgUp     : GotoXY (WhereX, 1);
    Ord ('c'), kbPgDn     : GotoXY (WhereX, GetYMax);
    Ord ('y'), kbCtrlPgUp : GotoXY (1, 1);
    Ord ('b'), kbCtrlPgDn : GotoXY (GetXMax, GetYMax);
    Ord ('z'), kbCtrlHome : ClrScr;
    Ord ('n'), kbCtrlEnd  : ClrEOL;
    Ord ('v'), kbIns      : InsLine;
    Ord ('t'), kbDel      : DelLine;
    Ord ('#')             : Beep;
    Ord ('*')             : Flash;
    kbTab                 : begin
                              NewX := ((WhereX - 1) div TabSize + 1) * TabSize + 1;
                              if NewX <= GetXMax then GotoXY (NewX, WhereY) else Writeln
                            end;
    kbCR                  : Writeln;
    kbBkSp                : Write (chBkSp, ' ', chBkSp);
    else                    Ch := Key2Char (Key);
                            if Ch <> #0 then Write (Ch)
  end
end;

procedure KeyDemoDraw;
begin
  Writeln ('Press some keys...')
end;

procedure KeyDemoKey (Key : TKey);
var Ch : Char;
begin
  Ch := Key2Char (Key);
  if Ch <> #0 then
    begin
      Write ('Normal key');
      if Ch in [' ' .. #126] then Write (' `', Ch, '''');
      Writeln (', ASCII #', Ord (Ch))
    end
  else
    Writeln ('Special key ', Ord (Key2Scan (Key)))
end;

procedure IOSelectPeriodical;
var
  CurrentTime : TimeStamp;
  s : String (8);
  i : Integer;
begin
  GetTimeStamp (CurrentTime);
  with CurrentTime do
    WriteStr (s, Hour : 2, ':', Minute : 2, ':', Second : 2);
  for i := 1 to Length (s) do
    if s [i] = ' ' then s [i] := '0';
  GotoXY (1, 10);
  Write ('The time is: ', s)
end;

procedure IOSelectDraw;
begin
  Writeln ('IOSelect is a way to handle I/O from/to several');
  Writeln ('places simultaneously, without having to use');
  Writeln ('threads or signal/interrupt handlers or waste');
  Writeln ('CPU time with busy waiting.');
  Writeln;
  Writeln ('This demo shows how IOSelect works in connection');
  Writeln ('with CRT. It displays a clock, but still reacts');
  Writeln ('to user input immediately.');
  IOSelectPeriodical
end;

procedure ModifierPeriodical;
const
  Pressed : array [Boolean] of String [8] = ('Released', 'Pressed');
  ModifierNames : array [1 .. 7] of record
    Modifier : Integer;
    Name     : String (17)
  end =
   ((shLeftShift,  'Left Shift'),
    (shRightShift, 'Right Shift'),
    (shLeftCtrl,   'Left Control'),
    (shRightCtrl,  'Right Control'),
    (shAlt,        'Alt (left)'),
    (shAltGr,      'AltGr (right Alt)'),
    (shExtra,      'Extra'));
var
  ShiftState, i : Integer;
begin
  ShiftState := GetShiftState;
  for i := 1 to 7 do
    with ModifierNames [i] do
      begin
        GotoXY (1, 4 + i);
        ClrEOL;
        Write (Name, ':');
        GotoXY (20, WhereY);
        Write (Pressed [ShiftState and Modifier <> 0])
      end
end;

procedure ModifierDraw;
begin
  Writeln ('Modifier keys (NOTE: only');
  Writeln ('accessible on some systems;');
  Writeln ('X11: only after key press):');
  ModifierPeriodical
end;

type
  PWindowList = ^TWindowList;
  TWindowList = record
    Next, Prev : PWindowList;
    Panel, FramePanel : TPanel;
    WindowType : Integer;
    x1, y1, xs, ys : Integer;
  end;

  TKeyProc = procedure (Key : TKey);
  TProcedure = procedure;

const
  (*@@fjf258*){$define Nil TKeyProc (nil)}{$define NIL TProcedure (nil)}
  MenuNameLength = 16;
  WindowTypes  : array [0 .. 8] of record
    DrawProc,
    PeriodicalProc : procedure;
    KeyProc        : TKeyProc;
    Name           : String (MenuNameLength);
    Color,
    Background,
    MinSizeX,
    MinSizeY       : Integer;
    RedrawAlways,
    WantCursor,
    ExactSize      : Boolean
  end =
  ((MainDraw         , NIL               , Nil        , 'CRT Demo'        , LightGreen, Blue     , 26,  7, False, False, True),
   (StatusDraw       , NIL               , StatusKey  , 'Status'          , White     , Red      , 60, 10, True,  True,  True),
   (TextAttrDemo     , NIL               , Nil        , 'Text Attributes' , White     , Blue     , 64, 16, False, False, True),
   (NormalCharSetDemo, NIL               , Nil        , 'Character Set'   , Black     , Green    , 53, 17, False, False, True),
   (PCCharSetDemo    , NIL               , Nil        , 'PC Character Set', Black     , Brown    , 53, 17, False, False, True),
   (KeyDemoDraw      , NIL               , KeyDemoKey , 'Keys'            , Blue      , LightGray, 29,  5, False, True,  False),
   (FKeyDemoDraw     , NIL               , FKeyDemoKey, 'Function Keys'   , Blue      , LightGray, 61, 21, False, True,  False),
   (ModifierDraw     , ModifierPeriodical, Nil        , 'Modifier Keys'   , Black     , Cyan     , 29, 11, True,  False, True),
   (IOSelectDraw     , IOSelectPeriodical, Nil        , 'IOSelect Demo'   , Yellow    , Magenta  , 50, 10, False, False, True));

  MinSizeX = 66;
  MinSizeY = 23;
  MenuMax = High (WindowTypes);
  MenuXSize = MenuNameLength + 4;
  MenuYSize = MenuMax + 2;

var
  WindowList : PWindowList = nil;

procedure RedrawFrame (const Moving : String; const Frame : TFrameChars);
begin
  with WindowList^, WindowTypes [WindowType] do
    begin
      PanelActivate (FramePanel);
      Window (x1, y1, x1 + xs - 1, y1 + ys - 1);
      ClrScr;
      if Moving = ''
        then FrameWin (' ' + Name + ' ', Frame)
        else FrameWin (' ' + Moving + ' ', Frame)
    end
end;

procedure DrawWindow (const Moving : String; const Frame : TFrameChars);
begin
  with WindowList^, WindowTypes [WindowType] do
    begin
      RedrawFrame (Moving, Frame);
      PanelActivate (Panel);
      Window (x1 + 2, y1 + 1, x1 + xs - 2, y1 + ys - 2);
      ClrScr;
      DrawProc
    end
end;

procedure CheckScreenSize;
var
  p : PWindowList;
  Changed : Boolean;
  LastPanel : TPanel;
begin
  LastPanel := GetActivePanel;
  PanelActivate (MainPanel);
  HideCursor;
  with GetScreenSize do
    begin
      Window (1, 1, X, Y);
      TextBackground (Blue);
      ClrScr;
      if (X < MinSizeX) or (Y < MinSizeY) then
        begin
          NormVideo;
          ClrScr;
          RestoreTerminal (True);
          Writeln (StdErr, 'Sorry, your screen is too small for this demo.');
          Writeln (StdErr, 'You need at least ', MinSizeX, 'x', MinSizeY, ' characters.');
          Halt (2)
        end;
      p := WindowList;
      if p <> nil then
        repeat
          with p^, WindowTypes [WindowType] do
            begin
              Changed := False;
              if x1 + xs - 1 > X then
                begin
                  xs := X - x1 + 1;
                  if xs < MinSizeX + 2 then
                    begin
                      xs := MinSizeX + 2;
                      x1 := X - xs + 1
                    end;
                  Changed := True
                end;
              if y1 + ys - 1 > Y then
                begin
                  ys := Y - y1 + 1;
                  if ys < MinSizeY + 2 then
                    begin
                      ys := MinSizeY + 2;
                      y1 := Y - ys + 1
                    end;
                  Changed := True
                end;
(*@@              if Changed and (p = WindowList) then DrawWindow ('', DoubleFrame); *)
              p := p^.Next
            end
        until p = WindowList
    end;
  PanelActivate (LastPanel)
end;

procedure SetTextMode (Mode : Integer);
begin
  TextMode (Mode);
  CheckScreenSize
end;

function GetKey = Key : TKey;
var
  NeedSelect : Boolean;
  SelectInput : array [1 .. 1] of PAnyFile;
  SelectValue : Integer;
  NextSelectTime : static MicroSecondTimeType = 0;
  LastPanel : TPanel;
  p : PWindowList;
begin
  LastPanel := GetActivePanel;
  NeedSelect := False;
  p := WindowList;
  repeat
    NeedSelect := @WindowTypes [p^.WindowType].PeriodicalProc <> nil;
    p := p^.Next
  until NeedSelect or (p = WindowList);
  p := WindowList;
  repeat
    with p^, WindowTypes [WindowType] do
      if RedrawAlways then
        begin
          PanelActivate (Panel);
          ClrScr;
          DrawProc
        end;
    p := p^.Next
  until p = WindowList;
  if NeedSelect then
    repeat
      CRTUpdate;
      (*@@constarray1*)SelectInput [1] := ((*@@*)PAnyFile( @Input));
      SelectValue := IOSelectRead (SelectInput, Max (0, NextSelectTime - GetMicroSecondTime));
      if SelectValue = 0 then
        begin
          NextSelectTime := GetMicroSecondTime + 100000;
          p := WindowList;
          repeat
            with p^, WindowTypes [WindowType] do
              if @PeriodicalProc <> nil then
                begin
                  PanelActivate (Panel);
                  PeriodicalProc
                end;
            p := p^.Next
          until p = WindowList
        end
    until SelectValue <> 0;
  Key := ReadKeyWord;
  if Key = kbScreenSizeChanged then CheckScreenSize;
  PanelActivate (LastPanel)
end;

function Menu = n : Integer;
var
  i, ax, ay : Integer;
  Key : TKey;
  Done : Boolean;
begin
  with GetScreenSize do
    begin
      ax := (X - MenuXSize) div 2 + 1;
      ay := (Y - MenuYSize) div 2 + 1
    end;
  PanelNew (ax, ay, ax + MenuXSize - 1, ay + MenuYSize - 1, False);
  SetControlChars (True);
  TextColor (Blue);
  TextBackground (LightGray);
  FrameWin (' Select Window ', DoubleFrame);
  IgnoreCursor;
  PanelNew (ax + 1, ay + 1, ax + MenuXSize - 2, ay + MenuYSize - 2, False);
  ClrScr;
  TextColor (Black);
  SetScroll (False);
  Done := False;
  n := 1;
  repeat
    for i := 1 to MenuMax do
      begin
        GotoXY (1, i);
        if i = n
          then TextBackGround (Green)
          else TextBackGround (LightGray);
        ClrEOL;
        Write (' ', WindowTypes [i].Name);
        ChangeTextAttr (2, i, 1, Red + $10 * GetTextBackground)
      end;
    Key := GetKey;
    case LoCaseKey (Key) of
      kbUp                   : if n = 1 then n := MenuMax else Dec (n);
      kbDown                 : if n = MenuMax then n := 1 else Inc (n);
      kbHome,
      kbPgUp,
      kbCtrlPgUp,
      kbCtrlHome             : n := 1;
      kbEnd,
      kbPgDn,
      kbCtrlPgDn,
      kbCtrlEnd              : n := MenuMax;
      kbCR                   : Done := True;
      kbEsc                  : begin
                                 n := - 1;
                                 Done := True
                               end;
      Ord ('a') .. Ord ('z') : begin
                                 i := MenuMax;
                                 while (i > 0) and (LoCase (Key2Char (Key)) <> LoCase (WindowTypes [i].Name [1])) do Dec (i);
                                 if i > 0 then
                                   begin
                                     n := i;
                                     Done := True
                                   end
                               end;
    end
  until Done;
  PanelDelete (GetActivePanel);
  PanelDelete (GetActivePanel)
end;

procedure NewWindow (WindowType, ax, ay : Integer);
var
  p : PWindowList;
  MaxX1, MaxY1 : Integer;
begin
  New (p);
  if WindowList = nil then
    begin
      p^.Prev := p;
      p^.Next := p
    end
  else
    begin
      RedrawFrame ('', SingleFrame);
      p^.Prev := WindowList;
      p^.Next := WindowList^.Next;
      p^.Prev^.Next := p;
      p^.Next^.Prev := p
    end;
  p^.WindowType := WindowType;
  WindowList := p;
  with p^, WindowTypes [WindowType] do
    begin
      with GetScreenSize do
        begin
          MaxX1 := X - MinSizeX - 1;
          MaxY1 := Y - MinSizeY - 1;
          if ax = 0 then x1 := Random (MaxX1) + 1 else x1 := Min (ax, MaxX1);
          if ay = 0 then y1 := Random (MaxY1) + 1 else y1 := Min (ay, MaxY1);
          xs := MinSizeX + 2;
          ys := MinSizeY + 2;
          if (ax = 0) and not ExactSize then
            begin
              Inc (xs, Random (X - x1 - xs + 2));
              Inc (ys, Random (Y - y1 - ys + 2))
            end
        end;
      PanelNew (1, 1, 1, 1, False);
      FramePanel := GetActivePanel;
      SetControlChars (True);
      TextColor (Color);
      TextBackground (Background);
      PanelNew (1, 1, 1, 1, False);
      SetPCCharSet (False);
      Panel := GetActivePanel;
      DrawWindow ('', DoubleFrame)
    end
end;

procedure OpenWindow;
var WindowType : Integer;
begin
  WindowType := Menu;
  if WindowType >= 0 then NewWindow (WindowType, 0, 0)
end;

procedure NextWindow;
begin
  RedrawFrame ('', SingleFrame);
  WindowList := WindowList^.Next;
  PanelTop (WindowList^.FramePanel);
  PanelTop (WindowList^.Panel);
  RedrawFrame ('', DoubleFrame)
end;

procedure PreviousWindow;
begin
  RedrawFrame ('', SingleFrame);
  PanelMoveAbove (WindowList^.Panel, MainPanel);
  PanelMoveAbove (WindowList^.FramePanel, MainPanel);
  WindowList := WindowList^.Prev;
  RedrawFrame ('', DoubleFrame)
end;

procedure CloseWindow;
var p : PWindowList;
begin
  if WindowList^.WindowType <> 0 then
    begin
      p := WindowList;
      NextWindow;
      PanelDelete (p^.FramePanel);
      PanelDelete (p^.Panel);
      p^.Next^.Prev := p^.Prev;
      p^.Prev^.Next := p^.Next;
      Dispose (p)
    end
end;

procedure MoveWindow;
var Done, Changed : Boolean;
begin
  with WindowList^ do
    begin
      Done := False;
      Changed := True;
      repeat
        if Changed then DrawWindow ('Move Window', SingleFrame);
        Changed := True;
        case LoCaseKey (GetKey) of
          Ord ('s'), kbLeft     : if x1 > 1 then Dec (x1);
          Ord ('d'), kbRight    : if x1 + xs - 1 < GetScreenSize.X then Inc (x1);
          Ord ('e'), kbUp       : if y1 > 1 then Dec (y1);
          Ord ('x'), kbDown     : if y1 + ys - 1 < GetScreenSize.Y then Inc (y1);
          Ord ('a'), kbHome     : x1 := 1;
          Ord ('f'), kbEnd      : x1 := GetScreenSize.X - xs + 1;
          Ord ('r'), kbPgUp     : y1 := 1;
          Ord ('c'), kbPgDn     : y1 := GetScreenSize.Y - ys + 1;
          Ord ('y'), kbCtrlPgUp : begin
                                    x1 := 1;
                                    y1 := 1
                                  end;
          Ord ('b'), kbCtrlPgDn : with GetScreenSize do
                                    begin
                                      x1 := X - xs + 1;
                                      y1 := Y - ys + 1
                                    end;
          kbCR,
          kbEsc                 : Done := True;
          else                    Changed := False
        end
      until Done;
      DrawWindow ('', DoubleFrame)
    end
end;

procedure ResizeWindow;
var Done, Changed : Boolean;
begin
  with WindowList^, WindowTypes [WindowType] do
    begin
      Done := False;
      Changed := True;
      repeat
        if Changed then DrawWindow ('Resize Window', SingleFrame);
        Changed := True;
        case LoCaseKey (GetKey) of
          Ord ('s'), kbLeft     : if xs > MinSizeX + 2 then Dec (xs);
          Ord ('d'), kbRight    : if x1 + xs - 1 < GetScreenSize.X then Inc (xs);
          Ord ('e'), kbUp       : if ys > MinSizeY + 2 then Dec (ys);
          Ord ('x'), kbDown     : if y1 + ys - 1 < GetScreenSize.Y then Inc (ys);
          Ord ('a'), kbHome     : xs := MinSizeX + 2;
          Ord ('f'), kbEnd      : xs := GetScreenSize.X - x1 + 1;
          Ord ('r'), kbPgUp     : ys := MinSizeY + 2;
          Ord ('c'), kbPgDn     : ys := GetScreenSize.Y - y1 + 1;
          Ord ('y'), kbCtrlPgUp : begin
                                    xs := MinSizeX + 2;
                                    ys := MinSizeY + 2
                                  end;
          Ord ('b'), kbCtrlPgDn : with GetScreenSize do
                                    begin
                                      xs := X - x1 + 1;
                                      ys := Y - y1 + 1
                                    end;
          kbCR,
          kbEsc                 : Done := True;
          else                    Changed := False
        end
      until Done;
      DrawWindow ('', DoubleFrame)
    end
end;

procedure ActivateCursor;
begin
  with WindowList^, WindowTypes [WindowType] do
    begin
      PanelActivate (Panel);
      if WantCursor
        then SetCursorShape (CursorShape)
        else HideCursor
    end;
  SetScroll (ScrollState)
end;

procedure Die;
begin
  RestoreTerminalClearCRT;
  Writeln (StdErr, 'You''re trying to kill me. Since I have break checking turned off,');
  Writeln (StdErr, 'I''m not dying, but I''ll do you a favour, and terminate now.');
  Halt (3)
end;

var
  Key : TKey;
  ScreenShot, Done : Boolean;

begin
  ScreenShot := ParamStr (1) = '--screenshot';
  if ParamCount <> Ord (ScreenShot) then
    begin
      RestoreTerminal (True);
      Writeln (StdErr, ParamStr (0), ': invalid argument `', ParamStr (Ord (ScreenShot) + 1), '''');
      Halt (1)
    end;
  Randomize;
  SetCRTUpdate (UpdateInput);
  MainPanel := GetActivePanel;
  CheckScreenSize;
  if ScreenShot then
    begin
      CursorShape := CursorBlock;
      NewWindow (6,      1,      1);
      NewWindow (2,      1, MaxInt);
      NewWindow (8, MaxInt,      1);
      NewWindow (3, MaxInt,     12);
      NewWindow (4, MaxInt,     30);
      NewWindow (5,      1,     27);
      KeyDemoKey (Ord ('f'));
      KeyDemoKey (246);
      KeyDemoKey (kbDown);
      NewWindow (7, MaxInt, MaxInt);
      NewWindow (0,      1,      2);
      NewWindow (1,      1,     16);
      ActivateCursor;
      OpenWindow
    end
  else
    NewWindow (0, 3, 2);
  Done := False;
  repeat
    ActivateCursor;
    Key := GetKey;
    if IsDeadlySignal (Key) then Die;
    case LoCaseKey (Key) of
      Ord ('3'), kbF3  : OpenWindow;
      Ord ('4'), kbF4  : CloseWindow;
      Ord ('5'), kbF5  : PreviousWindow;
      Ord ('6'), kbF6  : NextWindow;
      Ord ('7'), kbF7  : MoveWindow;
      Ord ('8'), kbF8  : ResizeWindow;
      Ord ('q'), kbEsc : Done := True;
      else
        if WindowList <> nil then
          with WindowList^, WindowTypes [WindowType] do
            if @KeyProc <> nil then
              begin
                TextColor (Color);
                TextBackground (Background);
                KeyProc (Key)
              end
    end
  until Done
end.
