
/*
 * Gnome-biff program
 * Version:  2.6
 * Copyright (C) 2000-2001  Nicolas Rougier
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <ctype.h>
#include <fcntl.h>

#include <errno.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <dirent.h>
#include <time.h>
#include <utime.h>

#include "protocol.h"
#include "gbiff.h"
#include "support.h"


/********************************************************************************
 * Function: Check the status of a FILE mailbox
 * Input   : Biff structure
 * Output  : /
 * Return  : STATUS_NEW_MAIL : new mail has arrived
 *           STATUS_OLD_MAIL : no new mail
 *           STATUS_NO_MAIL  : no mail
 *           -1              : error
 ********************************************************************************/
int FILE_checkmail (Biff *biff) {
  struct stat fileStat;
  static time_t mTime = 0;

  /* Check file modification time */
  if (stat (biff->address, &fileStat) != 0) {
    g_warning (_("Cannot open mailfile \"%s\" (%s).\n"), biff->address, strerror(errno));
    return (-1);
  }

  /* If modification time has not changed then only old mail */
  if (fileStat.st_mtime == mTime) {
    if (fileStat.st_size == 0) {
      biff->unread = 0;
      return (STATUS_NO_MAIL);
    }
    else {
      return (STATUS_OLD_MAIL);
    }
  }
  /* Else, save new modification time */
  else
    mTime = fileStat.st_mtime;
  /* If size is zero, then no mail at all */
  if (fileStat.st_size == 0) {
    biff->unread = 0;
    return (STATUS_NO_MAIL);
  }
  return (STATUS_NEW_MAIL);
}


/********************************************************************************
 * Function: Get headers from a FILE mailbox
 * Input   : Biff structure
 * Output  : Fill headers in the biff structure
 * Return  : 0 if OK, -1 else
 ********************************************************************************/
int FILE_getheaders (Biff *biff) {
  static int headerNB = 0;
  char line[256];
  char number[256];
  char sender[256];
  char subject[256];
  char date[256];
  FILE *file;
  char c;
  int senderFlag = 0;
  int subjectFlag = 0;
  int dateFlag = 0;
  int headerFlag = 0;
  int i;
  struct stat fileStat;
  struct utimbuf buf;

  /* Free previous headers */
  for (i=0; i<headerNB; i++) {
    free (biff->headers[i][NUMBER_INDEX]);
    free (biff->headers[i][SENDER_INDEX]);
    free (biff->headers[i][FROM_INDEX]);
    free (biff->headers[i][DATE_INDEX]);
  };
  headerNB = 0;


  /* Save last access time for resetting it at the end (to avoid confusing mutt) */
  if (stat (biff->address, &fileStat) != 0) {
    g_warning (_("Cannot open mailfile \"%s\" (%s).\n"), biff->address, strerror(errno));
    return (-1);
  }
  buf.actime = fileStat.st_atime;
  buf.modtime = fileStat.st_mtime;

  /* Open mail file */
  if ((file = fopen (biff->address, "r")) == NULL) {
    g_warning (_("Cannot open %s mailfile (%s).\n"), biff->address, strerror (errno));
    return (-1);
  }
  
  /* We do not want to collect all headers since they won't fit on screen */
  while ((fgets (line, 255, file)) && (headerNB < MAX_HEADER_IN_POPUP)) {
    /* New header */
    if ((char *) strstr (line, "From ") == line) {
      senderFlag = 0;			/* No sender yet        */
      subjectFlag = 0;			/* No subject yet       */
      dateFlag = 0;			/* No date yet          */
      headerFlag = 0;			/* No end of header yet */
      strcpy (sender, _("(unknown)\n"));/* Default sender       */
      strcpy (subject, _("(none)\n"));	/* Default subject      */
      strcpy (date, _("???\n"));	/* Default date         */
    };
  
    /* Get sender */
    if ((strstr(line, "From: ") == line) && (senderFlag == 0)) {
      strncpy (sender, line+strlen("From: "), 254);
      sender[255] = '\0';
      senderFlag = 1;
      /* Check if multilines sender */
      while ((c = (char) getc (file)) == '\t') {
	ungetc (c, file);
	fgets (line, 255, file);
      };
      ungetc (c, file);
    };

    /* Get subject */
    if ((strstr(line, "Subject: ") == line) && (strlen (line) > strlen ("Subject: \n")) && (subjectFlag == 0)) {
      strncpy (subject, line+strlen ("Subject: "), 254);
      subject[255] = '\0';
      subjectFlag = 1;
      /* Check if multilines subject */
      while ((c = (char) getc (file)) == '\t') {
	ungetc (c, file);
	fgets (line, 255, file);
      };
      ungetc (c, file);
    };
    
    /* Get date */
    if ((strstr(line, "Date: ") == line) && (dateFlag == 0)) {
      strncpy (date, line+strlen ("Date: "), 254);
      date[255] = '\0';
      dateFlag = 1;
      /* Check if multilines date */
      while ((c = (char) getc (file)) == '\t') {
	ungetc (c, file);
	fgets (line, 255, file);
      };
      ungetc (c, file);
    };

    /* Concatenate mail header */
    if (((char *) strstr(line, "\n") == line) && (headerFlag == 0)) {
      senderFlag = 1;
      subjectFlag = 1;
      dateFlag = 1;
      headerFlag = 1;

      sprintf (number, "%.2d", headerNB+1);
      biff->headers[headerNB][NUMBER_INDEX] = strdup (number);
      biff->headers[headerNB][SENDER_INDEX] = strdup (parse_ISO_8859 (sender));
      biff->headers[headerNB][FROM_INDEX] = strdup (parse_ISO_8859 (subject));
      biff->headers[headerNB][DATE_INDEX] = strdup (parse_ISO_8859 (date));
      headerNB++;
    };
  };

  /* Close mail file */
  if (fclose (file) != 0) {
    g_warning (_("Problem while closing %s mailfile (%s).\n"), biff->address, strerror (errno));
    return (-1);
  }

  utime (biff->address, &buf);
  biff->unread = headerNB;
  return (0);
}

/********************************************************************************
 * Function: Check a qmail-style maildir for new mail
 * Input   : Biff structure
 * Output  : /
 * Return  : STATUS_NEW_MAIL : new mail has arrived
 *           STATUS_OLD_MAIL : no new mail
 *           STATUS_NO_MAIL  : no mail
 *           -1              : error
 * Notes   : Added by Neil Muller, Febuary 2001
 ********************************************************************************/
int MAILDIR_checkmail (Biff *biff) {
  char *newdir;
  static time_t mTime=0;
  struct stat fileStat;
  DIR *dir;
  struct dirent *dent;
  int dirsize=0;

  newdir = malloc (strlen(biff->address)+5);
  if (!newdir) return -1;
  strcpy (newdir,biff->address);  /* biff-> address Maildir base */
  strcat (newdir,"/new"); 	  /* Maildir/new stores new mail */

  if ( (stat (newdir, &fileStat) != 0)||(!S_ISDIR(fileStat.st_mode)) ) {
    g_warning (_("Cannot open Maildir \"%s\" (%s).\n"), biff->address, strerror (errno));
    free(newdir);
    return (-1);
  }

  if ((dir = opendir (newdir)) == NULL) {
    g_warning (_("Cannot open maildir (%s).\n"), strerror (errno));
    return (-1);
  }
  while ((dent = readdir(dir))) {
    if (dent->d_name[0] != '.')
      dirsize++;
  }
  closedir(dir); 
   
  if (fileStat.st_mtime == mTime) {
    if (dirsize==0) {
      biff->unread = 0;
      free(newdir);
      return (STATUS_NO_MAIL);
    }
    else {
      free(newdir);
      return (STATUS_OLD_MAIL);
    }
  }
  /* Else, save new modification time */
  else
    mTime = fileStat.st_mtime;
  /* If size is zero, then no mail at all */
  if (dirsize == 0) {
    biff->unread = 0;
    free(newdir);
    return (STATUS_NO_MAIL);
  }
  free (newdir);
  return (STATUS_NEW_MAIL);
}

/********************************************************************************
 * Function: Fetch headers from a qmail-style maildir 
 * Input   : Biff structure
 * Output  : Fill headers in Biff structure
 * Return  : 0 OK, -1 if error
 * Notes   : Added by Neil Muller, Febuary 2001
 *           Large chunks ripper from FILE_getheaders
 ********************************************************************************/
int MAILDIR_getheaders (Biff *biff) {
  static int headerNB = 0;
  int i,flen;
  char *newdir,*filename;
  DIR *dir;
  FILE *file;
  struct dirent *dent;
  char line[256],date[256],subject[256],sender[256],number[256];
  
  /* Free previous headers */
  for (i=0;i<headerNB;i++) {
    free (biff->headers[i][NUMBER_INDEX]);
    free (biff->headers[i][SENDER_INDEX]);
    free (biff->headers[i][FROM_INDEX]);
    free (biff->headers[i][DATE_INDEX]);
  }
  headerNB=0;

  /* OK, open dir and start scanning files */
  newdir = malloc (strlen(biff->address)+5);
  if (!newdir) return -1;
  strcpy (newdir,biff->address);  /* biff-> address Maildir base */
  strcat (newdir,"/new"); 	  /* Maildir/new stores new mail */


  if ((dir = opendir (newdir)) == NULL) {
    g_warning (_("Cannot open maildir (%s).\n"), strerror (errno));
    return (-1);
  }

  /* We do not want to collect all headers since they won't fit on screen */
  while ((dent=readdir(dir)) && (headerNB<MAX_HEADER_IN_POPUP))
  {
    if (dent->d_name[0]=='.')
      continue;
    flen=strlen(newdir)+strlen(dent->d_name)+2;
    filename = malloc(flen);
    if (!filename)
      continue; /* Skip this one */
    sprintf(filename,"%s/%s",newdir,dent->d_name);
    file = fopen(filename,"r");
    if (file) {
      /* OK, extract header info from file */
      sprintf (number, "%.2d", headerNB+1);
      sprintf (date, _("???\n"));
      sprintf (subject, _("(none)\n"));
      sprintf (sender, _("(unknown)\n"));
      while (fgets(line,255,file)) {
       if (strstr(line,"\n")==line)
         break; /* End of headers, can skip rest of loop */
       if ((strstr(line,"From:")==line) && (strlen(line)>strlen("From: "))) {
         strncpy(sender,line+strlen("From: "),254);
	 sender[255] = '\0';
       }
       if ((strstr(line,"Subject:")==line) && (strlen(line)>strlen("Subject: "))) {
	 strncpy(subject,line+strlen("Subject: "),254);
	 subject[255] = '\0';
       }
       if ((strstr(line,"Date:")==line) && (strlen(line)>strlen("Date: "))) {
	  strncpy(date,line+strlen("Date: "),254);
	  date[255] = '\0';
       }
      } 
      biff->headers[headerNB][NUMBER_INDEX] = strdup(number);
      biff->headers[headerNB][SENDER_INDEX] = strdup (parse_ISO_8859 (sender));
      biff->headers[headerNB][FROM_INDEX] = strdup (parse_ISO_8859 (subject));
      biff->headers[headerNB][DATE_INDEX] = strdup (parse_ISO_8859 (date));
      headerNB++;
      fclose(file);
    }
    free(filename);  
  }
  closedir(dir);
  biff->unread=headerNB;
  free(newdir);
  return (0);
}


/********************************************************************************
 * Function: Check an MH-style maildir for new mail
 * Input   : Biff structure
 * Output  : /
 * Return  : STATUS_NEW_MAIL : new mail has arrived
 *           STATUS_OLD_MAIL : no new mail
 *           STATUS_NO_MAIL  : no mail
 *           -1              : error
 ********************************************************************************/
int MH_checkmail (Biff *biff) {
  FILE *file;
  struct stat fileStat;
  int status, found, i, j;
  char *filename, *buffer;
  char line[4096];
  static int unseen[MAX_HEADER_IN_POPUP];
  int unseen_tmp[MAX_HEADER_IN_POPUP];
  static int unseen_nb;
  int inf_bound, sup_bound, range_end, n;

  
  if (!(filename = malloc (strlen (biff->address) + strlen ("/.mh_sequences") + 1)))
    return -1;
  strcpy (filename, biff->address);
  if (filename[strlen(biff->address)-1] != '/')
    strcat (filename,"/");
  strcat (filename,".mh_sequences");

  if ((stat (biff->address, &fileStat) != 0)||(!S_ISDIR(fileStat.st_mode)) ) {
    g_warning (_("Cannot open maildir \"%s\" (%s).\n"), biff->address, strerror (errno));
    return (-1);
  }

  /* Open .mh_sequences file */
  if ((file = fopen (filename, "r")) == NULL) {
    g_warning (_("Cannot open MH sequences file : %s (%s).\n"), filename, strerror (errno));
    return (-1);
  }
  

  /* Analyze of the unseen sequence 
   * unseen sequences looks something like
   * unseen: 1-3, 7, 9-13, 16, 19
   */
  n = 0;
  inf_bound = 0;
  sup_bound = 0;
  range_end = 0;
  while (fgets (line, 4095, file)) {
    if (strstr (line, "unseen:")) {
      buffer = line + strlen ("unseen: ");

      while (*buffer != '\n') {
	/*
	 * First case : digit
	 */
	if (isdigit (*buffer)) {
	  inf_bound = 0;
	  while (isdigit (*buffer)) {
	    inf_bound *= 10;
	    inf_bound += (*buffer-'0');
	    buffer++;
	  }
	}
	/*
	 * Second case: number separator ' ' or '\n'
	 */
	else if ((*buffer == ' ') || (*buffer == '\n')) {
	  /* If range is 0 then we get a single number */
	  if (range_end == 0) {
	    if (n < MAX_HEADER_IN_POPUP)
	      unseen_tmp[n++] = inf_bound;
	  }
	  /* else, we reached the end of a range so reset the range flag */
	  else
	    range_end = 0;
	  inf_bound = 0;
	  buffer++;
	}
	/*
	 * Third case: range indicator '-'
	 */
	else if (*buffer == '-') {
	  buffer++;
	  sup_bound = 0;
	  while (isdigit (*buffer)) {
	    sup_bound *= 10;
	    sup_bound += (*buffer-'0');
	    buffer++;
	  }
	  if ((n + sup_bound - inf_bound + 1) >= MAX_HEADER_IN_POPUP) {
	    sup_bound -= (n + sup_bound - inf_bound + 1 - MAX_HEADER_IN_POPUP);
	  }
	  for (i=n; i<(n+sup_bound-inf_bound+1); i++)
	    unseen_tmp[i] = inf_bound + i - n;
	  n += (sup_bound - inf_bound + 1);
	  range_end = 1;
	  inf_bound = 0;
	}
	/*
	 * Else, we advance in the line
	 */
	else {
	  buffer++;
	}
      }
      if ((inf_bound > 0) && (n < MAX_HEADER_IN_POPUP))
	unseen_tmp[n++] = inf_bound;
    }
  }

  /* Close mail file
   *  => in case of problem we continue anyway
   *  => buf be careful, this may result in too many opened file descriptors
   */
  if (fclose (file) != 0)
    g_warning (_("Problem while closing %s mailfile (%s).\n"), biff->address, strerror (errno));

  /* Ok, ath this point we get all unseen mails number in unseen_tmp
   * Now we compare the saved unseen list to the new one
   * if we found an unseen in the new list that is not in the old
   * list then we got new mail
   */
  status = -1;
  for (i=0; i<n; i++) {
    found = 0;
    for (j=0; j<unseen_nb; j++) {
      if (unseen_tmp[i] == unseen[j]) {
	found = 1;
	break;
      }
    }
    if (found == 0) {
      status = STATUS_NEW_MAIL;
      break;
    }
  }

  /* Finding status */
  if ((status == -1) && (n > 0))
    status = STATUS_OLD_MAIL;
  else if ((status == -1) && (n == 0))
    status = STATUS_NO_MAIL;
  
  /* Number of unread mail */
  biff->unread = n;

  /* Saving new unseen list */
  for (i=0; i<n; i++)
    unseen[i] = unseen_tmp[i];
  unseen_nb = n;

  free (filename);
  return (status);
}


/********************************************************************************
 * Function: Fetch headers from an MH-style Maildir 
 * Input   : Biff structure
 * Output  : Fill headers in Biff structure
 * Return  : 0 OK, -1 if error
 ********************************************************************************/
int MH_getheaders (Biff *biff) {
  FILE *file;
  struct stat fileStat;
  int i;
  char *filename, *buffer;
  char line[4096];
  static int headerNB = 0;
  char date[256],subject[256],sender[256],number[256];
  int unseen[MAX_HEADER_IN_POPUP];
  int unseen_nb;
  int inf_bound, sup_bound, range_end;


  
  /* Free previous headers */
  for (i=0;i<headerNB;i++) {
    free (biff->headers[i][NUMBER_INDEX]);
    free (biff->headers[i][SENDER_INDEX]);
    free (biff->headers[i][FROM_INDEX]);
    free (biff->headers[i][DATE_INDEX]);
  }
  headerNB=0;


  if (!(filename = malloc (strlen (biff->address) + strlen ("/.mh_sequences") + 2)))
    return -1;
  strcpy (filename, biff->address);
  if (filename[strlen(biff->address)-1] != '/')
    strcat (filename,"/");
  strcat (filename,".mh_sequences");

  if ((stat (biff->address, &fileStat) != 0)||(!S_ISDIR(fileStat.st_mode)) ) {
    g_warning (_("Cannot open maildir \"%s\" (%s).\n"), biff->address, strerror (errno));
    return (-1);
  }

  /* Open .mh_sequences file */
  if ((file = fopen (filename, "r")) == NULL) {
    g_warning (_("Cannot open MH sequences file : %s (%s).\n"), filename, strerror (errno));
    return (-1);
  }
  

  /* Analyze of the unseen sequence 
   * unseen sequences looks something like
   * unseen: 1-3 7 9-13
   * WARNING: we analyze only the first 4095 bytes of the unseen lines,
   *          it will generally be sufficient to handle a hundred unseen mails
   */
  unseen_nb = 0;
  inf_bound = 0;
  sup_bound = 0;
  range_end = 0;
  while (fgets (line, 4095, file)) {
    if (strstr (line, "unseen:")) {
      buffer = line + strlen ("unseen: ");
      while (*buffer != '\n') {
	/*
	 * First case : digit
	 */
	if (isdigit (*buffer)) {
	  inf_bound = 0;
	  while (isdigit (*buffer)) {
	    inf_bound *= 10;
	    inf_bound += (*buffer-'0');
	    buffer++;
	  }
	}
	/*
	 * Second case: number separator ' ' or '\n'
	 */
	else if ((*buffer == ' ') || (*buffer == '\n')) {
	  /* If range is 0 then we get a single number */
	  if (range_end == 0) {
	    if (unseen_nb < MAX_HEADER_IN_POPUP)
	      unseen[unseen_nb++] = inf_bound;
	  }
	  /* else, we reached the end of a range so reset the range flag */
	  else
	    range_end = 0;
	  buffer++;
	  inf_bound = 0;
	}
	/*
	 * Third case: range indicator '-'
	 */
	else if (*buffer == '-') {
	  buffer++;
	  sup_bound = 0;
	  while (isdigit (*buffer)) {
	    sup_bound *= 10;
	    sup_bound += (*buffer-'0');
	    buffer++;
	  }
	  if ((unseen_nb + sup_bound - inf_bound + 1) >= MAX_HEADER_IN_POPUP) {
	    sup_bound -= (unseen_nb + sup_bound - inf_bound + 1 - MAX_HEADER_IN_POPUP);
	  }
	  for (i=unseen_nb; i<(unseen_nb+sup_bound-inf_bound+1); i++)
	    unseen[i] = inf_bound + i - unseen_nb;
	  unseen_nb += (sup_bound - inf_bound + 1);
	  range_end = 1;
	  inf_bound = 0;
	}
	/*
	 * Else, we advance in the line
	 */
	else {
	  buffer++;
	}
      }
      if ((inf_bound > 0) && (unseen_nb < MAX_HEADER_IN_POPUP))
	unseen[unseen_nb++] = inf_bound;
    }
  }
  /* Close mail file
   *  => in case of problem we continue anyway
   *  => buf be careful, this may result in too many opened file descriptors
   */
  if (fclose (file) != 0)
    g_warning (_("Problem while closing %s mailfile (%s).\n"), biff->address, strerror (errno));


  for (i=0; i<unseen_nb; i++) {
    if (biff->address[strlen(biff->address)-1] == '/') 
      sprintf (filename, "%s%d", biff->address, unseen[i]);
    else
      sprintf (filename, "%s/%d", biff->address, unseen[i]);

    file = fopen (filename, "r");
    if (file) {
      /* OK, extract header info from file */
      sprintf (number, "%.2d",headerNB+1);
      sprintf (date, _("???\n"));
      sprintf (subject, _("(none)\n"));
      sprintf (sender, _("(unknown)\n"));
      while (fgets(line,255,file)) {
       if (strstr(line,"\n")==line)
         break; /* End of headers, can skip rest of loop */
       if ((strstr(line,"From:")==line) && (strlen(line)>strlen("From: "))) {
         strncpy(sender,line+strlen("From: "),254);
	 sender[255] = '\0';
       }
       if ((strstr(line,"Subject:")==line) && (strlen(line)>strlen("Subject: "))) {
	 strncpy(subject,line+strlen("Subject: "),254);
	 subject[255] = '\0';
       }
       if ((strstr(line,"Date:")==line) && (strlen(line)>strlen("Date: "))) {
	  strncpy(date,line+strlen("Date: "),254);
	  date[255] = '\0';
       }
      } 
      biff->headers[headerNB][NUMBER_INDEX] = strdup(number);
      biff->headers[headerNB][SENDER_INDEX] = strdup (parse_ISO_8859 (sender));
      biff->headers[headerNB][FROM_INDEX] = strdup (parse_ISO_8859 (subject));
      biff->headers[headerNB][DATE_INDEX] = strdup (parse_ISO_8859 (date));
      headerNB++;
      /* Close mail file
       *  => in case of problem we continue anyway
       *  => buf be careful, this may result in too many opened file descriptors
       */
      if (fclose (file) != 0)
	g_warning (_("Problem while closing %s mailfile (%s).\n"), biff->address, strerror (errno));
    }
  }
  free (filename);
  biff->unread = headerNB;
  return (0);
}


/********************************************************************************
 * Function: Connect to a pop3 server and check the status of the mailbox
 * Input   : Biff structure
 * Output  : /
 * Return  : STATUS_NEW_MAIL : new mail has arrived
 *           STATUS_OLD_MAIL : no new mail
 *           STATUS_NO_MAIL  : no mail
 *           -1              : error
 ********************************************************************************/
int POP3_checkmail (Biff *biff) {
  int socketFD;
  char *msg;
  char command[64];
  /* Here I could have used lists for storing uidl but since gbiff won't display
     more than 100 headers an array was easier (and I don't like lists anyway).
     Furthermore, according to rfc 1939, an uidl is between 1 and 70 characters
     long but, in case some servers decide to do weird things an uidl is saved
     within a 128 characters long array.
  */
  static char saved_uidl[100][128];
  static int saved_uidl_nb = 0;
  char new_uidl[100][128];
  int new_uidl_nb;
  int i, j, status, found;
  char *address;
  int port;

  /* Get host and port */
  sscanf (strchr (biff->address, (int) ':')+1, "%d", &port);
  address = strdup (biff->address);
  *strchr(address,(int) ':') = '\0';

  /* Authorization state */
  if ((socketFD = socket_open (address, port)) == -1) {
    g_warning (_("Cannot connect to server.\n"));
    biff->unread = -1;
    biff->total = -1;
    free (address);
    return (-1);
  }
  msg = socket_read (socketFD);
  if (strncmp (msg,"+OK",3) != 0) {
    g_warning (_("Cannot initiate authorization state: %s\n"), msg);
    socket_close (socketFD);
    free (address);
    return (-1);
  }

  /* User identification */
  sprintf (command, "USER %.32s\r\n", biff->user);
  socket_write (socketFD, command);
  msg = socket_read (socketFD);
  if (strncmp (msg,"+OK",3) != 0) {
    g_warning (_("Cannot identify user: %s\n"), msg);
    biff->unread = -1;
    biff->total = -1;
    socket_close (socketFD);
    free (address);
    return (-1);
  }

  /* Password authentication */
  sprintf (command, "PASS %.32s\r\n", biff->password);
  socket_write (socketFD, command);
  msg = socket_read (socketFD);
  if (strncmp (msg,"+OK",3) != 0) {
    g_warning (_("Cannot authenticate user: %s\n"), msg);
    biff->unread = -1;
    biff->total = -1;
    socket_close (socketFD);
    free (address);
    return (-1);
  }

  /* Retrieving UIDL list */
  /* Each message on a pop3 server can be identified by its uidl, then, each time
     we check for new mail, we compare the server uidl list to our own saved list
     (from the previous checking).
     Then, any new uidl on the server means new message and any lacking uidl on
     the server means deleted message.
  */
  sprintf (command, "UIDL\r\n");
  socket_write (socketFD, command);
  msg = socket_read (socketFD);
  if (strncmp (msg,"+OK",3) != 0) {
    g_warning (_("Cannot retrieve unique-id listings: %s\n"), msg);
    socket_close (socketFD);
    free (address);
    return (-1);
  }
  new_uidl_nb = 0;
  do {
    msg = socket_read (socketFD);
    if (msg[0] != '.') {
      sscanf (msg, "%d %s", &i, new_uidl[new_uidl_nb]);
      new_uidl_nb++;
    }
  } while (msg[0] != '.');

  /* Now we compare the saved uidl list to the new one
   * if we found an uidl in the new list that is not in the old
   * list then we got new mail
   */
  status = -1;
  for (i=0; i<new_uidl_nb; i++) {
    found = 0;
    for (j=0; j<saved_uidl_nb; j++) {
      if (strcmp (new_uidl[i], saved_uidl[j]) == 0) {
	found = 1;
	break;
      }
    }
    if (found == 0) {
      status = STATUS_NEW_MAIL;
      break;
    }
  }

  /* Finding status */
  if ((status == -1) && (new_uidl_nb > 0))
    status = STATUS_OLD_MAIL;
  else if ((status == -1) && (new_uidl_nb == 0))
    status = STATUS_NO_MAIL;
  
  /* Number of unread mail */
  biff->unread = new_uidl_nb;

  /* Closing connection */
  socket_write (socketFD, "QUIT\r\n");
  msg = socket_read (socketFD);  
  socket_close (socketFD);
  
  /* Saving new uidl list */
  for (i=0; i<new_uidl_nb; i++)
    strcpy (saved_uidl[i], new_uidl[i]);
  saved_uidl_nb = new_uidl_nb;

  free (address);
  return (status);
}


/********************************************************************************
 * Function: Get headers from a POP3 mailbox
 * Input   : Biff structure
 * Output  : Fill headers in the biff structure
 * Return  : 0 if OK, -1 else
 ********************************************************************************/
int POP3_getheaders (Biff *biff) {
  static int headerNB = 0;
  int socketFD;
  char *msg;
  char command[64];
  int i;
  char number[256];
  char sender[256];
  char subject[256];
  char date[256];
  int senderFlag = 0;
  int subjectFlag = 0;
  int dateFlag = 0;
  int headerFlag = 0;
  int port;
  char *address;

  /* Free previous headers */
  for (i=0; i<headerNB; i++) {
    free (biff->headers[i][NUMBER_INDEX]);
    free (biff->headers[i][SENDER_INDEX]);
    free (biff->headers[i][FROM_INDEX]);
    free (biff->headers[i][DATE_INDEX]);
  }
  headerNB = 0;


  /* Get host and port */
  sscanf (strchr (biff->address, (int) ':')+1, "%d", &port);
  address = strdup (biff->address);
  *strchr (address, (int) ':') = 0;

  /* Authorization state */
  if ((socketFD = socket_open (address, port)) == -1) {
    g_warning (_("Cannot connect to server.\n"));
    free (address);
    return (-1);
  }
  msg = socket_read (socketFD);
  if (strncmp (msg,"+OK",3) != 0) {
    g_warning (_("Cannot initiate authorization state: %s\n"), msg);
    biff->unread = -1;
    socket_close (socketFD);
    free (address);
    return (-1);
  }

  /* User identification */
  sprintf (command, "USER %.32s\r\n", biff->user);
  socket_write (socketFD, command);
  msg = socket_read (socketFD);
  if (strncmp (msg,"+OK",3) != 0) {
    g_warning (_("Cannot identify user: %s\n"), msg);
    biff->unread = -1;
    socket_close (socketFD);
    free (address);
    return (-1);
  }

  /* Password authentication */
  sprintf (command, "PASS %.32s\r\n", biff->password);
  socket_write (socketFD, command);
  msg = socket_read (socketFD);
  if (strncmp (msg,"+OK",3) != 0) {
    g_warning (_("Cannot authenticate user: %s\n"), msg);
    biff->unread = -1;
    socket_close (socketFD);
    free (address);
    return (-1);
  }

  /* Retrieve message number */
  sprintf (command, "STAT\r\n");
  socket_write (socketFD, command);
  msg = socket_read (socketFD);
  if (strncmp (msg,"+OK",3) != 0) {
    g_warning (_("Cannot get message number: %s\n"), msg);
    biff->unread = -1;
    socket_close (socketFD);
    free (address);
    return (-1);
  }
  sscanf (msg,  "+OK %d", &headerNB);

  /* We do not want to collect all headers since they won't fit on screen */
  if (headerNB > MAX_HEADER_IN_POPUP) {
    headerNB = MAX_HEADER_IN_POPUP;
  };

  /* Scan each message header */
  for (i=0; i<headerNB; i++) {
    sprintf (command, "TOP %d 0\r\n", i+1);
    socket_write (socketFD, command);
    msg = socket_read (socketFD);
    if (strncmp (msg,"+OK",3) != 0) {
      g_warning (_("Cannot get message number: %s\n"), msg);
      biff->unread = -1;
      socket_close (socketFD);
      free (address);
      return (-1);
    }
    senderFlag = 0;			/* No sender yet        */
    subjectFlag = 0;			/* No subject yet       */
    dateFlag = 0;			/* No date yet          */
    headerFlag = 0;			/* No end of header yet */
    strcpy (sender, _("(unknown)\n"));	/* Default sender       */
    strcpy (subject, _("(none)\n"));	/* Default subject      */
    strcpy (date, _("???\n"));		/* Default date         */
    do {
      msg = socket_read (socketFD);
      if ((strstr(msg, "From: ") == msg) && (senderFlag == 0)) {
	strncpy (sender, msg+strlen("From: "), 254);
	sender[255] = '\0';
	senderFlag = 1;
      }
      else if ((strstr(msg, "Subject: ") == msg) && (strlen (msg) > strlen ("Subject: \r\n")) && (subjectFlag == 0)) {
	strncpy (subject, msg+strlen ("Subject: "),254);
	subject[255] = '\0';
	subjectFlag = 1;
      }
      else if ((strstr(msg, "Date: ") == msg) && (dateFlag == 0)) {
	strncpy (date, msg+strlen ("Date: "),254);
	date[255] = '\0';
	dateFlag = 1;
      }
    } while (msg[0] != '.');
    
    sprintf (number, "%.2d", i+1);
    biff->headers[i][NUMBER_INDEX] = strdup (number);
    biff->headers[i][SENDER_INDEX] = strdup (parse_ISO_8859 (sender));
    biff->headers[i][FROM_INDEX] = strdup (parse_ISO_8859 (subject));
    biff->headers[i][DATE_INDEX] = strdup (parse_ISO_8859 (date));
  }
  biff->unread = headerNB;
  
  
  /* Closing connection */
  socket_write (socketFD, "QUIT\r\n");
  msg = socket_read (socketFD);  
  socket_close (socketFD);

  free (address);
  return (0);
}


/********************************************************************************
 * Function: Check the status of an IMAP4 mailbox
 * Input   : Biff structure
 * Output  : /
 * Return  : STATUS_NEW_MAIL : new mail has arrived
 *           STATUS_OLD_MAIL : no new mail
 *           STATUS_NO_MAIL  : no mail
 *           -1              : error
 * Notes   : Added by Yu Cai, May 2001
 ********************************************************************************/
int IMAP4_checkmail (Biff *biff) {
 int socketFD;
  char *msg;
  char command[64];
  int status, found;
  int unreadmsg, totalmsg;
  char *address;
  int port;

  /* Get host and port */
  sscanf (strchr (biff->address, (int) ':')+1, "%d", &port);
  address = strdup (biff->address);
  *strchr(address,(int) ':') = '\0';
  
  /* Authorization state */
  if ((socketFD = socket_open (address, port)) == -1) {
    g_warning (_("Cannot connect to server.\n"));
    biff->unread = -1;
    free (address);
    return (-1);
  }
  msg = socket_read (socketFD);
  if (strncmp (msg,"* OK",4) != 0) {
    g_warning (_("Cannot initiate authorization state: %s\n"), msg);
    socket_close (socketFD);
    free (address);
    return (-1);
  }

  /* User identification */
  sprintf (command, "A001 LOGIN %.32s %.32s\r\n", biff->user,biff->password);
  socket_write (socketFD, command);
  msg = socket_read (socketFD);
  if (strncmp (msg,"A001 OK",7) != 0) {
    g_warning ("Cannot identify user: %s\n", msg);
    biff->unread = -1;
    socket_close (socketFD);
    return (-1);
  }

  /* Retrieving INBOX list */
  sprintf (command, "A002 SELECT INBOX\r\n");
  socket_write (socketFD, command);
  found = 0;
  unreadmsg = 0;
  totalmsg = 0;
  do {
    msg = socket_read (socketFD);
    if(strncmp(msg,"A002",4)!=0) {
      if(strstr(msg,"* OK [UNSEEN")){
	sscanf(msg,"* OK [UNSEEN %d] %*s",&unreadmsg);
      }
      if(strstr(msg,"EXISTS"))  // ? are there any better way to match
	sscanf(msg,"* %d EXISTS %*s",&totalmsg);
    }
    else {
      if (strncmp(msg,"A002 OK",7) != 0) {
	g_warning (_("Cannot select INBOX : %s\n"), msg);
	return (-1);
      }
      else {
	found = 1;
      }
    }
  } while(!found);
  
  if(unreadmsg) {
    status = STATUS_NEW_MAIL;
    /* Number of unread mail */
    biff->unread = 1; // ?? 
  }
  else {
    status = STATUS_OLD_MAIL;
    biff->unread = 0;
  }

  /* Closing connection */
  socket_write (socketFD, "A003 LOGOUT\r\n");
  msg = socket_read (socketFD);  
  socket_close (socketFD);

  return (status);
}


/********************************************************************************
 * Function: Get headers from an IMAP4 mailbox
 * Input   : Biff structure
 * Output  : Fill headers in the biff structure
 * Return  : 0 if OK, -1 else
 * Notes   : Added by Yu Cai, May 2001
 ********************************************************************************/
int IMAP4_getheaders (Biff *biff) {
  static int headerNB = 0;
  int socketFD;
  char *msg;
  char command[64];
  int i;
  char number[256];
  char sender[256];
  char subject[256];
  char date[256];
  int senderFlag = 0;
  int subjectFlag = 0;
  int dateFlag = 0;
  int headerFlag = 0;
  
  char status[256];
  char *pos,*pos1;
  int totalmsg,unreadmsg,found;
  char *address;
  int port;

  /* Free previous headers */
  for (i=0; i<headerNB; i++) {
    free (biff->headers[i][NUMBER_INDEX]);
    free (biff->headers[i][SENDER_INDEX]);
    free (biff->headers[i][FROM_INDEX]);
    free (biff->headers[i][DATE_INDEX]);
  }
  headerNB = 0;

  /* Get host and port */
  sscanf (strchr (biff->address, (int) ':')+1, "%d", &port);
  address = strdup (biff->address);
  *strchr(address,(int) ':') = '\0';

  /* Authorization state */
  if ((socketFD = socket_open (address, port)) == -1) {
    g_warning (_("Cannot connect to server.\n"));
    biff->unread = -1;
    free (address);
    return (-1);
  }
  msg = socket_read (socketFD);
  if (strncmp (msg,"* OK",4) != 0) {
    g_warning (_("Cannot initiate authorization state: %s\n"), msg);
    socket_close (socketFD);
    free (address);
    return (-1);
  }

  /* User identification */
  sprintf (command, "A001 LOGIN %.32s %.32s\r\n", biff->user,biff->password);
  socket_write (socketFD, command);
  msg = socket_read (socketFD);
  if (strncmp (msg,"A001 OK",7) != 0) {
    g_warning (_("Cannot identify user: %s\n"), msg);
    biff->unread = -1;
    socket_close (socketFD);
    free (address);
    return (-1);
  }

  /* Retrieving INBOX list */
  sprintf (command, "A002 SELECT INBOX\r\n");
  socket_write (socketFD, command);
  found = 0;
  unreadmsg = 0;
  totalmsg = 0;
  do {
    msg = socket_read (socketFD);
    if(strncmp(msg,"A002",4)!=0) {
      if(strstr(msg,"* OK [UNSEEN")){
	sscanf(msg,"* OK [UNSEEN %d] %*s",&unreadmsg);
      }
      if(strstr(msg,"EXISTS"))  // ? are there any better way to match
	sscanf(msg,"* %d EXISTS %*s",&totalmsg);      
    }
    else {
      if (strncmp(msg,"A002 OK",7) != 0) {
	g_warning (_("Cannot select INBOX : %s\n"), msg);
	socket_close (socketFD);
	free (address);
	return (-1);
      }
      else {
	found = 1;
      }
    }
  } while(!found);
  
  if(unreadmsg ==0) 
    unreadmsg = totalmsg+1; /* ignore the loop below */
  
  /* Scan each message header */
  for (i=unreadmsg; i<=totalmsg; i++) {
    sprintf(command,"A003 FETCH %d (FLAGS BODY.PEEK[HEADER.FIELDS (DATE FROM SUBJECT)])\r\n",i);
    socket_write (socketFD, command);
    msg = socket_read (socketFD);
    /*              g_warning ("Fetch result: %s\n", msg);  */
    
    if (strncmp(msg,"A003 NO",7)==0) {
      socket_close (socketFD);
      free (address);
      return (-1);
    }

    pos = strchr(msg,'(');
    pos1= strchr(pos+1,'(');
    pos = strchr(pos1+1,')');
    strncpy (status,pos1+1,pos-pos1-1);
    status[pos-pos1-1]='\0';
    if (strstr(status,"Deleted")    /* Deleted */
	|| strstr(status,"Seen")) { /* Seen */
      do {
	msg = socket_read (socketFD);
      } while(!strstr(msg,"A003 OK"));
      continue;
    }
                
    senderFlag = 0;                 /* No sender yet        */
    subjectFlag = 0;                /* No subject yet       */
    dateFlag = 0;                   /* No date yet          */
    headerFlag = 0;                 /* No end of header yet */
    strcpy (sender, "(unknown)\n"); /* Default sender       */
    strcpy (subject, "(none)\n");   /* Default subject      */
    strcpy (date, "???\n");         /* Default date         */

    found = 0;
    do {
      msg = socket_read (socketFD);
      /*                      g_warning ("Fetch inside result: %s\n", msg); */
      if ((strstr(msg, "From: ") == msg) && (senderFlag == 0)) {
	strncpy (sender, msg+strlen("From: "), 254);
	sender[255] = '\0';
	senderFlag = 1;
      }
      else if ((strstr(msg, "Subject: ") == msg) && (strlen (msg) > strlen ("Subject: "))
	       && (subjectFlag == 0)) {
	strncpy (subject, msg+strlen ("Subject: "),254);
	subject[255] = '\0';
	subjectFlag = 1;
      }
      else if ((strstr(msg, "Date: ") == msg) && (dateFlag == 0)) {
	strncpy (date, msg+strlen ("Date: "),254);
	date[255] = '\0';
	dateFlag = 1;
      }
      else if(strstr(msg,"A003 OK")) {
	found = 1;
      }
      
    } while (!found);
    
    sprintf (number, "%.2d", headerNB+1);
    biff->headers[headerNB][NUMBER_INDEX] = strdup (number);
    biff->headers[headerNB][SENDER_INDEX] = strdup (parse_ISO_8859 (sender));
    biff->headers[headerNB][FROM_INDEX] = strdup (parse_ISO_8859 (subject));
    biff->headers[headerNB][DATE_INDEX] = strdup (parse_ISO_8859 (date));
    /* We do not want to collect all headers since they won't fit on screen */
    headerNB ++;
    if (headerNB > MAX_HEADER_IN_POPUP) {
      headerNB = MAX_HEADER_IN_POPUP;
    }   
  }
  biff->unread = headerNB;
  
  /* Closing connection */
  socket_write (socketFD, "A004 LOGOUT\r\n");
  msg = socket_read (socketFD);  
  socket_close (socketFD);

  free (address);  
  return (0);
}


/********************************************************************************
 * Function: Open a connection on the given hostname and port
 * Input   : hostname and port
 * Output  : /
 * Return  : A file descriptor opened on the given host and port
 *           or -1 if connection is not possible
 ********************************************************************************/
int socket_open (char *hostname, unsigned short int port) {
  struct sockaddr_in sin;
  struct hostent *host;
  int socketFD;
  unsigned long int address;

  /* Create an endpoint for communication */
  if ((socketFD = socket (AF_INET, SOCK_STREAM, IPPROTO_IP)) == -1) {
    switch (errno) {
    case EPROTONOSUPPORT:
      g_warning (_("The  protocol type or the specified protocol is not supported within this domain.\n"));
      break;
    case ENFILE:
      g_warning (_("Not enough kernel memory to allocate a new socket structure.\n"));
      break;
    case EMFILE:
      g_warning (_("Process file table overflow.\n"));
      break;
    case EACCES:
      g_warning (_("Permission to create a socket of the specified type and/or protocol is denied.\n"));
      break;
    case ENOBUFS:
    case ENOMEM:
      g_warning (_("Insufficient memory is available.  The socket  cannot be created until sufficient resources are freed.\n"));
      break;
    case EINVAL:
      g_warning (_("Unknown protocol, or protocol family not available.\n"));
      break;
    default:
      g_warning (_("Unknown socket error.\n"));
      break;
    };
    return (-1);
  };

  /* Setting socket info for connection */
  memset ((char *)&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons (port);

  /* First, try to get the address by standard notation (e.g. 127.0.0.1) */
  if ((address = inet_addr (hostname)) == -1) {
    /* If it does not work, get the address by name (e.g. localhost.localdomain)*/
    if ((host = gethostbyname (hostname)) == 0) {
      switch (h_errno) {
      case HOST_NOT_FOUND:
	g_warning (_("The specified host is unknown.\n"));
	break;
      case NO_ADDRESS:
	g_warning (_("The requested name is valid but does not have an IP address.\n"));
	break;
      case NO_RECOVERY:
	g_warning (_("A non-recoverable name server error occurred.\n"));
	break;
      case TRY_AGAIN:
	g_warning (_("A temporary error occurred on an authoritative name server. Try again later.\n"));
	break;
      default:
	g_warning (_("Unknown gethostbyname error.\n"));
	break;
      };
      return (-1);
    };
    /* This way of filling sin_addr field avoid 'incompatible types in assignment' problems */
    memcpy ((void *) &sin.sin_addr, *(host->h_addr_list), host->h_length);
  }
  else {
    memcpy ((void *) &sin.sin_addr, (void *)&address, sizeof (address));
  };

  /* Initiate the connection on the socket */
  if ((connect (socketFD, (struct sockaddr *) &sin, sizeof (struct sockaddr_in))) == -1) {
    switch (errno) {
    case EBADF:
      g_warning (_("The file descriptor is not a valid index in the descriptor table.\n"));
      break;
    case EFAULT:
      g_warning (_("The socket structure address is outside the user's address space.\n"));
      break;
    case ENOTSOCK:
      g_warning (_("The file descriptor is not associated with a socket.\n"));
      break;
    case EISCONN:
      g_warning (_("The socket is already connected.\n"));
      break;
    case ECONNREFUSED:
      g_warning (_("No one listening on the remote address.\n"));
      break;
    case ETIMEDOUT:
      g_warning (_("Timeout while attempting connection.\n"));
      break;
    case ENETUNREACH:
      g_warning (_("Network is unreachable.\n"));
      break;
    case EADDRINUSE:
      g_warning (_("Local address is already in use.\n"));
      break;
    case EINPROGRESS:
      g_warning (_("The  socket is non-blocking and the connection cannot be completed immediately.\n"));
      break;
    case EALREADY:
      g_warning (_("The socket is non-blocking and a previous connection attempt has not yet been completed.\n"));
      break;
    case EAGAIN:
      g_warning (_("No more free local ports or insufficient entries in the routing cache.\n"));
      break;
    case EAFNOSUPPORT:
      g_warning (_("The passed address didn't have the correct address family in its sa_family field.\n"));
      break;
    case EACCES:
    case EPERM:
      g_warning (_("Connection request failed because of a local firewall rule.\n"));
      break;
    default:
      g_warning (_("Unknown connect error.\n"));
      break;
    };
    close (socketFD);
    return (-1);
  }
  return (socketFD);
};


/********************************************************************************
 * Function: Write a line on an opened socket
 * Input   : A socket file descriptor
 * Output  : /
 * Return  : 0 if OK, -1 else
 *******************************************************************************/
int socket_write (int socketFD, char *line) {

  if (write (socketFD, line, strlen (line)) <= 0) {
    g_warning (_("Problem occured while writing to the socket.\n"));
    return (-1);
  }
  return (0);
}


/********************************************************************************
 * Function: Read a line (terminated by '\n' from an opened socket
 * Input   : A socket file descriptor
 * Output  : /
 * Return  : A malloc'ed copy of the read line or 0 if problem
 *******************************************************************************/
char *socket_read (int socketFD) {
  static char buffer[1024];
  char *p;
  int bytes = 0;
  int status;

  p = buffer;
  while (((status = read (socketFD, p, 1)) > 0) && (*p++ != '\n') && (++bytes < 1023));
  *p = '\0';
  if (status <= 0) {
    /* g_warning (_("Problem occured while reading from the socket.\n")); */
    return (0);
  }
  /* In case of a buffer overflow we return what has been read so far */
  else if (bytes >= 1023)
    g_warning (_("Buffer overflow.\n"));

  return (buffer);
}



/********************************************************************************
 * Function: Empty any data stuck in socket and try to close it
 * Input   : A socket file descriptor
 * Output  : Close the specified socket
 * Return  : /
 *******************************************************************************/
void socket_close (int socketFD) {
  char *msg;

  fcntl (socketFD, F_SETFL, O_NONBLOCK);
  do {
    msg = socket_read (socketFD);
  } while (msg);
  close (socketFD);
}



/********************************************************************************
 * Function: Parse a text to remove any ISO-8859-?? encoding
 * Input   : Text string (terminated by 0)
 * Output  : /
 * Return  : Parsed line terminated by a newline or EOF which is replaced with '\0'
 ********************************************************************************/
char *parse_ISO_8859 (char *text) {
  static char *parsedText = 0;
  char *pText, *pParsedText, *p;
  int ISOmode, c, size;
  char code[5];

  if (text == (char *) -1) {
    free (parsedText);
    parsedText = 0;
    return (0);
  };
  
  if ((parsedText == 0) || (strlen (text) > strlen(parsedText)))
    parsedText = (char *) realloc (parsedText, strlen (text)+2);
  pText = text;
  pParsedText = parsedText;

  code[0] = '0';
  code[1] = 'x';
  code[4] = 0;

  ISOmode = 0;
  while ((*pText != 0) && (*pText != '\n')) {
    /* ASCII mode, copy character */
    if ((*pText != '=') && (*pText != '?')) {
      *pParsedText++ = *pText++;
    }
    /* ISO mode on */
    /* An iso encoding is something like that =?iso-8859-xx?C? 
       where xx is the number of the charset, and C the type of character
       (Q for quoted-printable and B for base64), so we look for =?iso-8859 
       and then the next '?'  this way we avoid to wonder if there
       is 1 or 2 digit in the charset (=?iso-8859-1?q? or =?iso-8859-15?q?)
    */
    else if ((strstr (pText, "=?iso-8859") == pText) || (strstr (pText, "=?ISO-8859") == pText)) {
       pText += strlen ("=?iso-8859");
      pText = strchr (pText, '?') +1;

      // Quoted printable characters
      if ((*pText == 'Q') || (*pText == 'q')) {
	ISOmode = 1;
	pText +=2;
      }
      // Base64 encoded characters
      // In this case, there is a function to decode the whole string,
      // so we do not enter in ISO mode.
      else if ((*pText == 'B') || (*pText == 'b')) {
	pText += 2;
	p = strstr (pText, "?=");
	size = parse_base64 (pText, (int) (p-pText-1), pParsedText);
	pParsedText += size;
	pText = p+2;
      }
    }
    /* ISO mode off */
    else if ((ISOmode == 1) && (strstr(pText, "?=") == pText)) {
      pText += 2;
      ISOmode = 0;
    }
    /* ISO mode, parse code */
    else if ((ISOmode == 1) && (*pText == '=')) {
      code[2] = *(pText+1);
      code[3] = *(pText+2);
      pText += 3;
      sscanf (code, "%i", &c);
      *pParsedText++ = (char) c;
    }
    /* ISO mode, copy character */
    else {
      *pParsedText++ = *pText++;
    };
  };
  *pParsedText = '\0';
    
  return (parsedText);
};



/********************************************************************************
 * Function: Decode a base64 string
 * Input   : buftode : text to decode
 *         : bufsize : size of the text
 *         : decbuf  : buffer for storing decoded text
 * Output  : /
 * Return  : Size fo the decoded text
 * Note    : Code by Michel Leunen <leu@rtbf.be>
 ********************************************************************************/
int parse_base64 (const char *buftodec, int bufsize, char *decbuf) {
  const char Base64Table[64]="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"; 
  int i=0; 
  unsigned char binbyte[4]; 
  int cpos[5]; 
  unsigned char *buftemp; 

  //Allocate space for the temporary string 
  buftemp = (unsigned char *) malloc (bufsize+1); 
  strncpy (buftemp, buftodec, bufsize);
  buftemp[bufsize] = '\0';

  if (fmod (bufsize,4) == 1)  { 
      buftemp[bufsize]   = '\0'; 
      buftemp[bufsize+1] = '\0'; 
      buftemp[bufsize+2] = '\0'; 
   } 
   if (fmod(bufsize,4) == 2) { 
      buftemp[bufsize]  ='\0'; 
      buftemp[bufsize+1]='\0'; 
   } 
   if (fmod(bufsize,4) == 3)
     buftemp[bufsize]='\0'; 
   while (i<bufsize)  { 
      cpos[0] = strchr (Base64Table,buftemp[i])-Base64Table; 
      cpos[1] = strchr (Base64Table,buftemp[i+1])-Base64Table; 
      cpos[2] = strchr (Base64Table,buftemp[i+2])-Base64Table; 
      cpos[3] = strchr (Base64Table,buftemp[i+3])-Base64Table; 
      binbyte[0] = ((cpos[0]<<2)|(cpos[1]>>4)); 
      binbyte[1] = ((cpos[1]<<4)|(cpos[2]>>2)); 
      binbyte[2] = (((cpos[2]&0x03)<<6)|(cpos[3]&0x3f)); 
      decbuf[i-(i/4)]= binbyte[0]; 
      decbuf[i-(i/4)+1]= binbyte[1]; 
      decbuf[i-(i/4)+2]= binbyte[2]; 
      i+=4;
   } 
   free (buftemp); 
   if (fmod(bufsize,4)==0)
     return bufsize*6/8; 
   if (fmod(bufsize,4)==1)
     return((bufsize+3)*6/8)-3; 
   if (fmod(bufsize,4)==2)
     return((bufsize+2)*6/8)-2; 
   if (fmod(bufsize,4)==3)
     return((bufsize+1)*6/8)-1; 
   return -1; 
}
