# Copyright (C) 2004,2005 by SICEm S.L. and Imendio AB
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gtk
import gobject

from gazpacho import gaction, util, widget

class CommandManager(object):
    """This class is the entry point accesing the commands.
    Every undoable action in Gazpacho is wrapped into a command.
    The stack of un/redoable actions is stored in the Project class
    so each project is independent on terms of undo/redo.
    
    The CommandManager knows how to perform the undo and redo actions.
    There is also a method for every type of command supported so
    you don't have to worry about creating the low level command
    objects.
    """
    def __init__(self, app):
        self.app = app
        
    def undo(self, project):
        """Undo the last command if there is such a command"""
        if project.undo_stack is None:
            return
        if project.prev_redo_item == -1:
            return
        
        cmd = project.undo_stack[project.prev_redo_item]
        cmd.undo()

        project.prev_redo_item -= 1
    
    def redo(self, project):
        """Redo the last undo command if there is such a command"""
        if not project.undo_stack: return
        # There is no more items to redo
        if project.prev_redo_item + 1 == len(project.undo_stack): return
    
        project.prev_redo_item += 1

        cmd = project.undo_stack[project.prev_redo_item]
        cmd.redo()

    #
    # for every possible command we have a method here
    #

    def delete(self, widget_instance):
        # internal children cannot be deleted. Should we notify the user?
        if widget_instance.internal_name is not None:
            return
        description = _("Delete %s") % widget_instance.name
        cmd = CommandCreateDelete(widget_instance, None,
                                  widget_instance.get_parent(), False,
                                  description, self.app)
        cmd.execute()
        cmd.push_undo(widget_instance.project)

    def create(self, klass, placeholder, project, parent=None,
               interactive=True):
        if placeholder:
            parent = util.get_parent(placeholder)
            if parent is None:
                return

        if project is None:
            project = parent.project

        widget_instance = widget.Widget(klass, project)
        widget_instance.create_gtk_widget(interactive)
        if widget_instance is None:
            return

        description = _("Create %s") % widget_instance.name

        cmd = CommandCreateDelete(widget_instance, placeholder, parent, True,
                                  description, self.app)
        cmd.execute()
        cmd.push_undo(widget_instance.project)

        if not self.app.palette.persistent_mode:
            self.app.palette.unselect_widget()

    def delete_placeholder(self, placeholder):
        parent = util.get_parent(placeholder)
        if len(parent.gtk_widget.get_children()) == 1:
            return
        
        description = _("Delete placeholder")
        cmd = CommandDeletePlaceholder(placeholder, parent, description,
                                       self.app)
        cmd.execute()
        cmd.push_undo(parent.project)
        
    def box_insert_placeholder(self, box, pos):
        description = _("Insert after")

        cmd = CommandInsertPlaceholder(box, pos, description, self.app)
        cmd.execute()
        cmd.push_undo(box.project)

    def set_property(self, property, value):
        gwidget = property.widget
        dsc = _('Setting %s of %s') % (property.klass.name, gwidget.name)
        cmd = CommandSetProperty(property, value, dsc, self.app)
        cmd.execute()
        gwidget.project.changed = True
        cmd.push_undo(gwidget.project)

    def set_translatable_property(self, property, value, comment,
                                  is_translatable, has_context):
        gwidget = property.widget
        dsc = _('Setting %s of %s') % (property.klass.name, gwidget.name)
        cmd = CommandSetTranslatableProperty(property, value,
                                             comment, is_translatable,
                                             has_context,
                                             dsc, self.app)
        cmd.execute()
        gwidget.project.changed = True
        cmd.push_undo(gwidget.project)

    def add_signal(self, widget_instance, signal):
        dsc = _('Add signal handler %s') % signal['handler']
        cmd = CommandAddRemoveSignal(True, signal, widget_instance, dsc,
                                     self.app)
        cmd.execute()
        cmd.push_undo(widget_instance.project)

    def remove_signal(self, widget_instance, signal):
        dsc = _('Remove signal handler %s') % signal['handler']
        cmd = CommandAddRemoveSignal(False, signal, widget_instance, dsc,
                                     self.app)
        cmd.execute()
        cmd.push_undo(widget_instance.project)

    def copy(self, widget_instance):
        """Add a copy of the widget to the clipboard.

        Note that it does not make sense to undo this operation
        """
        clipboard = self.app.get_clipboard()
        clipboard.add_widget(widget_instance)
        
    def cut(self, widget_instance):
        dsc = _('Cut widget %s into the clipboard') % widget_instance.name
        clipboard = self.app.get_clipboard()
        clipboard.add_widget(widget_instance)
        cmd = CommandCutPaste(widget_instance, widget_instance.project, None,
                              CommandCutPaste.CUT, dsc, self.app)
        cmd.execute()
        cmd.push_undo(widget_instance.project)
        
    def paste(self, placeholder, project):
        if project is None:
            raise ValueError("No project has been specified. Cannot paste "
                             "the widget")

        clipboard = self.app.get_clipboard()
        gwidget = clipboard.get_selected_widget(project)

        dsc = _('Paste widget %s from the clipboard') % gwidget.name
        cmd = CommandCutPaste(gwidget, project, placeholder,
                              CommandCutPaste.PASTE, dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)

    def add_action(self, values, parent, project):
        gact = gaction.GAction(parent, values['name'], values['label'],
                               values['tooltip'], values['stock_id'],
                               values['callback'], values['accelerator'])
        dsc = _('Add action %s') % gact.name
        cmd = CommandAddRemoveAction(parent, gact, True, dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)
    
    def remove_action(self, gact, project):
        dsc = _('Remove action %s') % gact.name
        cmd = CommandAddRemoveAction(gact.parent, gact, False, dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)
    
    def edit_action(self, gact, new_values, project):
        dsc = _('Edit action %s') % gact.name
        cmd = CommandEditAction(gact, new_values, dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)
        project.change_action_name(gact)

    def add_action_group(self, name, project):
        gaction_group = gaction.GActionGroup(name)
        dsc = _('Add action group %s') % gaction_group.name
        cmd = CommandAddRemoveActionGroup(gaction_group, project, True,
                                          dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)
    
    def remove_action_group(self, gaction_group, project):
        dsc = _('Remove action group %s') % gaction_group.name
        cmd = CommandAddRemoveActionGroup(gaction_group, project, False,
                                          dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)
        
    def edit_action_group(self, gaction_group, new_name, project):
        dsc = _('Edit action group %s') % gaction_group.name
        cmd = CommandEditActionGroup(gaction_group, new_name,
                                     dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)
        project.change_action_name(gaction_group)
    
class Command(object):
    """A command is the minimum unit of undo/redoable actions.

    It has a description so the user can see it on the Undo/Redo
    menu items.

    Every Command subclass should implement the 'execute' method in
    the following way:
        - After a command is executed, if the execute method is called
        again, their effects will be undone.
        - If we keep calling the execute method, the action will
        be undone, then redone, then undone, etc...
        - To acomplish this every command constructor will probably
        need to gather more data that it may seems necessary.

    After you execute a command in the usual way you should put that
    command in the command stack of that project and that's what
    the push_undo method does. Otherwise no undo will be available.
   
    Some commands unifies themselves. This means that if you execute
    several commands of the same type one after the other, they will be
    treated as only one big command in terms of undo/redo. In other words,
    they will collapse. For example, every time you change a letter on a
    widget's name it is a command but all these commands unifies to one
    command so if you undo that all the name will be restored to the old one.
    """
    def __init__(self, description=None, app=None):
        self._description = description
        self._app = app

    def get_description(self): return self._description
    def set_description(self, value): self._description = value
    description = property(get_description, set_description)

    def __repr__(self):
        return self._description

    def execute(self):
        """ This is the main method of the Command class.
        Note that it does not have any arguments so all the
        necessary data should be provided in the constructor.
        """
        pass

    def undo(self):
        """Convenience method that just call execute"""
        self.execute()
    
    def redo(self):
        """Convenience method that just call redo"""
        self.execute()
    
    def unifies(self, other):
        """True if self unifies with 'other'
        Unifying means that both commands can be treated as they
        would be only one command
        """
        return False
    
    def collapse(self, other):
        """Combine self and 'other' to form only one command.
        'other' should unifies with self but this method does not
        check that.
        """
        return False

    def push_undo(self, project):
        """Put the command in the project command stack
        It tries to collapse the command with the previous command
        """
        # If there are no "redo" items, and the last "undo" item unifies with
        # us, then we collapse the two items in one and we're done
        if project.prev_redo_item == len(project.undo_stack) - 1 and \
               project.prev_redo_item != -1:
            cmd = project.undo_stack[project.prev_redo_item]
            if cmd.unifies(self):
                cmd.collapse(self)
                return

        # We should now free al the 'redo' items
        project.undo_stack = project.undo_stack[:project.prev_redo_item + 1]

        # and then push the new undo item
        project.undo_stack.append(self)
        project.prev_redo_item += 1

        self._app.refresh_undo_and_redo()

class CommandCreateDelete(Command):
    def __init__(self, widget_instance=None, placeholder=None, parent=None,
                 create=True, description=None, application=None):
        Command.__init__(self, description, application)

        self._widget_instance = widget_instance
        self._placeholder = placeholder
        self._parent = parent
        self._create = create
        self._initial_creation = create
        self._ui_defs = {}
        
    def _create_execute(self):
        from gazpacho import placeholder

        widget_instance = self._widget_instance

        # if _delete_execute was called on a toolbar/menubar the uim removed
        # its ui definitions so we need to add them again
        if len(self._ui_defs.keys()) > 0:
            widget_instance.project.uim.add_widget(widget_instance,
                                                   self._ui_defs)
            self._ui_defs = {}
        
        if not widget_instance.gtk_widget.flags() & gtk.TOPLEVEL:
            if self._placeholder is None:
                for child in self._parent.gtk_widget.get_children():
                    if isinstance(child, placeholder.Placeholder):
                        self._placeholder = child
                        break
            widget.replace_widget(self._placeholder,
                                  widget_instance.gtk_widget,
                                  self._parent)

        self._widget_instance.project.add_widget(widget_instance.gtk_widget)
        self._widget_instance.project.selection_set(widget_instance.gtk_widget,
                                                    True)

        if isinstance(widget_instance.gtk_widget, gtk.Widget):
            widget_instance.gtk_widget.show_all()

        if widget_instance.is_toplevel():
            # we have to attach the accelerators groups so key shortcuts
            # keep working when this window has the focus. Only do
            # this the first time when creating a window, not when
            # redoing the creation since the accel group is already
            # set by then
            if self._initial_creation:
                for ag in self._app.get_accel_groups():
                    widget_instance.gtk_widget.add_accel_group(ag)
                self._initial_creation = False
                
            # make window management easier by making created windows
            # transient for the editor window
            app_window = self._app.get_window()
            widget_instance.gtk_widget.set_transient_for(app_window)

    def _delete_execute(self):
        from gazpacho import placeholder
        widget_instance = self._widget_instance
        
        if self._parent:
            if self._placeholder is None:
                self._placeholder = placeholder.Placeholder(self._app)

            widget.replace_widget(widget_instance.gtk_widget,
                                  self._placeholder, self._parent)

            widget_klass = widget_instance.klass
            self._parent.set_default_gtk_packing_properties(self._placeholder,
                                                            widget_klass)
        
        widget_instance.gtk_widget.hide()
        widget_instance.project.remove_widget(widget_instance.gtk_widget)

        # if we are removing a toolbar or a menubar we need to update the
        # uimanager
        if isinstance(widget_instance.gtk_widget, (gtk.Toolbar, gtk.MenuBar)):
            d = widget_instance.project.uim.get_ui(widget_instance)
            self._ui_defs = dict(d)
            widget_instance.project.uim.remove_widget(widget_instance)
        
        return True

    def execute(self):
        if self._create:
            retval = self._create_execute()
        else:
            retval = self._delete_execute()

        self._create = not self._create

        return retval

class CommandDeletePlaceholder(Command):
    def __init__(self, placeholder=None, parent=None, description=None, application=None):
        Command.__init__(self, description, application)

        self._placeholder = placeholder
        self._parent = parent
        self._create = False

        children = self._parent.gtk_widget.get_children()
        self._pos = children.index(placeholder)
        
    def _create_execute(self):
        from gazpacho import placeholder
        # create a placeholder and insert it at self._pos
        self._placeholder = placeholder.Placeholder(self._app)
        self._parent.gtk_widget.add(self._placeholder)
        self._parent.gtk_widget.reorder_child(self._placeholder, self._pos)

    def _delete_execute(self):
        self._placeholder.destroy()
        self._placeholder = None
        return True

    def execute(self):
        if self._create:
            retval = self._create_execute()
        else:
            retval = self._delete_execute()

        self._create = not self._create

        return retval

class CommandInsertPlaceholder(Command):
    def __init__(self, box, pos, description=None, application=None):
        Command.__init__(self, description, application)

        self._box = box
        self._pos = pos
        self._placeholder = None
        self._insert = True
        

    def _insert_execute(self):
        from gazpacho import placeholder
        # create a placeholder and insert it at self._pos
        self._placeholder = placeholder.Placeholder(self._app)
        self._box.gtk_widget.add(self._placeholder)
        self._box.gtk_widget.reorder_child(self._placeholder, self._pos)

    def _delete_execute(self):
        self._placeholder.destroy()
        self._placeholder = None
        return True

    def execute(self):
        if self._insert:
            retval = self._insert_execute()
        else:
            retval = self._delete_execute()

        self._insert = not self._insert

        return retval

class CommandSetProperty(Command):
    def __init__(self,  property=None, value=None, description=None,
                 application=None):
        Command.__init__(self, description, application)

        self._property = property
        self._value = value

    def execute(self):
        # we have to be careful with tbe name property
        if self._property.klass.id == 'name':
            # make sure the name is unique
            project = self._property.widget.project
            other = project.get_widget_by_name(self._value)
            if other:
                msg = _('There is another widget with the name "%s" in this '
                        'project' % self._value)
                self._app.show_message(msg)
                self._app.refresh_editor()
                return False

            # check the name is not empty
            if self._value == '':
                msg = _("The name can not be empty")
                self._app.show_message(msg)
                self._app.refresh_editor()
                return False
            
        new_value = self._value
        # store the current value for undo
        self._value = self._property.value
        self._property.set(new_value)
        # if the property is the name, we explicitily set the name of the
        # gwidget to trigger the notify signal so several parts of the
        # interface get updated
        if self._property.klass.id == 'name':
            self._property.widget.name = new_value

        return True

    def unifies(self, other):
        if isinstance(other, CommandSetProperty):
            return self._property == other._property
        return False

    def collapse(self, other):
        self._description = other._description
        other._description = None

        self._app.refresh_undo_and_redo()

class CommandSetTranslatableProperty(Command):
    def __init__(self,  property=None, value=None, comment=None,
                 is_translatable=False, has_context=False, description=None,
                 application=None):
        Command.__init__(self, description, application)

        self._property = property
        self._value = value
        self._comment = comment
        self._is_translatable = is_translatable
        self._has_context = has_context

    def execute(self):
        new_value = self._value
        new_comment = self._comment
        new_is_translatable = self._is_translatable
        new_has_context = self._has_context

        # store the current value for undo
        self._value = self._property.value
        self._comment = self._property.i18n_comment
        self._is_translatable = self._property.is_translatable
        self._has_context = self._property.has_i18n_context

        self._property.set(new_value)
        self._property.is_translatable = new_is_translatable
        self._property.i18n_comment = new_comment
        self._property.has_i18n_context = new_has_context

        return True

    def unifies(self, other):
        return False

    def collapse(self, other):
        return False

class CommandAddRemoveSignal(Command):
    def __init__(self, add=True, signal=None, widget_instance=None,
                 description=None, application=None):
        Command.__init__(self, description, application)
        self._add = add
        self._signal = signal
        self._widget_instance = widget_instance

    def execute(self):
        if self._add:
            self._widget_instance.add_signal_handler(self._signal)
        else:
            self._widget_instance.remove_signal_handler(self._signal)

        self._add = not self._add
        return True

class CommandCutPaste(Command):

    CUT, PASTE = range(2)
    
    def __init__(self, widget_instance, project, placeholder, operation,
                 description=None, application=None):
        Command.__init__(self, description, application)

        self._project = project
        self._placeholder = placeholder
        self._operation = operation
        self._widget_instance = widget_instance
        self._ui_defs = {}

    def execute(self):
        if self._operation == CommandCutPaste.CUT:
            self._execute_cut()
            self._operation = CommandCutPaste.PASTE
        else:
            self._execute_paste()
            self._operation = CommandCutPaste.CUT
        
    def _execute_cut(self):
        from gazpacho import placeholder
        widget_instance = self._widget_instance
        
        if not widget_instance.is_toplevel():
            parent = widget_instance.get_parent()

            if not self._placeholder:
                self._placeholder = placeholder.Placeholder(self._app)

            widget.replace_widget(widget_instance.gtk_widget,
                                  self._placeholder, parent)
            parent.set_default_gtk_packing_properties(self._placeholder,
                                                      widget_instance.klass)

        widget_instance.gtk_widget.hide()
        widget_instance.project.remove_widget(widget_instance.gtk_widget)

        # if we are removing a toolbar or a menubar we need to update the
        # uimanager
        if isinstance(widget_instance.gtk_widget, (gtk.Toolbar, gtk.MenuBar)):
            d = widget_instance.project.uim.get_ui(widget_instance)
            self._ui_defs = dict(d)
            widget_instance.project.uim.remove_widget(widget_instance)

    def _execute_paste(self):
        # if we did cut a toolbar/menubar the uim removed
        # its ui definitions so we need to add them again
        if len(self._ui_defs.keys()) > 0:
            self._widget_instance.project.uim.add_widget(self._widget_instance,
                                                         self._ui_defs)
            self._ui_defs = {}

        if self._widget_instance.is_toplevel():
            project = self._project
        else:
            parent = util.get_parent(self._placeholder)
            project = parent.project
            widget.replace_widget(self._placeholder,
                                  self._widget_instance.gtk_widget,
                                  parent)
            
        project.add_widget(self._widget_instance.gtk_widget)
        project.selection_set(self._widget_instance.gtk_widget, True)
        
        self._widget_instance.gtk_widget.show_all()

        # We need to store the project of a toplevel widget to use
        # when undoing the cut.
        self._project = project

class CommandAddRemoveAction(Command):
    def __init__(self, parent, gact, add=True,
                 description=None, application=None):
        Command.__init__(self, description, application)

        self.add = add
        self.gact = gact
        self.parent = parent
        
    def execute(self):
        if self.add:
            self._add_execute()
        else:
            self._remove_execute()

        self.add = not self.add
    
    def _add_execute(self):
        self.parent.append(self.gact)
    
    def _remove_execute(self):
        self.parent.remove(self.gact)

class CommandEditAction(Command):
    def __init__(self, gact, new_values, description=None,
                 application=None):
        Command.__init__(self, description, application)
        
        self.new_values = new_values
        self.gact = gact

    def execute(self):
        old_values = {
            'name' : self.gact.name,
            'label': self.gact.label,
            'stock_id': self.gact.stock_id,
            'tooltip': self.gact.tooltip,
            'accelerator': self.gact.accelerator,
            'callback': self.gact.callback,
            }
        self.gact.name = self.new_values['name']
        self.gact.label = self.new_values['label']
        self.gact.stock_id = self.new_values['stock_id']
        self.gact.tooltip = self.new_values['tooltip']
        self.gact.accelerator = self.new_values['accelerator']
        self.gact.callback = self.new_values['callback']
        self.gact.parent.update_action(self.gact, old_values['name'])
        self.new_values = old_values

class CommandAddRemoveActionGroup(Command):
    def __init__(self, gaction_group, project, add=True,
                 description=None, application=None):
        Command.__init__(self, description, application)

        self.add = add
        self.project = project
        self.gaction_group = gaction_group
        
    def execute(self):
        if self.add:
            self._add_execute()
        else:
            self._remove_execute()

        self.add = not self.add
    
    def _add_execute(self):
        self.project.add_action_group(self.gaction_group)
    
    def _remove_execute(self):
        self.project.remove_action_group(self.gaction_group)

class CommandEditActionGroup(Command):
    def __init__(self, gaction_group, new_name, description=None,
                 application=None):
        Command.__init__(self, description, application)
        
        self.new_name = new_name
        self.gaction_group = gaction_group

    def execute(self):
        old_name = self.gaction_group.name
        self.gaction_group.name = self.new_name
        self.new_name = old_name
        
class CommandStackView(gtk.ScrolledWindow):
    """This class is just a little TreeView that knows how
    to show the command stack of a project.
    It shows a plain list of all the commands performed by
    the user and also it mark the current command that
    would be redone if the user wanted so.
    Older commands are under newer commands on the list.
    """
    def __init__(self):
        gtk.ScrolledWindow.__init__(self)
        self._project = None

        self._model = gtk.ListStore(bool, str)
        self._treeview = gtk.TreeView(self._model)
        self._treeview.set_headers_visible(False)
        
        column = gtk.TreeViewColumn()
        renderer1 = gtk.CellRendererPixbuf()
        column.pack_start(renderer1, expand=False)
        column.set_cell_data_func(renderer1, self._draw_redo_position)

        renderer2 = gtk.CellRendererText()
        column.pack_start(renderer2, expand=True)
        column.add_attribute(renderer2, 'text', 1)
        
        self._treeview.append_column(column)

        self.add(self._treeview)

    def set_project(self, project):
        self._project = project
        self.update()
        
    def update(self):
        self._model.clear()
        if self._project is None:
            return
        
        i = 0
        for cmd in self._project.undo_stack:
            if i == self._project.prev_redo_item:
                redo = True
            else:
                redo = False
            self._model.insert(0, (redo, cmd.description))
            i += 1

    def _draw_redo_position(self, column, cell, model, iter):
        is_the_one = model.get_value(iter, 0)

        if is_the_one:
            stock_id = gtk.STOCK_JUMP_TO
        else:
            stock_id = None

        cell.set_property('stock-id', stock_id)
        
gobject.type_register(CommandStackView)
