/* Copyright (C) 2004 Damir Zucic */

/*=============================================================================

			    hyphob_function1.c

Purpose:
	Draw the hydrophobicity function F1.  The sequence stored to the
	main sequence buffer  is used  to calculate  the function value.

Input:
	(1) Pointer to the storage where the minimal function value will
	    be stored.
	(2) Pointer to the storage where the maximal function value will
	    be stored.
	(3) Pointer to RuntimeS structure.

Output:
	(1) Function F1 calculated and stored.
	(2) Return value.

Return value:
	(1) Positive on success.
	(2) Negative on failure.

Notes:
	(1) The function  F1 may be modified and used for many purposes.
	    Originally, it was introduced while searching for the method
	    which will be suitable for prediction of the porin secondary
	    structure.

========includes:============================================================*/

#include <stdio.h>

#include <math.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======calculate the hydrophobicity function F1:============================*/

int HyphobFunction1_ (double *min_functionP, double *max_functionP,
		      RuntimeS *runtimeSP)
{
int             residuesN, residueI;   /* Do not use size_t instead of int ! */
int		windowI, combinedI;
int		used_residuesN;
double		hydrophobicity;
double		function_value;
static double   weightA[19] =  {1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
				0.0, 0.0, 0.0, 0.0, 0.0,
				1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0};

/*------prepare some parameters:---------------------------------------------*/

/* The number of residues in sequence buffer: */
residuesN = (int) runtimeSP->residuesN;
if (residuesN == 0) return -1;

/*------calculate the function F1:-------------------------------------------*/

/* Initialize the extreme values: */
*min_functionP = +999999.0;
*max_functionP = -999999.0;

/* 20041022.1415: */
/* A window of  19 residues is scanned.  The weighted hydrophobicity */
/* is calculated and used as the final function value. The weighting */
/* sheme skips  5 central residues and uses  14 out of  19 residues. */

/* Initialize (reset) the function value: */
for (residueI = 0; residueI < residuesN; residueI++)
	{
	/* Reset the function value, it might be initialized before: */
	*(runtimeSP->function1P + residueI) = 0.0;
	}

/* Scan the sequence, ignoring 9 residues at each end: */
for (residueI = 9; residueI < residuesN - 9; residueI++)
	{
	/* Reset the number of residues used to prepare the raw values: */
	used_residuesN = 0;

        /* Reset the function value: */
	function_value = 0.0;

	/* Scan the sliding window: */
	for (windowI = 0; windowI < 19; windowI++)
		{
		/* Prepare and check the combined index: */
		combinedI = residueI + windowI - 9;
		if (combinedI < 0) continue;
		if (combinedI >= residuesN) continue;

		/* Prepare the hydrophobicity value: */
		hydrophobicity = *(runtimeSP->hydrophobicityP + combinedI);

		/* Update the function value: */
		function_value += hydrophobicity * weightA[windowI];

		/* Update the number of used residues: */
		used_residuesN++;
		}

	/* Check the number of used residues: */
	if (used_residuesN == 0) continue;

	/* Scale the function value (it should fit */
	/* to the same plot with other functions): */
	function_value /= 14.0;

	/* Store the function value: */
	*(runtimeSP->function1P + residueI) = function_value;

	/* Dummy values should not be used to determine extreme values: */
	if (used_residuesN < 19) continue;

	/* Find the extreme values: */
	if (function_value < *min_functionP) *min_functionP = function_value;
	if (function_value > *max_functionP) *max_functionP = function_value;

	/* End of residueI loop: */
	}

/*---------------------------------------------------------------------------*/

return 1;
}

/*===========================================================================*/


