/* -*- C -*-                                        vim:set ts=4 sw=4:

 * FILE: "/home/life/projects/garchiver-0.5/src/dir_listing.c"
 * LAST MODIFICATION: "Wed, 04 Jul 2001 12:34:11 +0200 (life)"

 * (C) 2001 by Danie Roux <droux@tuks.co.za>
 *
 * $Id: dir_listing.c,v 1.19 2001/07/04 12:56:53 uid43216 Exp $
 *
 * Implementation of the directory listing widget
 *
 */

#include "dir_listing.h"
#include <string.h>
#include <stdio.h>
#include <sys/stat.h>

#include <dlfcn.h>

#ifdef __cplusplus
extern "C" {
#endif /* C++ */

static void gtk_dir_listing_class_init (GtkDirListingClass *dl);
static void gtk_dir_listing_init (GtkDirListing *dl);

void new_folder (gpointer, GtkDirListing*);
void delete_folder (gpointer, GtkDirListing*);

GtkCTreeNode* find_node (GtkCTree *ctree, GtkCTreeNode *parent,
                         const gchar *to_find);
void ctree_node_path (GtkCTree *ctree, GtkCTreeNode *node, char **path);
void parse_dir (GtkCTree *ctree, GtkCTreeNode *parent, const char* dir,
                gboolean show_dotfiles);
void ctree_expand (GtkCTree *ctree, GList *node, GtkDirListing*);
void ctree_adjust (GtkCTree *ctree, GList *node, GtkDirListing*);
void ctree_collapse (GtkCTree *ctree, GList *node, gpointer na);
void on_ok_clicked (GtkObject *obj, GtkWidget *win);
void on_cancel_clicked (GtkObject *obj, GtkWidget *win);
void ctree_row_selected (GtkCTree *ctree, GList *node, gint col,
                         GtkDirListing *dl);
void popup_handler (GtkWidget *widget, GdkEvent *event, GtkDirListing*);

enum {DIR_SELECTED, DIR_FINAL_SELECTION, LAST_SIGNAL};

static guint dir_listing_signals[LAST_SIGNAL] = { 0 };

/* Pre condition: A widget called ctree must exist in the context this
 * gets used, as well as a variable new_node that will get the new_node
 */
#define INSERT_NODE(_parent_, _text_, _data_) \
    new_node = gtk_ctree_insert_node (ctree, _parent_, NULL, _text_, \
                                      10, open_dir, open_dir_mask, \
                                      closed_dir, closed_dir_mask, \
                                      FALSE, _parent_ == NULL ? TRUE : \
                                      FALSE);\
    gtk_ctree_node_set_row_data (ctree, new_node, _data_);

GdkPixmap *open_dir;
GdkPixmap *closed_dir;
GdkBitmap *open_dir_mask;
GdkBitmap *closed_dir_mask;

/* This was drawn by Brian Masney <masneyb@seul.org */
static char *open_dir_xpm[] = {
    "18 14 7 1",
    "   c   None",
    "#  c   #000000",
    "-  c   #ffffff",
    "*  c   gray50",
    "i  c   darkgray",
    "h  c   gray85",
    "j  c   yellow",
    "     ****         ",
    "    *hjhj*        ",
    "   *hjhjhj******* ",
    "  *jijijijijijij*#* ",
    "#############iji*#",
    "#-----------#jij*#",
    " #-hjhjhjhjhj#ji*#",
    " #-jhjhjhjhjh#ij*#",
    " #-hjhjhjhjhj#ji*#",
    "  #-hjhjhjhjhi#j*#",
    "  #-jhjhjhjhjh#i*#",
    "  #-hjhjhjhjhj#j*#",
    "  ***************#",
    "   ###############"
};

/* This was drawn by Brian Masney <masneyb@seul.org */
static char *dir_xpm[] = {
    "16 14 6 1",
    "   c   None",
    "#  c   #000000",
    "-  c   #ffffff",
    "*  c   gray50",
    "h  c   gray85",
    "j  c   yellow",
    "   ****         ",
    "  *hjhj*        ",
    " *hjhjhj******* ",
    "*-------------* ",
    "*-jhjhjhjhjhjh*#",
    "*-hjhjhjhjhjhj*#",
    "*-jhjhjhjhjhjh*#",
    "*-hjhjhjhjhjhj*#",
    "*-jhjhjhjhjhjh*#",
    "*-hjhjhjhjhjhj*#",
    "*-jhjhjhjhjhjh*#",
    "*-hjhjhjhjhjhj*#",
    "***************#",
    " ###############"
};

static char * del_xpm[] = {
"12 11 82 1",
"   c None",
".  c #2D322A",
"+  c #3E413C",
"@  c #31332F",
"#  c #262A23",
"$  c #262923",
"%  c #2A2D26",
"&  c #242622",
"*  c #7A8375",
"=  c #A3B199",
"-  c #A9B3A1",
";  c #60665B",
">  c #4C5347",
",  c #6F7867",
"'  c #93A188",
")  c #5F6758",
"!  c #1B1D18",
"~  c #292E27",
"{  c #B2BFAC",
"]  c #C0CAB8",
"^  c #ACB9A2",
"/  c #9CAA91",
"(  c #97A48C",
"_  c #98A68C",
":  c #8D9C82",
"<  c #1F251B",
"[  c #809477",
"}  c #A9B9A2",
"|  c #A9B9A4",
"1  c #92A589",
"2  c #889C7E",
"3  c #7F9076",
"4  c #74846D",
"5  c #5A6A56",
"6  c #161A15",
"7  c #272C25",
"8  c #60715C",
"9  c #60705C",
"0  c #596855",
"a  c #4F5A4A",
"b  c #485245",
"c  c #3E453C",
"d  c #232622",
"e  c #3B4738",
"f  c #687165",
"g  c #505A4E",
"h  c #454D44",
"i  c #465044",
"j  c #424B40",
"k  c #424A3F",
"l  c #3B4638",
"m  c #A1B09C",
"n  c #7C8E76",
"o  c #667563",
"p  c #687C63",
"q  c #60725B",
"r  c #556351",
"s  c #040404",
"t  c #A0B09C",
"u  c #7B8E76",
"v  c #667562",
"w  c #556350",
"x  c #050505",
"y  c #3C4839",
"z  c #9FAF9B",
"A  c #677663",
"B  c #3B4538",
"C  c #2D362A",
"D  c #83987D",
"E  c #7A8F74",
"F  c #72866D",
"G  c #6B8165",
"H  c #65785F",
"I  c #5C6D57",
"J  c #262C23",
"K  c #060606",
"L  c #21281F",
"M  c #252D23",
"N  c #242C22",
"O  c #232A21",
"P  c #222920",
"Q  c #1D221B",
"    .+@#$%  ",
"  &*=-;>,')!",
"  ~{]^/(__:#",
"  <[}|123456",
"   7890abcd ",
"   efghijkl ",
"   emnopqrls",
"   etuvpqwlx",
"   yzuApqwBx",
"   CDEFGHIJK",
"    LMNOPQs "};

GtkWidget*
gtk_dir_listing_new (void)
{
    GtkWidget *widget;

    widget = gtk_type_new (gtk_dir_listing_get_type ());

    g_assert (widget != NULL);

    gtk_dir_listing_show_dotfiles (GTK_DIR_LISTING (widget), FALSE);

    return widget;
}

GtkWidget*
gtk_dir_listing_new_with_dotfiles (gboolean show_dotfiles)
{
    GtkWidget *widget;

    widget = gtk_type_new (gtk_dir_listing_get_type ());

    g_assert (widget != NULL);

    gtk_dir_listing_show_dotfiles (GTK_DIR_LISTING (widget), show_dotfiles);

    return widget;
}

/* Must release the path when you're finished with it */
/* Returns a the path of the node as a string */
void
ctree_node_path (GtkCTree *ctree, GtkCTreeNode *node, char **path)
{
    GString *str = g_string_new ("");
    /* A cursor to iterate up the tree */
    GtkCTreeNode *cur = node;

    g_assert (node != NULL);
    g_assert (ctree != NULL);

    *path = (gchar*) gtk_ctree_node_get_row_data (ctree, node);
    if (*path == NULL) return; /* No data yet, the root node */

    g_string_assign (str, *path);

    while ((cur = ((GTK_CTREE_ROW (cur))->parent)))
      {
        g_string_prepend_c (str, '/');
        if ((GTK_CTREE_ROW (cur))->parent != NULL)
          {
            *path = (gchar*) gtk_ctree_node_get_row_data (ctree, cur);
            g_string_prepend (str, *path);
          }
      }
    *path = str->str;
    /* Leaves the memory intact */
    g_string_free (str, FALSE);
}

static int
dirs_only (const struct dirent *de)
{
    struct stat st;

    static int euid = 0;
    static int eguid = 0;

    if (euid == 0) euid = geteuid ();
    if (eguid == 0) eguid = getegid ();

    if (!strcmp(de->d_name, ".") || !(strcmp(de->d_name, ".."))) return 0;

    /* Probably a dangling link */
    if (stat (de->d_name, &st) == -1)
      {
        /* Realize the truth, there was no error */
        /* This is needed so that scandir doesn't think an error
         * occured */
        errno = 0;
        return 0;
      }

    if (S_ISDIR (st.st_mode))
      {
        if (st.st_mode & S_IROTH) return 1;
        if ((euid == st.st_uid) && (st.st_mode & S_IRUSR)) return 1;
        if ((eguid == st.st_gid) && (st.st_mode & S_IRGRP)) return 1;
      }

    return 0;
}

void
parse_dir (GtkCTree *ctree, GtkCTreeNode *parent, const char* dir,
           gboolean show_dotfiles)
{
    struct dirent **de;
    char *cols[2];
    GtkCTreeNode *sibling = NULL;
    GtkCTreeNode *new_sibling = NULL;

    char *cwd;
    int i;

    cwd = getcwd (NULL, 0);

    cols [0] = NULL;

    /* Remove dummy node that was created with the skim */
    if (((GTK_CTREE_ROW (parent))->children != NULL))
        gtk_ctree_remove_node (ctree, (GTK_CTREE_ROW
                                       (parent))->children);

    if (chdir (dir) == 0)
      {
        i = scandir (".", &de, dirs_only, alphasort);

        if (i >= 0)
          {
            int j;
            gchar *data;

            GtkCTreeNode *new_node;

            for (j = 0; j < i; j++)
              {
                /* If this is a dotfile, skip to next iteration */
                if ((!show_dotfiles) && (de[j]->d_name [0] == '.')) continue;

                cols [1] = de[j]->d_name;

                if (chdir (de[j]->d_name) == 0)
                  {
                    chdir (cwd);

                    /* If this is the root check if we are dealing with
                     * /proc or /dev before adding node */
                    if ((GTK_CTREE_ROW (parent))->parent == NULL)
                      {
                        if (!(strcmp (de[j]->d_name, "proc") == 0)
                            && !(strcmp (de[j]->d_name, "dev") == 0))
                          {
                            /* Keeps a copy for data */
                            data = g_strdup (de[j]->d_name);

                            /* Sets new_node */
                            INSERT_NODE (parent, cols, data);
                            new_sibling = new_node;
                          }
                        else new_sibling = NULL;
                      } else
                        {
                          /* Keeps a copy for data */
                          data = g_strdup (de[j]->d_name);
                          INSERT_NODE (parent, cols, data);
                        }

                      /* This way of entering new nodes keeps the right order
                       */
                      if (sibling && new_sibling)
                        {
                          GTK_CTREE_ROW (sibling)->sibling = new_sibling;
                          sibling = new_sibling;
                        }

                  }
              } /* for */
          }
        else
          {
            gchar* err = g_strerror (errno);
            g_warning ("Couldn't scan directory %s! Error: %s\n",
                     dir, err);
            free (err);
          }
        if (!de) free (de);
        chdir (cwd);
      }
    else
      {
        g_warning ("Couldn't change to directory %s!\n", dir);
      }
    free (cwd);
}

/* When this function finds a directory, it returns immediately, making
 * it fast to get those +'s working
 */
static gboolean
has_subdir (const gchar* path)
{
    DIR *dir;
    struct dirent *dirent;
    struct stat statbuf;
    gchar *npath;

    if ((dir = opendir (path)) != NULL)
      {
        while ((dirent = readdir (dir)) != NULL)
          {
            if (!(dirent->d_name[0] == '.' &&
                  (dirent->d_name [1] == '\0' || dirent->d_name [1] == '.')))
              {
                npath = g_strjoin ("/", path, dirent->d_name, NULL);
                if ((stat (npath, &statbuf) != -1) && (S_ISDIR (statbuf.st_mode)))
                  {
                    g_free (npath);
                    closedir (dir);
                    return TRUE;
                  }
                g_free (npath);
              }
          }
        closedir (dir);
      }
    return FALSE;
}

void
ctree_expand (GtkCTree *ctree, GList *node, GtkDirListing *dl)
{
    GtkCTreeNode *parent;
    GtkCTreeNode *child = NULL;

    GtkCTreeNode *new_node;

    char *tmp;
    char *cwd = getcwd (NULL, 0);
    char *fullpath;
    char *data;

    g_assert (ctree != NULL);
    g_assert (dl != NULL);

    gtk_clist_freeze (GTK_CLIST (ctree));

    if (node == NULL)
      {
        char *cols[2];
        cols [0] = NULL;
        cols [1] = "/";
        data = g_strdup ("/");
        /* Sets new_node */
        INSERT_NODE (NULL, cols, data);
        parent = new_node;
      }
    else parent = GTK_CTREE_NODE (node);

    ctree_node_path (ctree, parent, &tmp);

    /* Check if it is possible to change dir */
    if (chdir (tmp) == 0)
      {
        char *dummy[2];
        char *data;
        dummy [0] = NULL;
        dummy [1] = "dummy";

        /* Flesh out newly expanded branch */

        parse_dir (ctree, parent, tmp, dl->show_dotfiles);

        /* Skim each child of the node selected */
        for (child = (GTK_CTREE_ROW (parent))->children;
             child != NULL;
             child = GTK_CTREE_NODE_NEXT (child))
          {

            if (strcmp (tmp, "/") == 0)
              {
                fullpath = g_strconcat ("/", gtk_ctree_node_get_row_data
                                        (ctree, child), NULL);
              }
            else
              {
                fullpath = g_strjoin ("/", tmp, (gchar*)
                                      gtk_ctree_node_get_row_data
                                      (ctree, child), NULL);
              }

            if (strcmp (fullpath, "/dev") != 0
                && strcmp (fullpath, "/proc") != 0
                && has_subdir (fullpath))
              {
                data = g_strdup ("dummy");
                INSERT_NODE (child, dummy, data);
              }
            g_free (fullpath);
          }

        g_free (tmp);
      }

    chdir (cwd);
    free (cwd);
    gtk_clist_thaw (GTK_CLIST (ctree));
}

void
ctree_adjust (GtkCTree *ctree, GList *node, GtkDirListing* dl)
{
    GtkCTreeNode *first;
    GtkCTreeNode *last;

    if (node != NULL)
      {
        first = (GTK_CTREE_ROW (node))->children;
        if (first == NULL) return;

        last = first;

        while ((GTK_CTREE_ROW (last))->sibling != NULL)
            last = (GTK_CTREE_ROW (last))->sibling;

        while ((!(gtk_ctree_node_is_visible (ctree, last) ==
                 GTK_VISIBILITY_FULL))
               && (gtk_ctree_node_is_visible (ctree, first)))
          {
            dl->adjustment->value += 1;
            gtk_adjustment_value_changed (dl->adjustment);
          }

      }
}

void
insure_node_visible (GtkCTree *ctree, GList *node, gint na,
                           GtkDirListing* dl)
{
    int row;
    int increment;
    int i, visible_node_row = -1, current_node_row = -1;
    GtkCTreeNode *test_node;

    gboolean was_visible = FALSE;

    g_assert (ctree != NULL);
    g_assert (dl != NULL);
    g_assert (node != NULL);

    if (!(GTK_WIDGET_VISIBLE (GTK_WIDGET (dl)))) return;

    for (i = 0;; ++i)
      {
        test_node = gtk_ctree_node_nth (dl->ctree, i);

        if (test_node == NULL) break;

        if (GTK_CTREE_NODE (node) == test_node) current_node_row = i;
        if (gtk_ctree_node_is_visible (ctree, GTK_CTREE_NODE (test_node))
            == GTK_VISIBILITY_FULL)
          {
            visible_node_row = i;
          }
      }

    if (visible_node_row < 0)
      {
        //g_warning ("Can't find a visible row?");
        return;
      }
    if (current_node_row < 0)
      {
        //g_warning ("Can't find the current row? Possibly the root node got re-opened");
        return;
      }

    if (visible_node_row < current_node_row) increment = 1;
    else increment = -1;

    if (gtk_ctree_node_is_visible (ctree, GTK_CTREE_NODE
                                   (node)) ==
        GTK_VISIBILITY_PARTIAL)
      {
        was_visible = TRUE;
      }

    while (gtk_ctree_node_is_visible (ctree, GTK_CTREE_NODE
                                      (node)) !=
           GTK_VISIBILITY_FULL)
      {
        dl->adjustment->value += increment;
        gtk_adjustment_value_changed (dl->adjustment);

        if (was_visible &&
            gtk_ctree_node_is_visible (ctree, GTK_CTREE_NODE
                                       (node)) ==
            GTK_VISIBILITY_NONE)
          {
            increment = increment * -1;
          }
      }
}

void
gtk_dir_listing_show_dotfiles (GtkDirListing *dl, gboolean show)
{
    gchar *base_folder = NULL;

    dl->show_dotfiles = show;

    if (dl->node_selected == NULL)
        gtk_dir_listing_cd (GTK_DIR_LISTING (dl), "/");
    else
      {
        ctree_node_path (dl->ctree, dl->node_selected, &base_folder);
        g_assert (base_folder != NULL);

        gtk_ctree_collapse (dl->ctree, gtk_ctree_node_nth (dl->ctree, 0));
        gtk_ctree_expand (dl->ctree, gtk_ctree_node_nth (dl->ctree, 0));

        gtk_dir_listing_cd (GTK_DIR_LISTING (dl), base_folder);

        g_free (base_folder);
      }
}

void
gtk_dir_listing_cd (GtkDirListing *dl, const gchar *path)
{
    GtkCTreeNode *cursor, *found;
    gchar** tokens;
    gchar* token;
    int i;

    /* Get the root node / */
    cursor = gtk_ctree_node_nth (dl->ctree, 0);

    /* Make sure the root node is expanded */

    tokens = g_strsplit (path, "/", 0);

    /* Start at one, to skip the empty string the first / creates for us */
    for (i = 1; *(tokens+i) != NULL; ++i)
      {
        token = *(tokens+i);
        found = gtk_ctree_find_by_row_data_custom (dl->ctree, cursor, token, (GCompareFunc) &strcmp);

        if (found != NULL) cursor = found;
        else
            /* Can't continue, probably an old path read from the
               config files that doesn't exist anymore */
            break;

        if (!(GTK_CTREE_ROW (cursor))->expanded)
          {
            gtk_ctree_expand (dl->ctree, cursor);
          }
      }

    gtk_ctree_select (dl->ctree, cursor);

    g_strfreev (tokens);
}

void
ctree_collapse (GtkCTree *ctree, GList *node, gpointer na)
{
    GtkCTreeNode *children = NULL;

    gtk_clist_freeze (GTK_CLIST (ctree));

    for (;;)
      {
        children = (GTK_CTREE_ROW (node))->children;

        if (children == NULL || ((GTK_CTREE_ROW (children))->sibling ==
                                 NULL))
            break;

        gtk_ctree_remove_node (ctree, children);
      }

    gtk_clist_thaw (GTK_CLIST (ctree));
}

guint
gtk_dir_listing_get_type (void)
{
    static guint dl_type = 0;

    if (!dl_type)
      {
        static const GtkTypeInfo dl_info =
          {
            "GtkDirListing",
            sizeof (GtkDirListing),
            sizeof (GtkDirListingClass),
            (GtkClassInitFunc) gtk_dir_listing_class_init,
            (GtkObjectInitFunc) gtk_dir_listing_init,
            (GtkArgSetFunc) NULL,
            (GtkArgGetFunc) NULL
          };

        dl_type = gtk_type_unique (GTK_TYPE_VBOX, &dl_info);
      }
    return dl_type;
}

static void
gtk_dir_listing_class_init (GtkDirListingClass *dc)
{
    GtkObjectClass *object_class = (GtkObjectClass*) dc;

    dir_listing_signals[DIR_SELECTED] =
        gtk_signal_new ("dir_selected", GTK_RUN_FIRST,
                        object_class->type, GTK_SIGNAL_OFFSET
                        (GtkDirListingClass, dir_selected),
                        gtk_marshal_NONE__STRING, GTK_TYPE_NONE,
                        1, GTK_TYPE_STRING);

    dir_listing_signals[DIR_FINAL_SELECTION] =
        gtk_signal_new ("dir_final_selection", GTK_RUN_FIRST,
                        object_class->type, GTK_SIGNAL_OFFSET
                        (GtkDirListingClass, dir_final_selection),
                        gtk_marshal_NONE__STRING, GTK_TYPE_NONE,
                        1, GTK_TYPE_STRING);

    gtk_object_class_add_signals (object_class,
                                  dir_listing_signals,
                                  LAST_SIGNAL);

    ((GtkDirListingClass*)(object_class))->dir_selected = NULL;
    ((GtkDirListingClass*)(object_class))->dir_final_selection = NULL;
}

void
ctree_row_selected (GtkCTree *ctree, GList *node, gint col,
                    GtkDirListing *dl)
{
    char *path;

    g_assert (IS_DIR_LISTING (dl));
    g_assert (ctree != NULL);
    g_assert (node != NULL);

    ctree_node_path (ctree, GTK_CTREE_NODE (node), &path);
    if (path == NULL) return;

    gtk_signal_emit (GTK_OBJECT (dl),
                     dir_listing_signals[DIR_SELECTED], path);

    dl->node_selected = GTK_CTREE_NODE (node);

    free (path);
}

void
real_new_folder (gchar *new_folder, GtkDirListing *dl)
{
    gchar *base_folder = NULL;
    GtkCTree *ctree = NULL;
    char *cols[2];

    g_assert (dl != NULL);

    g_assert (IS_DIR_LISTING (dl));

    ctree = dl->ctree;

    cols [0] = NULL;

    if (new_folder != NULL) /* If it's NULL s/he doesn't want a folder */
      {
        ctree_node_path (dl->ctree, dl->node_selected, &base_folder);
        g_assert (base_folder != NULL);

        if (strchr (new_folder, '/'))
          {
            gnome_error_dialog
                (_("You are not allowed to put / in the folder name!"));
          }
        else
          {
            gchar* tmp = g_strjoin ("/", base_folder, new_folder, NULL);
            if (mkdir (tmp, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH |
                        S_IXOTH) != 0)
              {
                gchar *error = g_strerror (errno);
                gchar *msg = g_strjoin (NULL,
                                        _("Folder could not be created: "),
                                        error, ".", NULL);

                gnome_error_dialog (msg);

                g_free (msg);
              }
            else
              {
                GtkCTreeNode *last;
                GtkCTreeNode *new_node;
                gchar *data;

                cols [1] = new_folder;

                /* Copy for data */
                data = g_strdup (new_folder);

                if ((GTK_CTREE_ROW (dl->node_selected))->expanded)
                  {
                    INSERT_NODE (dl->node_selected, cols, data);
                  }
                else gtk_ctree_expand (dl->ctree,
                                       dl->node_selected);

                last = (GTK_CTREE_ROW (dl->node_selected))->children;
                g_assert (last != NULL);

                while ((GTK_CTREE_ROW (last))->sibling != NULL)
                    last = (GTK_CTREE_ROW (last))->sibling;

                gtk_ctree_select (dl->ctree, last);

              }
            g_free (tmp);
          }
        g_free (new_folder);
        g_free (base_folder);
      }
}

void
show_hide_folders (gpointer na, GtkDirListing *dl)
{
    g_assert (IS_DIR_LISTING (dl));

    gtk_dir_listing_show_dotfiles (dl, !dl->show_dotfiles);
}

void
new_folder (gpointer na, GtkDirListing *dl)
{
    GtkWidget *dialog;

    g_assert (IS_DIR_LISTING (dl));

    dialog = gnome_request_dialog (0, _("Name of new directory:"), NULL, 50,
                                   (GnomeStringCallback) real_new_folder,
                                   dl, NULL);

    gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
}

void
delete_folder (gpointer na, GtkDirListing *dl)
{
    GtkWidget *dialog;
    gchar *folder;

    ctree_node_path (dl->ctree, dl->node_selected, &folder);

    g_assert (folder != NULL);
    g_assert (IS_DIR_LISTING (dl));

    if (rmdir (folder) != 0) /* Error occured */
      {
        gchar *error = g_strerror (errno);
        gchar *msg = g_strjoin (NULL, _("Folder could not removed: "),
                                error, ".", NULL);

        dialog = gnome_error_dialog (msg);
        gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);

        g_free (msg);
      }
    else
      {
        gtk_ctree_remove_node (dl->ctree, dl->node_selected);
      }

    g_free (folder);
}

void
popup_handler (GtkWidget *widget, GdkEvent *event, GtkDirListing *dl)
{
    GdkEventButton *event_button;
    int row, col;
    GtkCTree *ctree;
    GtkCTreeNode *node;

    g_assert (IS_DIR_LISTING (dl));

    if (event->type == GDK_BUTTON_PRESS)
      {
        event_button = (GdkEventButton*) event;

        if (event_button->button == 3)
          {

            ctree = dl->ctree;
            g_assert (ctree != NULL);

            if (!gtk_clist_get_selection_info (GTK_CLIST (ctree),
                                               event_button->x,
                                               event_button->y,
                                               &row,
                                               &col))
              {
                dl->node_selected = NULL;
                return;
              }

            node = gtk_ctree_node_nth (GTK_CTREE (ctree), row);

            if (!node)
              {
                dl->node_selected = NULL;
                return;
              }

            /* This is VERY important. It sets dl->node_selected */
            gtk_ctree_select (dl->ctree, node);

            gnome_popup_menu_do_popup (dl->popup, NULL, NULL, event_button,
                                       dl);
          }
      }

}

static void
gtk_dir_listing_init (GtkDirListing *dl)
{
    GnomeUIInfo popup_menu_uiinfo[] = {
        GNOMEUIINFO_MENU_NEW_ITEM (N_("_New Folder"), NULL, new_folder,
                                   NULL),
        GNOMEUIINFO_ITEM (N_("_Delete Folder"), NULL, delete_folder,
                          del_xpm),
        GNOMEUIINFO_ITEM (N_("_Show/hide hidden folders"), NULL,
                             show_hide_folders, NULL),
        GNOMEUIINFO_END
    };

    GtkWidget *scrolled;

    char *cols[2];
    char *data;
    char *debug;
    GtkCTreeNode *new_node;
    GtkCTree *ctree;

    dl->show_dotfiles = FALSE;

    closed_dir = gdk_pixmap_colormap_create_from_xpm_d (NULL,
                                                        gdk_colormap_get_system(),
                                                        &closed_dir_mask,
                                                        NULL,
                                                        open_dir_xpm);

    open_dir = gdk_pixmap_colormap_create_from_xpm_d (NULL,
                                                      gdk_colormap_get_system(),
                                                      &closed_dir_mask,
                                                      NULL, dir_xpm);

    dl->ctree = GTK_CTREE (gtk_ctree_new (2, 1));

    dl->popup = gnome_popup_menu_new (popup_menu_uiinfo);

    gtk_ctree_set_line_style (GTK_CTREE (dl->ctree),
                              GTK_CTREE_LINES_DOTTED);

    gtk_clist_set_selection_mode (GTK_CLIST (dl->ctree),
                                  GTK_SELECTION_BROWSE);

    scrolled = gtk_scrolled_window_new (gtk_clist_get_hadjustment
                                        (GTK_CLIST (dl->ctree)),
                                        gtk_clist_get_vadjustment
                                        (GTK_CLIST (dl->ctree)));

    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);

    gtk_signal_connect (GTK_OBJECT (dl), "button_press_event",
                        popup_handler, dl);

    gtk_signal_connect (GTK_OBJECT (dl->ctree), "tree-expand",
                        ctree_expand, dl);

    dl->adjustment = gtk_scrolled_window_get_vadjustment
                        (GTK_SCROLLED_WINDOW (scrolled));

    gtk_signal_connect_after (GTK_OBJECT (dl->ctree), "tree-expand",
                              ctree_adjust, dl);

    gtk_signal_connect (GTK_OBJECT (dl->ctree), "tree-select-row",
                        ctree_row_selected, dl);

    gtk_signal_connect_after (GTK_OBJECT (dl->ctree), "tree-select-row",
                        insure_node_visible, dl);

    /* Emit the signal tree-expand. The handler will recognise this as
     * the first time and set up the tree. */
    gtk_signal_emit_by_name (GTK_OBJECT (dl->ctree), "tree-expand",
                             NULL, NULL);

    gtk_signal_connect (GTK_OBJECT (dl->ctree), "tree-collapse",
                        ctree_collapse, NULL);
    gtk_signal_connect (GTK_OBJECT (dl), "delete-event",
                        gtk_widget_destroy, NULL);

    gtk_widget_show (GTK_WIDGET (dl->ctree));

    gtk_widget_show (GTK_WIDGET (scrolled));

    gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET (dl->ctree));

    gtk_container_add (GTK_CONTAINER (dl), GTK_WIDGET (scrolled));
}

#ifdef __cplusplus
}
#endif
