''' -*- python -*-                        vim:set ts=4 sw=4:

    FILE: "/home/life/projects/garchiver/garchiver-0.5/src/archive.py"
    LAST MODIFICATION: "Sat, 22 Sep 2001 03:06:25 +0200 (life)"

    (C) 2000 by Danie Roux <droux@tuks.co.za>

$Id: archive.py,v 1.28 2001/06/27 11:02:12 droux Exp $

This is the base class of all archive types.
'''

# This is used by the function make_tempfile to create unique tempfiles.
import tempfile
import string
# And this for piping
import popen2
import os.path

import gettext
_ = gettext.gettext

class Archive:
    ''' Okay, you must call list first before any other methods can be
    used, since list sets the number of files, and this is needed to
    update the progress bar. This is not a problem actually, since the
    first thing you want to do is list the files to add to the gui. TODO
    conversion can make this statement void?
    '''

    def __init__ (self, name, question, message, error):
        ''' Inits the archive with it's filename, and give it references
            to callbacks it can call

            name: Full path and name to archive

            yes_no: A callback that can be used to ask a question It
                returns a boolean and accepts a string

            message: A callback that displays informative messages. Use
                this if you just want to let the user know about
                something. The rest of the application will continue.

            error: A callback that you must use if errors are detected. It
                Takes a string.

            i.e.
            ~~~~
            On error:
                self.error (_("Big bummer happened!"))

            Or:
                self.error (_("Error with a log."), log)

            Question:
                if self.yes_no (_("Should I totally nuke your hardrive?")):
                    wipe the drive
                else: 
                    the guy/girl is still sane
        '''

        self.name = os.path.abspath (name)

        # Progress of any action, except list, in archive should update
        # this. It is a percentage number ranging from 0.0 to 1.0 The gui
        # will connect a time out function that checks this value and
        # updates a progressbar accordingly
        self.progress = 0.0

        # Increment by which the progress bar has to grow
        self.progress_inc = 0

        # The number of files in archive, list should update this
        self.number_of_files = 0

        # Certain formats (read Tar) can't update in a nice way. For them
        # we have this boolean to tell whether the archive has been saved
        # or not. When files gets added to the archive, this gets set to
        # true, and the save button must be set active. When you then save
        # the archive, it gets re-created from scratch.
        # This can also be useful in the tree interface, I think?
        self.modified = 0

        # Set the interaction functions
        self.question = question
        self.message = message
        self.error = error

        self.standard_headers = [_("Name"), _("Size"), _("Path"),
            _("Date"), _("Time")]

        # Keeps a log of all the errors. And only messages that can be
        # seen as errors. Must begin a clean error log with each action
        self.error_log = ''

        # This is a log that should show EXACTLY what the person would
        # see is s/he was doing it in the console. Error messages
        # included. This log will be shown when get_log is called ()
        # Put the command that you executed at the top of the log
        self.log = ''

        # And keep one log previously. In other words, save the logs of
        # the last two actions. In the future it would be possible to go
        # through them all
        self.old_log = ''

    def available (self):
        ''' Returns true if the external program for this format is
        available. Else this format can not be used.
        '''
        raise NotImplementedError, 'Use one of the derived classes!'

    def make_tempfile (self, for_file = ''):
        ''' This is the function that must be called to create
        tempfiles. It just makes them look better! 

        for_file: This is the name of the file in the archive this file
        tempfile gets created for. Say you want to view
        /home/life/pics/splash.xpm, you send this whole string as for_file. Then
        you'll get a nice tempfile like this:
        /tmp/garchiver@1332.2_splash.xpm
        '''

        tmp = tempfile.mktemp ()

        # Too make the file look better, and to aid other programs by
        # making the extension a part of the tempfile
        # Also deletes any spaces in the file name, seeing that it makes
        # tar dump core when you want to delete a file
        tmp = tmp + '_' + os.path.basename (string.replace (for_file, " ", ""))

        return tmp [:string.find (tmp, '@')] + 'garchiver' + \
                    tmp [string.find (tmp, '@'):]

    def extract (self, to, with_path, files=[]):
        '''Call this with a path and a tuple of files

        to: Where to extract the archive to
        with_path: boolean which tells whether the files should be
            extracted with full paths or not.
        files: The individual files to extract. If this is the empty
            list [], extract everything
        '''
        raise NotImplementedError, 'Use one of the derived classes!'

    def new (self):
        ''' Creates a new file that holds the archive. This is not
        necessary for zip, but tar complains, so tar just touches a dummy
        file.
        '''
        pass

    def add (self, files, root_directory='', in_archive='/', flags=''):
        '''Call this with a tuple of files to add

        files: Tuple of files. CAN CONTAIN DIRECTORIES AS WELL!

        root_directory: After adding the files' starting directory
        should be this. i.e. /home/life/Makefile with root_directory
        /home/life/ Should become Makefile after adding it to the
        archive. And /home/life/src/garchiver/README should become
        src/garchiver/README

        Of course, if no root_directory was specified, you just add it
        as is.
        '''

        raise NotImplementedError, 'Use one of the derived classes!'

    def remove (self, files):
        '''Removes all the files in files from the archive

        Returns None if it could remove all the files, else it returns a
        list of files that could not be removed
        '''
        raise NotImplementedError, 'Use one of the derived classes!'

    def list_headers (self):
        '''Called by the GUI to determine the headers of the clist

        Returns a tuple that consists of the _extra_ headers you want in
        the clist.

        The standard headers are ['Name', 'Size', 'Path', 'Date', 'Time']
        Note: gettextize the names. i.e. _('Name') instead of 'Name'
        '''
        raise NotImplementedError, 'Use one of the derived classes!'

    def list (self):
        '''Returns a list of files in this archive

        It would be nice if some caching can be done. So that the whole
        thing does not have to executed and parsed just when 3 files
        gets added. But that's up to the implementation.

        returns: a list of tuples. The tuples has the following form:

        (Name, Size, Path, Date, Time ... and any extra headers you
        defined. See list_headers
        '''
        raise NotImplementedError, 'Use one of the derived classes!'

    def list_dir(self):
        ''' Returns a list of all the folders in the archive

        returns: A list of all folders
        '''
        raise NotImplementedError, 'Use one of the derived classes!'

    def view (self, file):
        ''' Called with a file to view

        file: File that has to be extracted to temporary directory so that
                it can be viewed
        returns: a name which is a temporary file that can be opened.

        Calling function should remove this file when done with it.
        '''
        raise NotImplementedError, 'Use one of the derived classes!'

    def get_log (self):
        ''' Returns a self.log so that the gui can display it. The
        reason we have this function is so that you can implement you're
        own way of logging, and return it here.
        '''
        raise NotImplementedError, 'Use one of the derived classes!'

    def get_previous_log (self):
        ''' Returns the previous log. This will eventually be used to be
        able to go back in all the logs.
        '''
        raise NotImplementedError, 'Use one of the derived classes!'

    def has_next_log (self):
        ''' Tests whether there is a next log. If this does not get
        overridden, it by default says false.
        '''
        return 0

    def has_previous_log (self):
        ''' By default no. '''
        return 0

    def get_previous_log (self):
        raise 'This should not have been called! Use has_previous_log'

    def get_next_log (self):
        raise 'This should not have been called! Use has_next_log'

    def save (self):
        ''' Primarily for tar files, this method recreates the archive
        with the list of files.

        It raises an error if you call this on an archive that does not
        need 'saving'.

        '''
        raise NotImplementedError, 'BUG: This archive does not need to be saved'
