/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla_prefs.h"
#include "mozilla_notifiers.h"
#include "mozilla_i18n.h"
#include "mozilla.h"
#include "prefs.h"
#include "nsBuildID.h"

#include <string.h>
#include <stdlib.h>
#include <libgnome/gnome-defs.h>
#include <locale.h>
#include <libgnome/gnome-i18n.h>
#include <sys/utsname.h>

void mozilla_prefs_set_default_font (void);
static int
mozilla_prefs_get_font_size (gchar *key, int lang_enc);
static gchar*
mozilla_prefs_get_font (gchar *name_key, gchar *size_key, int lang_enc);

extern const gchar *lang_encode_name[LANG_ENC_NUM];
extern const gchar *lang_encode_item[LANG_ENC_NUM];

/**
 * mozilla_prefs_set: set mozilla preferences
 */
void
mozilla_prefs_set (void)
{
	mozilla_notifiers_set_defaults();

	mozilla_preference_set_boolean ("mozilla.widget.raise-on-setfocus",
					FALSE);
	mozilla_preference_set_boolean ("browser.display.use_system_colors",
					FALSE);

	/* set default search engine */
	mozilla_preference_set ("keyword.URL",_("http://www.google.com/search?q="));
	mozilla_preference_set_boolean ("keyword.enabled", TRUE);
	mozilla_preference_set_boolean ("security.checkloaduri", FALSE); 

	/* while we have no UI */
	mozilla_preference_set_boolean ("wallet.captureForms", FALSE);

	/* deactivate mailcap and mime.types support */
	mozilla_preference_set ("helpers.global_mime_types_file", "");
	mozilla_preference_set ("helpers.global_mailcap_file", "");
	mozilla_preference_set ("helpers.private_mime_types_file", "");
	mozilla_preference_set ("helpers.private_mailcap_file", "");
	
	/* dont allow xpi installs from galeon, there are crashes */
	mozilla_preference_set_boolean ("xpinstall.enabled", FALSE);

	/* disable sucky XUL ftp view, have nice ns4-like html page instead */
	mozilla_preference_set_boolean ("network.dir.generate_html", TRUE);
	
	/* set the right accept encoding flags */
	mozilla_preference_set ("network.http.accept-encoding" ,
				"gzip, deflate, compress;q=0.9");

	mozilla_save_prefs ();
}

/**
 * mozilla_prefs_get_fonts: get fonts preferences
 */
void 
mozilla_prefs_get_fonts (LanguageFontInfo *info, int lang_enc)
{
	g_return_if_fail (info != NULL);

	/* get serif font */
	info->serif = mozilla_prefs_get_font 
		("font.name.serif","font.size.variable",
		 lang_enc);

	/* get sans serif font */
	info->sansserif = mozilla_prefs_get_font 
		("font.name.sans-serif","font.size.variable", 
		 lang_enc);

	/* get cursive font */
	info->cursive = mozilla_prefs_get_font
		("font.name.cursive", "font.size.variable", 
		 lang_enc);

	/* get fantasy font */
	info->fantasy = mozilla_prefs_get_font 
		("font.name.fantasy", "font.size.variable", 
		 lang_enc);

	/* get monospace font */
	info->monospace = mozilla_prefs_get_font 
		("font.name.monospace", "font.size.fixed", 
		 lang_enc);

	/* get font size */
        info->min_size = mozilla_prefs_get_font_size 
                ("font.min-size.variable", lang_enc);
}

static int
mozilla_prefs_get_font_size (gchar *key, int lang_enc)
{
	gchar key_buf[128];

	g_snprintf (key_buf, 128, "%s.%s",
		    key, lang_encode_item [lang_enc]);
	return mozilla_preference_get_int (key_buf);
}

static gchar*
mozilla_prefs_get_font (gchar *name_key, gchar *size_key, int lang_enc)
{
	gchar key_buf[128];
	gchar *font_name;
	gchar **font;
	gchar *result;
	int font_size;

	g_snprintf (key_buf, 128, "%s.%s",
		    name_key, lang_encode_item [lang_enc]);
	font_name = mozilla_preference_get (key_buf);
	g_snprintf (key_buf, 128, "%s.%s",
		    size_key, lang_encode_item [lang_enc]);
	font_size = mozilla_preference_get_int (key_buf);

	if (!font_name)
	{
		font_name = g_strdup ("*-*-*-*");
	}

	font = g_strsplit (font_name,"-",-1);
	result = g_strdup_printf("-%s-%s-*-*-*-*-%d-*-*-*-*-*-%s-%s",
				 font[0], font[1], font_size,
				 font[2], font [3]);

	g_strfreev (font);
	g_free (font_name);

	return result;
}

int
mozilla_prefs_set_font (gchar *name, gchar *key, int lang_enc)
{
	gchar *font_name;
	gchar **font;
	gchar key_buf[128];
	int size;

	if (!name) return 0;

	font = g_strsplit (name,"-",-1);
	font_name = g_strconcat(font[1],"-",font[2],"-",
				font[13],"-",font[14], NULL);
	g_snprintf (key_buf, 128, "%s.%s",
		    key, lang_encode_item [lang_enc]);

	if (strcmp(font_name, "*-*-*-*")!=0)
	{
		mozilla_preference_set (key_buf, font_name);
	}

	size = atoi (font[7]);

	g_strfreev (font);
	g_free (font_name);

	return size;
}

void
mozilla_prefs_set_font_size (gchar *key, int size, int lang_enc)
{
	gchar key_buf[128];
	g_snprintf (key_buf, 128, "%s.%s",
		    key, lang_encode_item [lang_enc]);
	mozilla_preference_set_int (key_buf, size);
}

void
mozilla_prefs_set_user_agent ()
{
	static gchar *default_user_agent = NULL;
	gchar *value;
	gchar *user_agent = NULL;
	struct utsname name;
	gchar *system;

	if (!default_user_agent)
	{
		if (uname (&name) >= 0)
		{
			system = g_strdup_printf ("%s %s",
						  name.sysname, 
						  name.machine);
		}
		else
		{
			system = g_strdup ("Unknown");
		}
		
		default_user_agent = g_strdup_printf
			("Mozilla/5.0 Galeon/" VERSION " (X11; %s; U;) Gecko/%d", 
			 system,
			 NS_BUILD_ID/100);

		g_free (system);
	}

	value = eel_gconf_get_string (CONF_NETWORK_USER_AGENT);
	
	/* now, build a valid user agent string */
	if (!value || !strcmp ("", value) 
		   || !strcmp ("default", value) 
	 	   || !strcmp ("Galeon", value)
	 	   || !strcmp (_("Default (recommended)"), value)
		   || !strcmp ("Default (recommended)", value))
	{
		user_agent = g_strdup (default_user_agent);
	}
	else
	{
		user_agent = g_strdup (value);
	}
				
	if (user_agent)
	{
		mozilla_preference_set ("general.useragent.override",
					user_agent);
		g_free (user_agent);
	}

	g_free (value);
}

