/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: FWWindow.cpp,v 1.134 2004/09/28 05:21:00 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"
#include "global.h"
#include "utils.h"

#include "FWWindow.h"
#include "ObjectManipulator.h"
#include "ObjectEditor.h"
#include "FWObjectClipboard.h"
#include "RuleSetView.h"
#include "RCSFileDialog.h"
#include "RCSFilePreview.h"
#include "execDialog.h"
#include "FWBSettings.h"
#include "PrefsDialog.h"
#include "LibExportDialog.h"
#include "FWBTree.h"
#include "ObjConflictResolutionDialog.h"
#include "findDialog.h"
#include "upgradePredicate.h"
#include "FWObjectPropertiesFactory.h"
#include "listOfLibraries.h"

#include "rcsfilesavedialog_q.h"

#include "aboutdialog_q.h"
#include "debugDialog.h"
#include "filePropDialog.h"

#include "instConf.h"
#include "instDialog.h"

#include "fwbuilder/FWReference.h"
#include "fwbuilder/Policy.h"
#include "fwbuilder/InterfacePolicy.h"
#include "fwbuilder/NAT.h"
#include "fwbuilder/Tools.h"
#include "fwbuilder/dns.h"
//#include "fwbuilder/crypto.h"
#include "fwbuilder/XMLTools.h"
#include "fwbuilder/Resources.h"
#include "fwbuilder/FWObjectDatabase.h"
#include "fwbuilder/FWException.h"
#include "fwbuilder/Management.h"
#include "fwbuilder/RuleElement.h"

#include "fwbuilder/Library.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/Host.h"
#include "fwbuilder/Network.h"
#include "fwbuilder/IPv4.h"
#include "fwbuilder/AddressRange.h"
#include "fwbuilder/ObjectGroup.h"

#include "fwbuilder/Resources.h"
#include "fwbuilder/FWReference.h"
#include "fwbuilder/Interface.h"
#include "fwbuilder/RuleSet.h"
#include "fwbuilder/InterfacePolicy.h"

#include "fwbuilder/CustomService.h"
#include "fwbuilder/IPService.h"
#include "fwbuilder/ICMPService.h"
#include "fwbuilder/TCPService.h"
#include "fwbuilder/UDPService.h"
#include "fwbuilder/ServiceGroup.h"

#include "fwbuilder/Interval.h"
#include "fwbuilder/IntervalGroup.h"


#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>

#ifndef _WIN32
#  include <unistd.h>     // for access(2)
#endif

#include <qaction.h>
#include <qlistview.h>
#include <qmessagebox.h>
#include <qapplication.h>
#include <qfileinfo.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qpixmap.h>
#include <qheader.h>
#include <qtabwidget.h>
#include <qcombobox.h>
#include <qcheckbox.h>
#include <qtextedit.h>
#include <qstringlist.h>
#include <qpopupmenu.h>
#include <qtoolbutton.h>
#include <qsplitter.h>
#include <qtextedit.h>
#include <qlayout.h>
#include <qapplication.h>
#include <qcursor.h>
#include <qsplitter.h>
#include <qtimer.h>
#include <qstatusbar.h>
#include <qlabel.h>

using namespace libfwbuilder;
using namespace std;



FWWindow::FWWindow() : FWBMainWindow_q()
{
    rcs             = NULL;
    systemFile      = true;
    visibleFirewall = NULL;
    delete treeFrame;

    autosaveTimer = new QTimer(this);

// om is a global var
    om=new ObjectManipulator( objInfoSplitter );

    om->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)7,
                                    (QSizePolicy::SizeType)7,
                                    0, 0,
                                    om->sizePolicy().hasHeightForWidth() ) );
    objInfoSplitter->moveToFirst( om );

//    connect( newObjectAction, SIGNAL( activated() ),
//             om,              SLOT( newObject() ) );

//    connect( backAction, SIGNAL( activated() ),
//             om,         SLOT( back() ) );

    connect( findAction, SIGNAL( activated() ),
             this,       SLOT(   search()    ) );

    delete infoFrame;

    oi=new QTextEdit( objInfoSplitter );
    oi->setReadOnly(true);

    objInfoSplitter->setResizeMode( oi, QSplitter::KeepSize );

    oi->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)7,
                                    (QSizePolicy::SizeType)7,
                                    0, 0,
                                    oi->sizePolicy().hasHeightForWidth() ) );
    objInfoSplitter->moveToLast( oi );

    oi->setGeometry( oi->geometry().x(), oi->geometry().y(),
                     oi->geometry().width(), st->getInfoWindowHeight() );

    if (st->getInfoStyle()!=0) oi->show();
    else oi->hide();

    om->show();

//    findObject->setMinimumSize( QSize( 0, 0 ) );
}

FWWindow::~FWWindow()
{
    if (rcs!=NULL) delete rcs;
}

void FWWindow::clearFirewallTabs()
{
    ruleSets->hide();

    while (ruleSets->count()!=0)
    {
        QWidget *p = ruleSets->page(0);
        ruleSets->removePage( p );
        delete p;
    }
    return;
}

void FWWindow::helpAbout()
{
    AboutDialog_q ad;
    ad.exec();
}

void FWWindow::debug()
{
    debugDialog dd(this);
    dd.exec();
}

void FWWindow::info(FWObject *obj)
{
//    select();

    if (st->getInfoStyle()!=0)
    {
        oi->clear();
        oi->setTextFormat(QTextEdit::RichText);

        QString s="";
        if (st->getInfoStyle()==2)
            s=FWObjectPropertiesFactory::getObjectPropertiesDetailed(obj)
                + QString("<hr height=\"0\">");

        s+=QString::fromUtf8(obj->getComment().c_str());
        oi->append(s);
        oi->setCursorPosition(0,0);
    }

//    mw->unselectRules();
}

bool FWWindow::saveIfModified()
{
    if (db()->isDirty())
    {
        switch (QMessageBox::information(this, "Firewall Builder",
            tr("Some objects have been modified but not saved.\n"
               "Do you want to save changes now ?"),
            tr("&Save"), tr("&Discard"), tr("&Cancel"),
            0,       // Enter = button 0
            2 ) ) {   // Escape == button 2

        case 0:
            save();
            break;
        case 1:  // discard
            db()->setDirty(false);
            break;
        case 2:  // cancel
            return(false);
        }
    }
    return true;
}

QString FWWindow::chooseNewFileName(const QString &fname,
                                    bool checkPresence,const QString &title)
{
    QString destdir;

    if (st->getWDir().isEmpty())
    {
        if (fname.isEmpty())
        {
/* need some reasonable default working directory.
 * on Unix will use current dir.
 * on Windows will use user's document dir.
 */
#if defined(Q_OS_WIN32) || defined(Q_OS_MACX)
            destdir = userDataDir.c_str();
#else
            destdir = "";
#endif
        } else
        {
            if (QFileInfo(fname).isDir()) destdir=fname;
            else                      destdir = fname.left( fname.findRev('/',-1) );
        }
    } else
    {
        destdir=st->getWDir();
    }

    QString fn = QFileDialog::getSaveFileName( destdir,
       tr( "FWB Files (*.fwb);;All Files (*)" ),
                                       this, 0, title );
    if ( fn.isEmpty() ) return "";

    QFileInfo finfo(fn);
    if (finfo.extension(false)!="fwb") fn=fn+".fwb";
    finfo.setFile(fn);

    if ( ! checkPresence || ! finfo.exists() ||
             QMessageBox::warning(
                 this,"Firewall Builder", 
                 tr("The file %1 already exists.\nDo you want to overwrite it ?")
                 .arg(fn.latin1()),
                 "&Yes", "&No", QString::null,
                 0, 1 )==0 )
    {
        return fn;
    }

    return "";
}

void FWWindow::setFileName(const QString &fname)
{
    systemFile=false;
    rcs->setFileName(fname);
    db()->setFileName(fname.latin1());

    QString caption = rcs->getFileName().section("/",-1,-1);
    if (rcs->isInRCS()) caption = caption + ", rev " + rcs->getSelectedRev();

    setCaption( QString("Firewall Builder: ")+caption );
}

void FWWindow::fileProp()
{
    if (rcs!=NULL)
    {
        filePropDialog fpd(this,rcs);
        fpd.exec();
    }
}

void FWWindow::fileNew()
{    
    if (!saveIfModified()) return;

    if (!systemFile && rcs!=NULL) fileClose();  // fileClose calls load(this)
    else  load(this);

    visibleFirewall = NULL;

    showFirewalls();

    QString nfn=chooseNewFileName(st->getWDir(),true,
                                  tr("Choose name and location for the new file"));

    if ( !nfn.isEmpty() )
    {
        setFileName(nfn);

        save();

        addToRCSAction->setEnabled( !rcs->isInRCS() && !rcs->isRO() && !rcs->isTemp());
        fileDiscardAction->setEnabled( rcs->isInRCS() && !rcs->isRO() && !rcs->isTemp());
        fileSaveAction->setEnabled( !rcs->isRO() && !rcs->isTemp() );

        setupAutoSave();
    }
}

void FWWindow::fileOpen()
{    
    if (!saveIfModified()) return;

    if (!systemFile && rcs!=NULL) fileClose();

    RCSFileDialog   fd(this, 0, true);

    if ( fd.exec()== QDialog::Accepted )
    {
        RCS *rcs = fd.getSelectedRev();

        if (rcs==NULL) return;

/***********************************************************************
 * TODO : add an option "RCS support"
 *
 * if opening read-only, do not checkout 
 * checkout may throw exception, need to catch it
 */
        try
        {
            rcs->co();

        } catch (FWException &ex)
        {
/* if there was an exception, abort operation. E.g. RCS::co may throw
 * exception */
            return;
        }
/***********************************************************************/

        load(this, rcs );
        showFirewalls();

        if (rcs->isTemp()) unlink(rcs->getFileName().latin1());
    }
}

void FWWindow::fileClose()
{
    if (fwbdebug) qDebug("FWWindow::fileClose(): start");

    if (oe->isVisible()) oe->hide();

    if (!saveIfModified()) return;

    if (fwbdebug) qDebug("FWWindow::fileClose(): will save");

/***********************************************************************
 * TODO : add an option "RCS support"
 */
    if (!systemFile && rcs && rcs->isCheckedOut() && !rcs->isTemp())
    {
        if (fwbdebug) qDebug("FWWindow::fileClose(): checking in");
        if (!checkin(true)) return;
        if (fwbdebug) qDebug("FWWindow::fileClose(): done");
        delete rcs;
    }
/***********************************************************************/

    rcs=NULL;

    if (fwbdebug) qDebug("FWWindow::fileClose(): clearing widgets");

    firewalls.clear();
    fwList->clear();
    clearFirewallTabs();
    ruleSetViews.clear();
    om->clearObjects();
    FWObjectClipboard::obj_clipboard->clear();

    if (fwbdebug) qDebug("FWWindow::fileClose(): loading standard objects");

    load(this);

    if (fwbdebug) qDebug("FWWindow::fileClose(): show firewalls");

    showFirewalls();

    if (fwbdebug) qDebug("FWWindow::fileClose(): all done");

    setupAutoSave();
}

void FWWindow::fileSave()
{
    QStatusBar *sb = statusBar();
    sb->message( tr("Saving data to file...") );
    save();
    sb->clear();
}

void FWWindow::fileSaveAs()
{
    if (oe->isVisible()) oe->hide();

/* we need to save data into the current file before we save it into a
 * new file, provided we do have current file
 */
    if (!systemFile && rcs &&
        !rcs->isRO() && !rcs->isTemp() && !rcs->getFileName().isEmpty() &&
        !saveIfModified()) return;

    if (!systemFile && rcs && rcs->isCheckedOut() && !rcs->isTemp())
    {
        if (fwbdebug) qDebug("FWWindow::fileSaveAs(): checking in");
        if (!checkin(true)) return;
        if (fwbdebug) qDebug("FWWindow::fileSaveAs(): done");
    }
    
    QString oldFileName = rcs->getFileName();
    if (rcs!=NULL) delete rcs;

    rcs = new RCS("");

    QString nfn=chooseNewFileName(oldFileName,true,
                                  tr("Choose name and location for the file"));

    if (!nfn.isEmpty())
    {
        setFileName(nfn);

        save();

        addToRCSAction->setEnabled( !rcs->isInRCS() && !rcs->isRO() && !rcs->isTemp());
        fileDiscardAction->setEnabled( rcs->isInRCS() && !rcs->isRO() && !rcs->isTemp());
        fileSaveAction->setEnabled( !rcs->isRO() && !rcs->isTemp() );
    }
}

void FWWindow::filePrint()
{
}

void FWWindow::fileExit()
{    
    if (saveIfModified())
    {
        if (!systemFile && rcs && rcs->isCheckedOut())
        {
            if (!checkin(true)) return;
            delete rcs;
        }
        qApp->quit();
    }
}

/*
 * discard changes done to the file and check out clean copy of the
 * head revision from RCS
 */
void FWWindow::fileDiscard()
{
    if (QMessageBox::warning(this, "Firewall Builder",
      tr("This operation discards all changes that have been saved\n"
         "into the file so far, closes it and replaces it with a clean\n"
         "copy of its head revision from RCS.\n"
         "\n"
         "All changes will be lost if you do this.\n\n"),
      tr("&Discard changes"),
      tr("&Cancel"), QString::null,
      1 )==0 )
    {
/* need to close the file without asking and saving, then reopen it again */
        
//        QString fname = rcs->getFileName();

        db()->setDirty(false);  // so it wont ask if user wants to save
        rcs->abandon();
        fileClose();

        load(this);
    }
}

void FWWindow::fileAddToRCS()
{
    if (!saveIfModified()) return;

    try
    {
        if (!rcs->isInRCS() && !rcs->isRO())
        {
            rcs->add();
            rcs->co();
            QMessageBox::information(
                this,"Firewall Builder", 
                tr("File %1 has been added to RCS.").arg(rcs->getFileName()),
                "&Continue", QString::null,QString::null,
                0, 1 );
        }
    }
    catch (FWException &ex)
    {
        QMessageBox::critical(
            this,"Firewall Builder", 
            tr("Error adding file to RCS:\n%1").arg(ex.toString().c_str()),
            "&Continue", QString::null,QString::null,
            0, 1 );
    }

    QString caption = rcs->getFileName().section("/",-1,-1);
    if (rcs->isInRCS()) caption = caption + ", rev " + rcs->getSelectedRev();
    if (rcs->isRO()) caption = caption + " " + tr("(read-only)");
    
    setCaption( QString("Firewall Builder: ")+caption );

    addToRCSAction->setEnabled( !rcs->isInRCS() && !rcs->isRO());
    fileDiscardAction->setEnabled( rcs->isInRCS() && !rcs->isRO() && !rcs->isTemp());
}

bool FWWindow::editingLibrary()
{
    return (rcs!=NULL && 
            ( rcs->getFileName().findRev(".fwl",-1)!=-1) );
}

void FWWindow::load(QWidget *dialogs_parent)
{
    if (fwbdebug) qDebug("FWWindow::load(): start");

    editingStandardLib = false;
    editingTemplateLib = false;

    QWidget *dlgp=NULL;
    if (dialogs_parent==NULL)
    {       
        if (isVisible()) dlgp=this;
    } else
    {
        dlgp=dialogs_parent;
    }

    MessageBoxUpgradePredicate upgrade_predicate(dlgp);

    fd->reset();

    if (fwbdebug) qDebug("FWWindow::load(): start 2");

    try
    {
// need to drop read-only flag on the database before I load new objects
        
        FWObjectDatabase::db->setReadOnly( false );

        if (fwbdebug) qDebug("FWWindow::load(): loading objects");

// always loading system objects
        FWObjectDatabase::db->load( sysfname, &upgrade_predicate, librespath);
        FWObjectDatabase::db->setFileName("");

        if (fwbdebug) qDebug("FWWindow::load(): create User library");

        FWObject *userLib=FWBTree::createNewLibrary( FWObjectDatabase::db );
        userLib->setName("User");
        userLib->setStr("color","#d2ffd0");

        if (fwbdebug) qDebug("FWWindow::load(): loading libraries");

        for (list<libData>::iterator i=addOnLibs->begin(); i!=addOnLibs->end(); ++i)
        {
            string libfname = i->path;
            if (libfname!=sysfname && i->load)
            {
                FWObjectDatabase *ndb = new FWObjectDatabase();
                ndb->load(libfname, &upgrade_predicate,librespath);
                FWObject *dobj = ndb->getById( FWObjectDatabase::getDeletedObjectsId());
                if (dobj) ndb->remove(dobj, false);

                MergeConflictRes mcr(dlgp);
                FWObjectDatabase::db->merge(ndb, &mcr);

                delete ndb;
            }
        }

        FWObjectDatabase::db->setDirty(false);
        FWObjectDatabase::db->setFileName("");

        if (fwbdebug) qDebug("FWWindow::load(): create RCS");

        rcs = new RCS("");
        systemFile=true;
/*
 * TODO: we should create new FWObjectDatabase object and assign db
 * instead of using singleton
 */
        objdb = FWObjectDatabase::db;

        setCaption( "Firewall Builder" );

        fileSaveAction->setEnabled( false );
        addToRCSAction->setEnabled( false );
        fileDiscardAction->setEnabled( false );
    
        if (fwbdebug) qDebug("FWWindow::load(): done");

    } catch(FWException &ex)
    {
        QMessageBox::warning(
            this,"Firewall Builder", 
            tr("Error loading file:\n%1").arg(ex.toString().c_str()),
            "&Continue", QString::null,QString::null,
            0, 1 );
    }

    if (fwbdebug) qDebug("FWWindow::load(): load objects in ObjectManager");

    om->loadObjects();

    if (fwbdebug) qDebug("FWWindow::load(): all done");

    setupAutoSave();
}

void FWWindow::load(QWidget *dialogs_parent,RCS *_rcs)
{
    fd->reset();

    editingStandardLib = false;
    editingTemplateLib = false;

    bool forceSave=false; // use this flag to force 'save' operation if file should be renamed

    QWidget *dlgp=NULL;
    if (dialogs_parent==NULL)
    {       
        if (isVisible()) dlgp=this;
    } else
    {
        dlgp=dialogs_parent;
    }

    MessageBoxUpgradePredicate upgrade_predicate(dlgp);

    assert(_rcs!=NULL);

    rcs = _rcs;

    try
    {
        /* load the data file */
        systemFile=false;

// need to drop read-only flag on the database before I load new objects
        FWObjectDatabase::db->setReadOnly( false );

// always loading system objects
        FWObjectDatabase::db->load( sysfname, &upgrade_predicate, librespath);
        FWObjectDatabase::db->setFileName("");

/* objects from a data file are in database ndb */
        FWObjectDatabase *ndb = new FWObjectDatabase();
        ndb->load(rcs->getFileName().latin1(), &upgrade_predicate,librespath);
        time_t   oldtimestamp = ndb->getTimeLastModified();

/* loadingLib is true if user wants to open a library or master library file */
        bool loadingLib         = editingLibrary();

/* if user opens library file, clear read-only flag so they can edit it */
        if (loadingLib)
        {
            list<FWObject*> ll = ndb->getByType(Library::TYPENAME);
            for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
            {
                if ((*i)->getId()==STANDARD_LIB) editingStandardLib=true;
                if ((*i)->getId()==TEMPLATE_LIB) editingTemplateLib=true;
                (*i)->setReadOnly( false );
            }
        } else
        {
/* preload libraries only if we do not edit a library file */
            for (list<libData>::iterator i=addOnLibs->begin();
                 i!=addOnLibs->end(); ++i)
            {
                string libfname = i->path;
                if (libfname!=sysfname && i->load)
                {
                    FWObjectDatabase *ndb1 = new FWObjectDatabase();
                    ndb1->load(libfname, &upgrade_predicate,librespath);
                    FWObject  *nlib1 = ndb1->getFirstByType(Library::TYPENAME);
                    if(nlib1==NULL)
                    {
                        qDebug("Error preloading library from file %s",
                               libfname.c_str());
                        assert(nlib1!=NULL);
                    }
                    string nlib1ID = nlib1->getId();
                    FWObject *dobj = ndb1->getById( FWObjectDatabase::getDeletedObjectsId());
                    if (dobj) ndb1->remove(dobj, false);

                    MergeConflictRes mcr(dlgp);
                    FWObjectDatabase::db->merge(ndb1, &mcr);

/* preloaded libraries are always read-only */
                    FWObjectDatabase::db->getById(nlib1ID)->setReadOnly(true);

                    delete ndb1;
                }
            }
        }

        MergeConflictRes mcr(dlgp);
        FWObjectDatabase::db->merge(ndb, &mcr);

        delete ndb;

        FWObjectDatabase::db->setFileName(rcs->getFileName().latin1());
        FWObjectDatabase::db->resetTimeLastModified(oldtimestamp);
        FWObjectDatabase::db->setDirty(false);

/*
 * TODO: we should create new FWObjectDatabase object and assign db
 * instead of using singleton
 */
        objdb = FWObjectDatabase::db;

/* this is a hack: 'Standard' library should be read-only. I have too
 * many files I already converted to the new API/DTD and I am too lazy
 * to convert them again, so I patch it up here.
 *
 * However, if I am editing standard library, it should not be read-only.
 */
        FWObject *slib = objdb->getById("syslib000");
        if (fwbdebug)
            qDebug("standard library read-only status: %d, editingStandardLib: %d",
                   slib->isReadOnly(), editingStandardLib);

        if (slib!=NULL ) slib->setReadOnly(! editingStandardLib);

/* if the file name has an old extension .xml, change it to .fwb and
 * warn the user
 */
        QString   fn = rcs->getFileName();
        QFileInfo ofinfo(fn);

        if ( ofinfo.extension(false)=="xml")
        {
            QString nfn=ofinfo.baseName(true) + ".fwb";

            bool needToRename = true;

/* need these dances with symlinks to fix bug #1008956: "Existing .fwb
 * file gets overwritten if has wrong extension"
 */
            QFileInfo nfinfo(nfn);
            if (nfinfo.exists() && ofinfo.isSymLink() && ofinfo.readLink()==nfn)
            {
// .xml file is a symlink pointing at .fwb file
// no need to rename
                needToRename = false;
            }

            if (needToRename)
            {
                bool firstDlg=true;
                if (nfinfo.exists())
                {
/* .fwb file exists but .xml is not a symlink
 * .fwb is a separate file with the same name.
 *
 * tell the user we need to rename old file but the new file exists,
 * then ask them to choose a new name. If the user chooses the same
 * name and agrees to overwrite the file, just use this name. If the
 * user hits cancel, tell them they need to choose a new name and open
 * "file save" dialog again.
 *
 * Show the first dialog only once. If user hits Cancel, they see
 * shorted version of the dialog and will be presented with "save
 * file" dialog again.
 */
                    do
                    {
                        if (firstDlg)
                        {
                            QMessageBox::warning(
                                this,"Firewall Builder", 
                                tr("Firewall Builder 2 uses file extension '.fwb' and \nneeds to rename old data file '%1' to '%2',\nbut file '%3' already exists.\nChoose a different name for the new file.")
                                .arg(fn).arg(nfn).arg(nfn),
                                "&Continue", QString::null,QString::null,
                                0, 1 );
                        }
                        firstDlg=false;

                        nfn=chooseNewFileName(fn,true,
                                              tr("Choose name and location for the new file"));
                        if (!nfn.isEmpty())    nfinfo.setFile(nfn);
                        else 
                        {
                            QMessageBox::warning(
                                this,"Firewall Builder", 
                                tr("Please choose a different name for the new file."),
                                "&Continue", QString::null,QString::null,
                                0, 1 );
                        }
                    } while (nfn.isEmpty());
                }

                rename(fn.latin1(), nfn.latin1());

                
                QMessageBox::warning(
                this,"Firewall Builder", 
                tr("Firewall Builder 2 uses file extension '.fwb'. Your data file '%1' \nhas been renamed '%2'")
                .arg(fn).arg(nfn),
                "&Continue", QString::null,QString::null,
                0, 1 );
                
            }

            fn = nfn;
        }

        rcs->setFileName(fn);
        db()->setFileName(fn.latin1());

        QString caption = rcs->getFileName().section("/",-1,-1);
        if (rcs->isInRCS()) caption = caption + ", rev " + rcs->getSelectedRev();
        if (rcs->isRO()) caption = caption + " " + tr("(read-only)");

        setCaption( QString("Firewall Builder: ")+caption );

        fileSaveAction->setEnabled( !rcs->isRO() && !rcs->isTemp());
        addToRCSAction->setEnabled( !rcs->isInRCS() && !rcs->isRO());
        fileDiscardAction->setEnabled( rcs->isInRCS() && !rcs->isRO());

    } catch(FWException &ex)
    {
        string trans = ex.getProperties()["failed_transformation"];
        string elem  = ex.getProperties()["failed_element"];

        if(!trans.empty() || !elem.empty())
        {
            QString msg = tr("Exception: %1").arg(ex.toString().c_str());
            if (!trans.empty())
		msg+="\n"+tr("Failed transformation : %1").arg(trans.c_str());
            if (!elem.empty())
		msg+="\n"+tr("XML element : %1").arg(elem.c_str());

             QMessageBox::warning(
                 this,"Firewall Builder", 
                 tr("Error loading file:\n%1").arg(msg),
                 "&Continue", QString::null,QString::null,
                 0, 1 );
        } else
             QMessageBox::warning(
                 this,"Firewall Builder", 
                 tr("Error loading file:\n%1").arg(ex.toString().c_str()),
                 "&Continue", QString::null,QString::null,
                 0, 1 );

        load(this);
        return;
    }

    db()->setReadOnly( rcs->isRO() || rcs->isTemp() );

// clear dirty flag for all objects, recursively
    if (!forceSave)  db()->setDirty(false);

    om->loadObjects();

    setupAutoSave();
}

bool FWWindow::checkin(bool unlock)
{
/* doing checkin only if we did checkout so rcs!=NULL */
    QString rlog="";

    if (rcs && rcs->isCheckedOut() && !rcs->isTemp())
    { 
        if (rcs->isDiff()) // if the file hasn't changed, do not need to ask for the comment
        {
            if ( ! st->getRCSLogState())
            {
                RCSFileSaveDialog_q fsd(this);
                if ( fsd.exec()== QDialog::Rejected ) return false;

                bool    empty_rcslog = fsd.nolog->isChecked();
                if (empty_rcslog)
                {
                    rlog = "";
                    st->setRCSLogState(true);
                } else              rlog = fsd.rcslog->text();
            }
        }

/***********************************************************************/
        try
        {
            if (fwbdebug)  qDebug("about to check the file in");
            rcs->ci(rlog,unlock);
            if (fwbdebug)  qDebug("done");
        }
        catch (FWException &ex)
        {
            QMessageBox::warning(
                this,"Firewall Builder", 
                tr("Error checking in file %1:\n%2")
                .arg(rcs->getFileName()).arg(ex.toString().c_str()),
                "&Continue", QString::null, QString::null,
                0, 1 );
        }
/***********************************************************************/
    } else
    {
        return false;
    }
    return true;
}

void FWWindow::save()
{
    if (fwbdebug)
        qDebug("FWWindow::save:  rcs=%p  rcs->isRO=%d  rcs->isTemp=%d rcs->getFileName=%s",
               rcs, rcs->isRO(), rcs->isTemp(), rcs->getFileName().latin1());

    if (!rcs->isRO() && !rcs->isTemp())
    {
        try
        {
            if (rcs->getFileName().isEmpty())
                fileSaveAs();  // eventually calls this method again
            else
            {
/* editingLibfile is true if user edits a library or master library file */
                bool editingLibfile=editingLibrary();

                if (st->getDontSaveStdLib())
                {
                    list<FWObject*> userLibs;
                    list<FWObject*> ll = mw->db()->getByType(Library::TYPENAME);
                    for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
                    {
                        if (fwbdebug) qDebug("FWWindow::save()  lib %s",
                                             (*i)->getName().c_str() );
/* if we are not editing a library file, skip preloaded libraries */
                        if (!editingLibfile &&
                            addOnLibs->isLoaded((*i)->getName().c_str()))
                        {
                            if (fwbdebug) qDebug("                   skip");
                            continue;
                        }
/* skip standard and template libraries unless we edit them */
                        QString s=(*i)->getId().c_str();
                        if (s==STANDARD_LIB && !editingStandardLib) continue;
                        if (s==TEMPLATE_LIB && !editingTemplateLib) continue;

                        if (fwbdebug) qDebug("                   add");
                        userLibs.push_back( *i );
                    }

                    QApplication::setOverrideCursor(QCursor( Qt::WaitCursor));
                    FWObjectDatabase *ndb = mw->db()->exportSubtree(userLibs);
                    QApplication::restoreOverrideCursor();

                    if (editingLibfile)
                    {
/* exported libraries are always read-only */
                        list<FWObject*> ll = ndb->getByType(Library::TYPENAME);
                        for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
                            if ((*i)->getId()!=STANDARD_LIB &&
                                (*i)->getId()!=DELETED_LIB) (*i)->setReadOnly( true );
                    }

                    ndb->resetTimeLastModified( db()->getTimeLastModified() );
                    ndb->saveFile( rcs->getFileName().latin1() );

                    delete ndb;
                } else  db()->saveFile( rcs->getFileName().latin1() );
            }
            db()->setDirty(false);
        }
        catch (FWException &ex)
        {
/* error saving the file. Since XMLTools does not return any useful
 * error message in the exception, let's check for obvious problems here
 */
            QString err;
            if (access( rcs->getFileName().latin1(), W_OK)!=0 && errno==EACCES)
                err=tr("File is read-only");
            else
                err=ex.toString().c_str();

            QMessageBox::warning(
                this,"Firewall Builder", 
                tr("Error saving file %1: %2")
                .arg(rcs->getFileName()).arg(err),
                "&Continue", QString::null, QString::null,
                0, 1 );
        }
    }
}

void FWWindow::loadLibrary(const string &libfpath)
{
    MessageBoxUpgradePredicate upgrade_predicate;

    try
    {
        FWObjectDatabase *ndb = new FWObjectDatabase();
        ndb->load(libfpath,  &upgrade_predicate,  librespath);

        MergeConflictRes mcr(this);
        db()->merge(ndb, &mcr);

        delete ndb;

    } catch(FWException &ex)
    {
        QMessageBox::warning(
            this,"Firewall Builder", 
            tr("Error loading file %1:\n%2").
                 arg(libfpath.c_str()).arg(ex.toString().c_str()),
            "&Continue", QString::null,QString::null,
            0, 1 );
    }
}

void FWWindow::fileImport()
{
    fd->reset();

    QString fname = QFileDialog::getOpenFileName( st->getWDir(), 
           "Firewall Builder 2 files (*.fwl);;FWB Files (*.fwb);;All Files (*)",
                          this, 0,
                          tr("Choose a file to import") );

    if (fname.isEmpty()) return;   // Cancel  - keep working with old file

    loadLibrary( fname.latin1() );

    om->loadObjects();
    showFirewalls();

    addOnLibs->add( fname.latin1() );
}

void FWWindow::fileExport()
{
    fd->reset();

    LibExportDialog ed;
    ed.exec();
}

void FWWindow::showFirewalls()
{
    list<FWObject*> fl;
    findFirewalls(db(), fl);
    fl.sort(FWObjectNameCmpPredicate());

    firewalls.clear();
    fwList->clear();
    clearFirewallTabs();
    ruleSetViews.clear();
    firewallName->setText("");

    insertRuleAction->setEnabled( fl.size()!=0 );
    moveRuleAction->setEnabled( fl.size()!=0 );
    moveRuleUpAction->setEnabled( fl.size()!=0 );
    moveRuleDownAction->setEnabled( fl.size()!=0 );
    addRuleAfterCurrentAction->setEnabled( fl.size()!=0 );
    removeRuleAction->setEnabled( fl.size()!=0 );
    copyRuleAction->setEnabled( fl.size()!=0 );
    cutRuleAction->setEnabled( fl.size()!=0 );
    pasteRuleAboveAction->setEnabled( fl.size()!=0 );
    pasteRuleBelowAction->setEnabled( fl.size()!=0 );

    compileAction->setEnabled( fl.size()!=0 );
    installAction->setEnabled( fl.size()!=0 );

    if (fl.size()==0)
    {
        fwList->insertItem( tr("No firewalls defined") );
        return;
    }

    for (list<FWObject*>::iterator m=fl.begin(); m!=fl.end(); m++)
        addFirewallToList( *m );

    fwList->setCurrentItem( 0 );
    openFirewall( 0 );
}

int  FWWindow::findFirewallInList(FWObject *f)
{
    vector<FWObject*>::iterator i;
    int n=0;
    for (i=firewalls.begin(); i!=firewalls.end(); i++,n++)
    {
        if ( (*i)->getId()==f->getId() ) return n;
    }
    return -1;
}

void FWWindow::addFirewallToList(FWObject *o)
{
    QString icn_filename =
        QString( Resources::global_res->getObjResourceStr(o,
                                                          "icon-tree").c_str() );

    int n=fwList->count();

    if (fwbdebug) qDebug("FWWindow::addFirewallToList %d %p %s",
                         n, o, o->getName().c_str() );

    fwList->insertItem( QPixmap::fromMimeSource( icn_filename ),
			QString::fromUtf8(o->getName().c_str()) );
    firewalls.push_back(o);

    fwList->setCurrentItem( n );
//    openFirewall( n );
}

void FWWindow::removeFirewallFromList(FWObject *o)
{
    if (fwbdebug) qDebug("FWWindow::removeFirewallFromList %p %s",
                         o, o->getName().c_str() );

    vector<FWObject*>::iterator i;
    int  n=0;
    for (i=firewalls.begin(); i!=firewalls.end(); i++,n++)
    {
        if ( (*i)->getId()==o->getId() )
        {
            fwList->removeItem(n);
            firewalls.erase( i );
            break;
        }
    }
}

void FWWindow::showFirewall(FWObject *obj)
{
    vector<FWObject*>::iterator i;
    int n=0;
    for (i=firewalls.begin(); i!=firewalls.end(); i++,n++)
    {
        if ( (*i)->getId()==obj->getId() )
        {
            fwList->setCurrentItem( n );
            openFirewall( n );
            return;
        }
    }
}

void FWWindow::ensureObjectVisible(FWReference *obj)
{
    FWObject *p=obj;
    while (p && Firewall::cast(p)==NULL ) p=p->getParent();
    if (p==NULL) return;  // something is broken

    if (p!=getVisibleFirewall()) showFirewall(p);

    p=obj;

    while (p && RuleSet::cast(p)==NULL ) p=p->getParent();
    if (p==NULL) return;  // something is broken

    RuleSetView *rsv = ruleSetViews[p];

    ruleSets->showPage(rsv);

    rsv->selectRE( obj );
}

void FWWindow::updateRuleSetView()
{
//    ruleSets->repaint();
    RuleSetView* rv=dynamic_cast<RuleSetView*>(ruleSets->currentPage());
    if (rv!=NULL) rv->updateAll();
}

void FWWindow::updateRuleOptions()
{
    RuleSetView* rv=dynamic_cast<RuleSetView*>(ruleSets->currentPage());
    if (rv!=NULL) rv->updateCurrentCell();
}

void FWWindow::updateFirewallName(FWObject *obj,const QString &oldName)
{
    QString icn_filename =
        Resources::global_res->getObjResourceStr(obj, "icon-tree").c_str();

    vector<FWObject*>::iterator i;
    int n = 0;
    for (i=firewalls.begin(); i!=firewalls.end(); i++,n++)
    {
        if ( (*i)->getId()==obj->getId())
        {
            fwList->changeItem(  QPixmap::fromMimeSource( icn_filename ),
                                 QString::fromUtf8(obj->getName().c_str()),
                                 n );
            return;
        }
    }
}

void FWWindow::deleteFirewall(FWObject *fw)
{
    if (fwbdebug) qDebug("FWWindow::deleteFirewall   - fw %s %s",
                         fw->getName().c_str(), fw->getId().c_str());

    removeFirewallFromList(fw);
#if 0
    map<int, FWObject*>::iterator i;
    for (i=firewalls.begin(); i!=firewalls.end(); i++)
    {
        if (fwbdebug) qDebug("FWWindow::deleteFirewall   - *i %d %s %s",
                             (*i).first,
                             (*i).second->getName().c_str(),
                             (*i).second->getId().c_str());

        if ( ((*i).second)->getId()==fw->getId())
        {
            fwList->clear();
            showFirewalls();
            break;
        }
    }
#endif
    if (visibleFirewall==fw)  visibleFirewall=NULL;
}

void FWWindow::reopenFirewall()
{
    int currentPage = ruleSets->currentPageIndex();

    clearFirewallTabs();
    ruleSetViews.clear();

    if (firewalls.size()==0) return;
    if (visibleFirewall==NULL) return;

    RuleSetView *rsv;

    Policy *pol=Policy::cast(visibleFirewall->getFirstByType(Policy::TYPENAME));
    ruleSets->addTab( rsv=new PolicyView(pol) , tr("Policy") );
    ruleSetViews[pol]=rsv;

    FWObjectTypedChildIterator j=visibleFirewall->findByType(Interface::TYPENAME);
    for ( ; j!=j.end(); ++j )
    {
        Interface       *intf = Interface::cast(*j);
        InterfacePolicy *ip   = InterfacePolicy::cast((*j)->getFirstByType(InterfacePolicy::TYPENAME));

        QString tabName;
        if ( !intf->getLabel().empty() )
            tabName=QString::fromUtf8(intf->getLabel().c_str());
        else
            tabName=QString::fromUtf8(intf->getName().c_str());
        ruleSets->addTab( rsv=new InterfacePolicyView(ip) , tabName );
        ruleSetViews[ip]=rsv;
    }

    NAT *nat  = NAT::cast(visibleFirewall->getFirstByType(NAT::TYPENAME));
    ruleSets->addTab( rsv=new NATView(nat) , tr("NAT") );
    ruleSetViews[nat]=rsv;

    ruleSets->setCurrentPage( currentPage );

    insertRuleAction->setEnabled( true );
    moveRuleAction->setEnabled( true );
    moveRuleUpAction->setEnabled( true );
    moveRuleDownAction->setEnabled( true );
    addRuleAfterCurrentAction->setEnabled( true );
    removeRuleAction->setEnabled( true );
    copyRuleAction->setEnabled( true );
    cutRuleAction->setEnabled( true );
    pasteRuleAboveAction->setEnabled( true );
    pasteRuleBelowAction->setEnabled( true );

    compileAction->setEnabled( true );
    installAction->setEnabled( true );

    ruleSets->show();
}

void FWWindow::openFirewall( int idx )
{
    FWObject *fw=firewalls[idx];
    visibleFirewall = fw;
    firewallName->setText( fw->getName().c_str() );
    reopenFirewall();
    om->openObject(fw);
}

void FWWindow::selectRules()
{
    om->unselect();

    insertRuleAction->setEnabled( true );
    moveRuleAction->setEnabled( true );
    moveRuleUpAction->setEnabled( true );
    moveRuleDownAction->setEnabled( true );
    addRuleAfterCurrentAction->setEnabled( true );
    removeRuleAction->setEnabled( true );
    copyRuleAction->setEnabled( true );
    cutRuleAction->setEnabled( true );
    pasteRuleAboveAction->setEnabled( true );
    pasteRuleBelowAction->setEnabled( true );

    compileAction->setEnabled( true );
    installAction->setEnabled( true );
}

void FWWindow::unselectRules()
{
    bool havePolicies = (ruleSets->count()!=0);
#if 0
/* commented this out so that when I hit "Edit" in the object's pop-down
   menu in a rule, ruleset wont lose focus when object editor is opened.
   If rule set loses focus, the object's background turns from "selected" color
   to white and user loses context (which object is shown in the object editor)
*/
    if (havePolicies)
    {
        RuleSetView* rv=dynamic_cast<RuleSetView*>(ruleSets->currentPage());
        if (rv) rv->unselect();
    }
#endif
    insertRuleAction->setEnabled( havePolicies );     // enabled if there are policies
    moveRuleAction->setEnabled( false );
    moveRuleUpAction->setEnabled( false );
    moveRuleDownAction->setEnabled( false );
    addRuleAfterCurrentAction->setEnabled( false );
    removeRuleAction->setEnabled( false );
    copyRuleAction->setEnabled( false );
    cutRuleAction->setEnabled( false );
    pasteRuleAboveAction->setEnabled( false );
    pasteRuleBelowAction->setEnabled( false );

    compileAction->setEnabled( havePolicies );
    installAction->setEnabled( havePolicies );

}

void FWWindow::editCopy()
{
    if (om->isSelected()) om->copyObj();
    else
        if (ruleSets->count()!=0)
            dynamic_cast<RuleSetView*>(ruleSets->currentPage())->copySelectedObject();
}

void FWWindow::editCut()
{
    if (om->isSelected()) om->cutObj();
    else
        if (ruleSets->count()!=0)
            dynamic_cast<RuleSetView*>(ruleSets->currentPage())->cutSelectedObject();
}

void FWWindow::editDelete()
{
    if (om->isSelected()) om->deleteObj();
    else
        if (ruleSets->count()!=0)
            dynamic_cast<RuleSetView*>(ruleSets->currentPage())->deleteSelectedObject();
}

void FWWindow::editPaste()
{
    if (om->isSelected()) om->pasteObj();
    else
        if (ruleSets->count()!=0)
            dynamic_cast<RuleSetView*>(ruleSets->currentPage())->pasteObject();
}

void FWWindow::editPrefs()
{
    PrefsDialog pd(this);
    pd.exec();
}

void FWWindow::closeEvent( QCloseEvent * ev)
{
    if (saveIfModified())
    {
        if (!systemFile && rcs && rcs->isCheckedOut())
        {
            if (!checkin(true))
            {
                ev->ignore();
                return;
            }
            delete rcs;
        }
    } else
    {
        ev->ignore();
        return;
    }
    FWBMainWindow_q::closeEvent(ev);
//    emit closed();
}

void FWWindow::compile()
{
    doCompile();
}

int FWWindow::doCompile()
{
    if (visibleFirewall==NULL) return -1;

    fileSave();

    Firewall  *fw=Firewall::cast(visibleFirewall);
    FWOptions *fwopt=fw->getOptionsObject();

/*
 * I should be able to specify custom compiler for firewall with
 * no platform (e.g. for experiments)
 */
    string compiler=fwopt->getStr("compiler");
    if (compiler=="") 
    {
        compiler=Resources::platform_res[fw->getStr("platform")]->getCompiler();
    }

    if (compiler=="")
    {
        QMessageBox::warning(
            this,"Firewall Builder", 
            tr("Firewall platform is not specified in this object.\n\
Can't compile firewall policy."),
            "&Continue", QString::null,QString::null,
            0, 1 );
        return -1;
    }

/*
 * On Unix compilers are installed in the standard place and are
 * accessible via PATH. On Windows and Mac they get installed in
 * unpredictable directories and need to be found
 *
 * first, check if user specified an absolute path for the compiler,
 * then check  if compiler is registsred in preferences, and if not,
 * look for it in appRootDir and if it is not there, rely on PATH
 */
#if defined(Q_OS_WIN32) ||  defined(Q_OS_MACX)

    if ( ! QFile::exists( compiler ) )
    {
        QString cmppath = st->getStr( QString("Compilers/")+compiler );
        if (!cmppath.isEmpty()) compiler=cmppath.latin1();
        else
        {
            /* try to find compiler in appRootDir. */

            if ( QFile::exists( appRootDir + FS_SEPARATOR + compiler ) )
                compiler = appRootDir + FS_SEPARATOR + compiler;
        }
    }
#endif

#if 0
// if we use WDir for the "-d" option for compiler
    QString wdir;
    if (st->getWDir().isEmpty())
    {
        QString of = rcs->getFileName();
        wdir = of.left( of.findRev('/',-1) );
    } else
    {
        wdir=st->getWDir();
    }
#endif

    QString wdir = getFileDir( rcs->getFileName() );

    QStringList args;

    args.push_back(compiler.c_str());

    args += QStringList::split(" ", fwopt->getStr("cmdline").c_str() );

    args.push_back("-f");
    args.push_back(db()->getFileName().c_str());

    if (wdir!="")
    {
        args.push_back("-d");
        args.push_back(wdir);
    }

    QString ofname = fwopt->getStr("output_file").c_str();
    if (!ofname.isEmpty()) 
    {
        args.push_back("-o");
        args.push_back(ofname);
    }

#if defined(Q_OS_WIN32) || defined(Q_OS_MACX)
    args.push_back("-r");
    args.push_back(respath);
#endif

    args.push_back( fw->getName().c_str() );

    execDialog dlg(this, args );

    int res= dlg.run();

    return res;
}

void FWWindow::install()
{
    if (visibleFirewall==NULL) return;

    Firewall   *fw  =Firewall::cast(visibleFirewall);
    FWOptions  *fwopt=fw->getOptionsObject();

    Management *mgmt=fw->getManagementObject();
    assert(mgmt!=NULL);
    PolicyInstallScript *pis   = mgmt->getPolicyInstallScript();

/* we don't care about ssh settings if external installer is to be used */

    if ( (/* ! pis->isEnabled() || */ pis->getCommand()=="") &&
         (st->getSSHPath().isEmpty() || st->getSCPPath().isEmpty()) )
    {
        QMessageBox::critical(this, "Firewall Builder",
   tr("Policy installer uses Secure Shell to communicate with the firewall.\n"
      "Please configure directory path to the secure file copy and secure \n"
      "shell utilities installed on your machine using Preferences dialog"),
                              tr("&Continue") );
        return;
    }

/* need to save settings so that if the user just changed ssh/scp, the
 * wrapper will pick changes up
 */
    st->save();

    QString ofname = fwopt->getStr("output_file").c_str();
    if (ofname.isEmpty()) ofname = QString(fw->getName().c_str()) + ".fw";

    QString fwfname = getFileDir( rcs->getFileName() ) + "/" + ofname;
    if ( !QFile::exists(fwfname) )
    {
/* need to recompile */
        fileSave();
        if ( doCompile()!=0 ) return;
    } else
    {
/* check if the time of the last modification in the database is newer
 * than timestamp on the file that contains generated firewall policy 
 */
        struct stat st;

        if ( stat (fwfname.latin1() , &st)==0 )
        {
            time_t tt = st.st_mtime;
            time_t lm = db()->getTimeLastModified();
            if (tt<lm)
            {
/* need to recompile, but first ask the user */

                switch (QMessageBox::information(this, "Firewall Builder",
  tr("Some objects have been modified since\n"
     "you compiled the policy last time.\n"
     "Do you want to recompile it before you install ?"),
     tr("&Compile"), tr("&Install old copy"), tr("&Cancel"),
      0,       // Enter = button 0
      2 ) ) {   // Escape == button 2

                case 0:  // recompile
                    fileSave();
                    if ( doCompile()!=0 ) return;
                    break;
                case 1:  // install old one
                    break;
                case 2:  // cancel
                    return;
                }
            }
        }
    }


//    fileSave();

    if ( /*! pis->isEnabled() || */ pis->getCommand()=="" )
    {
        instConf cnf;

        cnf.user          = fwopt->getStr("admUser").c_str();
        cnf.maddr         = fwopt->getStr("altAddress").c_str();
        cnf.activationCmd = fwopt->getStr("activationCmd").c_str();

        cnf.fwobj     = fw;
        cnf.fwbfile   = db()->getFileName().c_str();
        cnf.conffile  = ofname;
        cnf.diff_file = QString(fw->getName().c_str())+".diff";
	cnf.wdir      = getFileDir( rcs->getFileName() );
        cnf.diff_pgm  = Resources::platform_res[fw->getStr("platform")]->
            getResourceStr("/FWBuilderResources/Target/diff").c_str();

/* set this in instDialog now

        QString s=fwopt->getStr("firewall_dir").c_str();
        if (s.isEmpty()) s="/etc/fw";
        cnf.fwdir     = s;
*/

        cnf.diff_pgm = QString(appRootDir.c_str()) + cnf.diff_pgm;

#ifdef _WIN32
	cnf.diff_pgm = cnf.diff_pgm + ".exe";
#endif

        instDialog *id = new instDialog( &cnf );

        id->show();
        id->exec();

        delete id;

    } else
    {
        string inst_script=pis->getCommand();

        QString wdir = getFileDir( rcs->getFileName() );

        QStringList args;

        args.push_back(inst_script.c_str());

        args += QStringList::split(" ", pis->getArguments().c_str() );

        args.push_back("-f");
        args.push_back(db()->getFileName().c_str());

        if (wdir!="")
        {
            args.push_back("-d");
            args.push_back(wdir);
        }

        args.push_back( QString("/%1/%2")
                        .arg(QString::fromUtf8(fw->getLibrary()->getName().c_str()))
                        .arg(fw->getPath(true).c_str() ) );

        execDialog dlg(this, args );

        dlg.run();
    }
}

/*
 * info styles go like this:
 * 0 - collapsed
 * 1 - opened
 * 2 - opened, more information
 *  etc.
 *
 */
void FWWindow::changeInfoStyle()
{
    switch (st->getInfoStyle())
    {
    case 0:
        st->setInfoStyle(1);
        oi->show();
        break;
    case 1:
        st->setInfoStyle(2);
        oi->show();
        break;
    case 2:
        st->setInfoStyle(0);
        oi->hide();
        break;
    }

/* TODO: now need to send signal to the dialog telling it to change
 * according to the style
 */

    om->info();
}

void FWWindow::insertRule()
{
    if (visibleFirewall==NULL || ruleSets->count()==0) return;
    dynamic_cast<RuleSetView*>(ruleSets->currentPage())->insertRule();
}

void FWWindow::addRuleAfterCurrent()
{
    if (visibleFirewall==NULL || ruleSets->count()==0) return;
    dynamic_cast<RuleSetView*>(ruleSets->currentPage())->addRuleAfterCurrent();
}

void FWWindow::removeRule()
{
    if (visibleFirewall==NULL || ruleSets->count()==0) return;
    dynamic_cast<RuleSetView*>(ruleSets->currentPage())->removeRule();
}

void FWWindow::moveRule()
{
    if (visibleFirewall==NULL || ruleSets->count()==0) return;
    dynamic_cast<RuleSetView*>(ruleSets->currentPage())->moveRule();
}

void FWWindow::moveRuleUp()
{
    if (visibleFirewall==NULL || ruleSets->count()==0) return;
    dynamic_cast<RuleSetView*>(ruleSets->currentPage())->moveRuleUp();
}

void FWWindow::moveRuleDown()
{
    if (visibleFirewall==NULL || ruleSets->count()==0) return;
    dynamic_cast<RuleSetView*>(ruleSets->currentPage())->moveRuleDown();
}

void FWWindow::copyRule()
{
    if (visibleFirewall==NULL || ruleSets->count()==0) return;
    dynamic_cast<RuleSetView*>(ruleSets->currentPage())->copyRule();
}

void FWWindow::cutRule()
{
    if (visibleFirewall==NULL || ruleSets->count()==0) return;
    dynamic_cast<RuleSetView*>(ruleSets->currentPage())->cutRule();
}

void FWWindow::pasteRuleAbove()
{
    if (visibleFirewall==NULL || ruleSets->count()==0) return;
    dynamic_cast<RuleSetView*>(ruleSets->currentPage())->pasteRuleAbove();
}

void FWWindow::pasteRuleBelow()
{
    if (visibleFirewall==NULL || ruleSets->count()==0) return;
    dynamic_cast<RuleSetView*>(ruleSets->currentPage())->pasteRuleBelow();
}



void FWWindow::search()
{
    fd->show();
}

void FWWindow::showEvent( QShowEvent *ev)
{
    QString val = st->getStr("Layout/MainWindowSplitter");
    if (!val.isEmpty())
    {
        int  w1 = val.section(',',0,0).toInt();
        int  w2 = val.section(',',1,1).toInt();

        QValueList<int> sl;
        sl.push_back(w1);
        sl.push_back(w2);
        mainSplitter->setSizes( sl );
    }
    st->restoreGeometry(this, QRect(100,100,750,600) );
    QMainWindow::showEvent(ev);
}

void FWWindow::hideEvent( QHideEvent *ev)
{
    st->saveGeometry(this);
    QValueList<int> sl = mainSplitter->sizes();
    QString arg = QString("%1,%2").arg(sl[0]).arg(sl[1]);
    st->setStr("Layout/MainWindowSplitter", arg );
    QMainWindow::hideEvent(ev);
}
    
void FWWindow::newObject()
{
    om->newObject();
}

void FWWindow::setupAutoSave()
{
    if ( st->getBool("Environment/autoSaveFile") &&
         rcs!=NULL && rcs->getFileName()!="")
    {
        int p = st->getInt("Environment/autoSaveFilePeriod");
        autosaveTimer->start( p*1000*60 );
        connect( autosaveTimer, SIGNAL(timeout()), this, SLOT(fileSave()) );
    } else
        autosaveTimer->stop();
}


