/*
 *      Copyright (C) 1997-1999 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 *     This program is a small utility to manipulate certain fields of
 *     the volume table of a floppy tape cartridge.  To be used with
 *     the QIC-40/80/3010/3020 floppy-tape driver "ftape" for Linux.
 */

char src[] = "$RCSfile: libftvt.c,v $";
char rev[] = "$Revision: 1.6 $";
char dat[] = "$Date: 1999/03/04 09:29:03 $";

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <getopt.h>
#include <ctype.h>
#include <stdarg.h>
#include <time.h>
#include <libintl.h>
#define _(String) gettext (String)

#include <linux/ftape.h>
#include <linux/ftape-vendors.h>
#include <linux/zftape.h>
#include <linux/ftape-header-segment.h>

#include "ftvt.h"
#include "libftvt-version.h"

extern void *xmalloc(size_t size);
extern void *xrealloc(void *old, size_t size);

/* "Data section". This variables are shared between the library
 * functions. Essentially, this is the contents of the ftvt_ctrl
 * structure
 */
static int   verbose;
static int   tagged;
static int   fmt_code;
static int   first_seg;
static int   last_seg;
static int   max_volumes;
static const char *errstr;
static const char *drivetype;

/* Static functions, not available to the calling program
 */
static void ftvt_error(const char *fmt, ...)
	__attribute__ ((format (printf, 1, 2)));
static void ftvt_perror(const char *msg);
static int ftvt_read_header_segment(int tape_fd,
									int *first_data_segment,
									int *last_data_segment);
static char *ftvt_gets(FILE *fp, char *buffer, int *len);


/* real code starts here
 */

/* In order to reduce the noise reduced by this library, but
 * nevertheless give access to the error messages produced by the
 * library, we only print them out when verbose == 1, but always save
 * a pointer to the most recent message in "errstr". The contents of
 * the buffer is only valid until the nect library call happens.
 */
static void ftvt_error(const char *fmt, ...)
{
	va_list args;
	static char buffer[1024];

	va_start(args, fmt);
	vsnprintf(buffer, 1024, fmt, args);
	va_end(args);

	if (verbose) {
		fprintf(stderr, buffer);
	}
	errstr = buffer;
}

/* a version of "perror()" which uses our ftvt_error() function
 */
static void ftvt_perror(const char *msg)
{
	char buffer[1024];

	strncpy(buffer, msg, 1024-6);
	strcat(buffer, ": %s\n");

	ftvt_error(msg, strerror(errno));
}

/* Manipulate certain parameters. Only FTVT_VERBOSE and FTVT_TAGGED
 * are defined for "flags".
 *
 * FTVT_VERBOSE: Be verbose. If not set, no error messages will be
 *               printed to stderr, and no status messages. One
 *               probably should be a little bit more selective and
 *               allow several levels of verbosity.
 *
 * FTVT_TAGGED: ftvt_print() uses a tagged output format, i.e. pairs
 *              of keywords and values. This really only affects the
 *              function ftvt_print().
 */
void ftvt_set_ctrl(unsigned long flags)
{
	if (flags & FTVT_VERBOSE) {
		verbose = 1;
	} else {
		verbose = 0;
	}
	if (flags & FTVT_TAGGED) {
		tagged = 1;
	} else {
		tagged = 0;
	}
}

/* Query some information about the library and the tape
 * cartridge. Normally not needed, I think.
 */
const ftvt_ctrl *ftvt_get_ctrl(void)
{
	static ftvt_ctrl ctrl;

	memset(&ctrl, 0, sizeof(ctrl));
	if (verbose) {
		ctrl.flags |= FTVT_VERBOSE;
	}
	if (tagged) {
		ctrl.flags |= FTVT_TAGGED;
	}
	ctrl.version     = version_string;
	ctrl.errstr      = errstr;
	ctrl.drivetype   = drivetype;
	ctrl.fmt_code    = fmt_code;
	ctrl.first_seg   = first_seg;
	ctrl.last_seg    = last_seg;
	ctrl.max_volumes = max_volumes;
	return &ctrl;
}

/* Open the tape device, try to determine whether is is a floppy tape.
 *  
 * Returns the tape fd on success, -1 otherwise
 *
 * Side effects: "drivetype" is set to the vendor name of this tape
 *               drive
 */
int ftvt_open(const char *name, mode_t mode)
{
	int tape_fd;
	struct mtget mtget;
	static const vendor_struct vendors[] = QIC117_VENDORS;
	const vendor_struct *vendor;

	/* open the tape device
	 */	
	if ((tape_fd = open(name, mode)) == -1) {
		ftvt_perror(_("Error opening tape device"));
		return -1;
	}
	/* get its status
	 */
	if (ioctl(tape_fd, MTIOCGET, &mtget) == -1) {
		ftvt_perror(_("Error getting tape drive status"));
		(void)close(tape_fd);
		return -1;
	}
	if (GMT_DR_OPEN(mtget.mt_gstat)) {
		ftvt_error(_("Error: No tape cartridge present!\n"));
		(void)close(tape_fd);
		return -1;
	}
	if (GMT_WR_PROT(mtget.mt_gstat)) {
		ftvt_error(_("Warning: Write protected cartridge!\n"));
	}
	if (!GMT_ONLINE(mtget.mt_gstat)) {
		ftvt_error(_("Error: Tape drive is offline!\n"));
		(void)close(tape_fd);
		return -1;
	}
	if ((mtget.mt_type & MT_ISFTAPE_FLAG) == 0) {
		ftvt_error(_("Error: This is not a floppy tape drive!\n"));
		(void)close(tape_fd);
		return -1;
	}
	mtget.mt_type &= ~MT_ISFTAPE_FLAG; /* clear the flag bit */
	if (verbose > 0) {
		vendor = &vendors[0];
		while (vendor->vendor_id != UNKNOWN_VENDOR &&
			   vendor->vendor_id != mtget.mt_type) {
			vendor++;
		}
		if (verbose) {
			printf(_("Tape drive type: %s (0x%04lx)\n"),
				   vendor->name, mtget.mt_type);
		}
		drivetype = vendor->name;
	}
	return tape_fd;
}

/* Close the tape device.
 */
int ftvt_close(int tape_fd)
{
	const struct mtop rewind = { MTREW, 1 };
	int result = 0;

	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		ftvt_perror(_("Ioctl error rewinding tape"));
		result = -1;
	}
	if (close(tape_fd) == -1) {
		ftvt_perror(_("Error closing tape device"));
		result = -1;
	}
	return result;
}

/* Read the volume table. This function first reads the header segment
 * of the tape cartridge to determine the location of the volume table
 * segment, and then reads volume table segment.
 *
 * Arguments:
 *
 * int tape_fd  : The file descriptor previously return by ftvt_open()
 *
 * ftvt *volumes: Buffer to hold the volume table in. The size of the
 *                buffer must be FTVT_MAX_VOLUMES (i.e. 29*1024/128)
 *
 * u_int8_t     : The buffer to hold the contents of the volume table
 *                segment. It must be of size 29*1024.
 *
 * Return value : The number of volumes found in the volume table segment.
 *
 * Side effects : The internal variable "max_volumes" is set to the
 *                number of volumes which would fit into the volume
 *                table segment. Normally this should be FTVT_MAX_VOLUMES,
 *                but some cartridges use volume table segments which
 *                contain bad sectors.
 */
int ftvt_read(int tape_fd, ftvt *volumes, u_int8_t *buffer)
{
	const struct mtop rewind = { MTREW, 1 };
	struct mtftseg ft_seg;
	int vtbl_cnt; 
	int end_seg = 0;
	const char *ids[] = FTVT_IDS;

	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		ftvt_perror(_("Ioctl error rewinding tape"));
		return -1;
	}
	if ((fmt_code = ftvt_read_header_segment(tape_fd,
											 &first_seg, &last_seg)) == -1) {
		return -1;
	}
	if (verbose) {
		printf(_("Reading volume table segment ... "));
		fflush(stdout);
	}
	memset(volumes, 0, FTVT_MAX_VOLUMES * sizeof(ftvt));
	memset(&ft_seg, 0, sizeof(ft_seg));
	ft_seg.mt_data  = buffer;
	ft_seg.mt_segno = first_seg;
	ft_seg.mt_mode  = MT_FT_RD_SINGLE;
	if (ioctl(tape_fd, MTIOCRDFTSEG, &ft_seg) == -1) {
		ftvt_perror(_("Ioctl error reading volume table"));
		return -1;
	}
	if (ft_seg.mt_result < 0) {
		ftvt_error(_("Error reading volume table: %s\n"),
				   strerror(-ft_seg.mt_result));
		return -1;
	} 
	max_volumes = ft_seg.mt_result / FTVT_SIZE;
	if (ft_seg.mt_result != FT_SEGMENT_SIZE) {
		ftvt_error(_(
			"Warning: Short read() reading volume table: %d.\n"
			"Continuing, but you can use only %d volumes (instead of %d)\n"
			),
				ft_seg.mt_result, max_volumes, FT_SEGMENT_SIZE / FTVT_SIZE);
	}
	if (verbose) {
		printf(_("done.\n"));
	}
	vtbl_cnt = 0;
	if (fmt_code == fmt_big) {
		end_seg = first_seg;
	}
	while (memcmp(buffer, ids[0], 4) == 0 ||
		   memcmp(buffer, ids[1], 4) == 0 ||
		   memcmp(buffer, ids[2], 4) == 0 ||
		   memcmp(buffer, ids[3], 4) == 0) {

		memcpy(&volumes[vtbl_cnt].entry, buffer, FTVT_SIZE);
		if (memcmp(&volumes[vtbl_cnt].ftvt_sig, "VTBL", 4) == 0) {
			volumes[vtbl_cnt].fmt_code = fmt_code;
			if (fmt_code == fmt_big) {
				volumes[vtbl_cnt].start = end_seg + 1;
				volumes[vtbl_cnt].end   = (end_seg + 
										   volumes[vtbl_cnt].ftvt_space);
				end_seg = volumes[vtbl_cnt].end;
			} else {
				volumes[vtbl_cnt].start = volumes[vtbl_cnt].ftvt_start;
				volumes[vtbl_cnt].end   = volumes[vtbl_cnt].ftvt_end;
			}
			volumes[vtbl_cnt].fmt_code = fmt_code;
		}
		volumes[vtbl_cnt].num = vtbl_cnt;
		vtbl_cnt ++;
		buffer += FTVT_SIZE;
	}
	return vtbl_cnt;
}

/* Try to read the header segments. Return the format code of this
 * cartridge, and set *first_data_segment and *last_data_segment.
 */
static int ftvt_read_header_segment(int tape_fd,
									int *first_data_segment,
									int *last_data_segment)
{
	struct mtftseg ft_seg;
	int i;
	const unsigned long hseg_magic = FT_HSEG_MAGIC;
	u_int8_t hseg[FT_SEGMENT_SIZE];
	
	if (verbose) {
		printf(_("Reading header segment ... "));
		fflush(stdout);
	}
	for (i = 0; i < 64; i++) {
		memset(&ft_seg, 0, sizeof(ft_seg));
		ft_seg.mt_data  = hseg;
		ft_seg.mt_segno = i;
		ft_seg.mt_mode  = MT_FT_RD_SINGLE;
		if (ioctl(tape_fd, MTIOCRDFTSEG, &ft_seg) == -1) {
			ftvt_perror(_("Ioctl error reading header segment"));
			return -1;
		}
		if (ft_seg.mt_result == FT_SEGMENT_SIZE) {
			break;
		}
	}
	if (memcmp(&hseg_magic, hseg, 4)) {
		ftvt_error(_("Error: Bad magic number in header segment!\n"));
		return -1;
	}
	if (verbose) {
		printf(_("done.\n"));
	}
	if (hseg[FT_FMT_CODE] == fmt_big) {
		memcpy(first_data_segment, &hseg[FT_6_FRST_SEG], 4);
		memcpy(last_data_segment,  &hseg[FT_6_LAST_SEG], 4);
	} else {
		memcpy(first_data_segment, &hseg[FT_FRST_SEG], 2);
		memcpy(last_data_segment,  &hseg[FT_LAST_SEG], 2);
		*first_data_segment &= 0x0000ffff;
		*last_data_segment &= 0x0000ffff;
	}
	return (int)hseg[FT_FMT_CODE];
}

/* Print the volume table.
 *
 * Arguments:
 *
 * const ftvt *volumes: Pointer to the libraries volume table
 *                      representation
 *
 * int maxnum         : Number of entries in "volumes"
 *
 * Side effects: none
 */
void ftvt_print(const ftvt *volumes, int maxnum)
{
	int i;

	if (tagged) {
		printf("VTBL START %d %d\n", first_seg, last_seg);
		for (i = 0; i < maxnum; i++) {
			printf("ENTRY %d\n", i);
			ftvt_print_one_tagged(&volumes[i]);
			printf("ENTRY END\n");
		}
		printf("VTBL END\n");
		return;
	}
	printf("%3s %3s %*s %*s %*s %8s %8s\n",
		   "Nr", "Id", 14, "Label", 22, "Date", 15, "Start", "End", "Space");
	for (i = 0; i < 80; i++) {
		printf("-");
	}
	printf("\n");
	for (i = 0; i < maxnum; i++) {
		ftvt_print_one(&volumes[i]);
	}
}

/* Print one volume table entry.
 *
 * Arguments:
 *
 * const ftvt *volume: pointer to the volume table entry to print
 */
void ftvt_print_one(const ftvt *volume)
{
	char label[45];
	char sig[5];
	double usage;

	memcpy(label, volume->ftvt_desc, 44);
	label[22] = '\0';
	memcpy(sig, volume->ftvt_sig, 4);
	sig[4] = '\0';
	if (!strcmp(sig, "VTBL")) {
		usage = (double)(volume->end - volume->start + 1);
		usage = usage/(double)(last_seg - first_seg)*100.0;
		printf("%3d %4s \"%-*s\" %*s %8d %8d    %2.2f%%\n",
			   volume->num, sig, 22, label, 18,
			   ftvt_decode_date(volume->ftvt_date),
			   volume->start, volume->end, usage);
	} else {
		printf("%4d %4s", volume->num, sig);
	}
}

/* Print one volume table in tagged output format (i.e. keyword-value
 * pairs)
 *
 * Arguments:
 *
 * const ftvt *volume: pointer to the volume table entry to print
 */
void ftvt_print_one_tagged(const ftvt *volume)
{
	char sig[5];
	char label[45];
	int i;

	memcpy(sig, volume->ftvt_sig, 4);
	sig[4] = '\0';
	memcpy(label, volume->ftvt_desc, 44);
	label[44] = '\0';
	printf("SIGNATURE \"%s\"\n", sig);
	if (strcmp(sig, "VTBL") == 0) {
		printf("START %d\n", volume->start);
		printf("END %d\n", volume->end);
		printf("DESCRIPTION \"%s\"\n", label);
		printf("DATE \"%s\"\n", ftvt_decode_date(volume->ftvt_date));
		printf("FLAG_VENDOR_SPECIFIC %d\n", volume->entry.vendor_specific);
		printf("FLAG_MULTI_CARTRIDGE %d\n", volume->entry.multi_cartridge);
		printf("FLAG_NOT_VERIFIED %d\n", volume->entry.not_verified);
		printf("FLAG_REDIRECTION_INHIBIT %d\n",
			   volume->entry.inhibit_redirection);
		printf("FLAG_SEGMENT_SPANNING %d\n", volume->entry.segment_spanning);
		printf("FLAG_DIRECTORY_LAST %d\n", volume->entry.directory_last);
		printf("FLAG_RESERVED_6 %d\n", volume->entry.fl_reserved_6);
		printf("FLAG_RESERVED_7 %d\n", volume->entry.fl_reserved_7);
		printf("MULTI_CARTRIDGE_COUNT %d\n", volume->entry.m_no);
		printf("VENDOR_EXTENSION \"");
		for (i = 0; i < sizeof(volume->entry.ext)-1; i ++) {
			printf("0x%02x ", volume->entry.ext[i]);
		}
		printf("0x%02x\"\n", volume->entry.ext[i]);
		printf("PASSWORD \"");
		for (i = 0; i < sizeof(volume->entry.pwd)-1; i ++) {
			printf("0x%02x ", volume->entry.pwd[i]);
		}
		printf("0x%02x\"\n", volume->entry.pwd[i]);
		printf("DIRECTORY_SIZE %d\n", volume->entry.dir_size);
		printf("DATA_SIZE %Ld\n", volume->entry.data_size);
		printf("OS_VERSION 0x%04x\n", volume->entry.os_version);
		printf("SOURCE_DRIVE %.16s\n", volume->entry.source_drive);
		printf("DEVICE 0x%02x\n", volume->entry.device);
		printf("RESERVED_1 0x%02x\n", volume->entry.reserved_1);
		printf("COMPRESSION_FLAGS 0x%02x\n", volume->entry.cmpr);
		printf("FORMAT 0x%02x\n", volume->entry.format);
		printf("RESERVED_2 0x%02x\n", volume->entry.reserved_1);
		printf("RESERVED_3 0x%02x\n", volume->entry.reserved_1);
	}
}

/* A safe version of gets, which reallocs the buffer in case of a
 * buffer overrun. Internal library use only.
 */ 
static char *ftvt_gets(FILE *fp, char *buffer, int *len)
{
	char *p = buffer;
	int remaining = *len;
	int c;

	for (;;) {
		errno = 0;
		switch (c = getc(fp)) {
		case EOF: {
			*p = '\0';
			if (ferror(fp)) {
				ftvt_perror("getc()");
			}
			return NULL;
		}		
		case '\n': {
			*p = '\0';
			return buffer;				
		}
		default: {
			if (remaining <= 1) {
				buffer = xrealloc(buffer, 1024 + *len);
				p = buffer + *len - remaining;
				(*len) += 1024;
				remaining += 1024;
			}
			*(p++) = (char)c;
			break;
		}
		}
	}
}

/**************** parsing of tagged input data *******************************/

/* Internal use. Get a single value.
 *
 * tag : the keyword
 * conv: conversion characater (like d, u, L etc.) for the numeric value
 * dest: Location to store the value in.
 *
 */
#define GET_SINGLE_VALUE(tag, conv, dest)						\
{																\
	typeof(dest) value; /* cope with bitfields */				\
																\
	if (strncmp(#tag, buffer, sizeof(#tag)-1) == 0) {			\
		if (sscanf(buffer, #tag" %"#conv, &value) == 1) {		\
			(dest) = value;										\
		} else {												\
			ftvt_error(_("Corrupt volume input data: %s\n"),	\
					   buffer);									\
			return -1;											\
		}														\
		continue;												\
	}															\
}


/* Internal use only. Get a string value.
 *
 * tag:  the keyword.
 * dest: where to store the string in. May be NULL
 * max:  max. number of chars to copy.
 * action: C statement to execute.
 */
#define GET_STRING_VALUE(tag, dest, max, action)		\
{														\
	char *value;										\
														\
	if (strncmp(#tag, buffer, sizeof(#tag)-1) == 0) {	\
		value = buffer + sizeof(#tag);					\
		if (strchr(value, '\"')) {						\
			value = strchr(value, '\"') + 1;			\
		}												\
		if (strrchr(value, '\"')) {						\
			*strrchr(value, '\"') = '\0';				\
		}												\
		if (dest) memcpy((dest), value, max);			\
		action;											\
		continue;										\
	}													\
}

/* Parse a volume table fed into stdin of the calling process.
 * The stream must consist of certain keyword-value pairs.
 *
 * Arguments:
 *
 * ftvt *volumes: Pointer to the storage area for the input. Must
 *                consists of FTVT_MAX_VOLUMES entries. The function checks
 *                that no more than "max_volumes" entries are stored
 *                in volumes.
 *
 * Return value:  -1 on error, the number of volumes otherwise.
 */
int ftvt_parse_tagged(ftvt *volumes)
{
	char *buffer = xmalloc(1024);
	char *new;
	int len = 1024;
	int in_volume = 0;
	int num;
	ftvt *volume = NULL;
	int get_data = 0;
	int maxnum = 0;
	char dummy[6];

	while ((new = ftvt_gets(stdin, buffer, &len))) {
		buffer = new;
		if (*new == '#') {
			continue; /* comment */
		}
		if (sscanf(buffer, "VTBL %5s", dummy) == 1) {
			if (strcmp(dummy, "START") == 0) {
				get_data = 1; /* start of volume table */
				continue;
			} else if (strcmp(dummy, "END") == 0) {
				if (in_volume) {
					ftvt_error(_("Corrupt volume input data: %s\n"), buffer);
					return -1;
				}
				return maxnum;
			}
		}
		if (!get_data) {
			continue;
		}
		if (strncmp("ENTRY", buffer, sizeof("ENTRY")-1) == 0) {
			if (strstr(buffer, "END")) {
				if (!in_volume) {
					ftvt_error(_("Corrupt volume input data: %s\n"), buffer);
					return -1;
				}
				in_volume = 0;
			} else if (sscanf(buffer, "ENTRY %d", &num) == 1) {
				if (num == maxnum) {
					if (++(maxnum) > max_volumes) {
						ftvt_error(_("Too many volumes: %d (%d max)\n"), 
								maxnum, max_volumes);
						return -1;
					}
					volumes[num].fmt_code = fmt_code;
				} else if (num > maxnum) {
					ftvt_error(_("Corrupt volume input data: %s\n"), buffer);
					return -1;
				}
				in_volume = 1;
				volume = &volumes[num];
				volume->modified = 1;
				volume->num = num;
				memcpy(volume->ftvt_sig, "VTBL", 4);
			} else {
				ftvt_error(_("Corrupt volume input data: %s\n"), buffer);
				return -1;
			}
			continue;
		}
		if (!in_volume) { /* need the "ENTRY" tag */
			ftvt_error(_("Corrupt volume input data\n"));
			return -1;
		}
		if (strncmp("SIGNATURE", buffer, sizeof("SIGNATURE")-1) == 0) {
			if (!strstr(buffer, "VTBL")) {
				ftvt_error(_("Corrupt volume input data: %s\n"), buffer);
				return -1;
			}
			memcpy(volume->ftvt_sig, "VTBL", 4);
			continue;
		}
		GET_SINGLE_VALUE(START, i, volume->start);
		GET_SINGLE_VALUE(END, i, volume->end);
		GET_SINGLE_VALUE(FLAG_VENDOR_SPECIFIC, i,
						 volume->entry.vendor_specific);
		GET_SINGLE_VALUE(FLAG_MULTI_CARTRIDGE, i,
						 volume->entry.multi_cartridge);
		GET_SINGLE_VALUE(FLAG_NOT_VERIFIED, i,
						 volume->entry.not_verified);
		GET_SINGLE_VALUE(FLAG_REDIRECTION_INHIBIT, i,
						 volume->entry.inhibit_redirection);
		GET_SINGLE_VALUE(FLAG_SEGMENT_SPANNING, i,
						 volume->entry.segment_spanning);
		GET_SINGLE_VALUE(FLAG_DIRECTORY_LAST, i,
						 volume->entry.directory_last);
		GET_SINGLE_VALUE(FLAG_RESERVED_6, i,
						 volume->entry.fl_reserved_6);
		GET_SINGLE_VALUE(FLAG_RESERVED_7, i,
						 volume->entry.fl_reserved_7);
		GET_SINGLE_VALUE(MULTI_CARTRIDGE_COUNT, i,
						 volume->entry.m_no);
		GET_SINGLE_VALUE(DIRECTORY_SIZE, i,
						 volume->entry.dir_size);
		GET_SINGLE_VALUE(DEVICE, i,
						 volume->entry.device);
		GET_SINGLE_VALUE(RESERVED_1, i,
						 volume->entry.reserved_1);
		GET_SINGLE_VALUE(COMPRESSION_FLAGS, i,
						 volume->entry.cmpr);
		GET_SINGLE_VALUE(FORMAT, i,
						 volume->entry.format);
		GET_SINGLE_VALUE(RESERVED_2, i,
						 volume->entry.reserved_1);
		GET_SINGLE_VALUE(RESERVED_3, i,
						 volume->entry.reserved_1);
		GET_SINGLE_VALUE(DATA_SIZE, Lu,
						 volume->entry.data_size);
		GET_SINGLE_VALUE(OS_VERSION, i,
						 volume->entry.os_version);
		GET_STRING_VALUE(DESCRIPTION, volume->ftvt_desc, 44,
						 memset(volume->ftvt_desc + strlen(value), ' ',
								44 - strlen(value)));
		GET_STRING_VALUE(DATE, NULL, 1024,
						 { if (ftvt_set_date(volumes, maxnum, value, num)) {
							 return -1;
						 } });
		GET_STRING_VALUE(VENDOR_EXTENSION, NULL, 1024,
						 { char *p = value; int i; int val;
						 for (i = 0; i < sizeof(volume->entry.ext)-1; i ++) {
							 if (sscanf(p, "%i", &val) != 1) {
								 ftvt_error(
									 _("Corrupt volume input data: %s\n"),
									 buffer);
								 return -1;
							 }
							 volume->entry.ext[i] = val;
							 while (isspace(*p)) p++;
							 while (*p != '\0' && !isspace(*p)) p++;
						 } });
		GET_STRING_VALUE(PASSWORD, NULL, 1024,
						 { char *p = value; int i; int val;
						 for (i = 0; i < sizeof(volume->entry.pwd)-1; i ++) {
							 if (sscanf(p, "%i", &val) != 1) {
								 ftvt_error(
									 _("Corrupt volume input data: %s\n"),
									 buffer);
								 return -1;
							 }
							 volume->entry.pwd[i] = val;
							 while (isspace(*p)) p++;
							 while (*p != '\0' && !isspace(*p)) p++;
						 } });
		GET_STRING_VALUE(SOURCE_DRIVE, volume->entry.source_drive, 16,
						 /**/);
	}
	ftvt_error(_("Premature end of volume table\n"));
	return -1;
}

/* Decode the date stored in timestamp in the format described in the
 * various QIC-something standards. This is year 2000 proof, but will
 * overflow in year 2097 :-)
 */
char *ftvt_decode_date(u_int32_t timestamp)
{
	struct tm tapetm;
	time_t tapetime;
	static char date[18];

	memset(&tapetm, 0, sizeof(tapetm));
	
	tapetm.tm_year  = timestamp >> FT_YEAR_SHIFT;
	tapetm.tm_year += FT_YEAR_0 - 1900;
	timestamp      &= FT_TIME_MASK;

	tapetm.tm_sec   = timestamp % 60;
	timestamp      /= 60;
	tapetm.tm_min   = timestamp % 60;
	timestamp      /= 60;
	tapetm.tm_hour  = timestamp % 24;
	timestamp      /= 24;
	tapetm.tm_mday  = timestamp % 31 + 1;
	timestamp      /= 31;
	tapetm.tm_mon   = timestamp % 12;
	tapetm.tm_isdst = -1;
	if ((tapetime = mktime(&tapetm)) == (time_t)(-1)) {
		return _("invalid");
	} else {
		/* maybe we should use locale's time representation here.
		 */
		(void)strftime(date, 18, "%T %D", &tapetm);
		return date;
	}
}

/* Flush the volume table to tape. Note that we try to preserve as
 * much of the original volume table as possible. For this reason,
 * "buffer" must be a pointer to the area previously filled by
 * ftvt_read(). Only those entries are copied from "volumes" to
 * "buffer" which have been modified by the library. If no entry has
 * been modified, the volume table isn't writte back to tape UNLESS
 * "do_write" is set to "1"
 * 
 * Arguments:
 *
 * int tape_fd: the file descriptor previously obtained by ftvt_open()
 *
 * const ftvt *volumes: The libraries volume table representation.
 *
 * u_int8_t *buffer: The buffer previously filled by ftvt_read().
 *          This must be the same pointer as given to ftvt_read().
 *
 * int vtbl_cnt: The number of volumes in the volume table. Setting
 *          this to "0" and setting "do_write" to "1" will erase the
 *          volume table.
 *
 * int do_write: Flag. If set to "1", always write the volume table to
 *          tape even if no modified entries are found.
 *
 * Return value: -1 on error, 0 otherwise
 */
int ftvt_write(int tape_fd, const ftvt *volumes, u_int8_t *buffer,
			   int vtbl_cnt, int do_write)
{
	const struct mtop rewind = { MTREW, 1 };
	struct mtftseg ft_seg;
	int i;
	u_int8_t *p = buffer;

	for (i = 0; i < vtbl_cnt; i++) {
		if (volumes[i].modified) {
			ftvt_entry *vt = (ftvt_entry *)p;

			memcpy(p, &volumes[i].entry, FTVT_SIZE);
			if (volumes[i].fmt_code == fmt_big) {
				vt->size.scsi_segs = (volumes[i].end - volumes[i].start + 1);
			} else {
				vt->size.se.start  = volumes[i].start;
				vt->size.se.end    = volumes[i].end;
			}
			do_write ++;
		}
		p += FTVT_SIZE;
	}
	memset(p, 0, FTVT_SIZE * (max_volumes - vtbl_cnt));
	if (do_write) {
		if (verbose) {
			printf(_("Writing volume table segment ... "));
			fflush(stdout);
		}
		memset(&ft_seg, 0, sizeof(ft_seg));
		ft_seg.mt_data  = buffer;
		ft_seg.mt_segno = first_seg;
		ft_seg.mt_mode  = MT_FT_WR_SINGLE;
		if (ioctl(tape_fd, MTIOCWRFTSEG, &ft_seg) == -1) {
			ftvt_perror(_("Ioctl error writing volume table"));
			return -1;
		}
		if (ft_seg.mt_result != FT_SEGMENT_SIZE) {
			ftvt_error(_("Error: Short write() writing volume table: %d\n"),
					   ft_seg.mt_result);
			return -1;
		}
		if (verbose) {
			printf(_("done.\n"));
		}
	}
	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		ftvt_perror(_("Ioctl error rewinding tape"));
		return -1;
	}
	return 0;
}

/* Nearly a dummy function. Checks whether one additional volume entry
 * would fit into the volume table and set the format code of the
 * entry.  All other fields are uninitializes (maybe change this by
 * using memset() ...
 *
 * Arguments:
 *
 * ftvt *volumes: pointer to the libraries volume table representation.
 * 
 * in num_volumes: number of valid entries in "volumes", as returned
 *         by ftvt_read() or a previous call to ftvt_add()
 *
 * Return value: -1 on error, the new number of volumes otherwise.
 */
int ftvt_add_volume(ftvt *volumes, int num_volumes)
{
	if (++num_volumes > max_volumes) {
		ftvt_error(_("Too many volumes: %d (%d max)\n"), 
				num_volumes, max_volumes);
		return -1;
	}
	volumes[num_volumes - 1].fmt_code = fmt_code;
	return num_volumes;
}

/* Set the date field of the given volume table entry.
 * 
 * Arguments:
 *
 * ftvt *volumes: pointer to the libraries volume table representation.
 *
 * int maxnum: number of entries in "volumes" (as returned by
 *        ftvt_read() or ftvt_add()
 *
 * const char *date: a date string in the "%T %D" format. This seems
 *        to not be year 2000 proof, however, it will work until 2070.
 *
 * int vtbl_no: the number of the volume table entry to modify.
 *
 * Return value: -1 on error, 0 otherwise
 */
int ftvt_set_date(ftvt *volumes, int maxnum, const char *date, int vtbl_no)
{
	time_t raw_time;
	struct tm tapetime;
	struct tm *tapetimep = &tapetime;

	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		ftvt_error(_("Volume number too big or negative: %d\n"), 
				   vtbl_no);
		return -1;
	}
	if (date == NULL) {
		time(&raw_time);
		tapetimep = localtime(&raw_time);
	} else {
		(void)strptime(date, "%T %D", tapetimep);
	}
	volumes[vtbl_no].ftvt_date = FT_TIME_STAMP(tapetimep->tm_year + 1900, 
										  tapetimep->tm_mon, 
										  tapetimep->tm_mday-1,
										  tapetimep->tm_hour,
										  tapetimep->tm_min,
										  tapetimep->tm_sec);
	volumes[vtbl_no].modified = 1;
	return 0;
}

/* Set the label field of the given volume table entry.
 * 
 * Arguments:
 *
 * ftvt *volumes: pointer to the libraries volume table representation.
 *
 * int maxnum: number of entries in "volumes" (as returned by
 *        ftvt_read() or ftvt_add()
 *
 * const char *desc: The new '\0' terminated volume label. The
 *        function truncates the label to 44 bytes, or fills the label
 *        with spaces if too short (according to QIC standards)
 *
 * int vtbl_no: the number of the volume table entry to modify.
 *
 * Return value: -1 on error, 0 otherwise
 */
int ftvt_set_label(ftvt *volumes, int maxnum, const char *desc, int vtbl_no)
{
	int len;

	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		ftvt_error(_("Volume number too big or negative: %d\n"), vtbl_no);
		return -1;
	}
	strncpy(volumes[vtbl_no].ftvt_desc, desc, 44);
	if ((len = strlen(desc)) < 44) {
		memset(volumes[vtbl_no].ftvt_desc + len, ' ', 44 - len);
	}
	volumes[vtbl_no].modified = 1;
	return 0;
}

/* Set the segment bounds for a given volume table entry
 * 
 * Arguments:
 *
 * ftvt *volumes: pointer to the libraries volume table representation.
 *
 * int maxnum: number of entries in "volumes" (as returned by
 *        ftvt_read() or ftvt_add()
 *
 * int start, int end: start and end segments for this volume. The
 *        function performs some range checking against the dimensions
 *        of the inserted cartridge, but does NOT check for
 *        overlapping volumes.
 *
 * int vtbl_no: the number of the volume table entry to modify.
 *
 * Return value: -1 on error, 0 otherwise
 */
int ftvt_set_bounds(ftvt *volumes, int maxnum, int start, int end, int vtbl_no)
{
	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		ftvt_error(_("Volume number too big or negative: %d\n"), vtbl_no);
		return -1;
	}
	if (start > end) {
		ftvt_error(_("Start segment (%d) "
					 "should be less than end segment (%d)\n"),
				   start, end);
		return -1;
	}
	if (end > last_seg) {
		ftvt_error(_("End segment (%d) must be less than %d\n"),
				   end, last_seg);
	}
	volumes[vtbl_no].start    = start;
	volumes[vtbl_no].end      = end;
	volumes[vtbl_no].modified = 1;
	return 0;
}

/* Set the volume entry ID.
 * 
 * Arguments:
 *
 * ftvt *volumes: pointer to the libraries volume table representation.
 *
 * int maxnum: number of entries in "volumes" (as returned by
 *        ftvt_read() or ftvt_add()
 *
 * const char *id: a four byte identifier for the volume entry
 *        type. See QIC standards for explanation. Must be a '\0'
 *        terminated four byte string.
 *
 * int vtbl_no: the number of the volume table entry to modify.
 *
 * Return value: -1 on error, 0 otherwise
 */
int ftvt_set_id(ftvt *volumes, int maxnum, const char *id, int vtbl_no)
{
	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		ftvt_error(_("Volume number too big or negative: %d\n"), vtbl_no);
		return -1;
	}
	if (strlen(id) != 4) {
		ftvt_error(_("Volume ID must consist of exactly four characters\n"));
	}
	memcpy(volumes[vtbl_no].ftvt_sig, id, 4);
	volumes[vtbl_no].modified = 1;
	return 0;
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
