package freenet.node.states.request;

import freenet.*;
import freenet.node.*;
import freenet.node.rt.Routing;
import freenet.message.StoreData;
import freenet.support.Logger;
import java.util.Enumeration;

/** This class is basically just a data struct with all the variables
  * needed to process a request chain, and some utility methods.
  * @author tavin
  */
abstract class RequestState extends State {

    // these must be set in all constructors

    /** the hops to live remaining at this node */
    int hopsToLive;
    
    /** the key for this request chain */
    final Key searchKey;
    
    /** The Peer that routed the request to us (if applicable) */
    final Peer origPeer;
    
    /** the FeedbackToken used to communicate back to the initiator */
    final FeedbackToken ft;
    
    /** the RequestInitiator scheduled to start or restart the request */
    RequestInitiator ri;
    
    // things that happen during processing

    /** The last address this request was sent to */
    Peer lastPeer;

    /** The NodeReferences to route to for this key */
    Routing routes;

    
    int unreachable = 0,    // send failures
        restarted = 0,      // automatic restarts
        rejected = 0;       // no. of QueryRejected
        
    
    /** Used for starting a new chain.
      */
    RequestState(long id, int htl, Key key, Peer orig, FeedbackToken ft, RequestInitiator ri) {
        super(id);
        hopsToLive = htl;
        searchKey = key;
        origPeer = orig;
        this.ft = ft;
        this.ri = ri;
    }

    /** If one RequestState is derived from another, we must maintain
      * all state variables.
      */
    RequestState(RequestState ancestor) {
        super(ancestor.id());
        hopsToLive = ancestor.hopsToLive;
        searchKey = ancestor.searchKey;
        origPeer = ancestor.origPeer;
        this.ft = ancestor.ft;
        this.ri = ancestor.ri;
        lastPeer = ancestor.lastPeer;
        routes = ancestor.routes;
        unreachable = ancestor.unreachable;
        restarted = ancestor.restarted;
        rejected = ancestor.rejected;
    }


    /** @return  whether the message came from the original peer
      *          (also returns true if origPeer and mo.peerIdentity() are null) 
      */
    final boolean fromOrigPeer(Message mo) {
        return origPeer == null ? mo.peerIdentity() == null
                                : origPeer.equalsIdent(mo.peerIdentity());
    }
    
    /** @return  whether the message came from the last attempted peer
      */
    final boolean fromLastPeer(Message mo) {
        return lastPeer != null && lastPeer.equalsIdent(mo.peerIdentity());
    }
        
    /** Schedules the RequestInitiator on the ticker.
      * @param n       the node to schedule on
      * @param millis  in how many millisseconds to restart
      */
    final void scheduleRestart(Node n, long millis) {
        n.logger.log(this, "Rescheduling the restart to timeout in "
                           +millis+" millis on chain "+Long.toHexString(id),
                           Logger.DEBUG);
        if (ri != null) ri.cancel();
        ri = new RequestInitiator(this);
        n.schedule(millis, ri);
    }

    /** Cancel the RequestInitiator that has been scheduled
      * to restart this chain.
      */
    final void cancelRestart() {
        if (ri != null) {
            ri.cancel();
            ri = null;
        }
    }

    final void fail(Node n) {
        fail(n, null, null);
    }

    final void fail(Node n, String reason) {
        fail(n, reason, null);
    }

    /**
     * Send back a QueryRejected to the previous node. This eats 
     * any SendFailedException that might happen.
     * This also cancels the restart if possible.
     */
    final void fail(Node n, String reason, FieldSet otherFields) {
        cancelRestart();
        try {
            if (reason == null)
                reason = "(no reason given)";
            ft.queryRejected(n, hopsToLive, reason, otherFields,
                             unreachable, restarted, rejected);
        }
        catch (CommunicationException e) {
            n.logger.log(this, "I couldn't even fail right :-(",
                         e, Logger.DEBUG);
        }
    }
}


