package freenet.node.states.request;

import freenet.*;
import freenet.node.*;
import freenet.message.*;
import freenet.support.Logger;

/** After a request or insert has successfully completed transferring,
  * we will enter this state to wait for the StoreData (unless we were
  * the terminal node in the chain).
  * @author tavin
  */
public class AwaitingStoreData extends RequestState {

    private NoStoreData nosd;

    // he he .. and here we are again :)
    private TransferInsert xferIns;
    
    
    /** New AwaitingStoreData with a definite timeout.
      * @param nosd  the MO that governs the timeout
      *              -- should be already scheduled on the ticker
      */
    AwaitingStoreData(Pending ancestor, NoStoreData nosd) {
        super(ancestor);
        this.nosd = nosd;
    }

    AwaitingStoreData(TransferInsert ancestor, NoStoreData nosd) {
        this((Pending) ancestor, nosd);
        xferIns = ancestor;
    }

    public final String getName() {
        return "Awaiting StoreData";
    }
    
    // it could happen...
    public State receivedMessage(Node n, Accepted a) {
        return this;
    }

    // ignore.
    public State receivedMessage(Node n, QueryRestarted qr) {
        return this;
    }

    public State receivedMessage(Node n, QueryRejected qr) throws StateException {
        if (!fromLastPeer(qr)) {
            throw new BadStateException("QueryRejected from the wrong peer!");
        }
        // this is pretty much the same as if it timed out for a DataRequest,
        // but for an insert we need to go back to TransferInsertPending
        if (xferIns == null) {
            relayStoreData(n, null);
            return new RequestDone(this);
        }
        else {
            if (nosd != null) nosd.cancel();
            return xferIns.receivedMessage(n, qr);
        }
    }
    
    public State receivedMessage(Node n, NoStoreData nosd) throws BadStateException {
        if (this.nosd != nosd) {
            throw new BadStateException("Not my NoStoreData: "+nosd);
        }
        relayStoreData(n, null);
        return new RequestDone(this);
    }
    
    public State receivedMessage(Node n, StoreData sd) throws BadStateException {
        if (!fromLastPeer(sd)) {
            throw new BadStateException("StoreData from the wrong peer!");
        }
        // yay!
        routes.routeSucceeded();
        // Update global network load estimate stats.
        n.loadStats.storeRequestRateForNode(sd.dataSource(), sd.requestsPerHour());
        relayStoreData(n, sd);

        return new RequestDone(this);
    }

    public final void lost(Node n) {
        Core.diagnostics.occurrenceCounting("lostAwaitingStoreData", 1);
        // just like timing out with the NoStoreData
        relayStoreData(n, null);
    }

    private void relayStoreData(Node n, StoreData sd) {
        if (nosd != null) {
            nosd.cancel();
        }
        NodeReference nr = (sd == null ? null : sd.dataSource());
        long rate = -1;
        if (nr != null) {
            n.rt.reference(searchKey, nr);
            rate = sd.requestsPerHour();
        }
        try {
            ft.storeData(n, nr, rate);
        }
        catch (CommunicationException e) {
            n.logger.log(this, "Failed to relay StoreData to peer "+e.peer,
                         e, Logger.MINOR);
        }
    }
}


