package freenet.node.states.FCP;

import freenet.*;
import freenet.node.*;
import freenet.node.states.request.*;
import freenet.message.DataInsert;
import freenet.message.client.*;
import freenet.client.*;
import freenet.support.*;
import freenet.support.io.*;
import freenet.crypt.*;
import java.io.*;

/**
 * This is the state pertaining to InsertRequests in their
 * initial state.
 */

public class NewClientPut extends NewClientRequest {

    public NewClientPut(long id, ConnectionHandler source) {
        super(id, source);
    }
    
    /**
     * Returns the name.
     * @return "New ClientPut"
     */
    public String getName() {
        return "New ClientPut";
    }

    public State received(Node n, MessageObject mo) throws StateException {
        if (!(mo instanceof ClientPut)) {
            throw new BadStateException("expecting ClientPut");
        }

        n.diagnostics.occurrenceCounting("inboundClientRequests", 1);

        ClientPut cpmo = (ClientPut) mo;
            
        long dlen = cpmo.getDataLength();
        long mlen = cpmo.getMetadataLength();

        State s;
        DataInsert dim;
        
        Bucket data = null;
        try {
            // first, fully read the data from the client
            data = n.bf.makeBucket(dlen);
            OutputStream out = data.getOutputStream();
            try { copyFully(cpmo.getDataStream(), out, dlen); }
            finally { out.close(); }

            // initialize the ClientKey
            FreenetURI uri = cpmo.getURI();
            ClientKey ckey = 
                AbstractClientKey.createFromInsertURI(n.randSource, uri);
            
            String cipher = cpmo.otherFields.get("Cipher");
            if (cipher != null) ckey.setCipher(cipher);
            
            // prepare the insert
            long tmpLen;
            tmpLen = ckey.getPaddedLength(dlen);
            tmpLen = ckey.getTotalLength(tmpLen, ckey.getPartSize(tmpLen));

            Bucket ctBucket = null;
            DiscontinueInputStream ctStream = null;
            
            try {
                ctBucket = n.bf.makeBucket(tmpLen);
                ctStream = new FreeBucketInputStream(ckey.encode(data, mlen, ctBucket),
                                                     n.bf, ctBucket);
            
                // set up the FeedbackToken
                FeedbackToken ft;
                if (ckey instanceof ClientSVK) {
                    ClientSVK csvk = (ClientSVK) ckey;
                    ft = new ClientPutToken( id, source, csvk.getURI().toString(),
                                             Base64.encode(csvk.getPrivateKey()),
                                             Base64.encode(csvk.getPublicKeyFingerPrint()) );
                }
                else ft = new ClientPutToken( id, source, ckey.getURI().toString() );

                // set up the DataInsert
                // and the InsertPending chain

                FieldSet root = new FieldSet();
                ckey.getStorables().addTo(root);
                dim = new DataInsert(id, root, ctStream, ckey.getTotalLength());
            
                RequestInitiator ri = new RequestInitiator(id);
                s = new InsertPending(id, cpmo.getHTL(), ckey.getKey(), null, ft, ri);
                s = s.received(n, ri);

                ctBucket = null;
                ctStream = null;
            }
            finally {
                if (ctStream != null)
                    ctStream.close();
                else if (ctBucket != null)
                    n.bf.freeBucket(ctBucket);
            }
        }
        catch (KeyException e) {
            n.logger.log(this, "ClientPut got KeyException, sending URIError",
                         e, Logger.MINOR);
            sendMessage(new URIError(id, e.getMessage()));
            return null;
        }
        catch (Exception e) {
            n.logger.log(this, "ClientPut failed", e, Logger.MINOR);
            sendMessage(new Failed(id, e.getMessage()));
            return null;
        }
        finally {
            try {
                if (data != null) n.bf.freeBucket(data);
            }
            catch (IOException e) {}
        }

        // send the DataInsert
        try {
            s = s.received(n, dim);
        }
        catch (BadStateException e) {
            // key collision, or something
            dim.drop(n);
        }
        return s;
    }
}


