package freenet.node.states.FCP;

import freenet.*;
import freenet.node.*;
import freenet.message.client.*;
import freenet.client.*;
import freenet.crypt.*;
import freenet.support.io.*;
import java.io.*;

/** Serves as a communication relay between the state chain and
  * the FCP client after a ClientGet has been initiated.
  * @author tavin
  */
public class ClientGetToken extends FCPFeedbackToken {

    private ClientKey ckey;
    
    ClientGetToken(long id, ConnectionHandler source, ClientKey ckey) {
        super(id, source);
        this.ckey = ckey;
    }

    public OutputStream dataFound(Node n, Storables sto, long ctLength)
                                            throws SendFailedException {                            
        waiting = false;

        Document doc;
        try {
            doc = ckey.decode(sto, ctLength);
        }
        catch (DataNotValidIOException e) {
            ClientMessage msg;
            if (e.getCode() == Document.DOC_BAD_KEY) {
                msg = new URIError(id,
                    "Data found, but failed to decrypt with the decryption key given in the URI.");
            }
            else if (e.getCode() == Document.DOC_UNKNOWN_CIPHER) {
                msg = new Failed(id,
                    "Data found, but encrypted with an unsupported cipher.");
            }
            else if (e.getCode() == Document.DOC_BAD_LENGTH) {
                msg = new Failed(id,
                    "Data found, but the length given in the document header " +
                    "is inconsistent with the actual length of the key data.");
            }
            else if (e.getCode() == Document.DOC_BAD_STORABLES) {
                msg = new Failed(id, "Data found, but the storables were corrupt.");
            }
            else {
                msg = new Failed(id, "Data found, but the document header was corrupt.");
            }
            sendMessage(msg);
            return null;
        }
        
        OutputStream out;
        try {
            out = doc.decipheringOutputStream(
                new DataChunkOutputStream(doc.length(), ckey.getPartSize())
            );
        }
        catch (IOException e) {
            sendMessage(new Failed(id, e.getMessage()));
            return null;
        }

        // send DataFound message
        FieldSet fs = new FieldSet();
        fs.put("DataLength", Long.toHexString(doc.length()));
        if (doc.metadataLength() > 0)
            fs.put("MetadataLength", Long.toHexString(doc.metadataLength()));
        sendMessage(new DataFound(id, fs));
        
        // CBStripOutputStream(DecipherOutputStream(DataChunkOutputStream))
        return new CBStripOutputStream(out, ckey.getPartSize(), ckey.getControlLength());
    }

    // these are useless w/r/to ClientGet ..
    public void insertReply(Node n, long millis) {}
    public void storeData(Node n, NodeReference nr, long rate) {}

    /** Takes the unencrypted document and writes it in chunks to
      * the FCP client, driven ultimately by the SendData state.
      */
    public class DataChunkOutputStream extends OutputStream {

        private long length, pos = 0;
        private byte[] buffer;

        private DataChunkOutputStream(long length, long chunkSize) {
            super();
            this.length = length;
            buffer      = new byte[(int) chunkSize];
        }

        public void write(int b) throws IOException {            
            // we'll allow the padding bytes to be thrown at us
            // .. just do nothing once pos >= length
            if (pos < length) {
                buffer[(int) (pos++ % buffer.length)] = (byte) (b & 0xFF);
                if (pos % buffer.length == 0)
                    sendChunk(buffer.length);
                else if (pos == length)
                    sendChunk((int) (pos % buffer.length));
            }
        }

        public void write(byte[] buf, int off, int len) throws IOException {
            while (len > 0) {
                // we'll allow the padding bytes to be thrown at us
                // .. just do nothing once pos >= length
                if (pos == length) return;
                int n = (int) Math.min(buffer.length - pos % buffer.length, length - pos);
                if (n > len) n = len;
                System.arraycopy(buf, off, buffer, (int) (pos % buffer.length), n);
                pos += n;
                off += n;
                len -= n;
                if (pos % buffer.length == 0)
                    sendChunk(buffer.length);
                else if (pos == length)
                    sendChunk((int) (pos % buffer.length));
            }
        }

        private void sendChunk(int chunkSize) throws IOException {
            OutputStream out;
            try {
                out = sendMessage(
                    new DataChunk(id, chunkSize, pos == length)
                );
            }
            catch (SendFailedException e) {
                throw new IOException(e.getMessage());
            }
            out.write(buffer, 0, chunkSize);
            out.flush();
        }
    }
    
}



