package freenet.fs.dir;

import freenet.fs.acct.AccountingProcess;
import freenet.fs.acct.sys.*;
import freenet.support.*;
import freenet.support.Comparable;
import freenet.support.BinaryTree.Node;
import java.io.*;
import java.util.Enumeration;

/**
 * A binary tree that maps FileNumber to 64-bit ticket ID.
 * @author tavin
 */
final class KeyMap extends AccountingTree {

    KeyMap(AccountingProcess proc, Cache cache) {
        super(proc, new KeyEntry.Marshal(), cache);
    }


    /**
     * @return  the ticket ID stored under this key
     */
    final long get(FileNumber fn) {
        Node n = treeSearch(new KeyEntry(fn));
        return n == null ? -1 : ((KeyEntry) n.getObject()).ticketID;
    }

    /**
     * Attempts to map the given FileNumber to the given ticket ID.
     * Fails in the event of key collision.
     * @return  the ticketID given, or the ticketID of the entry
     *          collided with
     */
    final long put(FileNumber fn, long ticketID) {
        KeyEntry ke = new KeyEntry(fn, ticketID);
        Node n = treeInsert(new AccountingTreeNode(ke), false);
        return n == null ? ticketID : ((KeyEntry) n.getObject()).ticketID;
    }

    /**
     * @return  the ticket ID that was stored under this key,
     *          or -1 if there wasn't one
     */
    final long remove(FileNumber fn) {
        Node n = treeRemove(new KeyEntry(fn));
        return n == null ? -1 : ((KeyEntry) n.getObject()).ticketID;
    }

    
    /**
     * Enumerates FileNumber keys from a starting point.
     * @param fn  the file number to start the enumeration at
     * @param inclusive  true to include the starting-point arg if it exists
     * @param ascending  true for lexicographically ascending order
     */
    final Walk keys(FileNumber fn, boolean inclusive, boolean ascending) {
        Walk w = treeWalk(new KeyEntry(fn), inclusive, ascending);
        return new KeyEntry.FileNumberWalk(w);
    }

    /**
     * Enumerates all FileNumber keys.
     * @param ascending  true for lexicographically ascending order
     */
    final Walk keys(boolean ascending) {
        Walk w = treeWalk(ascending);
        return new KeyEntry.FileNumberWalk(w);
    }



    private static final class KeyEntry implements Comparable {
    
        private static final class FileNumberWalk implements Walk {
            private final Walk walk;
            FileNumberWalk(Walk walk) {
                this.walk = walk;
            }
            public final Object getNext() {
                Node n = (Node) walk.getNext();
                return n == null ? null : ((KeyEntry) n.getObject()).fn;
            }
        }
    
        private static final class Marshal implements AccountingTreeMarshal {
            public final Comparable readEntry(DataInput din, int len)
                                                throws IOException {
                int dirID = din.readUnsignedShort();
                byte[] key = new byte[len - 10];
                din.readFully(key);
                long ticketID = din.readLong();
                return new KeyEntry(new FileNumber(dirID, key), ticketID);
            }
            public final int getEntryLength(Comparable entry) {
                return 10 + ((KeyEntry) entry).fn.key.length;
            }
            public final void writeEntry(Comparable entry, DataOutput out)
                                                        throws IOException {
                KeyEntry ke = (KeyEntry) entry;
                out.writeShort(ke.fn.dirID);
                out.write(ke.fn.key);
                out.writeLong(ke.ticketID);
            }
        }
        
        
        final FileNumber fn;
        final long ticketID;
        

        KeyEntry(FileNumber fn, long ticketID) {
            this.fn = fn;
            this.ticketID = ticketID;
        }

        KeyEntry(FileNumber fn) {
            this(fn, -1);
        }

        public final int compareTo(Object o) {
            return compareTo((KeyEntry) o);
        }

        public final int compareTo(KeyEntry ke) {
            return fn.compareTo(ke.fn);
        }
    }
}


