package freenet.fs.dir;

import freenet.support.*;
import freenet.support.Comparable;
import freenet.support.Fields.ByteArrayComparator;
import java.util.Enumeration;

/**
 * This is basically an abstraction of a file name.
 * @author tavin
 */
public final class FileNumber implements Comparable {

    public static final FileNumber NIL = new FileNumber(new byte[0]);
    
    final int dirID;
    final byte[] key;

    final int hashCode;


    FileNumber(int dirID, FileNumber fn) {
        this.dirID = dirID;
        this.key = fn.key;
        hashCode = fn.hashCode;
    }
    
    FileNumber(int dirID, byte[] key) {
        this.dirID = dirID;
        this.key = key;
        hashCode = Fields.hashCode(key);
    }

    FileNumber(int dirID) {
        this(dirID, NIL);
    }

    /**
     * Wrap a byte array as a FileNumber key.
     */
    public FileNumber(byte[] key) {
        this(0, key);
    }
    
    /**
     * Wrap a string as a FileNumber key.
     */
    public FileNumber(String key) {
        this(0, key.getBytes());
    }

    /**
     * @return  String representation for log messages, etc.
     */
    public final String toString() {
        return "0x" + Integer.toHexString(dirID) + " : " + Fields.bytesToHex(key);
    }

    /**
     * @return  this key as a string
     */
    public final String getString() {
        return new String(key);
    }

    /**
     * @return  this key as a byte array
     */
    public final byte[] getByteArray() {
        return key;
    }


    public final int compareTo(Object o) {
        return compareTo((FileNumber) o);
    }

    public final int compareTo(FileNumber fn) {
        return dirID == fn.dirID ? ByteArrayComparator.compare(key, fn.key)
                                 : (dirID > fn.dirID ? 1 : -1);
    }

    
    public final boolean equals(Object o) {
        return o instanceof FileNumber && equals((FileNumber) o);
    }

    public final boolean equals(FileNumber fn) {
        return dirID == fn.dirID && Fields.byteArrayEqual(key, fn.key);
    }

    public final int hashCode() {
        return hashCode;
    }
    

    static final Enumeration filter(FilePattern pat, Enumeration enum) {
        return new WalkEnumeration(filter(pat, new EnumerationWalk(enum)));
    }
    
    static final Walk filter(FilePattern pat, Walk w) {
        return new FilterWalk(pat, w);
    }
    
    
    
    private static final class FilterWalk implements Walk {

        private final FilePattern pat;
        private final Walk walk;

        private boolean done = false;
        
        FilterWalk(FilePattern pat, Walk walk) {
            this.pat = pat;
            this.walk = walk;
        }

        public final Object getNext() {
            if (!done) {        
                FileNumber fn;
                while (null != (fn = (FileNumber) walk.getNext())) {
                    if (pat.matches(fn))
                        return fn;
                    else if (pat.isLimitedBy(fn))
                        break;
                }
                done = true;
            }
            return null;
        }
    }
}


