package freenet.diagnostics;
import freenet.support.*;
import freenet.FieldSet;
import java.lang.reflect.Constructor;
import java.util.Enumeration;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Date;
import java.text.DateFormat;
import java.text.FieldPosition;
import java.text.ParsePosition;
import java.io.DataOutputStream;
import java.io.DataInputStream;
import java.io.IOException;
/**
 * Superclass of different types of variables.
 *
 * @author oskar
 */

abstract class RandomVar {

    private static Hashtable types;

    public static void registerType(String name, Class type) 
        throws NoSuchMethodException {

        Constructor c = 
            type.getConstructor(new Class[] { java.lang.String.class,
                                              Integer.TYPE });
        types.put(name, c);
    }
    
    protected String name, comment;
    
    protected int period;
    
    protected EventList[] periodAggs;
    
    protected RandomVar(StandardDiagnostics sd, String name, int period,
                        String comment) {
        this.name = name;
        this.comment = comment;
        this.period = period;
        Vector v = new Vector();
        v.addElement(sd.newList(name));
        for (int i = 0 ; ; i++) {
            EventList el = sd.getList(name, period + i);
            if (el == null)
                break;
            else 
                v.addElement(el);
        }
        periodAggs = new EventList[v.size()];
        v.copyInto(periodAggs);
        //this.periodAggs = (EventList[]) v.toArray(new EventList[0]);
    }

    /**
     * Returns the type of the Var of this object.
     */
    public abstract String getType();
    /**
     * Returns the name given to this var.
     */
    public String getName() {
        return name;
    }

    public String getComment() {
        return comment;
    }

    public int aggregationPeriod() {
        return period;
    }

    public int aggregations() {
        return periodAggs.length - 1;
    }
    

    /**
     * Returns the events over a given period, or the latest occurrences
     * if eventPeriod < 0.
     */
    EventList getEvents(int eventPeriod) {
        int offset;
        if (eventPeriod < 0)
            offset = 0;
        else {
            offset = eventPeriod - period + 1;
            if (offset == 0) // zero offset means occurrences below, but here it
                // would mean that formatPeriod was one less than period
                offset = -1;
        }
        return (offset < 0 || offset >= periodAggs.length ? 
                null :
                periodAggs[offset]);
    }
    
    protected void add(VarEvent ve, long lastTime) {
        addTo(0, ve, lastTime, 
              ve.time() - StandardDiagnostics.periods[period]);
    }
  
    protected synchronized void addTo(int i, VarEvent ve, long lastTime, 
                                      long exptime) {
        //        System.out.println("addTo:" + i + " varevent: " + ve 
        //                   + "lists= " + periodAggs.length);
        periodAggs[i].open(this);

        // iterate backwards in time while future (!) events have occured.
        // this is 0(n), but obviously should only happen if people
        // reset their clock... 
        VarEvent ive;

        long etime = ve.time();
        Enumeration e = periodAggs[i].reverseElements();
        while (true) {
            if (!e.hasMoreElements()) {
                periodAggs[i].addFirst(ve);
                break;
            }
            ive = (VarEvent) e.nextElement();
            if (ive.time() < etime) {
                periodAggs[i].insertNext(ive, ve);
                break;
            } 
        }
        
        // remove entries that are older than one period, and previous
        // to last aggregation.
      
        for ( long time = periodAggs[i].getFirst().time() ; 
              time < exptime && time < lastTime ;
              time = periodAggs[i].getFirst().time()) {
            periodAggs[i].removeFirst();
        }

        periodAggs[i].close();
    }

    /**
     * Signals the end of a period of time.
     * @param type  The type of period that ends, from MINUTE, HOUR,...
     *              above. Note that when many roll over they are called
     *              one at the time in order, so at the end of a DECADE, 
     *              this is called for MINUTE, then HOUR, then DAY etc.
     * @param plength  The exact length of the period that passed, this
     *                 can't be determined from type because months and
     *                 years shift (who thought of that!!)
     * @param time     The millisecond time of the end of this period. 
     */
    public synchronized void endOf(StandardDiagnostics sd, int type, 
                                   long plength, long time) {
        int offset = type - period;
        //System.err.println(type + " - " + period);
        if (offset < 0) {
            //System.out.println(this + " got endof, offset " + offset);
            // do nothing
            return;
        }  else {
            // eat older events..
            long start = time - plength;
            periodAggs[offset].open(this);
            while(periodAggs[offset].getFirst() != null && 
                  periodAggs[offset].getFirst().time() <= start) {
                periodAggs[offset].removeFirst();
            }
            if (offset + 2 > periodAggs.length) {
                // first time we encounter this period, add a list
                EventList[] tmp = new EventList[offset + 2];
                System.arraycopy(periodAggs, 0, tmp, 0, periodAggs.length);
                tmp[offset + 1] = sd.newList(name, type);
                periodAggs = tmp;
            }
            //System.out.println(this + " about to call addTo. Offset = " + offset);
            addTo(offset + 1, aggregate(periodAggs[offset], time),
                  time, time - StandardDiagnostics.getPeriod(type + 1));
            periodAggs[offset].close();
        }
    }

    public synchronized double getValue(int vperiod, int type,
                                        long time) {
        EventList events;
        int offset = vperiod - period; 
        if (offset < 0) {
            events = periodAggs[0];
        } else if (offset + 1 >= periodAggs.length) {
            events = periodAggs[periodAggs.length - 1]; // the biggest we have
        } else {
            events = periodAggs[offset + 1];
        }
        return aggregate(events, time).getValue(type);
    }

    public abstract String[] headers();
    
    public abstract VarEvent aggregate(EventList al, long time);

    public abstract VarEvent readEvent(DataInputStream in) throws IOException;

    public String toString() {
        return getType() + ':' + name;
    }

}



